/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.dynamic_config.api.service;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.terracotta.dynamic_config.api.model.Cluster;
import org.terracotta.dynamic_config.api.model.ClusterState;
import org.terracotta.dynamic_config.api.model.Node;
import org.terracotta.dynamic_config.api.model.Setting;
import org.terracotta.dynamic_config.api.model.Stripe;
import org.terracotta.dynamic_config.api.model.UID;
import org.terracotta.dynamic_config.api.model.Version;
import org.terracotta.dynamic_config.api.service.MalformedClusterException;

public class ClusterValidator {
    private static final char[] FORBIDDEN_CTRL_CHARS = new char[]{'\u0000', '\u0001', '\u0002', '\u0003', '\u0004', '\u0005', '\u0006', '\u0007', '\b', '\t', '\n', '\u000b', '\f', '\r', '\u000e', '\u000f', '\u0010', '\u0011', '\u0012', '\u0013', '\u0014', '\u0015', '\u0016', '\u0017', '\u0018', '\u0019', '\u001a', '\u001b', '\u001c', '\u001d', '\u001e', '\u001f'};
    private static final char[] FORBIDDEN_FILE_CHARS = new char[]{':', '/', '\\', '<', '>', '\"', '|', '*', '?'};
    private static final char[] FORBIDDEN_ENDING_CHARS = new char[]{' ', '.'};
    private static final String[] FORBIDDEN_NAMES_NO_EXT = new String[]{"CON", "PRN", "AUX", "NUL", "COM1", "COM2", "COM3", "COM4", "COM5", "COM6", "COM7", "COM8", "COM9", "LPT1", "LPT2", "LPT3", "LPT4", "LPT5", "LPT6", "LPT7", "LPT8", "LPT9"};
    private static final char[] FORBIDDEN_DC_CHARS = new char[]{' ', ',', ':', '=', '%', '{', '}'};
    private final Cluster cluster;

    public ClusterValidator(Cluster cluster) {
        this.cluster = cluster;
    }

    public void validate(ClusterState clusterState) throws MalformedClusterException {
        this.validate(clusterState, Version.CURRENT);
    }

    public void validate(ClusterState clusterState, Version version) throws MalformedClusterException {
        this.validateNodeNames();
        this.validateNames(clusterState);
        this.validateAddresses();
        this.validateBackupDirs();
        this.validateDataDirs();
        this.validateSecurity();
        this.validateFailoverSetting(clusterState);
        if (version.amongst(EnumSet.of(Version.V2))) {
            this.validateStripeNames();
            this.validateUIDs();
        }
    }

    private void validateNames(ClusterState clusterState) {
        Stream.concat(Stream.of(this.cluster), this.cluster.descendants()).peek(o -> {
            if (clusterState == ClusterState.ACTIVATED && o.getName() == null) {
                throw new MalformedClusterException("Missing " + o.getScope().toString().toLowerCase() + " name");
            }
        }).filter(o -> o.getName() != null).forEach(o -> ClusterValidator.validateName(o.getName(), o.getScope().toString().toLowerCase()));
    }

    public static void validateName(String name, String scope) {
        String noExt;
        if (name.isEmpty()) {
            throw new MalformedClusterException("Empty " + scope.toLowerCase() + " name");
        }
        Character invalid = IntStream.range(0, name.length()).mapToObj(name::charAt).filter(c -> Arrays.binarySearch(FORBIDDEN_CTRL_CHARS, c.charValue()) >= 0 || Arrays.binarySearch(FORBIDDEN_FILE_CHARS, c.charValue()) >= 0 || Arrays.binarySearch(FORBIDDEN_DC_CHARS, c.charValue()) >= 0).findFirst().orElse(null);
        if (invalid != null) {
            throw new MalformedClusterException("Invalid character in " + scope + " name: '" + invalid + "'");
        }
        char last = name.charAt(name.length() - 1);
        if (Arrays.binarySearch(FORBIDDEN_ENDING_CHARS, last) >= 0) {
            throw new MalformedClusterException("Invalid ending character in " + scope + " name: '" + last + "'");
        }
        String string = noExt = name.lastIndexOf(".") == -1 ? name : name.substring(0, name.lastIndexOf("."));
        if (Arrays.binarySearch(FORBIDDEN_NAMES_NO_EXT, noExt) >= 0) {
            throw new MalformedClusterException("Invalid name for " + scope + ": '" + noExt + "' is a reserved word");
        }
    }

    private void validateUIDs() {
        HashMap<UID, String> discovered = new HashMap<UID, String>();
        if (this.cluster.getUID() == null) {
            throw new MalformedClusterException("Missing UID on cluster");
        }
        discovered.put(this.cluster.getUID(), "cluster");
        for (Stripe stripe : this.cluster.getStripes()) {
            String label = "stripe: " + stripe.getName();
            if (stripe.getUID() == null) {
                throw new MalformedClusterException("Missing UID on " + label);
            }
            String prev = discovered.put(stripe.getUID(), label);
            if (prev != null) {
                throw new MalformedClusterException("Duplicate UID for " + label + ". UID: " + stripe.getUID() + " was used on " + prev);
            }
            for (Node node : stripe.getNodes()) {
                label = "node: " + node.getName() + " in stripe: " + stripe.getName();
                if (node.getUID() == null) {
                    throw new MalformedClusterException("Missing UID on " + label);
                }
                prev = discovered.put(node.getUID(), label);
                if (prev == null) continue;
                throw new MalformedClusterException("Duplicate UID for " + label + ". UID: " + node.getUID() + " was used on " + prev);
            }
        }
    }

    private void validateAddresses() {
        this.checkDuplicateInternalAddresses();
        this.checkPublicAddressContent();
        this.checkDuplicatePublicAddresses();
        this.checkAllOrNoPublicAddresses();
    }

    private void checkAllOrNoPublicAddresses() {
        List nodesWithNoPublicAddresses = this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).filter(node -> !node.getPublicHostPort().isPresent()).map(Node::getName).collect(Collectors.toList());
        if (!nodesWithNoPublicAddresses.isEmpty() && nodesWithNoPublicAddresses.size() != this.cluster.getNodeCount()) {
            throw new MalformedClusterException("Nodes with names: " + nodesWithNoPublicAddresses + " don't have public addresses defined, but other nodes in the cluster do. Mutative operations on public addresses must be done simultaneously on every node in the cluster");
        }
    }

    private void checkPublicAddressContent() {
        this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).filter(node -> node.getPublicHostname().isConfigured() && !node.getPublicPort().isConfigured() || !node.getPublicHostname().isConfigured() && node.getPublicPort().isConfigured()).findFirst().ifPresent(node -> {
            throw new MalformedClusterException("Public address: '" + node.getPublicHostname().orDefault() + ":" + node.getPublicPort().orDefault() + "' of node with name: " + node.getName() + " isn't well-formed. Public hostname and port need to be set (or unset) together");
        });
    }

    private void checkDuplicateInternalAddresses() {
        this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).collect(Collectors.groupingBy(Node::getInternalHostPort, Collectors.toList())).entrySet().stream().filter(e -> ((List)e.getValue()).size() > 1).findAny().ifPresent(entry -> {
            throw new MalformedClusterException("Nodes with names: " + ((List)entry.getValue()).stream().map(Node::getName).collect(Collectors.joining(", ")) + " have the same address: '" + entry.getKey() + "'");
        });
    }

    private void checkDuplicatePublicAddresses() {
        this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).collect(Collectors.groupingBy(Node::getPublicHostPort, Collectors.toList())).entrySet().stream().filter(e -> ((Optional)e.getKey()).isPresent() && ((List)e.getValue()).size() > 1).findAny().ifPresent(entry -> {
            throw new MalformedClusterException("Nodes with names: " + ((List)entry.getValue()).stream().map(Node::getName).collect(Collectors.joining(", ")) + " have the same public address: '" + ((Optional)entry.getKey()).get() + "'");
        });
    }

    private void validateFailoverSetting(ClusterState clusterState) {
        if (clusterState == ClusterState.ACTIVATED && !this.cluster.getFailoverPriority().isConfigured() && this.cluster.getNodeCount() > 1) {
            throw new MalformedClusterException((Object)((Object)Setting.FAILOVER_PRIORITY) + " setting is not configured");
        }
    }

    private void validateNodeNames() {
        this.cluster.getNodes().stream().filter(node -> node.getName() == null).findAny().ifPresent(nodeName -> {
            throw new MalformedClusterException("Found node without name");
        });
        this.cluster.getNodes().stream().map(Node::getName).filter(Objects::nonNull).collect(Collectors.groupingBy(Function.identity(), Collectors.counting())).entrySet().stream().filter(e -> (Long)e.getValue() > 1L).map(Map.Entry::getKey).findAny().ifPresent(nodeName -> {
            throw new MalformedClusterException("Found duplicate node name: " + nodeName);
        });
    }

    private void validateStripeNames() {
        this.cluster.getStripes().stream().filter(stripe -> stripe.getName() == null).findAny().ifPresent(stripeName -> {
            throw new MalformedClusterException("Found stripe without name");
        });
        this.cluster.getStripes().stream().map(Stripe::getName).filter(Objects::nonNull).collect(Collectors.groupingBy(Function.identity(), Collectors.counting())).entrySet().stream().filter(e -> (Long)e.getValue() > 1L).map(Map.Entry::getKey).findAny().ifPresent(stripeName -> {
            throw new MalformedClusterException("Found duplicate stripe name: " + stripeName);
        });
    }

    private void validateDataDirs() {
        Set uniqueDataDirNames = this.cluster.getNodes().stream().map(node -> node.getDataDirs().orDefault().keySet()).collect(Collectors.toSet());
        if (uniqueDataDirNames.size() > 1) {
            throw new MalformedClusterException("Data directory names need to match across the cluster, but found the following mismatches: " + uniqueDataDirNames + ". Mutative operations on data dirs must be done simultaneously on every node in the cluster");
        }
    }

    private void validateBackupDirs() {
        List nodesWithBackupDirs = this.cluster.getNodes().stream().filter(node -> node.getBackupDir().isConfigured()).map(Node::getName).collect(Collectors.toList());
        if (!nodesWithBackupDirs.isEmpty() && nodesWithBackupDirs.size() != this.cluster.getNodeCount()) {
            throw new MalformedClusterException("Nodes: " + nodesWithBackupDirs + " currently have (or will have) backup directories defined, while some nodes in the cluster do not (or will not). Within a cluster, all nodes must have a backup directory defined or no backup directory defined.");
        }
    }

    private void validateSecurity() {
        boolean securityDirIsConfigured = this.validateSecurityDirs();
        this.validateSecurityRequirements(securityDirIsConfigured);
        this.validateAuditLogDir(securityDirIsConfigured);
    }

    private boolean validateSecurityDirs() {
        List nodesWithSecurityRootDirs = this.cluster.getNodes().stream().filter(node -> node.getSecurityDir().isConfigured()).map(Node::getName).collect(Collectors.toList());
        int count = nodesWithSecurityRootDirs.size();
        if (count > 0 && count != this.cluster.getNodeCount()) {
            throw new MalformedClusterException("Nodes: " + nodesWithSecurityRootDirs + " currently have (or will have) security root directories defined, while some nodes in the cluster do not (or will not). Within a cluster, all nodes must have a security root directory defined or no security root directory defined.");
        }
        return count > 0;
    }

    private void validateSecurityRequirements(boolean securityDirIsConfigured) {
        boolean minimumRequired;
        boolean bl = minimumRequired = this.cluster.getSecurityAuthc().isConfigured() || this.cluster.getSecuritySslTls().orDefault() != false || this.cluster.getSecurityWhitelist().orDefault() != false;
        if (securityDirIsConfigured) {
            if (!minimumRequired) {
                throw new MalformedClusterException("When security root directories are configured across the cluster at least one of " + (Object)((Object)Setting.SECURITY_AUTHC) + ", " + (Object)((Object)Setting.SECURITY_SSL_TLS) + " or " + (Object)((Object)Setting.SECURITY_WHITELIST) + " must also be configured.");
            }
            if (this.cluster.getSecurityAuthc().is("certificate") && !this.cluster.getSecuritySslTls().orDefault().booleanValue()) {
                throw new MalformedClusterException("When " + (Object)((Object)Setting.SECURITY_AUTHC) + "=certificate " + (Object)((Object)Setting.SECURITY_SSL_TLS) + " must be configured.");
            }
        } else if (minimumRequired) {
            throw new MalformedClusterException("There are no (or will be no) security root directories configured across the cluster. But " + (Object)((Object)Setting.SECURITY_AUTHC) + ", " + (Object)((Object)Setting.SECURITY_SSL_TLS) + ", and/or " + (Object)((Object)Setting.SECURITY_WHITELIST) + " is (or will be) configured.  When no security root directories are configured all other security settings should also be unconfigured (unset).");
        }
    }

    private void validateAuditLogDir(boolean securityDirIsConfigured) {
        List nodesWithAuditLogDirs = this.cluster.getNodes().stream().filter(node -> node.getSecurityAuditLogDir().isConfigured()).map(Node::getName).collect(Collectors.toList());
        int count = nodesWithAuditLogDirs.size();
        if (securityDirIsConfigured) {
            if (count > 0 && count != this.cluster.getNodeCount()) {
                throw new MalformedClusterException("Nodes: " + nodesWithAuditLogDirs + " currently have (or will have) audit log directories defined, while some nodes in the cluster do not (or will not). Within a cluster, all nodes must have an audit log directory defined or no audit log directory defined.");
            }
        } else if (count > 0) {
            throw new MalformedClusterException("There are no (or will be no) security root directories configured across the cluster. But nodes: " + nodesWithAuditLogDirs + " currently have (or will have) audit log directories defined.  When no security root directories are configured " + (Object)((Object)Setting.SECURITY_AUDIT_LOG_DIR) + " should also be unconfigured (unset) for all nodes in the cluster.");
        }
    }

    static {
        Arrays.sort(FORBIDDEN_CTRL_CHARS);
        Arrays.sort(FORBIDDEN_FILE_CHARS);
        Arrays.sort(FORBIDDEN_DC_CHARS);
        Arrays.sort(FORBIDDEN_ENDING_CHARS);
        Arrays.sort(FORBIDDEN_NAMES_NO_EXT);
    }
}

