//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.security.openid;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.transport.HttpClientTransportOverHTTP;
import org.eclipse.jetty.io.ClientConnector;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.ajax.JSON;
import org.eclipse.jetty.util.annotation.Name;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Holds the configuration for an OpenID Connect service.
 * <p>
 * This uses the OpenID Provider URL with the path {@link #CONFIG_PATH} to discover
 * the required information about the OIDC service.
 */
public class OpenIdConfiguration extends ContainerLifeCycle
{
/**
 * <p>Builder for {@link OpenIdConfiguration}.</p>
 * <p>The only required parameters are {@link #issuer}, {@link #clientId} and {@link #clientSecret}.</p>
 * <p>The {@link #authorizationEndpoint}, {@link #tokenEndpoint} and {@link #endSessionEndpoint}  will be retrieved
 * from the OpenID metadata at .well-known/openid-configuration if they are not explicitly specified.</p>
 */
    public static class Builder
    {
        private String issuer;
        private String clientId;
        private String clientSecret;

        private String authorizationEndpoint;
        private String tokenEndpoint;
        private String endSessionEndpoint;
        private String authenticationMethod;
        private HttpClient httpClient;

        private boolean authenticateNewUsers = false;
        private boolean logoutWhenIdTokenIsExpired = false;
        private final List<String> scopes = new ArrayList<>();

        /**
         * Create a Builder for an OpenID Configuration.
         */
        public Builder()
        {
        }

        /**
         * Create a Builder for an OpenID Configuration.
         * @param issuer The URL of the OpenID provider.
         * @param clientId OAuth 2.0 Client Identifier valid at the OpenID provider.
         * @param clientSecret The client secret known only by the Client and the OpenID provider.
         */
        public Builder(@Name("issuer") String issuer, @Name("clientId") String clientId, @Name("clientSecret") String clientSecret)
        {
            this();
            issuer(issuer);
            clientId(clientId);
            clientSecret(clientSecret);
        }

        /**
         * @param issuer The URL of the OpenID provider.
         */
        public Builder issuer(String issuer)
        {
            this.issuer = issuer;
            return this;
        }

        /**
         * @param clientId OAuth 2.0 Client Identifier valid at the OpenID provider.
         */
        public Builder clientId(String clientId)
        {
            this.clientId = clientId;
            return this;
        }

        /**
         * @param clientSecret The client secret known only by the Client and the OpenID provider.
         */
        public Builder clientSecret(String clientSecret)
        {
            this.clientSecret = clientSecret;
            return this;
        }

        /**
         * @param authorizationEndpoint the URL of the OpenID provider's authorization endpoint if configured.
         */
        public Builder authorizationEndpoint(String authorizationEndpoint)
        {
            this.authorizationEndpoint = authorizationEndpoint;
            return this;
        }

        /**
         * @param tokenEndpoint the URL of the OpenID provider's token endpoint if configured.
         */
        public Builder tokenEndpoint(String tokenEndpoint)
        {
            this.tokenEndpoint = tokenEndpoint;
            return this;
        }

        /**
         * @param endSessionEndpoint the URL of the OpenID provider's end session endpoint if configured.
         */
        public Builder endSessionEndpoint(String endSessionEndpoint)
        {
            this.endSessionEndpoint = endSessionEndpoint;
            return this;
        }

        /**
         * @param authenticationMethod Authentication method to use with the Token Endpoint.
         */
        public Builder authenticationMethod(String authenticationMethod)
        {
            this.authenticationMethod = authenticationMethod;
            return this;
        }

        /**
         * @param httpClient The {@link HttpClient} instance to use.
         */
        public Builder httpClient(HttpClient httpClient)
        {
            this.httpClient = httpClient;
            return this;
        }

        /**
         * @param authenticateNewUsers Whether to authenticate new users.
         */
        public Builder authenticateNewUsers(boolean authenticateNewUsers)
        {
            this.authenticateNewUsers = authenticateNewUsers;
            return this;
        }

        /**
         * @param scopes The scopes to request.
         */
        public Builder scopes(String... scopes)
        {
            if (scopes != null)
                Collections.addAll(this.scopes, scopes);
            return this;
        }

        /**
         * @param logoutWhenIdTokenIsExpired Whether to logout when the ID token is expired.
         */
        public Builder logoutWhenIdTokenIsExpired(boolean logoutWhenIdTokenIsExpired)
        {
            this.logoutWhenIdTokenIsExpired = logoutWhenIdTokenIsExpired;
            return this;
        }

        /**
         * @return a new {@link OpenIdConfiguration} instance.
         */
        public OpenIdConfiguration build()
        {
            if (issuer == null)
                throw new IllegalArgumentException("Issuer was not configured");
            if (clientId == null)
                throw new IllegalArgumentException("clientId was not configured");
            if (clientSecret == null)
                throw new IllegalArgumentException("clientSecret was not configured");

            return new OpenIdConfiguration(issuer, clientId, clientSecret, authorizationEndpoint, tokenEndpoint,
                endSessionEndpoint, authenticationMethod, httpClient, authenticateNewUsers, logoutWhenIdTokenIsExpired, scopes);
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(OpenIdConfiguration.class);
    private static final String CONFIG_PATH = "/.well-known/openid-configuration";
    private static final String AUTHORIZATION_ENDPOINT = "authorization_endpoint";
    private static final String TOKEN_ENDPOINT = "token_endpoint";
    private static final String END_SESSION_ENDPOINT = "end_session_endpoint";
    private static final String ISSUER = "issuer";

    private final HttpClient httpClient;
    private final String issuer;
    private final String clientId;
    private final String clientSecret;
    private final List<String> scopes = new ArrayList<>();
    private final String authenticationMethod;
    private String authorizationEndpoint;
    private String tokenEndpoint;
    private String endSessionEndpoint;
    private boolean authenticateNewUsers;
    private boolean logoutWhenIdTokenIsExpired;

    /**
     * Create an OpenID configuration for a specific OIDC provider.
     * @param provider The URL of the OpenID provider.
     * @param clientId OAuth 2.0 Client Identifier valid at the Authorization Server.
     * @param clientSecret The client secret known only by the Client and the Authorization Server.
     * @deprecated Use {@link Builder} instead.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public OpenIdConfiguration(String provider, String clientId, String clientSecret)
    {
        this(provider, null, null, clientId, clientSecret, null);
    }

    /**
     * Create an OpenID configuration for a specific OIDC provider.
     * @param issuer The URL of the OpenID provider.
     * @param authorizationEndpoint the URL of the OpenID provider's authorization endpoint if configured.
     * @param tokenEndpoint the URL of the OpenID provider's token endpoint if configured.
     * @param clientId OAuth 2.0 Client Identifier valid at the Authorization Server.
     * @param clientSecret The client secret known only by the Client and the Authorization Server.
     * @param httpClient The {@link HttpClient} instance to use.
     * @deprecated Use {@link Builder} instead.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public OpenIdConfiguration(String issuer, String authorizationEndpoint, String tokenEndpoint,
                               String clientId, String clientSecret, HttpClient httpClient)
    {
        this(issuer, authorizationEndpoint, tokenEndpoint, clientId, clientSecret, "client_secret_post", httpClient);
    }

    /**
     * Create an OpenID configuration for a specific OIDC provider.
     * @param issuer The URL of the OpenID provider.
     * @param authorizationEndpoint the URL of the OpenID provider's authorization endpoint if configured.
     * @param tokenEndpoint the URL of the OpenID provider's token endpoint if configured.
     * @param clientId OAuth 2.0 Client Identifier valid at the Authorization Server.
     * @param clientSecret The client secret known only by the Client and the Authorization Server.
     * @param authenticationMethod Authentication method to use with the Token Endpoint.
     * @param httpClient The {@link HttpClient} instance to use.
     * @deprecated Use {@link Builder} instead.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public OpenIdConfiguration(@Name("issuer") String issuer,
                               @Name("authorizationEndpoint") String authorizationEndpoint,
                               @Name("tokenEndpoint") String tokenEndpoint,
                               @Name("clientId") String clientId,
                               @Name("clientSecret") String clientSecret,
                               @Name("authenticationMethod") String authenticationMethod,
                               @Name("httpClient") HttpClient httpClient)
    {
        this(issuer, authorizationEndpoint, tokenEndpoint, null, clientId, clientSecret, authenticationMethod, httpClient);
    }

    /**
     * Create an OpenID configuration for a specific OIDC provider.
     * @param issuer The URL of the OpenID provider.
     * @param authorizationEndpoint the URL of the OpenID provider's authorization endpoint if configured.
     * @param tokenEndpoint the URL of the OpenID provider's token endpoint if configured.
     * @param endSessionEndpoint the URL of the OpenID provider's end session endpoint if configured.
     * @param clientId OAuth 2.0 Client Identifier valid at the Authorization Server.
     * @param clientSecret The client secret known only by the Client and the Authorization Server.
     * @param authenticationMethod Authentication method to use with the Token Endpoint.
     * @param httpClient The {@link HttpClient} instance to use.
     * @deprecated Use {@link Builder} instead.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public OpenIdConfiguration(@Name("issuer") String issuer,
                               @Name("authorizationEndpoint") String authorizationEndpoint,
                               @Name("tokenEndpoint") String tokenEndpoint,
                               @Name("endSessionEndpoint") String endSessionEndpoint,
                               @Name("clientId") String clientId,
                               @Name("clientSecret") String clientSecret,
                               @Name("authenticationMethod") String authenticationMethod,
                               @Name("httpClient") HttpClient httpClient)
    {
        this(issuer, clientId, clientSecret, authorizationEndpoint, tokenEndpoint, endSessionEndpoint, authenticationMethod, httpClient, false, false, Collections.emptyList());
    }

    /**
     * Create an OpenID configuration for a specific OIDC provider.
     * @param issuer The URL of the OpenID provider.
     * @param clientId OAuth 2.0 Client Identifier valid at the Authorization Server.
     * @param clientSecret The client secret known only by the Client and the Authorization Server.
     * @param authorizationEndpoint the URL of the OpenID provider's authorization endpoint if configured.
     * @param tokenEndpoint the URL of the OpenID provider's token endpoint if configured.
     * @param endSessionEndpoint the URL of the OpdnID provider's end session endpoint if configured.
     * @param authenticationMethod Authentication method to use with the Token Endpoint.
     * @param httpClient The {@link HttpClient} instance to use.
     * @param authenticateNewUsers Whether to authenticate new users.
     * @param logoutWhenIdTokenIsExpired Whether to logout when the ID token is expired.
     * @param scopes The scopes to request.
     */
    private OpenIdConfiguration(String issuer,
                                String clientId,
                                String clientSecret,
                                String authorizationEndpoint,
                                String tokenEndpoint,
                                String endSessionEndpoint,
                                String authenticationMethod,
                                HttpClient httpClient,
                                boolean authenticateNewUsers,
                                boolean logoutWhenIdTokenIsExpired,
                                List<String> scopes)
    {
        this.issuer = issuer;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.authorizationEndpoint = authorizationEndpoint;
        this.endSessionEndpoint = endSessionEndpoint;
        this.tokenEndpoint = tokenEndpoint;
        this.httpClient = httpClient != null ? httpClient : newHttpClient();
        this.authenticationMethod = authenticationMethod == null ? "client_secret_post" : authenticationMethod;
        this.authenticateNewUsers = authenticateNewUsers;
        this.logoutWhenIdTokenIsExpired = logoutWhenIdTokenIsExpired;
        this.scopes.addAll(scopes);

        if (this.issuer == null)
            throw new IllegalArgumentException("Issuer was not configured");

        installBean(this.httpClient);
    }

    @Override
    protected void doStart() throws Exception
    {
        super.doStart();

        if (authorizationEndpoint == null || tokenEndpoint == null)
        {
            Map<String, Object> discoveryDocument = fetchOpenIdConnectMetadata();
            processMetadata(discoveryDocument);
        }
    }

    /**
     * Process the OpenID Connect metadata discovered by {@link #fetchOpenIdConnectMetadata()}.
     * By default, only the {@link #AUTHORIZATION_ENDPOINT} and {@link #TOKEN_ENDPOINT} claims are extracted.
     * @see <a href="https://openid.net/specs/openid-connect-discovery-1_0.html">OpenID Connect Discovery 1.0</a>
     * @throws IllegalStateException if a required field is not present in the metadata.
     */
    protected void processMetadata(Map<String, Object> discoveryDocument)
    {
        authorizationEndpoint = (String)discoveryDocument.get(AUTHORIZATION_ENDPOINT);
        if (authorizationEndpoint == null)
            throw new IllegalStateException(AUTHORIZATION_ENDPOINT);

        tokenEndpoint = (String)discoveryDocument.get(TOKEN_ENDPOINT);
        if (tokenEndpoint == null)
            throw new IllegalStateException(TOKEN_ENDPOINT);

        // End session endpoint is optional.
        if (endSessionEndpoint == null)
            endSessionEndpoint = (String)discoveryDocument.get(END_SESSION_ENDPOINT);

        // We are lenient and not throw here as some major OIDC providers do not conform to this.
        if (!Objects.equals(discoveryDocument.get(ISSUER), issuer))
            LOG.warn("The issuer in the metadata is not correct.");
    }

    /**
     * Obtain the JSON metadata from OpenID Connect Discovery Configuration Endpoint.
     * @return a set of Claims about the OpenID Provider's configuration in JSON format.
     * @throws IllegalStateException if metadata could not be fetched from the OP.
     */
    protected Map<String, Object> fetchOpenIdConnectMetadata()
    {
        String provider = issuer;
        if (provider.endsWith("/"))
            provider = provider.substring(0, provider.length() - 1);

        try
        {
            Map<String, Object> result;
            String responseBody = httpClient.GET(provider + CONFIG_PATH).getContentAsString();
            Object parsedResult = new JSON().fromJSON(responseBody);

            if (parsedResult instanceof Map)
            {
                Map<?, ?> rawResult = (Map<?, ?>)parsedResult;
                result = rawResult.entrySet().stream()
                    .filter(entry -> entry.getValue() != null)
                    .collect(Collectors.toMap(it -> it.getKey().toString(), Map.Entry::getValue));
                if (LOG.isDebugEnabled())
                    LOG.debug("discovery document {}", result);
                return result;
            }
            else
            {
                LOG.warn("OpenID provider did not return a proper JSON object response. Result was '{}'", responseBody);
                throw new IllegalStateException("Could not parse OpenID provider's malformed response");
            }
        }
        catch (Exception e)
        {
            throw new IllegalStateException("invalid identity provider " + provider, e);
        }
    }

    public HttpClient getHttpClient()
    {
        return httpClient;
    }

    public String getAuthorizationEndpoint()
    {
        return authorizationEndpoint;
    }

    public String getClientId()
    {
        return clientId;
    }

    public String getClientSecret()
    {
        return clientSecret;
    }

    public String getIssuer()
    {
        return issuer;
    }

    public String getTokenEndpoint()
    {
        return tokenEndpoint;
    }

    public String getEndSessionEndpoint()
    {
        return endSessionEndpoint;
    }

    public String getAuthenticationMethod()
    {
        return authenticationMethod;
    }

    public List<String> getScopes()
    {
        return Collections.unmodifiableList(scopes);
    }

    public boolean isAuthenticateNewUsers()
    {
        return authenticateNewUsers;
    }

    public boolean isLogoutWhenIdTokenIsExpired()
    {
        return logoutWhenIdTokenIsExpired;
    }

    /**
     * @deprecated use {@link #getAuthorizationEndpoint()} instead.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public String getAuthEndpoint()
    {
        return authorizationEndpoint;
    }

    /**
     * @deprecated use {@link Builder} to configure the OpenID Configuration.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public void setAuthenticateNewUsers(boolean authenticateNewUsers)
    {
        this.authenticateNewUsers = authenticateNewUsers;
    }

    /**
     * @deprecated use {@link Builder} to configure the OpenID Configuration.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public void addScopes(String... scopes)
    {
        if (scopes != null)
            Collections.addAll(this.scopes, scopes);
    }

    /**
     * @deprecated use {@link Builder} to configure the OpenID Configuration.
     */
    @Deprecated(since = "12.1.0", forRemoval = true)
    public void setLogoutWhenIdTokenIsExpired(boolean logoutWhenIdTokenIsExpired)
    {
        this.logoutWhenIdTokenIsExpired = logoutWhenIdTokenIsExpired;
    }

    private static HttpClient newHttpClient()
    {
        ClientConnector connector = new ClientConnector();
        connector.setSslContextFactory(new SslContextFactory.Client(false));
        return new HttpClient(new HttpClientTransportOverHTTP(connector));
    }

    @Override
    public String toString()
    {
        return String.format("%s@%x{iss=%s, clientId=%s, authEndpoint=%s, authenticator=%s, tokenEndpoint=%s, scopes=%s, authNewUsers=%s}",
            TypeUtil.toShortName(getClass()), hashCode(), issuer, clientId, authorizationEndpoint, authenticationMethod, tokenEndpoint, scopes, authenticateNewUsers);
    }
}
