/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.eclipse.jetty.http.CompressedContentFormat;
import org.eclipse.jetty.http.GZIPContentDecoder;
import org.eclipse.jetty.io.ArrayByteBufferPool;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.toolchain.test.TestTracker;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class GZIPContentDecoderTest {
    @Rule
    public final TestTracker tracker = new TestTracker();
    ArrayByteBufferPool pool;
    AtomicInteger buffers = new AtomicInteger(0);

    @Before
    public void beforeClass() throws Exception {
        this.buffers.set(0);
        this.pool = new ArrayByteBufferPool(){

            public ByteBuffer acquire(int size, boolean direct) {
                GZIPContentDecoderTest.this.buffers.incrementAndGet();
                return super.acquire(size, direct);
            }

            public void release(ByteBuffer buffer) {
                GZIPContentDecoderTest.this.buffers.decrementAndGet();
                super.release(buffer);
            }
        };
    }

    @After
    public void afterClass() throws Exception {
        Assert.assertEquals((long)0L, (long)this.buffers.get());
    }

    @Test
    public void testCompresedContentFormat() throws Exception {
        Assert.assertTrue((boolean)CompressedContentFormat.tagEquals((String)"tag", (String)"tag"));
        Assert.assertTrue((boolean)CompressedContentFormat.tagEquals((String)"\"tag\"", (String)"\"tag\""));
        Assert.assertTrue((boolean)CompressedContentFormat.tagEquals((String)"\"tag\"", (String)"\"tag--gzip\""));
        Assert.assertFalse((boolean)CompressedContentFormat.tagEquals((String)"Zag", (String)"Xag--gzip"));
        Assert.assertFalse((boolean)CompressedContentFormat.tagEquals((String)"xtag", (String)"tag"));
    }

    @Test
    public void testStreamNoBlocks() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.close();
        byte[] bytes = baos.toByteArray();
        GZIPInputStream input = new GZIPInputStream((InputStream)new ByteArrayInputStream(bytes), 1);
        int read = input.read();
        Assert.assertEquals((long)-1L, (long)read);
    }

    @Test
    public void testStreamBigBlockOneByteAtATime() throws Exception {
        int read;
        String data = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data = data + data;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        baos = new ByteArrayOutputStream();
        GZIPInputStream input = new GZIPInputStream((InputStream)new ByteArrayInputStream(bytes), 1);
        while ((read = input.read()) >= 0) {
            baos.write(read);
        }
        Assert.assertEquals((Object)data, (Object)new String(baos.toByteArray(), StandardCharsets.UTF_8));
    }

    @Test
    public void testNoBlocks() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.close();
        byte[] bytes = baos.toByteArray();
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes));
        Assert.assertEquals((long)0L, (long)decoded.remaining());
    }

    @Test
    public void testSmallBlock() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes));
        Assert.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        decoder.release(decoded);
    }

    @Test
    public void testSmallBlockWithGZIPChunkedAtBegin() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        byte[] bytes1 = new byte[11];
        System.arraycopy(bytes, 0, bytes1, 0, bytes1.length);
        byte[] bytes2 = new byte[bytes.length - bytes1.length];
        System.arraycopy(bytes, bytes1.length, bytes2, 0, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes1));
        Assert.assertEquals((long)0L, (long)decoded.capacity());
        decoded = decoder.decode(ByteBuffer.wrap(bytes2));
        Assert.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        decoder.release(decoded);
    }

    @Test
    public void testSmallBlockWithGZIPChunkedAtEnd() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        byte[] bytes1 = new byte[bytes.length - 9];
        System.arraycopy(bytes, 0, bytes1, 0, bytes1.length);
        byte[] bytes2 = new byte[bytes.length - bytes1.length];
        System.arraycopy(bytes, bytes1.length, bytes2, 0, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes1));
        Assert.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        Assert.assertFalse((boolean)decoder.isFinished());
        decoder.release(decoded);
        decoded = decoder.decode(ByteBuffer.wrap(bytes2));
        Assert.assertEquals((long)0L, (long)decoded.remaining());
        Assert.assertTrue((boolean)decoder.isFinished());
        decoder.release(decoded);
    }

    @Test
    public void testSmallBlockWithGZIPTrailerChunked() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        byte[] bytes1 = new byte[bytes.length - 3];
        System.arraycopy(bytes, 0, bytes1, 0, bytes1.length);
        byte[] bytes2 = new byte[bytes.length - bytes1.length];
        System.arraycopy(bytes, bytes1.length, bytes2, 0, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes1));
        Assert.assertEquals((long)0L, (long)decoded.capacity());
        decoder.release(decoded);
        decoded = decoder.decode(ByteBuffer.wrap(bytes2));
        Assert.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        decoder.release(decoded);
    }

    @Test
    public void testTwoSmallBlocks() throws Exception {
        String data1 = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data1.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes1 = baos.toByteArray();
        String data2 = "1";
        baos = new ByteArrayOutputStream();
        output = new GZIPOutputStream(baos);
        output.write(data2.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes2 = baos.toByteArray();
        byte[] bytes = new byte[bytes1.length + bytes2.length];
        System.arraycopy(bytes1, 0, bytes, 0, bytes1.length);
        System.arraycopy(bytes2, 0, bytes, bytes1.length, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        ByteBuffer decoded = decoder.decode(buffer);
        Assert.assertEquals((Object)data1, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        Assert.assertTrue((boolean)decoder.isFinished());
        Assert.assertTrue((boolean)buffer.hasRemaining());
        decoder.release(decoded);
        decoded = decoder.decode(buffer);
        Assert.assertEquals((Object)data2, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        Assert.assertTrue((boolean)decoder.isFinished());
        Assert.assertFalse((boolean)buffer.hasRemaining());
        decoder.release(decoded);
    }

    @Test
    public void testBigBlock() throws Exception {
        String data = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data = data + data;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        String result = "";
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        while (buffer.hasRemaining()) {
            ByteBuffer decoded = decoder.decode(buffer);
            result = result + StandardCharsets.UTF_8.decode(decoded).toString();
            decoder.release(decoded);
        }
        Assert.assertEquals((Object)data, (Object)result);
    }

    @Test
    public void testBigBlockOneByteAtATime() throws Exception {
        String data = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data = data + data;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        String result = "";
        GZIPContentDecoder decoder = new GZIPContentDecoder(64);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        while (buffer.hasRemaining()) {
            ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(new byte[]{buffer.get()}));
            if (decoded.hasRemaining()) {
                result = result + StandardCharsets.UTF_8.decode(decoded).toString();
            }
            decoder.release(decoded);
        }
        Assert.assertEquals((Object)data, (Object)result);
        Assert.assertTrue((boolean)decoder.isFinished());
    }

    @Test
    public void testBigBlockWithExtraBytes() throws Exception {
        String data1 = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data1 = data1 + data1;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data1.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes1 = baos.toByteArray();
        String data2 = "HELLO";
        byte[] bytes2 = data2.getBytes(StandardCharsets.UTF_8);
        byte[] bytes = new byte[bytes1.length + bytes2.length];
        System.arraycopy(bytes1, 0, bytes, 0, bytes1.length);
        System.arraycopy(bytes2, 0, bytes, bytes1.length, bytes2.length);
        String result = "";
        GZIPContentDecoder decoder = new GZIPContentDecoder(64);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        while (buffer.hasRemaining()) {
            ByteBuffer decoded = decoder.decode(buffer);
            if (decoded.hasRemaining()) {
                result = result + StandardCharsets.UTF_8.decode(decoded).toString();
            }
            decoder.release(decoded);
            if (!decoder.isFinished()) continue;
            break;
        }
        Assert.assertEquals((Object)data1, (Object)result);
        Assert.assertTrue((boolean)buffer.hasRemaining());
        Assert.assertEquals((Object)data2, (Object)StandardCharsets.UTF_8.decode(buffer).toString());
    }
}

