/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http3.internal;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.util.concurrent.Executor;
import org.eclipse.jetty.http.MetaData;
import org.eclipse.jetty.http3.api.Stream;
import org.eclipse.jetty.http3.frames.DataFrame;
import org.eclipse.jetty.http3.frames.HeadersFrame;
import org.eclipse.jetty.http3.internal.HTTP3ErrorCode;
import org.eclipse.jetty.http3.internal.parser.MessageParser;
import org.eclipse.jetty.http3.internal.parser.ParserListener;
import org.eclipse.jetty.io.AbstractConnection;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.io.RetainableByteBufferPool;
import org.eclipse.jetty.quic.common.QuicStreamEndPoint;
import org.eclipse.jetty.util.thread.AutoLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class HTTP3StreamConnection
extends AbstractConnection {
    private static final Logger LOG = LoggerFactory.getLogger(HTTP3StreamConnection.class);
    private static final ByteBuffer EMPTY_DATA_FRAME = ByteBuffer.allocate(2);
    private final AutoLock lock = new AutoLock();
    private final RetainableByteBufferPool buffers;
    private final MessageParser parser;
    private boolean useInputDirectByteBuffers = true;
    private RetainableByteBuffer buffer;
    private boolean applicationMode;
    private boolean parserDataMode;
    private boolean dataDemand;
    private boolean dataStalled;
    private DataFrame dataFrame;
    private boolean dataLast;
    private boolean noData;
    private boolean remotelyClosed;

    public HTTP3StreamConnection(QuicStreamEndPoint endPoint, Executor executor, ByteBufferPool byteBufferPool, MessageParser parser) {
        super((EndPoint)endPoint, executor);
        this.buffers = RetainableByteBufferPool.findOrAdapt(null, (ByteBufferPool)byteBufferPool);
        this.parser = parser;
        parser.init(x$0 -> new MessageListener((ParserListener)x$0));
    }

    public QuicStreamEndPoint getEndPoint() {
        return (QuicStreamEndPoint)super.getEndPoint();
    }

    public boolean isUseInputDirectByteBuffers() {
        return this.useInputDirectByteBuffers;
    }

    public void setUseInputDirectByteBuffers(boolean useInputDirectByteBuffers) {
        this.useInputDirectByteBuffers = useInputDirectByteBuffers;
    }

    public void setApplicationMode(boolean mode) {
        this.applicationMode = mode;
    }

    public void onOpen() {
        super.onOpen();
        this.fillInterested();
    }

    protected boolean onReadTimeout(Throwable timeout) {
        return false;
    }

    public void onFillable() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("processing parserDataMode={} on {}", (Object)this.parserDataMode, (Object)this);
        }
        if (this.parserDataMode) {
            this.processDataFrames();
        } else {
            this.processNonDataFrames();
        }
    }

    private void processDataFrames() {
        this.processDataDemand();
        if (!this.parserDataMode) {
            if (this.buffer != null && this.buffer.hasRemaining()) {
                this.processNonDataFrames();
            } else {
                this.fillInterested();
            }
        }
    }

    private void processNonDataFrames() {
        try {
            this.tryAcquireBuffer();
            do {
                if (this.parseAndFill(true) == MessageParser.Result.NO_FRAME) {
                    this.tryReleaseBuffer(false);
                    return;
                }
                if (!this.remotelyClosed) continue;
                this.getEndPoint().getQuicSession().flush();
                this.tryReleaseBuffer(false);
                return;
            } while (!this.parserDataMode);
            if (this.buffer.hasRemaining()) {
                this.processDataFrames();
            } else if (this.applicationMode) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("skipping fill interest on {}", (Object)this);
                }
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("setting fill interest on {}", (Object)this);
                }
                this.fillInterested();
            }
            this.tryReleaseBuffer(false);
            return;
        }
        catch (Throwable x) {
            this.tryReleaseBuffer(true);
            long error = HTTP3ErrorCode.REQUEST_CANCELLED_ERROR.code();
            this.getEndPoint().close(error, x);
            this.parser.getListener().onStreamFailure(this.getEndPoint().getStreamId(), error, x);
            return;
        }
    }

    protected abstract void onDataAvailable(long var1);

    public Stream.Data readData() {
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("reading data on {}", (Object)this);
            }
            this.tryAcquireBuffer();
            switch (this.parseAndFill(false)) {
                case FRAME: {
                    if (this.parserDataMode) {
                        DataFrame frame = this.dataFrame;
                        this.dataFrame = null;
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("read data {} on {}", (Object)frame, (Object)this);
                        }
                        this.buffer.retain();
                        RetainableByteBuffer current = this.buffer;
                        this.tryReleaseBuffer(false);
                        return new Stream.Data(frame, () -> this.completeReadData(current));
                    }
                    this.tryReleaseBuffer(false);
                    return null;
                }
                case MODE_SWITCH: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("switching to parserDataMode=false on {}", (Object)this);
                    }
                    this.dataLast = true;
                    this.parserDataMode = false;
                    this.parser.setDataMode(false);
                    this.tryReleaseBuffer(false);
                    return null;
                }
                case NO_FRAME: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("read no data on {}", (Object)this);
                    }
                    this.tryReleaseBuffer(false);
                    return null;
                }
            }
            throw new IllegalStateException();
        }
        catch (Throwable x) {
            this.cancelDemand();
            this.tryReleaseBuffer(true);
            this.getEndPoint().close(HTTP3ErrorCode.REQUEST_CANCELLED_ERROR.code(), x);
            throw x;
        }
    }

    private void completeReadData(RetainableByteBuffer buffer) {
        buffer.release();
        if (LOG.isDebugEnabled()) {
            LOG.debug("released retained {}", (Object)buffer);
        }
    }

    public void demand() {
        boolean hasData;
        boolean process = false;
        try (AutoLock l = this.lock.lock();){
            hasData = !this.noData;
            this.dataDemand = true;
            if (this.dataStalled && hasData) {
                this.dataStalled = false;
                process = true;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("demand, wasStalled={} hasData={} on {}", new Object[]{process, hasData, this});
        }
        if (process) {
            this.processDataFrames();
        } else if (!hasData) {
            this.fillInterested();
        }
    }

    public boolean hasDemand() {
        try (AutoLock l = this.lock.lock();){
            boolean bl = this.dataDemand;
            return bl;
        }
    }

    private void cancelDemand() {
        try (AutoLock l = this.lock.lock();){
            this.dataDemand = false;
        }
    }

    private boolean isStalled() {
        try (AutoLock l = this.lock.lock();){
            boolean bl = this.dataStalled;
            return bl;
        }
    }

    private void setNoData(boolean noData) {
        try (AutoLock l = this.lock.lock();){
            this.noData = noData;
        }
    }

    private void processDataDemand() {
        while (true) {
            boolean process = true;
            try (AutoLock l = this.lock.lock();){
                if (LOG.isDebugEnabled()) {
                    LOG.debug("processing demand={}, last={} fillInterested={} on {}", new Object[]{this.dataDemand, this.dataLast, this.isFillInterested(), this});
                }
                if (this.dataDemand) {
                    if (this.dataLast || this.isFillInterested()) {
                        process = false;
                    } else {
                        this.dataDemand = false;
                    }
                } else {
                    this.dataStalled = true;
                    process = false;
                }
            }
            if (!process) {
                return;
            }
            this.onDataAvailable(this.getEndPoint().getStreamId());
        }
    }

    private void tryAcquireBuffer() {
        if (this.buffer == null) {
            this.buffer = this.buffers.acquire(this.getInputBufferSize(), this.isUseInputDirectByteBuffers());
            if (LOG.isDebugEnabled()) {
                LOG.debug("acquired {}", (Object)this.buffer);
            }
        }
    }

    private void tryReleaseBuffer(boolean force) {
        if (this.buffer != null) {
            if (this.buffer.hasRemaining() && force) {
                this.buffer.clear();
            }
            if (!this.buffer.hasRemaining()) {
                this.buffer.release();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("released {}", (Object)this.buffer);
                }
                this.buffer = null;
            }
        }
    }

    private MessageParser.Result parseAndFill(boolean setFillInterest) {
        try {
            int filled;
            if (LOG.isDebugEnabled()) {
                LOG.debug("parse+fill setFillInterest={} on {} with buffer {}", new Object[]{setFillInterest, this, this.buffer});
            }
            this.setNoData(false);
            do {
                ByteBuffer byteBuffer = this.buffer.getBuffer();
                MessageParser.Result result = this.parser.parse(byteBuffer);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("parsed {} on {} with buffer {}", new Object[]{result, this, this.buffer});
                }
                if (result == MessageParser.Result.FRAME || result == MessageParser.Result.MODE_SWITCH) {
                    return result;
                }
                if (this.buffer.isRetained()) {
                    this.buffer.release();
                    RetainableByteBuffer newBuffer = this.buffers.acquire(this.getInputBufferSize(), this.isUseInputDirectByteBuffers());
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("reacquired {} for retained {}", (Object)newBuffer, (Object)this.buffer);
                    }
                    this.buffer = newBuffer;
                    byteBuffer = this.buffer.getBuffer();
                }
                filled = this.fill(byteBuffer);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("filled {} on {} with buffer {}", new Object[]{filled, this, this.buffer});
            } while (filled > 0);
            if (filled == 0) {
                if (!this.remotelyClosed && this.getEndPoint().isStreamFinished()) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("detected end of stream on {}", (Object)this);
                    }
                    this.parser.parse(EMPTY_DATA_FRAME.slice());
                    return MessageParser.Result.FRAME;
                }
                this.setNoData(true);
                if (setFillInterest) {
                    this.fillInterested();
                }
            }
            return MessageParser.Result.NO_FRAME;
        }
        catch (Throwable x) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("parse+fill failure on {}", (Object)this, (Object)x);
            }
            throw x;
        }
    }

    private int fill(ByteBuffer byteBuffer) {
        try {
            return this.getEndPoint().fill(byteBuffer);
        }
        catch (IOException x) {
            throw new UncheckedIOException(x.getMessage(), x);
        }
    }

    public String toConnectionString() {
        return String.format("%s[demand=%b,stalled=%b,parserDataMode=%b]", super.toConnectionString(), this.hasDemand(), this.isStalled(), this.parserDataMode);
    }

    private class MessageListener
    extends ParserListener.Wrapper {
        private MessageListener(ParserListener listener) {
            super(listener);
        }

        @Override
        public void onHeaders(long streamId, HeadersFrame frame) {
            MetaData metaData = frame.getMetaData();
            if (metaData.isRequest()) {
                HTTP3StreamConnection.this.parserDataMode = true;
                HTTP3StreamConnection.this.parser.setDataMode(true);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("switching to parserDataMode=true for request {} on {}", (Object)metaData, (Object)this);
                }
            } else if (metaData.isResponse()) {
                MetaData.Response response = (MetaData.Response)metaData;
                if (response.getStatus() != 100) {
                    HTTP3StreamConnection.this.parserDataMode = true;
                    HTTP3StreamConnection.this.parser.setDataMode(true);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("switching to parserDataMode=true for response {} on {}", (Object)metaData, (Object)this);
                    }
                } else if (LOG.isDebugEnabled()) {
                    LOG.debug("staying in parserDataMode=false for response {} on {}", (Object)metaData, (Object)this);
                }
            } else if (!frame.isLast()) {
                frame = new HeadersFrame(metaData, true);
            }
            if (frame.isLast()) {
                this.shutdownInput();
            }
            super.onHeaders(streamId, frame);
        }

        @Override
        public void onData(long streamId, DataFrame frame) {
            if (HTTP3StreamConnection.this.dataFrame != null) {
                throw new IllegalStateException();
            }
            HTTP3StreamConnection.this.dataFrame = frame;
            if (frame.isLast()) {
                HTTP3StreamConnection.this.dataLast = true;
                this.shutdownInput();
            }
            super.onData(streamId, frame);
        }

        private void shutdownInput() {
            HTTP3StreamConnection.this.remotelyClosed = true;
            HTTP3StreamConnection.this.getEndPoint().shutdownInput(HTTP3ErrorCode.NO_ERROR.code());
        }
    }
}

