//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//
package org.eclipse.jetty.ee8.security.openid;

import java.io.IOException;
import java.io.Serial;
import java.io.Serializable;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.eclipse.jetty.ee8.nested.Authentication;
import org.eclipse.jetty.ee8.nested.Request;
import org.eclipse.jetty.ee8.nested.Response;
import org.eclipse.jetty.ee8.security.ServerAuthException;
import org.eclipse.jetty.ee8.security.UserAuthentication;
import org.eclipse.jetty.ee8.security.authentication.DeferredAuthentication;
import org.eclipse.jetty.ee8.security.authentication.LoginAuthenticator;
import org.eclipse.jetty.ee8.security.authentication.SessionAuthentication;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.security.Authenticator;
import org.eclipse.jetty.security.LoginService;
import org.eclipse.jetty.security.UserIdentity;
import org.eclipse.jetty.security.openid.OpenIdConfiguration;
import org.eclipse.jetty.security.openid.OpenIdCredentials;
import org.eclipse.jetty.security.openid.OpenIdLoginService;
import org.eclipse.jetty.util.Fields;
import org.eclipse.jetty.util.URIUtil;
import org.eclipse.jetty.util.UrlEncoded;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Implements authentication using OpenId Connect on top of OAuth 2.0.
 *
 * <p>The OpenIdAuthenticator redirects unauthenticated requests to the OpenID Connect Provider. The End-User is
 * eventually redirected back with an Authorization Code to the path set by {@link #setRedirectPath(String)} within the context.
 * The Authorization Code is then used to authenticate the user through the {@link OpenIdCredentials} and {@link OpenIdLoginService}.
 * </p>
 * <p>
 * Once a user is authenticated the OpenID Claims can be retrieved through an attribute on the session with the key {@link #CLAIMS}.
 * The full response containing the OAuth 2.0 Access Token can be obtained with the session attribute {@link #RESPONSE}.
 * </p>
 * <p>{@link SessionAuthentication} is then used to wrap Authentication results so that they are associated with the session.</p>
 */
public class OpenIdAuthenticator extends LoginAuthenticator {

    private static final Logger LOG = LoggerFactory.getLogger(OpenIdAuthenticator.class);

    public static final String CLAIMS = "org.eclipse.jetty.security.openid.claims";

    public static final String RESPONSE = "org.eclipse.jetty.security.openid.response";

    public static final String ISSUER = "org.eclipse.jetty.security.openid.issuer";

    public static final String REDIRECT_PATH = "org.eclipse.jetty.security.openid.redirect_path";

    public static final String LOGOUT_REDIRECT_PATH = "org.eclipse.jetty.security.openid.logout_redirect_path";

    public static final String ERROR_PAGE = "org.eclipse.jetty.security.openid.error_page";

    public static final String J_URI = "org.eclipse.jetty.security.openid.URI";

    public static final String J_POST = "org.eclipse.jetty.security.openid.POST";

    public static final String J_METHOD = "org.eclipse.jetty.security.openid.METHOD";

    public static final String J_SECURITY_CHECK = "/j_security_check";

    public static final String ERROR_PARAMETER = "error_description_jetty";

    private static final String CSRF_MAP = "org.eclipse.jetty.security.openid.csrf_map";

    @Deprecated
    public static final String CSRF_TOKEN = "org.eclipse.jetty.security.openid.csrf_token";

    private final SecureRandom _secureRandom = new SecureRandom();

    private OpenIdConfiguration _openIdConfiguration;

    private String _redirectPath;

    private String _logoutRedirectPath;

    private String _errorPage;

    private String _errorPath;

    private String _errorQuery;

    private boolean _alwaysSaveUri;

    public OpenIdAuthenticator() {
        this(null, J_SECURITY_CHECK, null);
    }

    public OpenIdAuthenticator(OpenIdConfiguration configuration) {
        this(configuration, J_SECURITY_CHECK, null);
    }

    public OpenIdAuthenticator(OpenIdConfiguration configuration, String errorPage) {
        this(configuration, J_SECURITY_CHECK, errorPage);
    }

    public OpenIdAuthenticator(OpenIdConfiguration configuration, String redirectPath, String errorPage) {
        this(configuration, redirectPath, errorPage, null);
    }

    public OpenIdAuthenticator(OpenIdConfiguration configuration, String redirectPath, String errorPage, String logoutRedirectPath) {
        _openIdConfiguration = configuration;
        setRedirectPath(redirectPath);
        if (errorPage != null)
            setErrorPage(errorPage);
        if (logoutRedirectPath != null)
            setLogoutRedirectPath(logoutRedirectPath);
    }

    @Override
    public void setConfiguration(AuthConfiguration authConfig) {
        if (_openIdConfiguration == null) {
            LoginService loginService = authConfig.getLoginService();
            if (!(loginService instanceof OpenIdLoginService))
                throw new IllegalArgumentException("invalid LoginService " + loginService);
            this._openIdConfiguration = ((OpenIdLoginService) loginService).getConfiguration();
        }
        String redirectPath = authConfig.getInitParameter(REDIRECT_PATH);
        if (redirectPath != null)
            setRedirectPath(redirectPath);
        String error = authConfig.getInitParameter(ERROR_PAGE);
        if (error != null)
            setErrorPage(error);
        String logout = authConfig.getInitParameter(LOGOUT_REDIRECT_PATH);
        if (logout != null)
            setLogoutRedirectPath(logout);
        super.setConfiguration(new OpenIdAuthConfiguration(_openIdConfiguration, authConfig));
    }

    @Override
    public String getAuthMethod() {
        return Authenticator.OPENID_AUTH;
    }

    @Deprecated
    public void setAlwaysSaveUri(boolean alwaysSave) {
        _alwaysSaveUri = alwaysSave;
    }

    @Deprecated
    public boolean isAlwaysSaveUri() {
        return _alwaysSaveUri;
    }

    public void setRedirectPath(String redirectPath) {
        if (redirectPath == null) {
            LOG.warn("redirect path must not be null, defaulting to " + J_SECURITY_CHECK);
            redirectPath = J_SECURITY_CHECK;
        } else if (!redirectPath.startsWith("/")) {
            LOG.warn("redirect path must start with /");
            redirectPath = "/" + redirectPath;
        }
        _redirectPath = redirectPath;
    }

    public void setLogoutRedirectPath(String logoutRedirectPath) {
        if (logoutRedirectPath == null) {
            LOG.warn("redirect path must not be null, defaulting to /");
            logoutRedirectPath = "/";
        } else if (!logoutRedirectPath.startsWith("/")) {
            LOG.warn("redirect path must start with /");
            logoutRedirectPath = "/" + logoutRedirectPath;
        }
        _logoutRedirectPath = logoutRedirectPath;
    }

    public void setErrorPage(String path) {
        if (path == null || path.trim().length() == 0) {
            _errorPath = null;
            _errorPage = null;
        } else {
            if (!path.startsWith("/")) {
                LOG.warn("error-page must start with /");
                path = "/" + path;
            }
            _errorPage = path;
            _errorPath = path;
            _errorQuery = "";
            int queryIndex = _errorPath.indexOf('?');
            if (queryIndex > 0) {
                _errorPath = _errorPage.substring(0, queryIndex);
                _errorQuery = _errorPage.substring(queryIndex + 1);
            }
        }
    }

    @Override
    public UserIdentity login(String username, Object credentials, ServletRequest request) {
        if (LOG.isDebugEnabled())
            LOG.debug("login {} {} {}", username, credentials, request);
        UserIdentity user = super.login(username, credentials, request);
        if (user != null) {
            HttpSession session = ((HttpServletRequest) request).getSession();
            Authentication cached = new SessionAuthentication(getAuthMethod(), user, credentials);
            synchronized (session) {
                session.setAttribute(SessionAuthentication.__J_AUTHENTICATED, cached);
                session.setAttribute(CLAIMS, ((OpenIdCredentials) credentials).getClaims());
                session.setAttribute(RESPONSE, ((OpenIdCredentials) credentials).getResponse());
                session.setAttribute(ISSUER, _openIdConfiguration.getIssuer());
            }
        }
        return user;
    }

    @Override
    public void logout(ServletRequest request) {
        attemptLogoutRedirect(request);
        logoutWithoutRedirect(request);
    }

    private void logoutWithoutRedirect(ServletRequest request) {
        super.logout(request);
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpSession session = httpRequest.getSession(false);
        if (session == null)
            return;
        synchronized (session) {
            session.removeAttribute(SessionAuthentication.__J_AUTHENTICATED);
            session.removeAttribute(CLAIMS);
            session.removeAttribute(RESPONSE);
            session.removeAttribute(ISSUER);
        }
    }

    /**
     * <p>This will attempt to redirect the request to the end_session_endpoint, and finally to the {@link #REDIRECT_PATH}.</p>
     *
     * <p>If end_session_endpoint is defined the request will be redirected to the end_session_endpoint, the optional
     * post_logout_redirect_uri parameter will be set if {@link #REDIRECT_PATH} is non-null.</p>
     *
     * <p>If the end_session_endpoint is not defined then the request will be redirected to {@link #REDIRECT_PATH} if it is a
     * non-null value, otherwise no redirection will be done.</p>
     *
     * @param request the request to redirect.
     */
    private void attemptLogoutRedirect(ServletRequest request) {
        try {
            Request baseRequest = Objects.requireNonNull(Request.getBaseRequest(request));
            Response baseResponse = baseRequest.getResponse();
            String endSessionEndpoint = _openIdConfiguration.getEndSessionEndpoint();
            String redirectUri = null;
            if (_logoutRedirectPath != null) {
                StringBuilder sb = URIUtil.newURIBuilder(request.getScheme(), request.getServerName(), request.getServerPort());
                sb.append(baseRequest.getContextPath());
                sb.append(_logoutRedirectPath);
                redirectUri = sb.toString();
            }
            HttpSession session = baseRequest.getSession(false);
            if (endSessionEndpoint == null || session == null) {
                if (redirectUri != null)
                    baseResponse.sendRedirect(redirectUri, true);
                return;
            }
            Object openIdResponse = session.getAttribute(OpenIdAuthenticator.RESPONSE);
            if (!(openIdResponse instanceof Map)) {
                if (redirectUri != null)
                    baseResponse.sendRedirect(redirectUri, true);
                return;
            }
            @SuppressWarnings("rawtypes")
            String idToken = (String) ((Map) openIdResponse).get("id_token");
            baseResponse.sendRedirect(endSessionEndpoint + "?id_token_hint=" + UrlEncoded.encodeString(idToken, StandardCharsets.UTF_8) + ((redirectUri == null) ? "" : "&post_logout_redirect_uri=" + UrlEncoded.encodeString(redirectUri, StandardCharsets.UTF_8)), true);
        } catch (Throwable t) {
            LOG.warn("failed to redirect to end_session_endpoint", t);
        }
    }

    @Override
    public void prepareRequest(ServletRequest request) {
        //if this is a request resulting from a redirect after auth is complete
        //(ie its from a redirect to the original request uri) then due to
        //browser handling of 302 redirects, the method may not be the same as
        //that of the original request. Replace the method and original post
        //params (if it was a post).
        //
        //See Servlet Spec 3.1 sec 13.6.3
        HttpServletRequest httpRequest = (HttpServletRequest) request;
        HttpSession session = httpRequest.getSession(false);
        if (session == null)
            //not authenticated yet
            return;
        String juri;
        String method;
        synchronized (session) {
            if (session.getAttribute(SessionAuthentication.__J_AUTHENTICATED) == null)
                //not authenticated yet
                return;
            juri = (String) session.getAttribute(J_URI);
            if (juri == null || juri.length() == 0)
                //no original uri saved
                return;
            method = (String) session.getAttribute(J_METHOD);
            if (method == null || method.length() == 0)
                //didn't save original request method
                return;
        }
        StringBuffer buf = httpRequest.getRequestURL();
        if (httpRequest.getQueryString() != null)
            buf.append("?").append(httpRequest.getQueryString());
        if (!juri.equals(buf.toString()))
            //this request is not for the same url as the original
            return;
        // Restore the original request's method on this request.
        if (LOG.isDebugEnabled())
            LOG.debug("Restoring original method {} for {} with method {}", method, juri, httpRequest.getMethod());
        Request baseRequest = Objects.requireNonNull(Request.getBaseRequest(request));
        baseRequest.setMethod(method);
    }

    private boolean hasExpiredIdToken(HttpSession session) {
        if (session != null) {
            Map<String, Object> claims = (Map) session.getAttribute(CLAIMS);
            if (claims != null)
                return OpenIdCredentials.checkExpiry(claims);
        }
        return false;
    }

    @Override
    public Authentication validateRequest(ServletRequest req, ServletResponse res, boolean mandatory) throws ServerAuthException {
        final HttpServletRequest request = (HttpServletRequest) req;
        final HttpServletResponse response = (HttpServletResponse) res;
        final Request baseRequest = Request.getBaseRequest(request);
        final Response baseResponse = baseRequest.getResponse();
        if (LOG.isDebugEnabled())
            LOG.debug("validateRequest({},{},{})", req, res, mandatory);
        String uri = request.getRequestURI();
        if (uri == null)
            uri = "/";
        HttpSession session = request.getSession(false);
        if (_openIdConfiguration.isLogoutWhenIdTokenIsExpired() && hasExpiredIdToken(session)) {
            // After logout, fall through to the code below and send another login challenge.
            logoutWithoutRedirect(req);
            // If we expired a valid authentication we do not want to defer authentication,
            // we want to try re-authenticate the user.
            mandatory = true;
        }
        mandatory |= isJSecurityCheck(uri);
        if (!mandatory)
            return new DeferredAuthentication(this);
        if (isErrorPage(baseRequest.getPathInContext()) && !DeferredAuthentication.isDeferred(response))
            return new DeferredAuthentication(this);
        try {
            // Get the Session.
            if (session == null)
                session = request.getSession(true);
            if (request.isRequestedSessionIdFromURL()) {
                sendError(request, response, "Session ID must be a cookie to support OpenID authentication");
                return Authentication.SEND_FAILURE;
            }
            // Handle a request for authentication.
            if (isJSecurityCheck(uri)) {
                String state = request.getParameter("state");
                if (state == null) {
                    sendError(request, response, "auth failed: no state parameter");
                    return Authentication.SEND_FAILURE;
                }
                // Handle error response defined by Section 3.1.2.6 of OpenID Connect Core 1.0.
                String errorCode = request.getParameter("error");
                if (errorCode != null) {
                    String errorDescription = request.getParameter("error_description");
                    String errorUri = request.getParameter("error_uri");
                    Fields fields = new Fields();
                    fields.add("error", errorCode);
                    if (errorDescription != null)
                        fields.add("error_description", errorDescription);
                    if (errorUri != null)
                        fields.add("error_uri", errorUri);
                    StringBuilder errorMessage = new StringBuilder();
                    errorMessage.append("auth failed: ").append(errorCode);
                    if (errorDescription != null)
                        errorMessage.append(" - ").append(errorDescription);
                    fields.add(ERROR_PARAMETER, errorMessage.toString());
                    sendError(request, response, fields);
                    return Authentication.SEND_FAILURE;
                }
                String authCode = request.getParameter("code");
                if (authCode == null) {
                    sendError(request, response, "auth failed: no code parameter");
                    return Authentication.SEND_FAILURE;
                }
                // Verify anti-forgery state token.
                UriRedirectInfo uriRedirectInfo;
                synchronized (session) {
                    uriRedirectInfo = removeAndClearCsrfMap(session, state);
                }
                if (uriRedirectInfo == null) {
                    sendError(request, response, "auth failed: invalid state parameter");
                    return Authentication.SEND_FAILURE;
                }
                // Attempt to login with the provided authCode.
                OpenIdCredentials credentials = new OpenIdCredentials(authCode, getRedirectUri(request));
                credentials.redeemAuthCode(_openIdConfiguration);
                if (credentials.getErrorFields() != null) {
                    sendError(request, response, credentials.getErrorFields());
                    return Authentication.SEND_FAILURE;
                }
                UserIdentity user = login(null, credentials, request);
                if (user == null) {
                    sendError(request, response, "auth failed: no user identity");
                    return Authentication.SEND_FAILURE;
                }
                OpenIdAuthentication openIdAuth = new OpenIdAuthentication(getAuthMethod(), user);
                if (LOG.isDebugEnabled())
                    LOG.debug("authenticated {}->{}", openIdAuth, uriRedirectInfo.getUri());
                // Save redirect info in session so original request can be restored after redirect.
                synchronized (session) {
                    session.setAttribute(J_URI, uriRedirectInfo.getUri());
                    session.setAttribute(J_METHOD, uriRedirectInfo.getMethod());
                    session.setAttribute(J_POST, uriRedirectInfo.getFormParameters());
                }
                // Redirect to the original URI.
                response.setContentLength(0);
                baseResponse.sendRedirect(uriRedirectInfo.getUri(), true);
                return openIdAuth;
            }
            // Look for cached authentication in the Session.
            Authentication authentication = (Authentication) session.getAttribute(SessionAuthentication.__J_AUTHENTICATED);
            if (authentication != null) {
                // Has authentication been revoked?
                if (authentication instanceof Authentication.User && _loginService != null && !_loginService.validate(((Authentication.User) authentication).getUserIdentity())) {
                    if (LOG.isDebugEnabled())
                        LOG.debug("auth revoked {}", authentication);
                    logoutWithoutRedirect(req);
                } else {
                    synchronized (session) {
                        String jUri = (String) session.getAttribute(J_URI);
                        if (jUri != null) {
                            // Check if the request is for the same url as the original and restore params if it was a post.
                            if (LOG.isDebugEnabled())
                                LOG.debug("auth retry {}->{}", authentication, jUri);
                            StringBuffer buf = request.getRequestURL();
                            if (request.getQueryString() != null)
                                buf.append("?").append(request.getQueryString());
                            if (jUri.equals(buf.toString())) {
                                @SuppressWarnings("unchecked")
                                Fields jPost = (Fields) session.getAttribute(J_POST);
                                if (jPost != null) {
                                    if (LOG.isDebugEnabled())
                                        LOG.debug("auth rePOST {}->{}", authentication, jUri);
                                    baseRequest.setContentFields(jPost);
                                }
                                session.removeAttribute(J_URI);
                                session.removeAttribute(J_METHOD);
                                session.removeAttribute(J_POST);
                            }
                        }
                    }
                    if (LOG.isDebugEnabled())
                        LOG.debug("auth {}", authentication);
                    return authentication;
                }
            }
            // If we can't send challenge.
            if (DeferredAuthentication.isDeferred(response)) {
                if (LOG.isDebugEnabled())
                    LOG.debug("auth deferred {}", session.getId());
                return Authentication.UNAUTHENTICATED;
            }
            // Send the the challenge.
            String challengeUri = getChallengeUri(baseRequest);
            if (LOG.isDebugEnabled())
                LOG.debug("challenge {}->{}", session.getId(), challengeUri);
            baseResponse.sendRedirect(challengeUri, true);
            return Authentication.SEND_CONTINUE;
        } catch (IOException e) {
            throw new ServerAuthException(e);
        }
    }

    /**
     * Report an error case either by redirecting to the error page if it is defined, otherwise sending a 403 response.
     * If the message parameter is not null, a query parameter with a key of {@link #ERROR_PARAMETER} and value of the error
     * message will be logged and added to the error redirect URI if the error page is defined.
     * @param request the request.
     * @param response the response.
     * @param message the reason for the error or null.
     * @throws IOException if sending the error fails for any reason.
     */
    private void sendError(HttpServletRequest request, HttpServletResponse response, String message) throws IOException {
        Fields fields = new Fields();
        fields.add(ERROR_PARAMETER, message);
        sendError(request, response, fields);
    }

    /**
     * Report an error case either by redirecting to the error page if it is defined, otherwise sending a 403 response.
     * If the message parameter is not null, a query parameter with a key of {@link #ERROR_PARAMETER} and value of the error
     * message will be logged and added to the error redirect URI if the error page is defined.
     * @param request the request.
     * @param response the response.
     * @param fields the list of query parameters to be included in the error redirect.
     * @throws IOException if sending the error fails for any reason.
     */
    private void sendError(HttpServletRequest request, HttpServletResponse response, Fields fields) throws IOException {
        final Request baseRequest = Request.getBaseRequest(request);
        final Response baseResponse = Objects.requireNonNull(baseRequest).getResponse();
        if (LOG.isDebugEnabled())
            LOG.debug("OpenId authentication FAILED: {}", fields);
        if (_errorPage == null) {
            if (LOG.isDebugEnabled())
                LOG.debug("auth failed 403");
            if (response != null)
                response.sendError(HttpServletResponse.SC_FORBIDDEN);
        } else {
            if (LOG.isDebugEnabled())
                LOG.debug("auth failed {}", _errorPage);
            String redirectUri = URIUtil.addPaths(request.getContextPath(), _errorPage);
            if (fields != null) {
                String query = _errorQuery;
                for (Fields.Field f : fields) {
                    query = URIUtil.addQueries(query, UrlEncoded.encodeString(f.getName()) + "=" + UrlEncoded.encodeString(f.getValue()));
                }
                redirectUri = URIUtil.addPathQuery(URIUtil.addPaths(request.getContextPath(), _errorPath), query);
            }
            baseResponse.sendRedirect(redirectUri, true);
        }
    }

    public boolean isJSecurityCheck(String uri) {
        int jsc = uri.indexOf(_redirectPath);
        if (jsc < 0)
            return false;
        int e = jsc + _redirectPath.length();
        if (e == uri.length())
            return true;
        char c = uri.charAt(e);
        return c == ';' || c == '#' || c == '/' || c == '?';
    }

    public boolean isErrorPage(String pathInContext) {
        return pathInContext != null && (pathInContext.equals(_errorPath));
    }

    private String getRedirectUri(HttpServletRequest request) {
        final StringBuilder redirectUri = URIUtil.newURIBuilder(request.getScheme(), request.getServerName(), request.getServerPort());
        redirectUri.append(request.getContextPath());
        redirectUri.append(_redirectPath);
        return redirectUri.toString();
    }

    protected String getChallengeUri(Request request) {
        HttpSession session = request.getSession();
        String antiForgeryToken;
        synchronized (session) {
            Map<String, UriRedirectInfo> csrfMap = ensureCsrfMap(session);
            antiForgeryToken = new BigInteger(130, _secureRandom).toString(32);
            csrfMap.put(antiForgeryToken, new UriRedirectInfo(request));
        }
        // any custom scopes requested from configuration
        StringBuilder scopes = new StringBuilder();
        for (String s : _openIdConfiguration.getScopes()) {
            scopes.append(" ").append(s);
        }
        return _openIdConfiguration.getAuthorizationEndpoint() + "?client_id=" + UrlEncoded.encodeString(_openIdConfiguration.getClientId(), StandardCharsets.UTF_8) + "&redirect_uri=" + UrlEncoded.encodeString(getRedirectUri(request), StandardCharsets.UTF_8) + "&scope=openid" + UrlEncoded.encodeString(scopes.toString(), StandardCharsets.UTF_8) + "&state=" + antiForgeryToken + "&response_type=code";
    }

    @Override
    public boolean secureResponse(ServletRequest req, ServletResponse res, boolean mandatory, Authentication.User validatedUser) {
        return req.isSecure();
    }

    private UriRedirectInfo removeAndClearCsrfMap(HttpSession session, String csrf) {
        @SuppressWarnings("unchecked")
        Map<String, UriRedirectInfo> csrfMap = (Map<String, UriRedirectInfo>) session.getAttribute(CSRF_MAP);
        if (csrfMap == null)
            return null;
        UriRedirectInfo uriRedirectInfo = csrfMap.get(csrf);
        csrfMap.clear();
        return uriRedirectInfo;
    }

    private Map<String, UriRedirectInfo> ensureCsrfMap(HttpSession session) {
        @SuppressWarnings("unchecked")
        Map<String, UriRedirectInfo> csrfMap = (Map<String, UriRedirectInfo>) session.getAttribute(CSRF_MAP);
        if (csrfMap == null) {
            csrfMap = new MRUMap(64);
            session.setAttribute(CSRF_MAP, csrfMap);
        }
        return csrfMap;
    }

    private static class MRUMap extends LinkedHashMap<String, UriRedirectInfo> {

        @Serial
        private static final long serialVersionUID = 5375723072014233L;

        private final int _size;

        private MRUMap(int size) {
            _size = size;
        }

        @Override
        protected boolean removeEldestEntry(Map.Entry<String, UriRedirectInfo> eldest) {
            return size() > _size;
        }
    }

    private static class UriRedirectInfo implements Serializable {

        @Serial
        private static final long serialVersionUID = 139567755844461433L;

        private final String _uri;

        private final String _method;

        private final Fields _formParameters;

        public UriRedirectInfo(Request request) {
            _uri = request.getRequestURI();
            _method = request.getMethod();
            if (MimeTypes.Type.FORM_ENCODED.is(request.getContentType()) && HttpMethod.POST.is(request.getMethod())) {
                Fields formParameters = new Fields(true);
                request.extractFormParameters(formParameters);
                _formParameters = formParameters;
            } else {
                _formParameters = null;
            }
        }

        public String getUri() {
            return _uri;
        }

        public String getMethod() {
            return _method;
        }

        public Fields getFormParameters() {
            return _formParameters;
        }
    }

    /**
     * This Authentication represents a just completed OpenId Connect authentication.
     * Subsequent requests from the same user are authenticated by the presents
     * of a {@link SessionAuthentication} instance in their session.
     */
    public static class OpenIdAuthentication extends UserAuthentication implements Authentication.ResponseSent {

        public OpenIdAuthentication(String method, UserIdentity userIdentity) {
            super(method, userIdentity);
        }

        @Override
        public String toString() {
            return "OpenId" + super.toString();
        }
    }
}
