/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.io;

import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ReadOnlyBufferException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Objects;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.Content;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.Retainable;
import org.eclipse.jetty.util.Blocker;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IteratingCallback;
import org.eclipse.jetty.util.IteratingNestedCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public interface RetainableByteBuffer
extends Retainable {
    public static final RetainableByteBuffer EMPTY = new NonRetainableByteBuffer(BufferUtil.EMPTY_BUFFER);

    public static Mutable wrap(ByteBuffer byteBuffer) {
        return new FixedCapacity(byteBuffer);
    }

    public static Mutable wrap(ByteBuffer byteBuffer, Retainable retainable) {
        return new FixedCapacity(byteBuffer, retainable);
    }

    public static Mutable wrap(ByteBuffer byteBuffer, final Runnable releaser) {
        return new FixedCapacity(byteBuffer){

            @Override
            public boolean release() {
                boolean released = super.release();
                if (released) {
                    releaser.run();
                }
                return released;
            }
        };
    }

    default public boolean isMutable() {
        return !this.getByteBuffer().isReadOnly();
    }

    default public Mutable asMutable() throws ReadOnlyBufferException {
        if (!this.isMutable() || this.isRetained()) {
            throw new ReadOnlyBufferException();
        }
        RetainableByteBuffer retainableByteBuffer = this;
        if (retainableByteBuffer instanceof Mutable) {
            Mutable mutable = (Mutable)retainableByteBuffer;
            return mutable;
        }
        throw new ReadOnlyBufferException();
    }

    default public boolean releaseAndRemove() {
        return this.release();
    }

    default public boolean appendTo(ByteBuffer buffer) {
        return this.remaining() == BufferUtil.append(buffer, this.getByteBuffer());
    }

    default public boolean appendTo(RetainableByteBuffer buffer) {
        return this.appendTo(buffer.getByteBuffer());
    }

    default public RetainableByteBuffer copy() {
        ByteBuffer byteBuffer = this.getByteBuffer();
        ByteBuffer copy = BufferUtil.copy(byteBuffer);
        return new FixedCapacity(copy);
    }

    default public byte get() throws BufferUnderflowException {
        return this.getByteBuffer().get();
    }

    default public byte get(long index) throws IndexOutOfBoundsException {
        ByteBuffer buffer = this.getByteBuffer();
        return buffer.get(buffer.position() + Math.toIntExact(index));
    }

    default public int get(byte[] bytes, int offset, int length) {
        ByteBuffer b = this.getByteBuffer();
        if (b == null || !b.hasRemaining()) {
            return 0;
        }
        length = Math.min(length, b.remaining());
        b.get(bytes, offset, length);
        return length;
    }

    public ByteBuffer getByteBuffer() throws BufferOverflowException;

    default public boolean isDirect() {
        return this.getByteBuffer().isDirect();
    }

    default public int remaining() {
        return this.getByteBuffer().remaining();
    }

    default public boolean hasRemaining() {
        return this.getByteBuffer().hasRemaining();
    }

    default public boolean isEmpty() {
        return !this.hasRemaining();
    }

    default public long size() {
        return this.remaining();
    }

    default public long maxSize() {
        return this.capacity();
    }

    default public int capacity() {
        return this.getByteBuffer().capacity();
    }

    default public void clear() {
        BufferUtil.clear(this.getByteBuffer());
    }

    default public long skip(long length) {
        if (length == 0L) {
            return 0L;
        }
        ByteBuffer byteBuffer = this.getByteBuffer();
        length = Math.min((long)byteBuffer.remaining(), length);
        byteBuffer.position(byteBuffer.position() + Math.toIntExact(length));
        return length;
    }

    default public void limit(long size) {
        ByteBuffer byteBuffer = this.getByteBuffer();
        size = Math.min(size, (long)byteBuffer.remaining());
        byteBuffer.limit(byteBuffer.position() + Math.toIntExact(size));
    }

    default public RetainableByteBuffer slice() {
        return this.slice(Long.MAX_VALUE);
    }

    default public RetainableByteBuffer slice(long length) {
        int size = this.remaining();
        ByteBuffer byteBuffer = this.getByteBuffer();
        int limit = byteBuffer.limit();
        byteBuffer.limit(byteBuffer.position() + Math.toIntExact(Math.min(length, (long)size)));
        ByteBuffer slice = byteBuffer.slice();
        byteBuffer.limit(limit);
        if (length > (long)size) {
            slice.limit(size);
        }
        if (!this.canRetain()) {
            return new NonRetainableByteBuffer(slice);
        }
        this.retain();
        return RetainableByteBuffer.wrap(slice, this);
    }

    default public RetainableByteBuffer take(long length) {
        if (this.isEmpty() || length == 0L) {
            return EMPTY;
        }
        RetainableByteBuffer slice = this.slice(length);
        this.skip(length);
        if (slice.isRetained()) {
            RetainableByteBuffer copy = slice.copy();
            slice.release();
            return copy;
        }
        return slice;
    }

    default public RetainableByteBuffer takeFrom(long skip) {
        if (this.isEmpty() || skip > this.size()) {
            return EMPTY;
        }
        RetainableByteBuffer slice = this.slice();
        slice.skip(skip);
        this.limit(skip);
        if (slice.isRetained()) {
            RetainableByteBuffer copy = slice.copy();
            slice.release();
            return copy;
        }
        return slice;
    }

    default public RetainableByteBuffer take() {
        return this.take(Long.MAX_VALUE);
    }

    default public void putTo(ByteBuffer toInfillMode) throws BufferOverflowException {
        toInfillMode.put(this.getByteBuffer());
    }

    default public void writeTo(Content.Sink sink, boolean last, Callback callback) {
        sink.write(last, this.getByteBuffer(), callback);
    }

    default public void writeTo(Content.Sink sink, boolean last) throws IOException {
        try (Blocker.Callback callback = Blocker.callback();){
            sink.write(last, this.getByteBuffer(), callback);
            callback.block();
        }
    }

    default public String toDetailString() {
        return this.toString();
    }

    public static class FixedCapacity
    extends Abstract
    implements Mutable {
        private final ByteBuffer _byteBuffer;
        private int _flipPosition = -1;

        public FixedCapacity(ByteBuffer byteBuffer) {
            this(byteBuffer, new Retainable.ReferenceCounter());
        }

        public FixedCapacity(ByteBuffer byteBuffer, Retainable retainable) {
            super(retainable);
            this._byteBuffer = Objects.requireNonNull(byteBuffer);
        }

        @Override
        public void clear() {
            super.clear();
            this._byteBuffer.clear();
            this._flipPosition = 0;
        }

        @Override
        public Mutable asMutable() {
            if (!this.isMutable() || this.isRetained()) {
                throw new ReadOnlyBufferException();
            }
            return this;
        }

        @Override
        public int remaining() {
            if (this._flipPosition < 0) {
                return super.remaining();
            }
            return this._byteBuffer.position() - this._flipPosition;
        }

        @Override
        public boolean hasRemaining() {
            if (this._flipPosition < 0) {
                return super.hasRemaining();
            }
            return this._flipPosition > 0 || this._byteBuffer.position() > 0;
        }

        @Override
        public boolean isDirect() {
            return this._byteBuffer.isDirect();
        }

        @Override
        public int capacity() {
            return this._byteBuffer.capacity();
        }

        @Override
        public byte get(long index) throws IndexOutOfBoundsException {
            int offset = this._flipPosition < 0 ? this._byteBuffer.position() : this._flipPosition;
            return this._byteBuffer.get(offset + Math.toIntExact(index));
        }

        @Override
        public void limit(long size) {
            if (this._flipPosition < 0) {
                super.limit(size);
            } else {
                this._byteBuffer.position(this._flipPosition + Math.toIntExact(Math.min(size, this.size())));
            }
        }

        @Override
        public ByteBuffer getByteBuffer() {
            if (this._flipPosition >= 0) {
                BufferUtil.flipToFlush(this._byteBuffer, this._flipPosition);
                this._flipPosition = -1;
            }
            return this._byteBuffer;
        }

        @Override
        public boolean append(ByteBuffer bytes) throws ReadOnlyBufferException {
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            int length = bytes.remaining();
            int space = this._byteBuffer.remaining();
            if (space == 0) {
                return length == 0;
            }
            if (length > space) {
                int position = this._byteBuffer.position();
                this._byteBuffer.put(position, bytes, bytes.position(), space);
                this._byteBuffer.position(position + space);
                bytes.position(bytes.position() + space);
                return false;
            }
            if (length > 0) {
                this._byteBuffer.put(bytes);
            }
            return true;
        }

        @Override
        public boolean append(RetainableByteBuffer bytes) throws ReadOnlyBufferException {
            assert (!this.isRetained());
            return bytes.remaining() == 0 || this.append(bytes.getByteBuffer());
        }

        @Override
        public Mutable add(ByteBuffer bytes) throws ReadOnlyBufferException {
            int space;
            int length;
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            if ((length = bytes.remaining()) > (space = this._byteBuffer.remaining())) {
                throw new BufferOverflowException();
            }
            if (length > 0) {
                this._byteBuffer.put(bytes);
            }
            return this;
        }

        @Override
        public Mutable add(RetainableByteBuffer bytes) throws ReadOnlyBufferException {
            assert (!this.isRetained());
            if (bytes instanceof DynamicCapacity) {
                int space;
                DynamicCapacity dynamic = (DynamicCapacity)bytes;
                int length = bytes.remaining();
                if (length > (space = this._byteBuffer.remaining())) {
                    throw new BufferOverflowException();
                }
                if (length > 0) {
                    for (RetainableByteBuffer buffer : dynamic._buffers) {
                        buffer.retain();
                        this.add(buffer);
                    }
                }
                bytes.release();
                return this;
            }
            this.add(bytes.getByteBuffer());
            bytes.release();
            return this;
        }

        @Override
        public Mutable put(byte b) {
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            this._byteBuffer.put(b);
            return this;
        }

        @Override
        public Mutable put(long index, byte b) {
            int remaining;
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            if (index > (long)(remaining = this._byteBuffer.position() - this._flipPosition)) {
                throw new IndexOutOfBoundsException();
            }
            this._byteBuffer.put(this._flipPosition + Math.toIntExact(index), b);
            return this;
        }

        @Override
        public Mutable putShort(short s) {
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            this._byteBuffer.putShort(s);
            return this;
        }

        @Override
        public Mutable putInt(int i) {
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            this._byteBuffer.putInt(i);
            return this;
        }

        @Override
        public Mutable putLong(long l) {
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            this._byteBuffer.putLong(l);
            return this;
        }

        @Override
        public Mutable put(byte[] bytes, int offset, int length) {
            assert (!this.isRetained());
            if (this._flipPosition < 0) {
                this._flipPosition = BufferUtil.flipToFill(this._byteBuffer);
            }
            this._byteBuffer.put(bytes, offset, length);
            return this;
        }

        @Override
        protected void addValueMarker(StringBuilder builder, boolean beginning) {
            if (beginning) {
                if (this._flipPosition >= 0) {
                    builder.append("<<~").append(this._flipPosition).append('-').append(this._byteBuffer.position()).append('/').append(this._byteBuffer.capacity()).append('<');
                } else {
                    builder.append("<<").append(this._byteBuffer.position()).append('-').append(this._byteBuffer.limit()).append('/').append(this._byteBuffer.capacity()).append('<');
                }
            } else {
                builder.append(">>>");
            }
        }
    }

    public static interface Mutable
    extends RetainableByteBuffer {
        default public long space() {
            return this.capacity() - this.remaining();
        }

        default public boolean isFull() {
            return this.space() == 0L;
        }

        public Mutable add(ByteBuffer var1) throws ReadOnlyBufferException, BufferOverflowException;

        public Mutable add(RetainableByteBuffer var1) throws ReadOnlyBufferException, BufferOverflowException;

        public boolean append(ByteBuffer var1) throws ReadOnlyBufferException;

        public boolean append(RetainableByteBuffer var1) throws ReadOnlyBufferException;

        public Mutable put(byte var1);

        public Mutable putShort(short var1);

        public Mutable putInt(int var1);

        public Mutable putLong(long var1);

        public Mutable put(byte[] var1, int var2, int var3);

        default public Mutable put(byte[] bytes) {
            return this.put(bytes, 0, bytes.length);
        }

        public Mutable put(long var1, byte var3);
    }

    public static class NonRetainableByteBuffer
    extends FixedCapacity {
        public NonRetainableByteBuffer(ByteBuffer byteBuffer) {
            super(byteBuffer, NON_RETAINABLE);
        }
    }

    public static class DynamicCapacity
    extends Abstract
    implements Mutable {
        private static final Logger LOG = LoggerFactory.getLogger(DynamicCapacity.class);
        private final ByteBufferPool.Sized _pool;
        private final long _maxSize;
        private final List<RetainableByteBuffer> _buffers;
        private final int _minRetainSize;
        private Mutable _aggregate;

        public DynamicCapacity() {
            this(null, false, -1L, -1, -1);
        }

        public DynamicCapacity(ByteBufferPool.Sized sizedPool) {
            this(null, sizedPool, -1L, -1);
        }

        public DynamicCapacity(ByteBufferPool.Sized sizedPool, long maxSize) {
            this(null, sizedPool, maxSize, -1);
        }

        public DynamicCapacity(ByteBufferPool.Sized sizedPool, long maxSize, int minRetainSize) {
            this(null, sizedPool, maxSize, minRetainSize);
        }

        public DynamicCapacity(ByteBufferPool pool) {
            this(null, pool instanceof ByteBufferPool.Sized ? (sized = (ByteBufferPool.Sized)pool) : new ByteBufferPool.Sized(pool), -1L, -1);
            ByteBufferPool.Sized sized;
        }

        public DynamicCapacity(ByteBufferPool pool, boolean direct, long maxSize) {
            this(pool, direct, maxSize, -1, -1);
        }

        public DynamicCapacity(ByteBufferPool pool, boolean direct, long maxSize, int aggregationSize) {
            this(pool, direct, maxSize, aggregationSize, -1);
        }

        public DynamicCapacity(ByteBufferPool pool, boolean direct, long maxSize, int aggregationSize, int minRetainSize) {
            this(null, new ByteBufferPool.Sized(pool, direct, maxSize > 0L && maxSize < 8192L ? (int)maxSize : aggregationSize), maxSize, minRetainSize);
        }

        private DynamicCapacity(List<RetainableByteBuffer> buffers, ByteBufferPool.Sized pool, long maxSize, int minRetainSize) {
            this._pool = pool == null ? ByteBufferPool.SIZED_NON_POOLING : pool;
            this._maxSize = maxSize < 0L ? Long.MAX_VALUE : maxSize;
            this._buffers = buffers == null ? new ArrayList() : buffers;
            this._minRetainSize = minRetainSize;
            if (this._pool.getSize() == 0 && this._maxSize >= Integer.MAX_VALUE && this._minRetainSize != 0) {
                throw new IllegalArgumentException("must always retain if cannot aggregate");
            }
        }

        private void checkNotReleased() {
            if (this.getRetained() <= 0) {
                throw new IllegalStateException("Already released");
            }
        }

        public long getMaxSize() {
            return this._maxSize;
        }

        public int getAggregationSize() {
            return this._pool.getSize();
        }

        public int getMinRetainSize() {
            return this._minRetainSize;
        }

        @Override
        public boolean isMutable() {
            return true;
        }

        @Override
        public Mutable asMutable() {
            if (this.isRetained()) {
                throw new ReadOnlyBufferException();
            }
            return this;
        }

        @Override
        public ByteBuffer getByteBuffer() throws BufferOverflowException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("getByteBuffer {}", (Object)this);
            }
            this.checkNotReleased();
            return switch (this._buffers.size()) {
                case 0 -> BufferUtil.EMPTY_BUFFER;
                case 1 -> this._buffers.get(0).getByteBuffer();
                default -> {
                    long size = this.size();
                    if (size > Integer.MAX_VALUE) {
                        throw new BufferOverflowException();
                    }
                    int length = (int)size;
                    Mutable combined = this._pool.acquire(length);
                    ByteBuffer byteBuffer = combined.getByteBuffer();
                    BufferUtil.flipToFill(byteBuffer);
                    for (RetainableByteBuffer buffer : this._buffers) {
                        byteBuffer.put(buffer.getByteBuffer());
                        buffer.release();
                    }
                    BufferUtil.flipToFlush(byteBuffer, 0);
                    this._buffers.clear();
                    this._buffers.add(combined);
                    this._aggregate = null;
                    yield combined.getByteBuffer();
                }
            };
        }

        @Override
        public RetainableByteBuffer take(long length) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("take {} {}", (Object)this, (Object)length);
            }
            this.checkNotReleased();
            if (this._buffers.isEmpty() || length == 0L) {
                return EMPTY;
            }
            this._aggregate = null;
            if (this._buffers.size() == 1) {
                RetainableByteBuffer buffer = this._buffers.get(0);
                if (length > buffer.size() / 2L && !buffer.isRetained()) {
                    RetainableByteBuffer tail = buffer.takeFrom(length);
                    this._buffers.set(0, tail);
                    return buffer;
                }
                return buffer.take(length);
            }
            ArrayList<RetainableByteBuffer> buffers = new ArrayList<RetainableByteBuffer>(this._buffers.size());
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = i.next();
                long size = buffer.size();
                if (length >= size) {
                    buffers.add(buffer);
                    i.remove();
                    if ((length -= size) != 0L) continue;
                    break;
                }
                if (length > buffer.size() / 2L && !buffer.isRetained()) {
                    RetainableByteBuffer tail = buffer.takeFrom(length);
                    buffers.add(buffer);
                    i.set(tail);
                    break;
                }
                buffers.add(buffer.take(length));
                break;
            }
            return new DynamicCapacity(buffers, this._pool, this._maxSize, this._minRetainSize);
        }

        @Override
        public RetainableByteBuffer takeFrom(long skip) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("take {} {}", (Object)this, (Object)skip);
            }
            this.checkNotReleased();
            if (this._buffers.isEmpty() || skip > this.size()) {
                return EMPTY;
            }
            this._aggregate = null;
            if (this._buffers.size() == 1) {
                RetainableByteBuffer buffer = this._buffers.get(0);
                if (skip > buffer.size() / 2L || buffer.isRetained()) {
                    return buffer.takeFrom(skip);
                }
                this._buffers.set(0, buffer.take(skip));
                return buffer;
            }
            ArrayList<RetainableByteBuffer> buffers = new ArrayList<RetainableByteBuffer>(this._buffers.size());
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = i.next();
                long size = buffer.size();
                if (skip >= size) {
                    skip -= size;
                    continue;
                }
                if (skip == 0L) {
                    buffers.add(buffer);
                    i.remove();
                    continue;
                }
                if (skip > buffer.size() / 2L || buffer.isRetained()) {
                    buffers.add(buffer.takeFrom(skip));
                } else {
                    i.set(buffer.take(skip));
                    buffers.add(buffer);
                }
                skip = 0L;
            }
            return new DynamicCapacity(buffers, this._pool, this._maxSize, this._minRetainSize);
        }

        public byte[] takeByteArray() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("takeByteArray {}", (Object)this);
            }
            this.checkNotReleased();
            return switch (this._buffers.size()) {
                case 0 -> BufferUtil.EMPTY_BUFFER.array();
                case 1 -> {
                    RetainableByteBuffer buffer = this._buffers.get(0);
                    this._aggregate = null;
                    this._buffers.clear();
                    byte[] array = !(buffer instanceof Pooled) && !buffer.isRetained() && !buffer.isDirect() ? buffer.getByteBuffer().array() : BufferUtil.toArray(buffer.getByteBuffer());
                    buffer.release();
                    yield array;
                }
                default -> {
                    long size = this.size();
                    if (size > Integer.MAX_VALUE) {
                        throw new BufferOverflowException();
                    }
                    int length = (int)size;
                    byte[] array = new byte[length];
                    int offset = 0;
                    for (RetainableByteBuffer buffer : this._buffers) {
                        int remaining = buffer.remaining();
                        buffer.get(array, offset, remaining);
                        offset += remaining;
                        buffer.release();
                    }
                    this._buffers.clear();
                    this._aggregate = null;
                    yield array;
                }
            };
        }

        @Override
        public byte get() throws BufferUnderflowException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("get {}", (Object)this);
            }
            this.checkNotReleased();
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = (RetainableByteBuffer)i.next();
                if (buffer.isEmpty()) {
                    buffer.release();
                    i.remove();
                    continue;
                }
                byte b = buffer.get();
                if (buffer.isEmpty()) {
                    buffer.release();
                    i.remove();
                }
                return b;
            }
            throw new BufferUnderflowException();
        }

        @Override
        public byte get(long index) throws IndexOutOfBoundsException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("get {} {}", (Object)this, (Object)index);
            }
            this.checkNotReleased();
            for (RetainableByteBuffer buffer : this._buffers) {
                long size = buffer.size();
                if (index < size) {
                    return buffer.get(Math.toIntExact(index));
                }
                index -= size;
            }
            throw new IndexOutOfBoundsException();
        }

        @Override
        public int get(byte[] bytes, int offset, int length) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("get array {} {}", (Object)this, (Object)length);
            }
            this.checkNotReleased();
            int got = 0;
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (length > 0 && i.hasNext()) {
                RetainableByteBuffer buffer = (RetainableByteBuffer)i.next();
                int l = buffer.get(bytes, offset, length);
                got += l;
                offset += l;
                length -= l;
                if (!buffer.isEmpty()) continue;
                buffer.release();
                i.remove();
            }
            return got;
        }

        @Override
        public boolean isDirect() {
            return this._pool.isDirect();
        }

        @Override
        public boolean hasRemaining() {
            this.checkNotReleased();
            for (RetainableByteBuffer rbb : this._buffers) {
                if (rbb.isEmpty()) continue;
                return true;
            }
            return false;
        }

        @Override
        public long skip(long length) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("skip {} {}", (Object)this, (Object)length);
            }
            this.checkNotReleased();
            long skipped = 0L;
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (length > 0L && i.hasNext()) {
                RetainableByteBuffer buffer = (RetainableByteBuffer)i.next();
                long skip = buffer.skip(length);
                skipped += skip;
                length -= skip;
                if (!buffer.isEmpty()) continue;
                buffer.release();
                i.remove();
            }
            return skipped;
        }

        @Override
        public void limit(long limit) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("limit {} {}", (Object)this, (Object)limit);
            }
            this.checkNotReleased();
            Iterator<RetainableByteBuffer> i = this._buffers.iterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = i.next();
                long size = buffer.size();
                if (limit == 0L) {
                    buffer.release();
                    i.remove();
                    continue;
                }
                if (limit < size) {
                    buffer.limit(limit);
                    limit = 0L;
                    continue;
                }
                limit -= size;
            }
        }

        @Override
        public Mutable slice() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("slice {}", (Object)this);
            }
            this.checkNotReleased();
            ArrayList<RetainableByteBuffer> buffers = new ArrayList<RetainableByteBuffer>(this._buffers.size());
            for (RetainableByteBuffer rbb : this._buffers) {
                buffers.add(rbb.slice());
            }
            return this.newSlice(buffers);
        }

        @Override
        public Mutable slice(long length) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("slice {} {}", (Object)this, (Object)length);
            }
            this.checkNotReleased();
            ArrayList<RetainableByteBuffer> buffers = new ArrayList<RetainableByteBuffer>(this._buffers.size());
            Iterator<RetainableByteBuffer> i = this._buffers.iterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = i.next();
                long size = buffer.size();
                if (size > length || !i.hasNext()) {
                    buffers.add(buffer.slice(length));
                    break;
                }
                buffers.add(buffer.slice());
                length -= size;
            }
            return this.newSlice(buffers);
        }

        private Mutable newSlice(List<RetainableByteBuffer> buffers) {
            return new DynamicCapacity(buffers, this._pool, this._maxSize, this._minRetainSize);
        }

        @Override
        public long space() {
            return this.maxSize() - this.size();
        }

        @Override
        public boolean isFull() {
            return this.size() >= this.maxSize();
        }

        @Override
        public RetainableByteBuffer copy() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("copy {}", (Object)this);
            }
            this.checkNotReleased();
            ArrayList<RetainableByteBuffer> buffers = new ArrayList<RetainableByteBuffer>(this._buffers.size());
            for (RetainableByteBuffer rbb : this._buffers) {
                buffers.add(rbb.copy());
            }
            return new DynamicCapacity(buffers, this._pool, this._maxSize, this._minRetainSize);
        }

        @Override
        public int remaining() {
            long size = this.size();
            return size > Integer.MAX_VALUE ? Integer.MAX_VALUE : Math.toIntExact(size);
        }

        @Override
        public long size() {
            this.checkNotReleased();
            long length = 0L;
            for (RetainableByteBuffer buffer : this._buffers) {
                length += (long)buffer.remaining();
            }
            return length;
        }

        @Override
        public int capacity() {
            long maxSize = this.maxSize();
            return maxSize > Integer.MAX_VALUE ? Integer.MAX_VALUE : Math.toIntExact(maxSize);
        }

        @Override
        public long maxSize() {
            return this._maxSize;
        }

        @Override
        public boolean release() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("release {}", (Object)this);
            }
            this.checkNotReleased();
            if (super.release()) {
                for (RetainableByteBuffer buffer : this._buffers) {
                    buffer.release();
                }
                this._buffers.clear();
                this._aggregate = null;
                return true;
            }
            return false;
        }

        @Override
        public boolean releaseAndRemove() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("release {}", (Object)this);
            }
            if (super.release()) {
                for (RetainableByteBuffer buffer : this._buffers) {
                    buffer.releaseAndRemove();
                }
                this._buffers.clear();
                this._aggregate = null;
                return true;
            }
            return false;
        }

        @Override
        public void clear() {
            if (LOG.isDebugEnabled()) {
                LOG.debug("clear {}", (Object)this);
            }
            this.checkNotReleased();
            if (this._buffers.isEmpty()) {
                return;
            }
            this._aggregate = null;
            for (RetainableByteBuffer rbb : this._buffers) {
                rbb.release();
            }
            this._buffers.clear();
        }

        @Override
        public boolean append(ByteBuffer bytes) {
            Mutable mutable;
            RetainableByteBuffer retainableByteBuffer;
            long size;
            long space;
            boolean existing;
            if (LOG.isDebugEnabled()) {
                LOG.debug("append BB {} <- {}", (Object)this, (Object)BufferUtil.toDetailString(bytes));
            }
            this.checkNotReleased();
            if (this.isRetained()) {
                throw new IllegalStateException("Cannot append to a retained instance");
            }
            if (bytes == null) {
                return true;
            }
            int length = bytes.remaining();
            if (length == 0) {
                return true;
            }
            boolean bl = existing = this._aggregate != null;
            if (existing) {
                if (BufferUtil.append(this._aggregate.getByteBuffer(), bytes) == length) {
                    return true;
                }
                this._aggregate = null;
            }
            if ((space = this._maxSize - (size = this.size())) <= 0L) {
                return false;
            }
            if (!existing && !this._buffers.isEmpty() && (retainableByteBuffer = this._buffers.get(this._buffers.size() - 1)) instanceof Mutable && (mutable = (Mutable)retainableByteBuffer).isMutable() && mutable.space() >= (long)length && !mutable.isRetained()) {
                this._aggregate = mutable;
                this.checkAggregateLimit(space);
            } else {
                int aggregateSize = this._pool.getSize();
                if (aggregateSize == 0 && this._buffers.isEmpty() && this._maxSize < Integer.MAX_VALUE) {
                    aggregateSize = (int)this._maxSize;
                }
                if ((long)(aggregateSize = Math.max(length, aggregateSize)) > space) {
                    aggregateSize = (int)space;
                }
                this._aggregate = this._pool.acquire(aggregateSize, this._pool.isDirect());
                this.checkAggregateLimit(space);
                this._buffers.add(this._aggregate);
            }
            return this._aggregate.append(bytes);
        }

        private void checkAggregateLimit(long space) {
            if ((long)this._aggregate.capacity() > space) {
                ByteBuffer byteBuffer = this._aggregate.getByteBuffer();
                int limit = byteBuffer.limit();
                byteBuffer.limit(limit + Math.toIntExact(space));
                byteBuffer = byteBuffer.slice();
                byteBuffer.limit(limit);
                this._aggregate = RetainableByteBuffer.wrap(byteBuffer, this._aggregate);
            }
        }

        private boolean shouldAggregate(RetainableByteBuffer buffer, long size) {
            if (this._minRetainSize > 0) {
                return size < (long)this._minRetainSize;
            }
            if (this._minRetainSize == -1) {
                if (this._aggregate != null && size < 128L) {
                    return true;
                }
                if (buffer instanceof FixedCapacity) {
                    return size < (long)(buffer.capacity() / 64);
                }
                return size < 128L;
            }
            return false;
        }

        @Override
        public boolean append(RetainableByteBuffer retainableBytes) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("append RBB {} {}", (Object)this, (Object)retainableBytes);
            }
            this.checkNotReleased();
            if (this.isRetained()) {
                throw new IllegalStateException("Cannot append to a retained instance");
            }
            if (retainableBytes instanceof DynamicCapacity) {
                DynamicCapacity dynamicCapacity = (DynamicCapacity)retainableBytes;
                Iterator<RetainableByteBuffer> i = dynamicCapacity._buffers.iterator();
                while (i.hasNext()) {
                    RetainableByteBuffer buffer = i.next();
                    if (!this.append(buffer)) {
                        return false;
                    }
                    buffer.release();
                    i.remove();
                }
                return true;
            }
            if (retainableBytes == null) {
                return true;
            }
            long length = retainableBytes.remaining();
            if (length == 0L) {
                return true;
            }
            if (this._aggregate != null && this._aggregate.space() >= length && length * 100L < retainableBytes.maxSize()) {
                return this._aggregate.append(retainableBytes.getByteBuffer());
            }
            if (this.shouldAggregate(retainableBytes, length)) {
                return this.append(retainableBytes.getByteBuffer());
            }
            this._aggregate = null;
            long space = this._maxSize - this.size();
            if (length <= space) {
                this._buffers.add(retainableBytes.slice());
                retainableBytes.skip(length);
                return true;
            }
            if (space == 0L) {
                return false;
            }
            length = space;
            this._buffers.add(retainableBytes.slice(length));
            retainableBytes.skip(length);
            return false;
        }

        @Override
        public Mutable add(ByteBuffer bytes) throws ReadOnlyBufferException, BufferOverflowException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("add BB {} <- {}", (Object)this, (Object)BufferUtil.toDetailString(bytes));
            }
            this.checkNotReleased();
            this.add(RetainableByteBuffer.wrap(bytes));
            return this;
        }

        @Override
        public Mutable add(RetainableByteBuffer bytes) throws ReadOnlyBufferException, BufferOverflowException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("add RBB {} <- {}", (Object)this, (Object)bytes);
            }
            this.checkNotReleased();
            long size = this.size();
            long space = this._maxSize - size;
            long length = bytes.size();
            if (space < length) {
                throw new BufferOverflowException();
            }
            if (this.shouldAggregate(bytes, length) && this.append(bytes)) {
                bytes.release();
                return this;
            }
            this._buffers.add(bytes);
            this._aggregate = null;
            return this;
        }

        @Override
        public Mutable put(byte b) {
            this.checkNotReleased();
            this.ensure(1).put(b);
            return this;
        }

        @Override
        public Mutable put(long index, byte b) {
            this.checkNotReleased();
            for (RetainableByteBuffer buffer : this._buffers) {
                long size = buffer.size();
                if (index < size) {
                    buffer.asMutable().put(index, b);
                    return this;
                }
                index -= size;
            }
            throw new IndexOutOfBoundsException();
        }

        @Override
        public Mutable putShort(short s) {
            this.checkNotReleased();
            this.ensure(2).putShort(s);
            return this;
        }

        @Override
        public Mutable putInt(int i) {
            this.checkNotReleased();
            this.ensure(4).putInt(i);
            return this;
        }

        @Override
        public Mutable putLong(long l) {
            this.checkNotReleased();
            this.ensure(8).putLong(l);
            return this;
        }

        @Override
        public Mutable put(byte[] bytes, int offset, int length) {
            long space;
            this.checkNotReleased();
            if (length >= 16 && this._aggregate != null && (long)length > (space = this._aggregate.space()) && (long)(length / 2) <= space) {
                int s = (int)space;
                this._aggregate.put(bytes, offset, s);
                offset += s;
                length -= s;
            }
            this.ensure(length).put(bytes, offset, length);
            return this;
        }

        private Mutable ensure(int needed) throws BufferOverflowException {
            int aggregateSize;
            Mutable mutable;
            RetainableByteBuffer retainableByteBuffer;
            long size;
            long space;
            if (LOG.isDebugEnabled()) {
                LOG.debug("ensure {} {}", (Object)this, (Object)needed);
            }
            if ((space = this._maxSize - (size = this.size())) < (long)needed) {
                throw new BufferOverflowException();
            }
            if (this._aggregate != null) {
                if (this._aggregate.space() >= (long)needed) {
                    return this._aggregate;
                }
            } else if (!this._buffers.isEmpty() && (retainableByteBuffer = this._buffers.get(this._buffers.size() - 1)) instanceof Mutable && (mutable = (Mutable)retainableByteBuffer).isMutable() && mutable.space() >= (long)needed && !mutable.isRetained()) {
                this._aggregate = mutable;
                return this._aggregate;
            }
            this._aggregate = (aggregateSize = this._pool.getSize()) == 0 && this._buffers.isEmpty() && this._maxSize < Integer.MAX_VALUE ? this._pool.acquire(Math.toIntExact(this._maxSize)) : (needed > aggregateSize ? this._pool.acquire(needed) : this._pool.acquire());
            this.checkAggregateLimit(space);
            this._buffers.add(this._aggregate);
            return this._aggregate;
        }

        @Override
        public boolean appendTo(ByteBuffer to) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("appendTo BB {} -> {}", (Object)this, (Object)BufferUtil.toDetailString(to));
            }
            this.checkNotReleased();
            this._aggregate = null;
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = (RetainableByteBuffer)i.next();
                if (!buffer.appendTo(to)) {
                    return false;
                }
                buffer.release();
                i.remove();
            }
            return true;
        }

        @Override
        public boolean appendTo(RetainableByteBuffer to) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("appendTo RBB {} -> {}", (Object)this, (Object)to);
            }
            this.checkNotReleased();
            this._aggregate = null;
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = (RetainableByteBuffer)i.next();
                if (!buffer.appendTo(to)) {
                    return false;
                }
                buffer.release();
                i.remove();
            }
            return true;
        }

        @Override
        public void putTo(ByteBuffer toInfillMode) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("putTo BB {} -> {}", (Object)this, (Object)toInfillMode);
            }
            this.checkNotReleased();
            this._aggregate = null;
            ListIterator<RetainableByteBuffer> i = this._buffers.listIterator();
            while (i.hasNext()) {
                RetainableByteBuffer buffer = (RetainableByteBuffer)i.next();
                buffer.putTo(toInfillMode);
                buffer.release();
                i.remove();
            }
        }

        @Override
        public void writeTo(final Content.Sink sink, final boolean last, Callback callback) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("writeTo {} -> {} {} {}", this, sink, last, callback);
            }
            this.checkNotReleased();
            this._aggregate = null;
            switch (this._buffers.size()) {
                case 0: {
                    callback.succeeded();
                    break;
                }
                case 1: {
                    RetainableByteBuffer buffer = this._buffers.get(0);
                    buffer.writeTo(sink, last, Callback.from(this::clear, callback));
                    break;
                }
                default: {
                    if (!last && sink instanceof EndPoint) {
                        EndPoint endPoint = (EndPoint)sink;
                        ByteBuffer[] buffers = new ByteBuffer[this._buffers.size()];
                        int i = 0;
                        for (RetainableByteBuffer rbb : this._buffers) {
                            buffers[i++] = rbb.getByteBuffer();
                        }
                        endPoint.write(Callback.from(this::clear, callback), buffers);
                        return;
                    }
                    new IteratingNestedCallback(this, callback){
                        int _index;
                        RetainableByteBuffer _buffer;
                        boolean _lastWritten;
                        final /* synthetic */ DynamicCapacity this$0;
                        {
                            this.this$0 = this$0;
                            super(arg0);
                        }

                        @Override
                        protected IteratingCallback.Action process() {
                            if (this._index < this.this$0._buffers.size()) {
                                this._buffer = this.this$0._buffers.get(this._index++);
                                this._lastWritten = last && this._index == this.this$0._buffers.size();
                                this._buffer.writeTo(sink, this._lastWritten, this);
                                return IteratingCallback.Action.SCHEDULED;
                            }
                            if (last && !this._lastWritten) {
                                this._buffer = null;
                                this._lastWritten = true;
                                sink.write(true, BufferUtil.EMPTY_BUFFER, this);
                                return IteratingCallback.Action.SCHEDULED;
                            }
                            this.this$0._buffers.clear();
                            return IteratingCallback.Action.SUCCEEDED;
                        }

                        @Override
                        protected void onSuccess() {
                            this._buffer = Retainable.release(this._buffer);
                        }

                        @Override
                        protected void onCompleteFailure(Throwable x) {
                            this._buffer = Retainable.release(this._buffer);
                        }
                    }.iterate();
                }
            }
        }

        @Override
        protected void addExtraStringInfo(StringBuilder builder) {
            super.addExtraStringInfo(builder);
            builder.append(",pool=");
            builder.append(this._pool);
            builder.append(",minRetain=");
            builder.append(this._minRetainSize);
            builder.append(",buffers=");
            builder.append(this._buffers.size());
        }

        @Override
        protected void addValueString(StringBuilder builder) {
            for (RetainableByteBuffer buffer : this._buffers) {
                builder.append('@');
                builder.append(Integer.toHexString(System.identityHashCode(buffer)));
                if (buffer instanceof Abstract) {
                    Abstract abstractBuffer = (Abstract)buffer;
                    builder.append("/r=");
                    builder.append(abstractBuffer.getRetained());
                    abstractBuffer.addValueString(builder);
                    continue;
                }
                builder.append("???");
            }
        }

        @Override
        protected void addValueMarker(StringBuilder builder, boolean beginning) {
            if (beginning) {
                builder.append("<<").append(this._buffers.size()).append('<');
            } else {
                builder.append(">>>");
            }
        }
    }

    public static class Pooled
    extends FixedCapacity {
        private final ByteBufferPool _pool;

        public Pooled(ByteBufferPool pool, ByteBuffer byteBuffer) {
            super(byteBuffer);
            this._pool = pool;
        }

        protected Pooled(ByteBufferPool pool, ByteBuffer byteBuffer, Retainable retainable) {
            super(byteBuffer, retainable);
            this._pool = pool;
        }

        @Override
        public boolean releaseAndRemove() {
            return this._pool.releaseAndRemove(this);
        }

        @Override
        public RetainableByteBuffer slice(long length) {
            int size = this.remaining();
            ByteBuffer byteBuffer = this.getByteBuffer();
            int limit = byteBuffer.limit();
            byteBuffer.limit(byteBuffer.position() + Math.toIntExact(Math.min(length, (long)size)));
            ByteBuffer slice = byteBuffer.slice();
            byteBuffer.limit(limit);
            if (length > (long)size) {
                slice.limit(size);
            }
            if (!this.canRetain()) {
                return new NonRetainableByteBuffer(slice);
            }
            this.retain();
            return new Pooled(this._pool, slice, this);
        }

        @Override
        public RetainableByteBuffer copy() {
            Mutable copy = this._pool.acquire(this.remaining(), this.isDirect());
            copy.asMutable().append(this.getByteBuffer().slice());
            return copy;
        }
    }

    public static abstract class Abstract
    extends Retainable.Wrapper
    implements Mutable {
        public Abstract() {
            this(new Retainable.ReferenceCounter());
        }

        public Abstract(Retainable retainable) {
            super(retainable);
        }

        @Override
        public String toString() {
            StringBuilder builder = new StringBuilder();
            this.addStringInfo(builder);
            return builder.toString();
        }

        @Override
        public String toDetailString() {
            StringBuilder builder = new StringBuilder();
            this.addStringInfo(builder);
            builder.append("={");
            this.addValueString(builder);
            builder.append("}");
            return builder.toString();
        }

        protected void addStringInfo(StringBuilder builder) {
            builder.append(this.getClass().getSimpleName());
            builder.append("@");
            builder.append(Integer.toHexString(System.identityHashCode(this)));
            builder.append("[");
            builder.append(this.size());
            builder.append("/");
            builder.append(this.maxSize());
            builder.append(",d=");
            builder.append(this.isDirect());
            this.addExtraStringInfo(builder);
            builder.append(",r=");
            builder.append(this.getRetained());
            builder.append("]");
        }

        protected void addExtraStringInfo(StringBuilder builder) {
        }

        protected void addValueString(StringBuilder builder) {
            this.addValueMarker(builder, true);
            long size = this.size();
            if (size <= 48L) {
                int i = 0;
                while ((long)i < size) {
                    BufferUtil.appendDebugByte(builder, this.get(i));
                    ++i;
                }
            } else {
                int i;
                for (i = 0; i < 24; ++i) {
                    BufferUtil.appendDebugByte(builder, this.get(i));
                }
                builder.append("...");
                for (i = 0; i < 24; ++i) {
                    BufferUtil.appendDebugByte(builder, this.get(size - 24L + (long)i));
                }
            }
            this.addValueMarker(builder, false);
        }

        protected void addValueMarker(StringBuilder builder, boolean beginning) {
            builder.append(beginning ? "<<<" : ">>>");
        }
    }

    public static class Wrapper
    extends Retainable.Wrapper
    implements Mutable {
        public Wrapper(RetainableByteBuffer wrapped) {
            super(wrapped);
        }

        @Override
        public RetainableByteBuffer getWrapped() {
            return (RetainableByteBuffer)super.getWrapped();
        }

        @Override
        public boolean releaseAndRemove() {
            return this.getWrapped().releaseAndRemove();
        }

        @Override
        public boolean isRetained() {
            return this.getWrapped().isRetained();
        }

        @Override
        public ByteBuffer getByteBuffer() {
            return this.getWrapped().getByteBuffer();
        }

        @Override
        public boolean isDirect() {
            return this.getWrapped().isDirect();
        }

        @Override
        public int remaining() {
            return this.getWrapped().remaining();
        }

        @Override
        public boolean hasRemaining() {
            return this.getWrapped().hasRemaining();
        }

        @Override
        public int capacity() {
            return this.getWrapped().capacity();
        }

        @Override
        public void clear() {
            this.getWrapped().clear();
        }

        @Override
        public String toString() {
            return "%s@%x{%s}".formatted(this.getClass().getSimpleName(), this.hashCode(), this.getWrapped().toString());
        }

        @Override
        public boolean appendTo(ByteBuffer buffer) {
            return this.getWrapped().appendTo(buffer);
        }

        @Override
        public boolean appendTo(RetainableByteBuffer buffer) {
            return this.getWrapped().appendTo(buffer);
        }

        @Override
        public RetainableByteBuffer copy() {
            return this.getWrapped().copy();
        }

        @Override
        public RetainableByteBuffer slice(long length) {
            return this.getWrapped().slice(length);
        }

        @Override
        public byte get(long index) {
            return this.getWrapped().get(index);
        }

        @Override
        public int get(byte[] bytes, int offset, int length) {
            return this.getWrapped().get(bytes, offset, length);
        }

        @Override
        public boolean isEmpty() {
            return this.getWrapped().isEmpty();
        }

        @Override
        public void putTo(ByteBuffer toInfillMode) throws BufferOverflowException {
            this.getWrapped().putTo(toInfillMode);
        }

        @Override
        public long skip(long length) {
            return this.getWrapped().skip(length);
        }

        @Override
        public RetainableByteBuffer slice() {
            return this.getWrapped().slice();
        }

        @Override
        public void writeTo(Content.Sink sink, boolean last, Callback callback) {
            this.getWrapped().writeTo(sink, last, callback);
        }

        @Override
        public Mutable asMutable() {
            return this;
        }

        @Override
        public boolean isFull() {
            return this.getWrapped().asMutable().isFull();
        }

        @Override
        public long space() {
            return this.getWrapped().asMutable().space();
        }

        @Override
        public boolean append(ByteBuffer bytes) throws ReadOnlyBufferException {
            return this.getWrapped().asMutable().append(bytes);
        }

        @Override
        public boolean append(RetainableByteBuffer bytes) throws ReadOnlyBufferException {
            return this.getWrapped().asMutable().append(bytes);
        }

        @Override
        public Mutable add(ByteBuffer bytes) throws ReadOnlyBufferException, BufferOverflowException {
            this.getWrapped().asMutable().add(bytes);
            return this;
        }

        @Override
        public Mutable add(RetainableByteBuffer bytes) throws ReadOnlyBufferException, BufferOverflowException {
            this.getWrapped().asMutable().add(bytes);
            return this;
        }

        @Override
        public Mutable put(byte b) {
            this.getWrapped().asMutable().put(b);
            return this;
        }

        @Override
        public Mutable put(long index, byte b) {
            this.getWrapped().asMutable().put(index, b);
            return this;
        }

        @Override
        public Mutable putShort(short s) {
            this.getWrapped().asMutable().putShort(s);
            return this;
        }

        @Override
        public Mutable putInt(int i) {
            this.getWrapped().asMutable().putInt(i);
            return this;
        }

        @Override
        public Mutable putLong(long l) {
            this.getWrapped().asMutable().putLong(l);
            return this;
        }

        @Override
        public Mutable put(byte[] bytes, int offset, int length) {
            this.getWrapped().asMutable().put(bytes, offset, length);
            return this;
        }

        @Override
        public String toDetailString() {
            return this.getWrapped().toDetailString();
        }
    }
}

