/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http.content;

import java.io.IOException;
import java.time.Instant;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.eclipse.jetty.http.CompressedContentFormat;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.http.content.HttpContent;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.Content;
import org.eclipse.jetty.io.IOResources;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.resource.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachingHttpContentFactory
implements HttpContent.Factory {
    private static final Logger LOG = LoggerFactory.getLogger(CachingHttpContentFactory.class);
    private static final int DEFAULT_MAX_CACHED_FILE_SIZE = 0x8000000;
    private static final int DEFAULT_MAX_CACHED_FILES = 2048;
    private static final long DEFAULT_MAX_CACHE_SIZE = 0x10000000L;
    private final HttpContent.Factory _authority;
    private final ConcurrentHashMap<String, CachingHttpContent> _cache = new ConcurrentHashMap();
    private final AtomicLong _cachedSize = new AtomicLong();
    private final ByteBufferPool.Sized _bufferPool;
    private int _maxCachedFileSize = 0x8000000;
    private int _maxCachedFiles = 2048;
    private long _maxCacheSize = 0x10000000L;

    public CachingHttpContentFactory(HttpContent.Factory authority, ByteBufferPool.Sized bufferPool) {
        this._authority = authority;
        this._bufferPool = bufferPool != null ? bufferPool : ByteBufferPool.SIZED_NON_POOLING;
    }

    protected ConcurrentMap<String, CachingHttpContent> getCache() {
        return this._cache;
    }

    public long getCachedSize() {
        return this._cachedSize.get();
    }

    public int getCachedFiles() {
        return this._cache.size();
    }

    public int getMaxCachedFileSize() {
        return this._maxCachedFileSize;
    }

    public void setMaxCachedFileSize(int maxCachedFileSize) {
        this._maxCachedFileSize = maxCachedFileSize;
        this.shrinkCache();
    }

    public long getMaxCacheSize() {
        return this._maxCacheSize;
    }

    public void setMaxCacheSize(long maxCacheSize) {
        this._maxCacheSize = maxCacheSize;
        this.shrinkCache();
    }

    public int getMaxCachedFiles() {
        return this._maxCachedFiles;
    }

    public void setMaxCachedFiles(int maxCachedFiles) {
        this._maxCachedFiles = maxCachedFiles;
        this.shrinkCache();
    }

    private void shrinkCache() {
        int numCacheEntries = this._cache.size();
        while (numCacheEntries > 0 && (numCacheEntries > this._maxCachedFiles || this._cachedSize.get() > this._maxCacheSize)) {
            TreeSet<CachingHttpContent> sorted = new TreeSet<CachingHttpContent>((c1, c2) -> {
                long delta = NanoTime.elapsed(c2.getLastAccessedNanos(), c1.getLastAccessedNanos());
                if (delta != 0L) {
                    return delta < 0L ? -1 : 1;
                }
                delta = c1.getContentLengthValue() - c2.getContentLengthValue();
                if (delta != 0L) {
                    return delta < 0L ? -1 : 1;
                }
                return c1.getKey().compareTo(c2.getKey());
            });
            sorted.addAll(this._cache.values());
            for (CachingHttpContent content : sorted) {
                if (this._cache.size() <= this._maxCachedFiles && this._cachedSize.get() <= this._maxCacheSize) break;
                this.removeFromCache(content);
            }
            numCacheEntries = this._cache.size();
        }
    }

    protected void removeFromCache(CachingHttpContent content) {
        CachingHttpContent removed = this._cache.remove(content.getKey());
        if (removed != null) {
            long contentLengthValue = removed.getContentLengthValue();
            removed.release();
            this._cachedSize.addAndGet(-contentLengthValue);
        }
    }

    public void flushCache() {
        for (CachingHttpContent content : this._cache.values()) {
            this.removeFromCache(content);
        }
    }

    protected boolean isCacheable(HttpContent httpContent) {
        if (httpContent == null) {
            return true;
        }
        if (httpContent.getResource().isDirectory()) {
            return false;
        }
        if (this._maxCachedFiles <= 0) {
            return false;
        }
        long len = httpContent.getContentLengthValue();
        return len <= (long)this._maxCachedFileSize && len <= this._maxCacheSize;
    }

    @Override
    public HttpContent getContent(String path) throws IOException {
        HttpContent httpContent;
        CachingHttpContent cachingHttpContent = this._cache.get(path);
        if (cachingHttpContent != null) {
            cachingHttpContent.setLastAccessedNanos(NanoTime.now());
            if (cachingHttpContent.isValid()) {
                return cachingHttpContent instanceof NotFoundHttpContent ? null : cachingHttpContent;
            }
            this.removeFromCache(cachingHttpContent);
        }
        if (!this.isCacheable(httpContent = this._authority.getContent(path))) {
            return httpContent;
        }
        AtomicBoolean added = new AtomicBoolean();
        cachingHttpContent = this._cache.computeIfAbsent(path, key -> {
            try {
                CachingHttpContent cachingContent = httpContent == null ? this.newNotFoundContent((String)key) : this.newCachedContent((String)key, httpContent);
                long contentLengthValue = cachingContent.getContentLengthValue();
                if (contentLengthValue < 0L) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Content at path '{}' with unknown length is not cacheable: {}", (Object)path, (Object)httpContent);
                    }
                    return null;
                }
                added.set(true);
                this._cachedSize.addAndGet(contentLengthValue);
                return cachingContent;
            }
            catch (Throwable x) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Content at path '{}' is not cacheable: {}", path, httpContent, x);
                }
                return null;
            }
        });
        if (added.get()) {
            this.shrinkCache();
        }
        return cachingHttpContent instanceof NotFoundHttpContent ? null : cachingHttpContent;
    }

    protected CachingHttpContent newCachedContent(String p, HttpContent httpContent) {
        return new CachedHttpContent(p, httpContent);
    }

    protected CachingHttpContent newNotFoundContent(String p) {
        return new NotFoundHttpContent(p);
    }

    protected static interface CachingHttpContent
    extends HttpContent {
        public long getLastAccessedNanos();

        public void setLastAccessedNanos(long var1);

        public String getKey();

        public boolean isValid();

        public void release();
    }

    protected static class NotFoundHttpContent
    implements CachingHttpContent {
        private volatile long _lastAccessed;
        private final String _key;

        public NotFoundHttpContent(String key) {
            this._key = key;
            this._lastAccessed = NanoTime.now();
        }

        @Override
        public String getKey() {
            return this._key;
        }

        @Override
        public long getLastAccessedNanos() {
            return this._lastAccessed;
        }

        @Override
        public void setLastAccessedNanos(long nanosTime) {
            this._lastAccessed = nanosTime;
        }

        @Override
        public HttpField getContentType() {
            return null;
        }

        @Override
        public String getCharacterEncoding() {
            return null;
        }

        @Override
        public MimeTypes.Type getMimeType() {
            return null;
        }

        @Override
        public HttpField getContentEncoding() {
            return null;
        }

        @Override
        public HttpField getContentLength() {
            return null;
        }

        @Override
        public long getContentLengthValue() {
            return 0L;
        }

        @Override
        public Instant getLastModifiedInstant() {
            return null;
        }

        @Override
        public HttpField getLastModified() {
            return null;
        }

        @Override
        public HttpField getETag() {
            return null;
        }

        @Override
        public Resource getResource() {
            return null;
        }

        @Override
        public void writeTo(Content.Sink sink, long offset, long length, Callback callback) {
            sink.write(true, BufferUtil.EMPTY_BUFFER, callback);
        }

        @Override
        public Set<CompressedContentFormat> getPreCompressedContentFormats() {
            return null;
        }

        @Override
        public void release() {
        }

        @Override
        public boolean isValid() {
            return true;
        }
    }

    protected class CachedHttpContent
    extends HttpContent.Wrapper
    implements CachingHttpContent {
        private final RetainableByteBuffer _buffer;
        private final String _cacheKey;
        private final HttpField _etagField;
        private volatile long _lastAccessed;
        private final Set<CompressedContentFormat> _compressedFormats;
        private final String _characterEncoding;
        private final MimeTypes.Type _mimeType;
        private final HttpField _contentLength;
        private final Instant _lastModifiedInstant;
        private final HttpField _lastModified;

        public CachedHttpContent(String key, HttpContent httpContent) {
            super(httpContent);
            this._cacheKey = key;
            HttpField etagField = httpContent.getETag();
            String eTagValue = httpContent.getETagValue();
            if (StringUtil.isNotBlank(eTagValue)) {
                etagField = new PreEncodedHttpField(HttpHeader.ETAG, eTagValue);
            }
            this._etagField = etagField;
            this._contentLength = httpContent.getContentLength();
            long contentLengthValue = httpContent.getContentLengthValue();
            if (contentLengthValue < 0L) {
                throw new IllegalArgumentException("Resource length is unknown");
            }
            if (contentLengthValue > (long)CachingHttpContentFactory.this._maxCachedFileSize) {
                throw new IllegalArgumentException("Resource is too large: length " + contentLengthValue + " > " + CachingHttpContentFactory.this._maxCachedFileSize);
            }
            this._buffer = IOResources.toRetainableByteBuffer(httpContent.getResource(), CachingHttpContentFactory.this._bufferPool);
            this._characterEncoding = httpContent.getCharacterEncoding();
            this._compressedFormats = httpContent.getPreCompressedContentFormats();
            this._mimeType = httpContent.getMimeType();
            this._lastModifiedInstant = httpContent.getLastModifiedInstant();
            this._lastModified = httpContent.getLastModified();
            this._lastAccessed = NanoTime.now();
        }

        @Override
        public long getLastAccessedNanos() {
            return this._lastAccessed;
        }

        @Override
        public void setLastAccessedNanos(long nanosTime) {
            this._lastAccessed = nanosTime;
        }

        @Override
        public String getKey() {
            return this._cacheKey;
        }

        @Override
        public void writeTo(Content.Sink sink, long offset, long length, Callback callback) {
            try {
                this._buffer.retain();
                sink.write(true, BufferUtil.slice(this._buffer.getByteBuffer(), (int)offset, (int)length), Callback.from(this._buffer::release, callback));
            }
            catch (Throwable x) {
                this._buffer.release();
                callback.failed(x);
            }
        }

        @Override
        public void release() {
            this._buffer.release();
        }

        @Override
        public Set<CompressedContentFormat> getPreCompressedContentFormats() {
            return this._compressedFormats;
        }

        @Override
        public HttpField getETag() {
            return this._etagField;
        }

        @Override
        public String getCharacterEncoding() {
            return this._characterEncoding;
        }

        @Override
        public MimeTypes.Type getMimeType() {
            return this._mimeType;
        }

        @Override
        public HttpField getContentLength() {
            return this._contentLength;
        }

        @Override
        public long getContentLengthValue() {
            return this._buffer.remaining();
        }

        @Override
        public Instant getLastModifiedInstant() {
            return this._lastModifiedInstant;
        }

        @Override
        public HttpField getLastModified() {
            return this._lastModified;
        }

        @Override
        public boolean isValid() {
            return true;
        }
    }
}

