/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.service.auth;

import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.impl.DefaultJwtBuilder;
import io.jsonwebtoken.impl.compression.GzipCompressionCodec;
import io.jsonwebtoken.jackson.io.JacksonSerializer;
import io.jsonwebtoken.security.Keys;
import io.jsonwebtoken.security.SecurityException;
import io.quarkus.runtime.annotations.RegisterForReflection;
import io.vertx.core.Vertx;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.crypto.SecretKey;
import org.eclipse.hono.auth.Authorities;
import org.eclipse.hono.service.auth.AuthTokenFactory;
import org.eclipse.hono.service.auth.JwtSupport;
import org.eclipse.hono.service.auth.SignatureSupportingConfigProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@RegisterForReflection(targets={DefaultJwtBuilder.class, JacksonSerializer.class, GzipCompressionCodec.class})
public final class JjwtBasedAuthTokenFactory
extends JwtSupport
implements AuthTokenFactory {
    private static final Logger LOG = LoggerFactory.getLogger(JjwtBasedAuthTokenFactory.class);
    private static final String PROPERTY_JWK_ALG = "alg";
    private static final String PROPERTY_JWK_CRV = "crv";
    private static final String PROPERTY_JWK_E = "e";
    private static final String PROPERTY_JWK_KEY_OPS = "key_ops";
    private static final String PROPERTY_JWK_K = "k";
    private static final String PROPERTY_JWK_KID = "kid";
    private static final String PROPERTY_JWK_KTY = "kty";
    private static final String PROPERTY_JWK_N = "n";
    private static final String PROPERTY_JWK_USE = "use";
    private static final String PROPERTY_JWK_X = "x";
    private static final String PROPERTY_JWK_Y = "y";
    private static final Map<String, String> CURVE_OID_TO_NIST_NAME = Map.of("1.2.840.10045.3.1.7", "P-256", "1.3.132.0.34", "P-384", "1.3.132.0.35", "P-521");
    private final JsonObject jwkSet = new JsonObject();
    private final SignatureSupportingConfigProperties config;
    private final Duration tokenLifetime;
    private final String signingKeyId;

    public JjwtBasedAuthTokenFactory(Vertx vertx, SignatureSupportingConfigProperties config) {
        super(vertx);
        Objects.requireNonNull(config);
        try {
            if (config.getSharedSecret() != null) {
                byte[] bytes = JjwtBasedAuthTokenFactory.getBytes(config.getSharedSecret());
                SecretKey secretKey = Keys.hmacShaKeyFor((byte[])bytes);
                this.signingKeyId = this.addSecretKey(secretKey);
                LOG.info("using shared secret [{} bytes] for signing/validating tokens", (Object)bytes.length);
            } else if (config.getKeyPath() != null && config.getCertPath() != null) {
                this.signingKeyId = this.addPrivateKey(config.getKeyPath(), config.getCertPath());
                LOG.info("using key pair [private: {}, cert: {}] for signing/verifying tokens", (Object)config.getKeyPath(), (Object)config.getCertPath());
            } else {
                throw new IllegalArgumentException("configuration does not specify any key material for signing tokens");
            }
            this.tokenLifetime = Duration.ofSeconds(config.getTokenExpiration());
            LOG.info("using token lifetime of {} seconds", (Object)this.tokenLifetime.getSeconds());
            this.config = config;
            this.createJwk();
        }
        catch (SecurityException e) {
            throw new IllegalArgumentException("failed to create factory for configured key material", e);
        }
    }

    @Override
    public Duration getTokenLifetime() {
        return this.tokenLifetime;
    }

    private void createJwk() {
        JsonArray keyArray = this.getValidatingKeys().stream().map(entry -> {
            JsonObject jwk = new JsonObject();
            String keyId = (String)entry.getKey();
            JwtSupport.KeySpec keySpec = (JwtSupport.KeySpec)entry.getValue();
            jwk.put(PROPERTY_JWK_USE, (Object)"sig");
            jwk.put(PROPERTY_JWK_KEY_OPS, (Object)"verify");
            Key patt5564$temp = keySpec.key;
            if (patt5564$temp instanceof SecretKey) {
                SecretKey secretKey = (SecretKey)patt5564$temp;
                jwk.put(PROPERTY_JWK_KTY, (Object)"oct");
                jwk.put(PROPERTY_JWK_K, (Object)keySpec.key.getEncoded());
            } else {
                this.addPublicKey(jwk, keySpec);
            }
            jwk.put(PROPERTY_JWK_KID, (Object)keyId);
            jwk.put(PROPERTY_JWK_ALG, (Object)keySpec.algorithm.getValue());
            return jwk;
        }).collect(JsonArray::new, JsonArray::add, JsonArray::addAll);
        this.jwkSet.put("keys", (Object)keyArray);
        if (LOG.isInfoEnabled()) {
            LOG.info("successfully created JWK set:{}{}", (Object)System.lineSeparator(), (Object)this.jwkSet.encodePrettily());
        }
    }

    private void addPublicKey(JsonObject jwk, JwtSupport.KeySpec keySpec) {
        Key key = keySpec.key;
        if (key instanceof RSAPublicKey) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)key;
            this.addRsaPublicKey(jwk, rsaPublicKey);
        } else {
            key = keySpec.key;
            if (key instanceof ECPublicKey) {
                ECPublicKey ecPublicKey = (ECPublicKey)key;
                this.addEcPublicKey(jwk, ecPublicKey);
            } else {
                throw new IllegalArgumentException("unsupported key type [%s], must be RSA or EC".formatted(keySpec.key.getAlgorithm()));
            }
        }
        jwk.put(PROPERTY_JWK_KTY, (Object)keySpec.key.getAlgorithm());
    }

    private void addRsaPublicKey(JsonObject jwk, RSAPublicKey rsaPublicKey) {
        Base64.Encoder encoder = Base64.getUrlEncoder().withoutPadding();
        jwk.put(PROPERTY_JWK_N, (Object)encoder.encodeToString(rsaPublicKey.getModulus().toByteArray()));
        jwk.put(PROPERTY_JWK_E, (Object)encoder.encodeToString(rsaPublicKey.getPublicExponent().toByteArray()));
    }

    private String getNistCurveName(ECPublicKey ecPublicKey) throws GeneralSecurityException {
        AlgorithmParameters parameters = AlgorithmParameters.getInstance("EC");
        parameters.init(ecPublicKey.getParams());
        ECGenParameterSpec spec = parameters.getParameterSpec(ECGenParameterSpec.class);
        String oid = spec.getName();
        return Optional.ofNullable(CURVE_OID_TO_NIST_NAME.get(oid)).orElseThrow(() -> new IllegalArgumentException("unsupported curve [%s]".formatted(oid)));
    }

    private void addEcPublicKey(JsonObject jwk, ECPublicKey ecPublicKey) {
        try {
            jwk.put(PROPERTY_JWK_CRV, (Object)this.getNistCurveName(ecPublicKey));
            Base64.Encoder encoder = Base64.getUrlEncoder().withoutPadding();
            jwk.put(PROPERTY_JWK_X, (Object)encoder.encodeToString(ecPublicKey.getW().getAffineX().toByteArray()));
            jwk.put(PROPERTY_JWK_Y, (Object)encoder.encodeToString(ecPublicKey.getW().getAffineY().toByteArray()));
        }
        catch (GeneralSecurityException e) {
            throw new IllegalArgumentException("cannot serialize EC based public key", e);
        }
    }

    @Override
    public String createToken(String authorizationId, Authorities authorities) {
        Objects.requireNonNull(authorizationId);
        JwtSupport.KeySpec signingKeySpec = this.getSigningKey(this.signingKeyId);
        JwtBuilder builder = Jwts.builder().signWith(signingKeySpec.key, signingKeySpec.algorithm).setHeaderParam(PROPERTY_JWK_KID, (Object)this.signingKeyId).setIssuer(this.config.getIssuer()).setSubject(authorizationId).setExpiration(Date.from(Instant.now().plus(this.tokenLifetime)));
        Optional.ofNullable(authorities).map(Authorities::asMap).ifPresent(authMap -> authMap.forEach((arg_0, arg_1) -> ((JwtBuilder)builder).claim(arg_0, arg_1)));
        Optional.ofNullable(this.config.getAudience()).ifPresent(arg_0 -> ((JwtBuilder)builder).setAudience(arg_0));
        return builder.compact();
    }

    @Override
    public JsonObject getValidatingJwkSet() {
        return this.jwkSet;
    }
}

