/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.service;

import io.vertx.core.CompositeFuture;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.http.HttpServer;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.ext.healthchecks.HealthCheckHandler;
import io.vertx.ext.web.Router;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.eclipse.hono.config.ServerConfig;
import org.eclipse.hono.service.HealthCheckProvider;
import org.eclipse.hono.service.HealthCheckServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

public final class VertxBasedHealthCheckServer
implements HealthCheckServer {
    private static final Logger LOG = LoggerFactory.getLogger(VertxBasedHealthCheckServer.class);
    private static final String URI_LIVENESS_PROBE = "/liveness";
    private static final String URI_READINESS_PROBE = "/readiness";
    private static final int DEFAULT_PORT = 8088;
    private boolean bindSecureServerToLoopbackDeviceAllowed = false;
    private boolean bindInsecureServerToLoopbackDeviceAllowed = false;
    private HttpServer server;
    private HttpServer insecureServer;
    private final HealthCheckHandler readinessHandler;
    private final HealthCheckHandler livenessHandler;
    private final Vertx vertx;
    private final ServerConfig config;
    private final Router router;
    private final List<Handler<Router>> additionalResources = new ArrayList<Handler<Router>>();

    public VertxBasedHealthCheckServer(Vertx vertx, ServerConfig config) {
        this.vertx = Objects.requireNonNull(vertx);
        this.config = Objects.requireNonNull(config);
        this.readinessHandler = HealthCheckHandler.create((Vertx)this.vertx);
        this.livenessHandler = HealthCheckHandler.create((Vertx)this.vertx);
        this.router = Router.router((Vertx)this.vertx);
    }

    void setBindToLoopbackDeviceAllowed(boolean secureServerAllowed, boolean insecureServerAllowed) {
        this.bindSecureServerToLoopbackDeviceAllowed = secureServerAllowed;
        this.bindInsecureServerToLoopbackDeviceAllowed = insecureServerAllowed;
    }

    @Override
    public void registerHealthCheckResources(HealthCheckProvider serviceInstance) {
        serviceInstance.registerLivenessChecks(this.livenessHandler);
        serviceInstance.registerReadinessChecks(this.readinessHandler);
    }

    @Autowired(required=false)
    @Qualifier(value="healthchecks")
    public void setAdditionalResources(List<Handler<Router>> resourceProviders) {
        Objects.requireNonNull(resourceProviders);
        this.additionalResources.addAll(resourceProviders);
    }

    public Future<Void> start() {
        this.registerAdditionalResources();
        return CompositeFuture.all(this.bindSecureHttpServer(), this.bindInsecureHttpServer()).map(ok -> {
            if (this.insecureServer == null && this.server == null) {
                throw new IllegalStateException("neither secure nor insecure server has been started");
            }
            return null;
        }).recover(error -> {
            LOG.error("failed to start Health Check server", error);
            return Future.failedFuture((Throwable)error);
        });
    }

    private Future<Void> bindInsecureHttpServer() {
        Promise result = Promise.promise();
        if ("127.0.0.1".equals(this.config.getInsecurePortBindAddress())) {
            if (this.bindInsecureServerToLoopbackDeviceAllowed) {
                LOG.warn("insecure health checks HTTP server will bind to loopback device only");
            } else {
                LOG.info("won't start insecure health checks HTTP server: no bind address configured.");
                return Future.succeededFuture();
            }
        }
        HttpServerOptions options = new HttpServerOptions().setPort(this.config.getInsecurePort(8088)).setHost(this.config.getInsecurePortBindAddress());
        this.insecureServer = this.vertx.createHttpServer(options);
        this.router.get(URI_READINESS_PROBE).handler((Handler)this.readinessHandler);
        this.router.get(URI_LIVENESS_PROBE).handler((Handler)this.livenessHandler);
        this.insecureServer.requestHandler((Handler)this.router).listen(startAttempt -> {
            if (startAttempt.succeeded()) {
                LOG.info("successfully started insecure health checks HTTP server");
                LOG.info("readiness probe available at http://{}:{}{}", new Object[]{options.getHost(), this.insecureServer.actualPort(), URI_READINESS_PROBE});
                LOG.info("liveness probe available at http://{}:{}{}", new Object[]{options.getHost(), this.insecureServer.actualPort(), URI_LIVENESS_PROBE});
                result.complete();
            } else {
                LOG.warn("failed to start insecure health checks HTTP server: {}", (Object)startAttempt.cause().getMessage());
                result.fail(startAttempt.cause());
            }
        });
        return result.future();
    }

    private Future<Void> bindSecureHttpServer() {
        if (this.config.isSecurePortEnabled()) {
            if ("127.0.0.1".equals(this.config.getBindAddress())) {
                if (this.bindSecureServerToLoopbackDeviceAllowed) {
                    LOG.warn("secure health checks HTTP server will bind to loopback device only");
                } else {
                    LOG.info("won't start secure health checks HTTP server: no bind address configured.");
                    return Future.failedFuture((String)"no bind address configured for secure server");
                }
            }
            Promise result = Promise.promise();
            HttpServerOptions options = new HttpServerOptions().setPort(this.config.getPort(8088)).setHost(this.config.getBindAddress()).setKeyCertOptions(this.config.getKeyCertOptions()).setSsl(true);
            this.server = this.vertx.createHttpServer(options);
            this.router.get(URI_READINESS_PROBE).handler((Handler)this.readinessHandler);
            this.router.get(URI_LIVENESS_PROBE).handler((Handler)this.livenessHandler);
            this.server.requestHandler((Handler)this.router).listen(startAttempt -> {
                if (startAttempt.succeeded()) {
                    LOG.info("successfully started secure health checks HTTP server");
                    LOG.info("readiness probe available at https://{}:{}{}", new Object[]{options.getHost(), this.server.actualPort(), URI_READINESS_PROBE});
                    LOG.info("liveness probe available at https://{}:{}{}", new Object[]{options.getHost(), this.server.actualPort(), URI_LIVENESS_PROBE});
                    result.complete();
                } else {
                    LOG.warn("failed to start secure health checks HTTP server: {}", (Object)startAttempt.cause().getMessage());
                    result.fail(startAttempt.cause());
                }
            });
            return result.future();
        }
        LOG.warn("cannot start secure health checks HTTP server: no key material configured");
        return Future.succeededFuture();
    }

    private void registerAdditionalResources() {
        this.additionalResources.forEach(handler -> {
            LOG.info("registering additional resource: {}", handler);
            handler.handle((Object)this.router);
        });
        this.additionalResources.clear();
    }

    public Future<Void> stop() {
        Promise serverStopTracker = Promise.promise();
        if (this.server != null) {
            LOG.info("closing secure health check HTTP server [{}:{}]", (Object)this.config.getBindAddress(), (Object)this.server.actualPort());
            this.server.close((Handler)serverStopTracker);
        } else {
            serverStopTracker.complete();
        }
        Promise insecureServerStopTracker = Promise.promise();
        if (this.insecureServer != null) {
            LOG.info("closing insecure health check HTTP server [{}:{}]", (Object)this.config.getInsecurePortBindAddress(), (Object)this.insecureServer.actualPort());
            this.insecureServer.close((Handler)insecureServerStopTracker);
        } else {
            insecureServerStopTracker.complete();
        }
        return CompositeFuture.all((Future)serverStopTracker.future(), (Future)insecureServerStopTracker.future()).map(ok -> null).recover(t -> Future.failedFuture((Throwable)t));
    }

    int getInsecurePort() {
        return Optional.ofNullable(this.insecureServer).map(s -> s.actualPort()).orElse(-1);
    }

    int getPort() {
        return Optional.ofNullable(this.server).map(s -> s.actualPort()).orElse(-1);
    }
}

