/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.client.util;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.eclipse.hono.client.ServerErrorException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CachingClientFactory<T> {
    private static final Logger log = LoggerFactory.getLogger(CachingClientFactory.class);
    private static final int WAITING_CREATION_REQUESTS_COMPLETION_BATCH_SIZE_DEFAULT = 10;
    private final Vertx vertx;
    private final Predicate<T> livenessCheck;
    private final Map<String, T> activeClients = new HashMap<String, T>();
    private final Map<String, Deque<CreationRequest>> waitingCreationRequests = new HashMap<String, Deque<CreationRequest>>();
    private int waitingCreationRequestsCompletionBatchSize = 10;

    public CachingClientFactory(Vertx vertx, Predicate<T> livenessCheck) {
        this.vertx = vertx;
        this.livenessCheck = Objects.requireNonNull(livenessCheck);
    }

    public void onDisconnect() {
        ServerErrorException connectionLostException = new ServerErrorException(503, "no connection to service");
        this.activeClients.clear();
        this.waitingCreationRequests.keySet().forEach(key -> this.failCreationRequests((String)key, connectionLostException));
    }

    void setWaitingCreationRequestsCompletionBatchSize(int batchSize) {
        this.waitingCreationRequestsCompletionBatchSize = batchSize;
    }

    public T removeClient(String key) {
        return this.activeClients.remove(key);
    }

    public T removeClient(String key, Handler<T> postProcessor) {
        T client = this.removeClient(key);
        if (client != null) {
            postProcessor.handle(client);
        }
        return client;
    }

    public T getClient(String key) {
        return this.activeClients.get(key);
    }

    public void getOrCreateClient(String key, Supplier<Future<T>> clientInstanceSupplier, Handler<AsyncResult<T>> result) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(clientInstanceSupplier);
        Objects.requireNonNull(result);
        this.getOrCreateClient(new CreationRequest(key, clientInstanceSupplier, result));
    }

    private void getOrCreateClient(CreationRequest creationRequest) {
        Deque requestsForKey = this.waitingCreationRequests.computeIfAbsent(creationRequest.key, k -> new ArrayDeque());
        if (requestsForKey.isEmpty()) {
            T sender = this.activeClients.get(creationRequest.key);
            if (sender != null && this.livenessCheck.test(sender)) {
                log.debug("reusing cached client [key: {}]", (Object)creationRequest.key);
                creationRequest.complete(sender);
                return;
            }
        } else {
            log.debug("delaying client creation request, previous requests still being finished for [{}] ({} waiting creation requests for all keys)", (Object)creationRequest.key, (Object)this.waitingCreationRequests.size());
            requestsForKey.add(creationRequest);
            return;
        }
        requestsForKey.add(creationRequest);
        log.debug("creating new client for [key: {}]", (Object)creationRequest.key);
        try {
            Future creationAttempt = creationRequest.clientInstanceSupplier.get();
            if (creationAttempt == null) {
                throw new NullPointerException("clientInstanceSupplier result is null");
            }
            creationAttempt.onComplete(ar -> {
                if (creationAttempt.succeeded()) {
                    log.debug("successfully created new client for [key: {}]", (Object)creationRequest.key);
                    Object newClient = creationAttempt.result();
                    this.completeCreationRequests(creationRequest.key, newClient);
                } else {
                    this.failCreationRequests(creationRequest.key, creationAttempt.cause());
                }
            });
        }
        catch (Exception ex) {
            log.error("exception creating new client for [key: {}]", (Object)creationRequest.key, (Object)ex);
            this.activeClients.remove(creationRequest.key);
            this.failCreationRequests(creationRequest.key, new ServerErrorException(500, String.format("exception creating new client for [key: %s]: %s", creationRequest.key, ex.getMessage())));
        }
    }

    private void failCreationRequests(String key, Throwable cause) {
        this.activeClients.remove(key);
        Deque requestsForKey = this.waitingCreationRequests.computeIfAbsent(key, k -> new ArrayDeque());
        int count = requestsForKey.size();
        while (!requestsForKey.isEmpty()) {
            ((CreationRequest)requestsForKey.removeFirst()).fail(cause);
        }
        if (count > 0 && log.isDebugEnabled()) {
            log.debug("failed {} concurrent requests to create new client for [key: {}]: {}", new Object[]{count, key, cause.getMessage()});
        }
    }

    private void completeCreationRequests(String key, T newClient) {
        CreationRequest req;
        this.activeClients.put(key, newClient);
        Deque requestsForKey = this.waitingCreationRequests.computeIfAbsent(key, k -> new ArrayDeque());
        for (int i = 0; i <= this.waitingCreationRequestsCompletionBatchSize && (req = (CreationRequest)requestsForKey.pollFirst()) != null; ++i) {
            req.complete(newClient);
        }
        if (!requestsForKey.isEmpty()) {
            log.trace("decoupling completion of remaining waiting creation requests");
            this.vertx.runOnContext(v -> this.completeCreationRequests(key, newClient));
        }
    }

    private class CreationRequest {
        final String key;
        final Supplier<Future<T>> clientInstanceSupplier;
        final Handler<AsyncResult<T>> result;

        CreationRequest(String key, Supplier<Future<T>> clientInstanceSupplier, Handler<AsyncResult<T>> result) {
            this.key = key;
            this.clientInstanceSupplier = clientInstanceSupplier;
            this.result = result;
        }

        void complete(T createdInstance) {
            this.result.handle((Object)Future.succeededFuture(createdInstance));
        }

        void fail(Throwable cause) {
            log.debug("failed to create new client for [key: {}]: {}", (Object)this.key, (Object)cause.getMessage());
            this.result.handle((Object)Future.failedFuture((Throwable)cause));
        }
    }
}

