/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.io.ByteArrayInputStream;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.util.DatagramReader;
import org.eclipse.californium.scandium.util.DatagramWriter;

public class CertificateMessage
extends HandshakeMessage {
    private static final Logger LOGGER = Logger.getLogger(CertificateMessage.class.getCanonicalName());
    private static final int CERTIFICATE_LENGTH_BITS = 24;
    private static final int CERTIFICATE_LIST_LENGTH = 24;
    private Certificate[] certificateChain;
    private List<byte[]> encodedChain;
    private int messageLength;
    private byte[] rawPublicKeyBytes = null;

    public CertificateMessage(Certificate[] certificateChain, boolean useRawPublicKey) {
        this.certificateChain = certificateChain;
        if (useRawPublicKey) {
            this.rawPublicKeyBytes = certificateChain[0].getPublicKey().getEncoded();
        }
    }

    public CertificateMessage(byte[] rawPublicKeyBytes) {
        this.rawPublicKeyBytes = rawPublicKeyBytes;
    }

    @Override
    public HandshakeType getMessageType() {
        return HandshakeType.CERTIFICATE;
    }

    @Override
    public int getMessageLength() {
        if (this.rawPublicKeyBytes == null) {
            if (this.encodedChain == null) {
                this.messageLength = 3;
                this.encodedChain = new ArrayList<byte[]>(this.certificateChain.length);
                for (Certificate cert : this.certificateChain) {
                    try {
                        byte[] encoded = cert.getEncoded();
                        this.encodedChain.add(encoded);
                        this.messageLength += encoded.length + 3;
                    }
                    catch (CertificateEncodingException e) {
                        this.encodedChain = null;
                        LOGGER.log(Level.SEVERE, "Could not encode the certificate.", e);
                    }
                }
            }
        } else {
            this.messageLength = 6 + this.rawPublicKeyBytes.length;
        }
        return this.messageLength;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        if (this.rawPublicKeyBytes == null) {
            sb.append("\t\tCertificates Length: " + (this.getMessageLength() - 3) + "\n");
            int index = 0;
            for (Certificate cert : this.certificateChain) {
                sb.append("\t\t\tCertificate Length: " + this.encodedChain.get(index).length + "\n");
                sb.append("\t\t\tCertificate: " + cert.toString() + "\n");
                ++index;
            }
        } else {
            sb.append("\t\tRaw Public Key:\n");
            sb.append("\t\t\t" + this.getPublicKey().toString() + "\n");
        }
        return sb.toString();
    }

    public Certificate[] getCertificateChain() {
        return this.certificateChain;
    }

    public void verifyCertificate(Certificate[] trustedCertificates) throws HandshakeException {
        if (this.rawPublicKeyBytes == null) {
            boolean verified = false;
            X509Certificate peerCertificate = (X509Certificate)this.certificateChain[0];
            try {
                peerCertificate.checkValidity();
            }
            catch (Exception e) {
                AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.CERTIFICATE_EXPIRED);
                throw new HandshakeException("Certificate not valid.", alert);
            }
            if (this.isSelfSigned(peerCertificate)) {
                LOGGER.info("Peer used self-signed certificate.");
                return;
            }
            try {
                verified = this.validateKeyChain(peerCertificate, this.certificateChain, trustedCertificates);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
            if (!verified) {
                AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_CERTIFICATE);
                throw new HandshakeException("Certificate could not be verified.", alert);
            }
        }
    }

    public boolean validateKeyChain(X509Certificate certificate, Certificate[] intermediateCertificates, Certificate[] trustedCertificates) {
        for (Certificate cert : intermediateCertificates) {
            X509Certificate intermediateCertificate = (X509Certificate)cert;
            if (!certificate.getIssuerX500Principal().equals(intermediateCertificate.getSubjectX500Principal())) continue;
            try {
                certificate.verify(intermediateCertificate.getPublicKey());
            }
            catch (Exception e) {
                continue;
            }
            if (this.isSelfSigned(intermediateCertificate) || certificate.equals(intermediateCertificate)) continue;
            return this.validateKeyChain(intermediateCertificate, intermediateCertificates, trustedCertificates);
        }
        for (Certificate cert : trustedCertificates) {
            X509Certificate trustedCertificate = (X509Certificate)cert;
            if (!certificate.getIssuerX500Principal().equals(trustedCertificate.getSubjectX500Principal())) continue;
            try {
                certificate.verify(trustedCertificate.getPublicKey());
            }
            catch (Exception e) {
                continue;
            }
            if (this.isSelfSigned(trustedCertificate)) {
                return true;
            }
            if (certificate.equals(trustedCertificate)) continue;
            return this.validateKeyChain(trustedCertificate, intermediateCertificates, trustedCertificates);
        }
        return false;
    }

    private boolean isSelfSigned(X509Certificate certificate) {
        try {
            certificate.verify(certificate.getPublicKey());
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        if (this.rawPublicKeyBytes == null) {
            writer.write(this.getMessageLength() - 3, 24);
            for (byte[] encoded : this.encodedChain) {
                writer.write(encoded.length, 24);
                writer.writeBytes(encoded);
            }
        } else {
            writer.write(this.getMessageLength() - 3, 24);
            writer.write(this.rawPublicKeyBytes.length, 24);
            writer.writeBytes(this.rawPublicKeyBytes);
        }
        return writer.toByteArray();
    }

    public static HandshakeMessage fromByteArray(byte[] byteArray, boolean useRawPublicKey) {
        CertificateMessage message;
        DatagramReader reader = new DatagramReader(byteArray);
        int certificateChainLength = reader.read(24);
        if (useRawPublicKey) {
            int certificateLength = reader.read(24);
            byte[] rawPublicKey = reader.readBytes(certificateLength);
            message = new CertificateMessage(rawPublicKey);
        } else {
            ArrayList<Certificate> certs = new ArrayList<Certificate>();
            CertificateFactory certificateFactory = null;
            while (certificateChainLength > 0) {
                int certificateLength = reader.read(24);
                byte[] certificate = reader.readBytes(certificateLength);
                certificateChainLength -= 3 + certificateLength;
                try {
                    if (certificateFactory == null) {
                        certificateFactory = CertificateFactory.getInstance("X.509");
                    }
                    Certificate cert = certificateFactory.generateCertificate(new ByteArrayInputStream(certificate));
                    certs.add(cert);
                }
                catch (CertificateException e) {
                    LOGGER.log(Level.SEVERE, "Could not generate the certificate.", e);
                    break;
                }
            }
            message = new CertificateMessage(certs.toArray(new X509Certificate[certs.size()]), useRawPublicKey);
        }
        return message;
    }

    public PublicKey getPublicKey() {
        PublicKey publicKey = null;
        if (this.rawPublicKeyBytes == null) {
            publicKey = this.certificateChain[0].getPublicKey();
        } else {
            X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(this.rawPublicKeyBytes);
            try {
                publicKey = KeyFactory.getInstance("EC").generatePublic(publicKeySpec);
            }
            catch (Exception e) {
                LOGGER.log(Level.SEVERE, "Could not reconstruct the server's public key.", e);
            }
        }
        return publicKey;
    }
}

