/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.Random;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.InternalMessageObserverAdapter;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.stack.AbstractLayer;
import org.eclipse.californium.core.network.stack.ReliabilityLayerParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReliabilityLayer
extends AbstractLayer {
    protected static final Logger LOGGER = LoggerFactory.getLogger(ReliabilityLayer.class);
    private final Random rand = new Random();
    private final ReliabilityLayerParameters defaultReliabilityLayerParameters;
    private final AtomicInteger counter = new AtomicInteger();

    public ReliabilityLayer(NetworkConfig config) {
        this.defaultReliabilityLayerParameters = ReliabilityLayerParameters.builder().applyConfig(config).build();
        LOGGER.info("ReliabilityLayer uses ACK_TIMEOUT={}, ACK_RANDOM_FACTOR={}, and ACK_TIMEOUT_SCALE={} as default", this.defaultReliabilityLayerParameters.getAckTimeout(), Float.valueOf(this.defaultReliabilityLayerParameters.getAckRandomFactor()), Float.valueOf(this.defaultReliabilityLayerParameters.getAckTimeoutScale()));
    }

    @Override
    public void sendRequest(final Exchange exchange, final Request request) {
        LOGGER.debug("{} send request, failed transmissions: {}", (Object)exchange, (Object)exchange.getFailedTransmissionCount());
        if (request.getType() == null) {
            request.setType(CoAP.Type.CON);
        }
        if (request.getType() == CoAP.Type.CON) {
            LOGGER.debug("{} prepare retransmission for {}", (Object)exchange, (Object)request);
            this.prepareRetransmission(exchange, new RetransmissionTask(exchange, request){

                @Override
                public void retransmit() {
                    ReliabilityLayer.this.sendRequest(exchange, request);
                }
            });
        }
        this.lower().sendRequest(exchange, request);
    }

    @Override
    public void sendResponse(final Exchange exchange, final Response response) {
        LOGGER.debug("{} send response {}, failed transmissions: {}", exchange, response, exchange.getFailedTransmissionCount());
        CoAP.Type respType = response.getType();
        if (respType == null) {
            CoAP.Type reqType = exchange.getCurrentRequest().getType();
            if (exchange.getCurrentRequest().acknowledge()) {
                response.setType(CoAP.Type.ACK);
                response.setMID(exchange.getCurrentRequest().getMID());
            } else {
                response.setType(reqType);
            }
            LOGGER.trace("{} switched response message type from {} to {} (request was {})", new Object[]{exchange, respType, response.getType(), reqType});
        } else if (respType == CoAP.Type.ACK || respType == CoAP.Type.RST) {
            response.setMID(exchange.getCurrentRequest().getMID());
        }
        if (response.getType() == CoAP.Type.CON) {
            LOGGER.debug("{} prepare retransmission for {}", (Object)exchange, (Object)response);
            this.prepareRetransmission(exchange, new RetransmissionTask(exchange, response){

                @Override
                public void retransmit() {
                    ReliabilityLayer.this.sendResponse(exchange, response);
                }
            });
        }
        this.lower().sendResponse(exchange, response);
    }

    private void prepareRetransmission(final Exchange exchange, final RetransmissionTask task) {
        if (this.executor.isShutdown()) {
            LOGGER.info("Endpoint is being destroyed: skipping retransmission");
            return;
        }
        exchange.setRetransmissionHandle(null);
        this.updateRetransmissionTimeout(exchange, task.getReliabilityLayerParameters());
        task.message.addMessageObserver(new InternalMessageObserverAdapter(){

            @Override
            public void onSent(boolean retransmission) {
                task.message.removeMessageObserver(this);
                if (!exchange.isComplete()) {
                    exchange.execute(new Runnable(){

                        @Override
                        public void run() {
                            task.startTimer();
                        }
                    });
                }
            }
        });
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        if (request.isDuplicate()) {
            if (exchange.getSendNanoTimestamp() > request.getNanoTimestamp()) {
                int count = this.counter.incrementAndGet();
                LOGGER.debug("{}: {} duplicate request {}, server sent response delayed, ignore request", count, exchange, request);
                return;
            }
            exchange.retransmitResponse();
            Response currentResponse = exchange.getCurrentResponse();
            if (currentResponse != null) {
                CoAP.Type type = currentResponse.getType();
                if (type == CoAP.Type.NON || type == CoAP.Type.CON) {
                    if (request.isConfirmable()) {
                        EmptyMessage ack = EmptyMessage.newACK(request);
                        this.sendEmptyMessage(exchange, ack);
                    }
                    if (type == CoAP.Type.CON) {
                        if (currentResponse.isAcknowledged()) {
                            LOGGER.debug("{} request duplicate: ignore, response already acknowledged!", (Object)exchange);
                        } else {
                            int failedCount = exchange.getFailedTransmissionCount() + 1;
                            exchange.setFailedTransmissionCount(failedCount);
                            LOGGER.debug("{} request duplicate: retransmit response, failed: {}, response: {}", exchange, failedCount, currentResponse);
                            currentResponse.retransmitting();
                            this.sendResponse(exchange, currentResponse);
                        }
                        return;
                    }
                }
                LOGGER.debug("{} respond with the current response to the duplicate request", (Object)exchange);
                this.lower().sendResponse(exchange, currentResponse);
            } else if (exchange.getCurrentRequest().isAcknowledged()) {
                LOGGER.debug("{} duplicate request was acknowledged but no response computed yet. Retransmit ACK", (Object)exchange);
                EmptyMessage ack = EmptyMessage.newACK(request);
                this.sendEmptyMessage(exchange, ack);
            } else if (exchange.getCurrentRequest().isRejected()) {
                LOGGER.debug("{} duplicate request was rejected. Reject again", (Object)exchange);
                EmptyMessage rst = EmptyMessage.newRST(request);
                this.sendEmptyMessage(exchange, rst);
            } else {
                LOGGER.debug("{} server has not yet decided what to do with the request. We ignore the duplicate.", (Object)exchange);
            }
        } else {
            exchange.setCurrentRequest(request);
            this.upper().receiveRequest(exchange, request);
        }
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        exchange.setFailedTransmissionCount(0);
        exchange.setRetransmissionHandle(null);
        if (response.getType() == CoAP.Type.CON) {
            EmptyMessage empty;
            boolean ack = true;
            if (response.isDuplicate()) {
                if (response.getNanoTimestamp() < exchange.getSendNanoTimestamp()) {
                    int count = this.counter.incrementAndGet();
                    LOGGER.debug("{}: {} duplicate response {}, server sent ACK delayed, ignore response", count, exchange, response);
                    return;
                }
                if (response.isRejected()) {
                    ack = false;
                    LOGGER.debug("{} reject duplicate CON response, request canceled.", (Object)exchange);
                } else {
                    LOGGER.debug("{} acknowledging duplicate CON response", (Object)exchange);
                }
            } else if (exchange.getRequest().isCanceled()) {
                ack = false;
                LOGGER.debug("{} reject CON response, request canceled.", (Object)exchange);
            } else {
                LOGGER.debug("{} acknowledging CON response", (Object)exchange);
            }
            if (ack) {
                empty = EmptyMessage.newACK(response);
                response.setAcknowledged(true);
            } else {
                empty = EmptyMessage.newRST(response);
                response.setRejected(true);
            }
            this.sendEmptyMessage(exchange, empty);
        }
        if (response.isDuplicate()) {
            if (response.getType() != CoAP.Type.CON) {
                LOGGER.debug("{} ignoring duplicate response", (Object)exchange);
            }
        } else {
            exchange.getCurrentRequest().setAcknowledged(true);
            exchange.setCurrentResponse(response);
            this.upper().receiveResponse(exchange, response);
        }
    }

    @Override
    public void receiveEmptyMessage(Exchange exchange, EmptyMessage message) {
        Message currentMessage;
        String type;
        exchange.setFailedTransmissionCount(0);
        exchange.setRetransmissionHandle(null);
        if (exchange.isOfLocalOrigin()) {
            type = "request";
            currentMessage = exchange.getCurrentRequest();
        } else {
            type = "response";
            currentMessage = exchange.getCurrentResponse();
        }
        int observer = currentMessage.getMessageObservers().size();
        if (message.getType() == CoAP.Type.ACK) {
            LOGGER.debug("{} acknowledge {} for {} {} ({} msg observer)", exchange, message, type, currentMessage, observer);
            currentMessage.setAcknowledged(true);
        } else if (message.getType() == CoAP.Type.RST) {
            LOGGER.debug("{} reject {} for {} {} ({} msg observer)", exchange, message, type, currentMessage, observer);
            currentMessage.setRejected(true);
        } else {
            LOGGER.warn("{} received empty message that is neither ACK nor RST: {}", (Object)exchange, (Object)message);
            return;
        }
        this.upper().receiveEmptyMessage(exchange, message);
    }

    protected void updateRetransmissionTimeout(Exchange exchange, ReliabilityLayerParameters reliabilityLayerParameters) {
        int timeout = exchange.getFailedTransmissionCount() == 0 ? this.getRandomTimeout(reliabilityLayerParameters.getAckTimeout(), reliabilityLayerParameters.getAckRandomFactor()) : (int)(reliabilityLayerParameters.getAckTimeoutScale() * (float)exchange.getCurrentTimeout());
        exchange.setCurrentTimeout(timeout);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int getRandomTimeout(int ackTimeout, float randomFactor) {
        if ((double)randomFactor <= 1.0) {
            return ackTimeout;
        }
        int delta = (int)((float)ackTimeout * randomFactor) - ackTimeout;
        Random random = this.rand;
        synchronized (random) {
            return ackTimeout + this.rand.nextInt(delta + 1);
        }
    }

    protected abstract class RetransmissionTask
    implements Runnable {
        private final Exchange exchange;
        private final Message message;

        public RetransmissionTask(Exchange exchange, Message message) {
            this.exchange = exchange;
            this.message = message;
        }

        public ReliabilityLayerParameters getReliabilityLayerParameters() {
            ReliabilityLayerParameters parameters = this.message.getReliabilityLayerParameters();
            if (parameters == null) {
                parameters = ReliabilityLayer.this.defaultReliabilityLayerParameters;
            }
            return parameters;
        }

        public void startTimer() {
            if (!this.exchange.isComplete()) {
                int timeout = this.exchange.getCurrentTimeout();
                ScheduledFuture<?> f = ReliabilityLayer.this.executor.schedule(this, (long)timeout, TimeUnit.MILLISECONDS);
                this.exchange.setRetransmissionHandle(f);
            }
        }

        @Override
        public void run() {
            this.exchange.execute(new Runnable(){

                @Override
                public void run() {
                    RetransmissionTask.this.retry();
                }
            });
        }

        private void retry() {
            try {
                this.exchange.setRetransmissionHandle(null);
                if (this.exchange.isComplete()) {
                    LOGGER.debug("Timeout: for {}, {}", (Object)this.exchange, (Object)this.message);
                    return;
                }
                int failedCount = this.exchange.getFailedTransmissionCount() + 1;
                this.exchange.setFailedTransmissionCount(failedCount);
                LOGGER.debug("Timeout: for {} retry {} of {}", this.exchange, failedCount, this.message);
                if (this.message.isAcknowledged()) {
                    LOGGER.trace("Timeout: for {} message already acknowledged, cancel retransmission of {}", (Object)this.exchange, (Object)this.message);
                    return;
                }
                if (this.message.isRejected()) {
                    LOGGER.trace("Timeout: for {} message already rejected, cancel retransmission of {}", (Object)this.exchange, (Object)this.message);
                    return;
                }
                if (this.message.isCanceled()) {
                    LOGGER.trace("Timeout: for {}, {} is canceled, do not retransmit", (Object)this.exchange, (Object)this.message);
                    return;
                }
                if (failedCount <= this.getReliabilityLayerParameters().getMaxRetransmit()) {
                    LOGGER.debug("Timeout: for {} retransmit message, failed: {}, message: {}", this.exchange, failedCount, this.message);
                    this.message.retransmitting();
                    if (this.message.isCanceled()) {
                        LOGGER.trace("Timeout: for {}, {} got canceled, do not retransmit", (Object)this.exchange, (Object)this.message);
                        return;
                    }
                    this.retransmit();
                } else {
                    LOGGER.debug("Timeout: for {} retransmission limit reached, exchange failed, message: {}", (Object)this.exchange, (Object)this.message);
                    this.exchange.setTimedOut(this.message);
                }
            }
            catch (Exception e) {
                LOGGER.error("Exception for {} in MessageObserver: {}", this.exchange, e.getMessage(), e);
            }
        }

        public abstract void retransmit();
    }
}

