/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.util;

import java.security.GeneralSecurityException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateParsingException;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import javax.security.auth.x500.X500Principal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertPathUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(CertPathUtil.class);
    private static final String TYPE_X509 = "X.509";
    private static final String SERVER_AUTHENTICATION = "1.3.6.1.5.5.7.3.1";
    private static final String CLIENT_AUTHENTICATION = "1.3.6.1.5.5.7.3.2";
    private static final int KEY_USAGE_SIGNATURE = 0;
    private static final int KEY_USAGE_CERTIFICATE_SIGNING = 5;

    public static boolean canBeUsedToVerifySignature(X509Certificate cert) {
        if (cert.getBasicConstraints() < 0) {
            LOGGER.debug("certificate: {}, not for CA!", (Object)cert.getSubjectDN());
            return false;
        }
        if (cert.getKeyUsage() != null && !cert.getKeyUsage()[5]) {
            LOGGER.debug("certificate: {}, not for certificate signing!", (Object)cert.getSubjectDN());
            return false;
        }
        return true;
    }

    public static boolean canBeUsedForAuthentication(X509Certificate cert, boolean client) {
        if (cert.getKeyUsage() != null && !cert.getKeyUsage()[0]) {
            LOGGER.debug("certificate: {}, not for signing!", (Object)cert.getSubjectDN());
            return false;
        }
        try {
            List<String> list = cert.getExtendedKeyUsage();
            if (list != null && !list.isEmpty()) {
                LOGGER.trace("certificate: {}", (Object)cert.getSubjectDN());
                String authentication = client ? CLIENT_AUTHENTICATION : SERVER_AUTHENTICATION;
                boolean foundUsage = false;
                for (String extension : list) {
                    LOGGER.trace("   extkeyusage {}", (Object)extension);
                    if (!authentication.equals(extension)) continue;
                    foundUsage = true;
                }
                if (!foundUsage) {
                    LOGGER.debug("certificate: {}, not for {}!", (Object)cert.getSubjectDN(), (Object)(client ? "client" : "server"));
                    return false;
                }
            } else {
                LOGGER.debug("certificate: {}, no extkeyusage!", (Object)cert.getSubjectDN());
            }
        }
        catch (CertificateParsingException e) {
            LOGGER.warn("x509 certificate:", e);
        }
        return true;
    }

    public static CertPath generateCertPath(List<X509Certificate> certificateChain) {
        if (certificateChain == null) {
            throw new NullPointerException("Certificate chain must not be null!");
        }
        return CertPathUtil.generateCertPath(certificateChain, certificateChain.size());
    }

    public static CertPath generateCertPath(List<X509Certificate> certificateChain, int size) {
        if (certificateChain == null) {
            throw new NullPointerException("Certificate chain must not be null!");
        }
        if (size > certificateChain.size()) {
            throw new IllegalArgumentException("size must not be larger then certificate chain!");
        }
        try {
            if (!certificateChain.isEmpty()) {
                int last = certificateChain.size() - 1;
                X500Principal issuer = null;
                for (int index = 0; index <= last; ++index) {
                    X509Certificate cert = certificateChain.get(index);
                    LOGGER.debug("Current Subject DN: {}", (Object)cert.getSubjectX500Principal().getName());
                    if (issuer != null && !issuer.equals(cert.getSubjectX500Principal())) {
                        LOGGER.debug("Actual Issuer DN: {}", (Object)cert.getSubjectX500Principal().getName());
                        throw new IllegalArgumentException("Given certificates do not form a chain");
                    }
                    issuer = cert.getIssuerX500Principal();
                    LOGGER.debug("Expected Issuer DN: {}", (Object)issuer.getName());
                    if (!issuer.equals(cert.getSubjectX500Principal()) || index == last) continue;
                    throw new IllegalArgumentException("Given certificates do not form a chain, root is not the last!");
                }
                if (size < certificateChain.size()) {
                    ArrayList<X509Certificate> temp = new ArrayList<X509Certificate>();
                    for (int index = 0; index < size; ++index) {
                        temp.add(certificateChain.get(index));
                    }
                    certificateChain = temp;
                }
            }
            CertificateFactory factory = CertificateFactory.getInstance(TYPE_X509);
            return factory.generateCertPath(certificateChain);
        }
        catch (CertificateException e) {
            throw new IllegalArgumentException("could not create X.509 certificate factory", e);
        }
    }

    public static CertPath generateValidatableCertPath(List<X509Certificate> certificateChain, List<X500Principal> certificateAuthorities) {
        if (certificateChain == null) {
            throw new NullPointerException("Certificate chain must not be null!");
        }
        int size = certificateChain.size();
        if (size > 1) {
            int last;
            X509Certificate cert;
            int truncate = size;
            if (certificateAuthorities != null && !certificateAuthorities.isEmpty()) {
                truncate = 0;
                for (int index = 0; index < size; ++index) {
                    X509Certificate certificate = certificateChain.get(index);
                    if (!certificateAuthorities.contains(certificate.getIssuerX500Principal())) continue;
                    truncate = index + 1;
                    break;
                }
            }
            if (truncate == size && (cert = certificateChain.get(last = size - 1)).getIssuerX500Principal().equals(cert.getSubjectX500Principal())) {
                truncate = last;
            }
            size = truncate;
        }
        return CertPathUtil.generateCertPath(certificateChain, size);
    }

    public static CertPath validateCertificatePath(boolean truncateCertificatePath, CertPath certPath, X509Certificate[] trustedCertificates) throws GeneralSecurityException {
        X509Certificate trust;
        CertPath verifyCertPath;
        if (trustedCertificates == null) {
            throw new CertPathValidatorException("certificates are not trusted!");
        }
        List<? extends Certificate> list = certPath.getCertificates();
        if (list.isEmpty()) {
            return certPath;
        }
        List<X509Certificate> chain = CertPathUtil.toX509CertificatesList(list);
        int size = chain.size();
        int last = size - 1;
        X509Certificate root = (X509Certificate)list.get(last);
        HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
        if (trustedCertificates.length == 0) {
            if (last == 0) {
                if (!root.getIssuerX500Principal().equals(root.getSubjectX500Principal())) {
                    return certPath;
                }
                ++last;
            }
            trustAnchors.add(new TrustAnchor(root, null));
            verifyCertPath = CertPathUtil.generateCertPath(chain, last);
        } else if (truncateCertificatePath) {
            trust = null;
            for (int index = 0; index < size; ++index) {
                X509Certificate certificate = chain.get(index);
                trust = CertPathUtil.search(certificate.getIssuerX500Principal(), trustedCertificates);
                if (trust == null) continue;
                size = index + 1;
                break;
            }
            if (trust != null) {
                trustAnchors.add(new TrustAnchor(trust, null));
            } else {
                trustAnchors.add(new TrustAnchor(trustedCertificates[0], null));
            }
            certPath = verifyCertPath = CertPathUtil.generateCertPath(chain, size);
        } else {
            trust = CertPathUtil.search(root.getIssuerX500Principal(), trustedCertificates);
            if (trust == null) {
                trust = CertPathUtil.search(root.getSubjectX500Principal(), trustedCertificates);
            }
            if (trust != null) {
                trustAnchors.add(new TrustAnchor(trust, null));
            } else {
                trustAnchors.add(new TrustAnchor(trustedCertificates[0], null));
            }
            verifyCertPath = CertPathUtil.generateCertPath(chain, size);
        }
        if (LOGGER.isDebugEnabled()) {
            List<X509Certificate> validateChain = CertPathUtil.toX509CertificatesList(verifyCertPath.getCertificates());
            LOGGER.debug("verify: certificate path {} (orig. {})", (Object)validateChain.size(), (Object)list.size());
            for (X509Certificate certificate : validateChain) {
                LOGGER.debug("   cert: {}", (Object)certificate.getSubjectX500Principal());
            }
            for (TrustAnchor anchor : trustAnchors) {
                LOGGER.debug("   trust: {}", (Object)anchor.getTrustedCert().getIssuerX500Principal());
            }
        }
        CertPathValidator validator = CertPathValidator.getInstance("PKIX");
        PKIXParameters params = new PKIXParameters(trustAnchors);
        params.setRevocationEnabled(false);
        validator.validate(verifyCertPath, params);
        return certPath;
    }

    public static List<X509Certificate> toX509CertificatesList(List<? extends Certificate> certificates) {
        if (certificates == null) {
            throw new NullPointerException("Certificates list must not be null!");
        }
        ArrayList<X509Certificate> chain = new ArrayList<X509Certificate>(certificates.size());
        for (Certificate certificate : certificates) {
            if (!(certificate instanceof X509Certificate)) {
                throw new IllegalArgumentException("Given certificate is not X.509! " + certificate);
            }
            chain.add((X509Certificate)certificate);
        }
        return chain;
    }

    public static List<X500Principal> toSubjects(List<X509Certificate> certificates) {
        if (certificates != null) {
            ArrayList<X500Principal> subjects = new ArrayList<X500Principal>(certificates.size());
            for (X509Certificate certificate : certificates) {
                subjects.add(certificate.getSubjectX500Principal());
            }
            return subjects;
        }
        return Collections.emptyList();
    }

    private static X509Certificate search(X500Principal subject, X509Certificate[] certificates) {
        for (int index = 0; index < certificates.length; ++index) {
            X509Certificate trust = certificates[index];
            if (trust == null || !subject.equals(trust.getSubjectX500Principal())) continue;
            return trust;
        }
        return null;
    }
}

