/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.californium.elements.auth.X509CertPath;
import org.eclipse.californium.elements.util.Asn1DerDecoder;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.CertificateType;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CertificateMessage
extends HandshakeMessage {
    private static final String CERTIFICATE_TYPE_X509 = "X.509";
    private static final Logger LOGGER = LoggerFactory.getLogger(CertificateMessage.class.getCanonicalName());
    private static final int CERTIFICATE_LENGTH_BITS = 24;
    private static final int CERTIFICATE_LIST_LENGTH = 24;
    private CertPath certPath;
    private List<byte[]> encodedChain;
    private byte[] rawPublicKeyBytes;
    private int length = 3;

    public CertificateMessage(List<X509Certificate> certificateChain, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (certificateChain == null) {
            throw new NullPointerException("Certificate chain must not be null");
        }
        this.setCertificateChain(certificateChain);
        this.calculateLength();
    }

    private CertificateMessage(CertPath peerCertChain, InetSocketAddress peerAddress) {
        super(peerAddress);
        this.certPath = peerCertChain;
        this.calculateLength();
    }

    public CertificateMessage(byte[] rawPublicKeyBytes, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (rawPublicKeyBytes == null) {
            throw new NullPointerException("Raw public key byte array must not be null");
        }
        this.rawPublicKeyBytes = Arrays.copyOf(rawPublicKeyBytes, rawPublicKeyBytes.length);
        this.length += this.rawPublicKeyBytes.length;
    }

    private void setCertificateChain(List<X509Certificate> chain) {
        this.certPath = X509CertPath.generateCertPath(false, chain);
    }

    @Override
    public HandshakeType getMessageType() {
        return HandshakeType.CERTIFICATE;
    }

    private void calculateLength() {
        if (this.certPath != null && this.encodedChain == null) {
            this.encodedChain = new ArrayList<byte[]>(this.certPath.getCertificates().size());
            try {
                for (Certificate certificate : this.certPath.getCertificates()) {
                    byte[] encoded = certificate.getEncoded();
                    this.encodedChain.add(encoded);
                    this.length += 3 + encoded.length;
                }
            }
            catch (CertificateEncodingException e) {
                this.encodedChain = null;
                LOGGER.error("Could not encode certificate chain", e);
            }
        }
    }

    @Override
    public int getMessageLength() {
        return this.length;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        if (this.rawPublicKeyBytes == null && this.certPath != null) {
            sb.append("\t\tCertificate chain length: ").append(this.getMessageLength() - 3).append(StringUtil.lineSeparator());
            int index = 0;
            for (Certificate certificate : this.certPath.getCertificates()) {
                sb.append("\t\t\tCertificate Length: ").append(this.encodedChain.get(index).length).append(StringUtil.lineSeparator());
                sb.append("\t\t\tCertificate: ").append(certificate).append(StringUtil.lineSeparator());
                ++index;
            }
        } else if (this.rawPublicKeyBytes != null && this.certPath == null) {
            sb.append("\t\tRaw Public Key: ");
            sb.append(this.getPublicKey().toString());
            sb.append(StringUtil.lineSeparator());
        }
        return sb.toString();
    }

    public CertPath getCertificateChain() {
        return this.certPath;
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        if (this.rawPublicKeyBytes == null) {
            writer.write(this.getMessageLength() - 3, 24);
            for (byte[] encoded : this.encodedChain) {
                writer.write(encoded.length, 24);
                writer.writeBytes(encoded);
            }
        } else {
            writer.write(this.rawPublicKeyBytes.length, 24);
            writer.writeBytes(this.rawPublicKeyBytes);
        }
        return writer.toByteArray();
    }

    public static CertificateMessage fromReader(DatagramReader reader, CertificateType certificateType, InetSocketAddress peerAddress) throws HandshakeException {
        if (CertificateType.RAW_PUBLIC_KEY == certificateType) {
            LOGGER.debug("Parsing RawPublicKey CERTIFICATE message");
            int certificateLength = reader.read(24);
            byte[] rawPublicKey = reader.readBytes(certificateLength);
            return new CertificateMessage(rawPublicKey, peerAddress);
        }
        if (CertificateType.X_509 == certificateType) {
            return CertificateMessage.readX509CertificateMessage(reader, peerAddress);
        }
        throw new IllegalArgumentException("Certificate type " + (Object)((Object)certificateType) + " not supported!");
    }

    private static CertificateMessage readX509CertificateMessage(DatagramReader reader, InetSocketAddress peerAddress) throws HandshakeException {
        LOGGER.debug("Parsing X.509 CERTIFICATE message");
        ArrayList<Certificate> certs = new ArrayList<Certificate>();
        int certificateChainLength = reader.read(24);
        DatagramReader rangeReader = reader.createRangeReader(certificateChainLength);
        try {
            CertificateFactory factory = CertificateFactory.getInstance(CERTIFICATE_TYPE_X509);
            while (rangeReader.bytesAvailable()) {
                int certificateLength = rangeReader.read(24);
                certs.add(factory.generateCertificate(rangeReader.createRangeInputStream(certificateLength)));
            }
            return new CertificateMessage(factory.generateCertPath(certs), peerAddress);
        }
        catch (CertificateException e) {
            throw new HandshakeException("Cannot parse X.509 certificate chain provided by peer", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_CERTIFICATE, peerAddress), e);
        }
    }

    public PublicKey getPublicKey() {
        PublicKey publicKey = null;
        if (this.rawPublicKeyBytes == null) {
            if (this.certPath != null && !this.certPath.getCertificates().isEmpty()) {
                publicKey = this.certPath.getCertificates().get(0).getPublicKey();
            }
        } else {
            X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(this.rawPublicKeyBytes);
            try {
                String keyAlgorithm = Asn1DerDecoder.readSubjectPublicKeyAlgorithm(this.rawPublicKeyBytes);
                publicKey = KeyFactory.getInstance(keyAlgorithm).generatePublic(publicKeySpec);
            }
            catch (GeneralSecurityException e) {
                LOGGER.error("Could not reconstruct the peer's public key", e);
            }
        }
        return publicKey;
    }
}

