/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.CoAPMessageFormatException;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageFormatException;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.coap.Token;
import org.eclipse.californium.core.network.CoapStackFactory;
import org.eclipse.californium.core.network.Endpoint;
import org.eclipse.californium.core.network.EndpointContextMatcherFactory;
import org.eclipse.californium.core.network.EndpointManager;
import org.eclipse.californium.core.network.EndpointObserver;
import org.eclipse.californium.core.network.EndpointReceiver;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.InMemoryMessageExchangeStore;
import org.eclipse.californium.core.network.Matcher;
import org.eclipse.californium.core.network.MessageExchangeStore;
import org.eclipse.californium.core.network.Outbox;
import org.eclipse.californium.core.network.RandomTokenGenerator;
import org.eclipse.californium.core.network.TcpMatcher;
import org.eclipse.californium.core.network.TokenGenerator;
import org.eclipse.californium.core.network.UdpMatcher;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.interceptors.MessageInterceptor;
import org.eclipse.californium.core.network.serialization.DataParser;
import org.eclipse.californium.core.network.serialization.DataSerializer;
import org.eclipse.californium.core.network.serialization.TcpDataParser;
import org.eclipse.californium.core.network.serialization.TcpDataSerializer;
import org.eclipse.californium.core.network.serialization.UdpDataParser;
import org.eclipse.californium.core.network.serialization.UdpDataSerializer;
import org.eclipse.californium.core.network.stack.CoapStack;
import org.eclipse.californium.core.network.stack.CoapTcpStack;
import org.eclipse.californium.core.network.stack.CoapUdpStack;
import org.eclipse.californium.core.observe.InMemoryObservationStore;
import org.eclipse.californium.core.observe.NotificationListener;
import org.eclipse.californium.core.observe.ObservationStore;
import org.eclipse.californium.core.server.MessageDeliverer;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.EndpointContext;
import org.eclipse.californium.elements.EndpointContextMatcher;
import org.eclipse.californium.elements.MessageCallback;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.UDPConnector;
import org.eclipse.californium.elements.util.ClockUtil;
import org.eclipse.californium.elements.util.DaemonThreadFactory;
import org.eclipse.californium.elements.util.ExecutorsUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CoapEndpoint
implements Endpoint {
    private static final Logger LOGGER = LoggerFactory.getLogger(CoapEndpoint.class.getCanonicalName());
    protected final CoapStack coapstack;
    private final Connector connector;
    private final String scheme;
    private final int multicastBaseMid;
    private final NetworkConfig config;
    private final Matcher matcher;
    private final DataSerializer serializer;
    private final DataParser parser;
    private final MessageExchangeStore exchangeStore;
    private final ObservationStore observationStore;
    private ExecutorService executor;
    private ScheduledExecutorService secondaryExecutor;
    private volatile boolean started;
    private List<EndpointObserver> observers = new CopyOnWriteArrayList<EndpointObserver>();
    private List<MessageInterceptor> interceptors = new CopyOnWriteArrayList<MessageInterceptor>();
    private List<NotificationListener> notificationListeners = new CopyOnWriteArrayList<NotificationListener>();
    private final EndpointReceiver endpointStackReceiver = new EndpointReceiver(){

        @Override
        public void receiveRequest(Exchange exchange, Request request) {
            exchange.setEndpoint(CoapEndpoint.this);
            CoapEndpoint.this.coapstack.receiveRequest(exchange, request);
        }

        @Override
        public void receiveResponse(Exchange exchange, Response response) {
            exchange.setEndpoint(CoapEndpoint.this);
            response.setRTT(exchange.calculateRTT());
            CoapEndpoint.this.coapstack.receiveResponse(exchange, response);
        }

        @Override
        public void receiveEmptyMessage(Exchange exchange, EmptyMessage message) {
            exchange.setEndpoint(CoapEndpoint.this);
            CoapEndpoint.this.coapstack.receiveEmptyMessage(exchange, message);
        }

        @Override
        public void reject(Message message) {
            EmptyMessage rst = EmptyMessage.newRST(message);
            CoapEndpoint.this.coapstack.sendEmptyMessage(null, rst);
        }
    };
    private static CoapStackFactory defaultCoapStackFactory;

    protected CoapEndpoint(Connector connector, boolean applyConfiguration, NetworkConfig config, TokenGenerator tokenGenerator, ObservationStore store, MessageExchangeStore exchangeStore, EndpointContextMatcher endpointContextMatcher, CoapStackFactory coapStackFactory, Object customStackArgument) {
        this.config = config;
        this.connector = connector;
        this.connector.setRawDataReceiver(new InboxImpl());
        this.scheme = CoAP.getSchemeForProtocol(connector.getProtocol());
        this.multicastBaseMid = config.getInt("MULTICAST_BASE_MID");
        if (tokenGenerator == null) {
            tokenGenerator = new RandomTokenGenerator(config);
        }
        if (coapStackFactory == null) {
            coapStackFactory = CoapEndpoint.getDefaultCoapStackFactory();
        }
        this.exchangeStore = null != exchangeStore ? exchangeStore : new InMemoryMessageExchangeStore(config, tokenGenerator, endpointContextMatcher);
        ObservationStore observationStore = this.observationStore = null != store ? store : new InMemoryObservationStore(config);
        if (null == endpointContextMatcher) {
            endpointContextMatcher = EndpointContextMatcherFactory.create(connector, config);
        }
        if (applyConfiguration) {
            if (connector instanceof UDPConnector) {
                UDPConnector udpConnector = (UDPConnector)connector;
                udpConnector.setReceiverThreadCount(config.getInt("NETWORK_STAGE_RECEIVER_THREAD_COUNT"));
                udpConnector.setSenderThreadCount(config.getInt("NETWORK_STAGE_SENDER_THREAD_COUNT"));
                udpConnector.setReceiveBufferSize(config.getInt("UDP_CONNECTOR_RECEIVE_BUFFER"));
                udpConnector.setSendBufferSize(config.getInt("UDP_CONNECTOR_SEND_BUFFER"));
                udpConnector.setReceiverPacketSize(config.getInt("UDP_CONNECTOR_DATAGRAM_SIZE"));
            } else {
                throw new IllegalArgumentException("Connector must be a UDPConnector to use apply configuration!");
            }
        }
        Executor exchangeExecutionHandler = new Executor(){

            @Override
            public void execute(Runnable command) {
                ExecutorService exchangeExecutor = CoapEndpoint.this.executor;
                if (exchangeExecutor == null) {
                    LOGGER.error("Executor not ready for exchanges!", new Throwable("exchange execution failed!"));
                } else {
                    exchangeExecutor.execute(command);
                }
            }
        };
        this.connector.setEndpointContextMatcher(endpointContextMatcher);
        LOGGER.info("{} uses {}", (Object)this.getClass().getSimpleName(), (Object)endpointContextMatcher.getName());
        this.coapstack = coapStackFactory.createCoapStack(connector.getProtocol(), config, new OutboxImpl(), customStackArgument);
        if (CoAP.isTcpProtocol(connector.getProtocol())) {
            this.matcher = new TcpMatcher(config, new NotificationDispatcher(), tokenGenerator, this.observationStore, this.exchangeStore, exchangeExecutionHandler, endpointContextMatcher);
            this.serializer = new TcpDataSerializer();
            this.parser = new TcpDataParser();
        } else {
            this.matcher = new UdpMatcher(config, new NotificationDispatcher(), tokenGenerator, this.observationStore, this.exchangeStore, exchangeExecutionHandler, endpointContextMatcher);
            this.serializer = new UdpDataSerializer();
            this.parser = new UdpDataParser();
        }
    }

    @Override
    public synchronized void start() throws IOException {
        if (this.started) {
            LOGGER.debug("Endpoint at {} is already started", (Object)this.getUri());
            return;
        }
        if (!this.coapstack.hasDeliverer()) {
            this.setMessageDeliverer(new EndpointManager.ClientMessageDeliverer());
        }
        if (this.executor == null) {
            LOGGER.info("Endpoint [{}] requires an executor to start, using default single-threaded daemon executor", (Object)this.getUri());
            final ScheduledExecutorService executorService = ExecutorsUtil.newSingleThreadScheduledExecutor(new DaemonThreadFactory(":CoapEndpoint-" + this.connector + '#'));
            this.setExecutors(executorService, executorService);
            this.addObserver(new EndpointObserver(){

                @Override
                public void started(Endpoint endpoint) {
                }

                @Override
                public void stopped(Endpoint endpoint) {
                }

                @Override
                public void destroyed(Endpoint endpoint) {
                    ExecutorsUtil.shutdownExecutorGracefully(1000L, executorService);
                }
            });
        }
        try {
            LOGGER.debug("Starting endpoint at {}", (Object)this.getUri());
            this.started = true;
            this.matcher.start();
            this.connector.start();
            this.coapstack.start();
            for (EndpointObserver obs : this.observers) {
                obs.started(this);
            }
            LOGGER.info("Started endpoint at {}", (Object)this.getUri());
        }
        catch (IOException e) {
            this.stop();
            throw e;
        }
    }

    @Override
    public synchronized void stop() {
        if (!this.started) {
            LOGGER.info("Endpoint at {} is already stopped", (Object)this.getUri());
        } else {
            LOGGER.info("Stopping endpoint at {}", (Object)this.getUri());
            this.started = false;
            this.connector.stop();
            this.matcher.stop();
            for (EndpointObserver obs : this.observers) {
                obs.stopped(this);
            }
        }
    }

    @Override
    public synchronized void destroy() {
        LOGGER.info("Destroying endpoint at {}", (Object)this.getUri());
        if (this.started) {
            this.stop();
        }
        this.connector.destroy();
        this.coapstack.destroy();
        for (EndpointObserver obs : this.observers) {
            obs.destroyed(this);
        }
    }

    @Override
    public void clear() {
        this.matcher.clear();
    }

    @Override
    public boolean isStarted() {
        return this.started;
    }

    @Override
    public void setExecutors(ScheduledExecutorService mainExecutor, ScheduledExecutorService secondaryExecutor) {
        if (mainExecutor == null || secondaryExecutor == null) {
            throw new IllegalArgumentException("executors must not be null");
        }
        if (this.executor == mainExecutor && this.secondaryExecutor == secondaryExecutor) {
            return;
        }
        if (this.started) {
            throw new IllegalStateException("endpoint already started!");
        }
        this.executor = mainExecutor;
        this.secondaryExecutor = secondaryExecutor;
        this.coapstack.setExecutors(mainExecutor, this.secondaryExecutor);
        this.exchangeStore.setExecutor(this.secondaryExecutor);
        this.observationStore.setExecutor(this.secondaryExecutor);
    }

    @Override
    public void addNotificationListener(NotificationListener lis) {
        this.notificationListeners.add(lis);
    }

    @Override
    public void removeNotificationListener(NotificationListener lis) {
        this.notificationListeners.remove(lis);
    }

    @Override
    public void addObserver(EndpointObserver observer) {
        this.observers.add(observer);
    }

    @Override
    public void removeObserver(EndpointObserver observer) {
        this.observers.remove(observer);
    }

    @Override
    public void addInterceptor(MessageInterceptor interceptor) {
        this.interceptors.add(interceptor);
    }

    @Override
    public void removeInterceptor(MessageInterceptor interceptor) {
        this.interceptors.remove(interceptor);
    }

    @Override
    public List<MessageInterceptor> getInterceptors() {
        return Collections.unmodifiableList(this.interceptors);
    }

    @Override
    public void sendRequest(final Request request) {
        if (!this.started) {
            request.cancel();
            return;
        }
        request.prepareDestinationContext();
        InetSocketAddress destinationAddress = request.getDestinationContext().getPeerAddress();
        if (request.isMulticast()) {
            if (0 >= this.multicastBaseMid) {
                LOGGER.warn("multicast messaging to destination {} is not enabled! Please enable it configuring \"MULTICAST_BASE_MID\" greater than 0", (Object)destinationAddress);
                return;
            }
            if (request.getType() == CoAP.Type.CON) {
                LOGGER.warn(" CON request to multicast destination {} is not allowed, as per RFC 7252, 8.1, a client MUST use NON message type for multicast requests ", (Object)destinationAddress);
                return;
            }
            if (request.hasMID() && request.getMID() < this.multicastBaseMid) {
                LOGGER.warn("multicast request to group {} has mid {} which is not in the MULTICAST_MID range [{}-65535]", destinationAddress, request.getMID(), this.multicastBaseMid);
                return;
            }
        } else if (0 < this.multicastBaseMid && request.getMID() >= this.multicastBaseMid) {
            LOGGER.warn("request has mid {}, which is in the MULTICAST_MID range [{}-65535]", destinationAddress, request.getMID(), this.multicastBaseMid);
            return;
        }
        final Exchange exchange = new Exchange(request, Exchange.Origin.LOCAL, this.executor);
        exchange.execute(new Runnable(){

            @Override
            public void run() {
                CoapEndpoint.this.coapstack.sendRequest(exchange, request);
            }
        });
    }

    @Override
    public void sendResponse(final Exchange exchange, final Response response) {
        if (!this.started) {
            response.cancel();
            return;
        }
        if (exchange.checkOwner()) {
            this.coapstack.sendResponse(exchange, response);
        } else {
            exchange.execute(new Runnable(){

                @Override
                public void run() {
                    CoapEndpoint.this.coapstack.sendResponse(exchange, response);
                }
            });
        }
    }

    @Override
    public void sendEmptyMessage(final Exchange exchange, final EmptyMessage message) {
        if (!this.started) {
            message.cancel();
            return;
        }
        if (exchange.checkOwner()) {
            this.coapstack.sendEmptyMessage(exchange, message);
        } else {
            exchange.execute(new Runnable(){

                @Override
                public void run() {
                    CoapEndpoint.this.coapstack.sendEmptyMessage(exchange, message);
                }
            });
        }
    }

    @Override
    public void setMessageDeliverer(MessageDeliverer deliverer) {
        this.coapstack.setDeliverer(deliverer);
    }

    @Override
    public InetSocketAddress getAddress() {
        return this.connector.getAddress();
    }

    @Override
    public URI getUri() {
        URI uri = null;
        try {
            InetSocketAddress socketAddress = this.getAddress();
            String scheme = CoAP.getSchemeForProtocol(this.connector.getProtocol());
            String host = socketAddress.getAddress().getHostAddress();
            try {
                uri = new URI(scheme, null, host, socketAddress.getPort(), null, null, null);
            }
            catch (URISyntaxException e) {
                try {
                    host = host.replaceAll("[-._~]", "");
                    uri = new URI(scheme, null, host, socketAddress.getPort(), null, null, null);
                }
                catch (URISyntaxException e2) {
                    LOGGER.warn("URI", e);
                }
            }
        }
        catch (IllegalArgumentException e) {
            LOGGER.warn("URI", e);
        }
        return uri;
    }

    @Override
    public NetworkConfig getConfig() {
        return this.config;
    }

    public Connector getConnector() {
        return this.connector;
    }

    @Override
    public void cancelObservation(Token token) {
        this.matcher.cancelObserve(token);
    }

    private void runInProtocolStage(final Runnable task) {
        try {
            this.executor.execute(new Runnable(){

                @Override
                public void run() {
                    try {
                        task.run();
                    }
                    catch (Throwable t) {
                        LOGGER.error("exception in protocol stage thread: {}", (Object)t.getMessage(), (Object)t);
                    }
                }
            });
        }
        catch (RejectedExecutionException e) {
            LOGGER.debug("{} execute:", (Object)this, (Object)e);
        }
    }

    private static synchronized CoapStackFactory getDefaultCoapStackFactory() {
        if (defaultCoapStackFactory == null) {
            defaultCoapStackFactory = new CoapStackFactory(){

                @Override
                public CoapStack createCoapStack(String protocol, NetworkConfig config, Outbox outbox, Object customStackArgument) {
                    if (CoAP.isTcpProtocol(protocol)) {
                        return new CoapTcpStack(config, outbox);
                    }
                    return new CoapUdpStack(config, outbox);
                }
            };
        }
        return defaultCoapStackFactory;
    }

    public static synchronized void setDefaultCoapStackFactory(CoapStackFactory newFactory) {
        if (defaultCoapStackFactory != null) {
            throw new IllegalStateException("Default coap-stack-factory already set!");
        }
        if (newFactory == null) {
            throw new NullPointerException("new coap-stack-factory must not be null!");
        }
        defaultCoapStackFactory = newFactory;
    }

    public static class Builder {
        private NetworkConfig config = null;
        private InetSocketAddress bindAddress = null;
        private boolean applyConfiguration = true;
        private Connector connector = null;
        private ObservationStore observationStore = null;
        private MessageExchangeStore exchangeStore = null;
        private EndpointContextMatcher endpointContextMatcher = null;
        private TokenGenerator tokenGenerator;
        private CoapStackFactory coapStackFactory;
        private Object customStackArgument;

        public Builder setNetworkConfig(NetworkConfig config) {
            this.config = config;
            return this;
        }

        public Builder setPort(int port) {
            if (this.bindAddress != null || this.connector != null) {
                throw new IllegalArgumentException("bind address already defined!");
            }
            this.bindAddress = new InetSocketAddress(port);
            return this;
        }

        public Builder setInetSocketAddress(InetSocketAddress address) {
            if (this.bindAddress != null || this.connector != null) {
                throw new IllegalArgumentException("bind address already defined!");
            }
            this.bindAddress = address;
            return this;
        }

        public Builder setConnector(Connector connector) {
            if (this.bindAddress != null || this.connector != null) {
                throw new IllegalArgumentException("bind address already defined!");
            }
            this.connector = connector;
            this.applyConfiguration = false;
            return this;
        }

        public Builder setConnectorWithAutoConfiguration(UDPConnector connector) {
            if (this.bindAddress != null || this.connector != null) {
                throw new IllegalArgumentException("bind address already defined!");
            }
            this.connector = connector;
            return this;
        }

        public Builder setObservationStore(ObservationStore store) {
            this.observationStore = store;
            return this;
        }

        public Builder setMessageExchangeStore(MessageExchangeStore exchangeStore) {
            this.exchangeStore = exchangeStore;
            return this;
        }

        public Builder setEndpointContextMatcher(EndpointContextMatcher endpointContextMatcher) {
            this.endpointContextMatcher = endpointContextMatcher;
            return this;
        }

        public Builder setTokenGenerator(TokenGenerator tokenGenerator) {
            this.tokenGenerator = tokenGenerator;
            return this;
        }

        public Builder setCoapStackFactory(CoapStackFactory coapStackFactory) {
            this.coapStackFactory = coapStackFactory;
            return this;
        }

        public Builder setCustomCoapStackArgument(Object customStackArgument) {
            this.customStackArgument = customStackArgument;
            return this;
        }

        public CoapEndpoint build() {
            if (this.config == null) {
                this.config = NetworkConfig.getStandard();
            }
            if (this.connector == null) {
                if (this.bindAddress == null) {
                    this.bindAddress = new InetSocketAddress(0);
                }
                this.connector = new UDPConnector(this.bindAddress);
            }
            if (this.tokenGenerator == null) {
                this.tokenGenerator = new RandomTokenGenerator(this.config);
            }
            if (this.observationStore == null) {
                this.observationStore = new InMemoryObservationStore(this.config);
            }
            if (this.endpointContextMatcher == null) {
                this.endpointContextMatcher = EndpointContextMatcherFactory.create(this.connector, this.config);
            }
            if (this.exchangeStore == null) {
                this.exchangeStore = new InMemoryMessageExchangeStore(this.config, this.tokenGenerator, this.endpointContextMatcher);
            }
            if (this.coapStackFactory == null) {
                this.coapStackFactory = CoapEndpoint.getDefaultCoapStackFactory();
            }
            return new CoapEndpoint(this.connector, this.applyConfiguration, this.config, this.tokenGenerator, this.observationStore, this.exchangeStore, this.endpointContextMatcher, this.coapStackFactory, this.customStackArgument);
        }
    }

    private class ExchangeCallback
    extends SendingCallback {
        private final Exchange exchange;

        public ExchangeCallback(Exchange exchange, Message message) {
            super(message);
            if (null == exchange) {
                throw new NullPointerException("exchange must not be null");
            }
            this.exchange = exchange;
        }

        @Override
        protected void onContextEstablished(EndpointContext context, long nanoTimestamp) {
            this.exchange.setSendNanoTimestamp(nanoTimestamp);
            this.exchange.setEndpointContext(context);
        }
    }

    private class SendingCallback
    implements MessageCallback {
        private final Message message;

        public SendingCallback(Message message) {
            if (null == message) {
                throw new NullPointerException("message must not be null");
            }
            this.message = message;
        }

        @Override
        public void onConnecting() {
            this.message.onConnecting();
        }

        @Override
        public void onDtlsRetransmission(int flight) {
            this.message.onDtlsRetransmission(flight);
        }

        @Override
        public final void onContextEstablished(EndpointContext context) {
            long now = ClockUtil.nanoRealtime();
            this.message.setNanoTimestamp(now);
            this.onContextEstablished(context, now);
        }

        @Override
        public void onSent() {
            this.message.setSent(true);
        }

        @Override
        public void onError(Throwable error) {
            this.message.setSendError(error);
        }

        protected void onContextEstablished(EndpointContext context, long nanoTimestamp) {
        }
    }

    private class InboxImpl
    implements RawDataChannel {
        private InboxImpl() {
        }

        @Override
        public void receiveData(final RawData raw) {
            if (raw.getEndpointContext() == null) {
                throw new IllegalArgumentException("received message that does not have a endpoint context");
            }
            if (raw.getEndpointContext().getPeerAddress() == null) {
                throw new IllegalArgumentException("received message that does not have a source address");
            }
            if (raw.getEndpointContext().getPeerAddress().getPort() == 0) {
                throw new IllegalArgumentException("received message that does not have a source port");
            }
            CoapEndpoint.this.runInProtocolStage(new Runnable(){

                @Override
                public void run() {
                    InboxImpl.this.receiveMessage(raw);
                }
            });
        }

        private void receiveMessage(RawData raw) {
            Message msg = null;
            try {
                msg = CoapEndpoint.this.parser.parseMessage(raw);
                if (CoAP.isRequest(msg.getRawCode())) {
                    this.receiveRequest((Request)msg);
                } else if (CoAP.isResponse(msg.getRawCode())) {
                    this.receiveResponse((Response)msg);
                } else if (CoAP.isEmptyMessage(msg.getRawCode())) {
                    this.receiveEmptyMessage((EmptyMessage)msg);
                } else {
                    LOGGER.debug("silently ignoring non-CoAP message from {}", (Object)raw.getEndpointContext());
                }
            }
            catch (CoAPMessageFormatException e) {
                if (e.isConfirmable() && e.hasMid()) {
                    this.reject(raw, e);
                    LOGGER.debug("rejected malformed message from [{}], reason: {}", (Object)raw.getEndpointContext(), (Object)e.getMessage());
                } else {
                    LOGGER.debug("discarding malformed message from [{}]", (Object)raw.getEndpointContext());
                }
            }
            catch (MessageFormatException e) {
                LOGGER.debug("discarding malformed message from [{}]", (Object)raw.getEndpointContext());
            }
        }

        private void reject(RawData raw, CoAPMessageFormatException cause) {
            EmptyMessage rst = new EmptyMessage(CoAP.Type.RST);
            rst.setMID(cause.getMid());
            rst.setDestinationContext(raw.getEndpointContext());
            CoapEndpoint.this.coapstack.sendEmptyMessage(null, rst);
        }

        private void receiveRequest(Request request) {
            request.setScheme(CoapEndpoint.this.scheme);
            if (!CoapEndpoint.this.started) {
                LOGGER.debug("not running, drop request {}", (Object)request);
                return;
            }
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.receiveRequest(request);
            }
            if (!request.isCanceled()) {
                CoapEndpoint.this.matcher.receiveRequest(request, CoapEndpoint.this.endpointStackReceiver);
            }
        }

        private void receiveResponse(Response response) {
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.receiveResponse(response);
            }
            if (!response.isCanceled()) {
                CoapEndpoint.this.matcher.receiveResponse(response, CoapEndpoint.this.endpointStackReceiver);
            }
        }

        private void receiveEmptyMessage(EmptyMessage message) {
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.receiveEmptyMessage(message);
            }
            if (!message.isCanceled()) {
                if ((message.getType() == CoAP.Type.CON || message.getType() == CoAP.Type.NON) && message.hasMID()) {
                    LOGGER.debug("responding to ping from {}", (Object)message.getSourceContext());
                    CoapEndpoint.this.endpointStackReceiver.reject(message);
                } else {
                    CoapEndpoint.this.matcher.receiveEmptyMessage(message, CoapEndpoint.this.endpointStackReceiver);
                }
            }
        }
    }

    public class OutboxImpl
    implements Outbox {
        @Override
        public void sendRequest(Exchange exchange, Request request) {
            this.assertMessageHasDestinationAddress(request);
            exchange.setCurrentRequest(request);
            CoapEndpoint.this.matcher.sendRequest(exchange);
            for (MessageInterceptor messageInterceptor : CoapEndpoint.this.interceptors) {
                messageInterceptor.sendRequest(request);
            }
            request.setReadyToSend();
            if (!CoapEndpoint.this.started) {
                request.cancel();
            }
            if (request.isCanceled() || request.getSendError() != null) {
                exchange.executeComplete();
            } else {
                RawData message = CoapEndpoint.this.serializer.serializeRequest(request, new ExchangeCallback(exchange, request));
                CoapEndpoint.this.connector.send(message);
            }
        }

        @Override
        public void sendResponse(Exchange exchange, Response response) {
            this.assertMessageHasDestinationAddress(response);
            exchange.setCurrentResponse(response);
            CoapEndpoint.this.matcher.sendResponse(exchange);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendResponse(response);
            }
            response.setReadyToSend();
            if (!CoapEndpoint.this.started) {
                response.cancel();
            }
            if (response.isCanceled() || response.getSendError() != null) {
                if (null != exchange) {
                    exchange.executeComplete();
                }
            } else {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeResponse(response, new ExchangeCallback(exchange, response)));
            }
        }

        @Override
        public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
            this.assertMessageHasDestinationAddress(message);
            CoapEndpoint.this.matcher.sendEmptyMessage(exchange, message);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendEmptyMessage(message);
            }
            message.setReadyToSend();
            if (!CoapEndpoint.this.started) {
                message.cancel();
            }
            if (message.isCanceled() || message.getSendError() != null) {
                if (null != exchange) {
                    exchange.executeComplete();
                }
            } else if (exchange != null) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeEmptyMessage(message, new ExchangeCallback(exchange, message)));
            } else {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeEmptyMessage(message, new SendingCallback(message)));
            }
        }

        private void assertMessageHasDestinationAddress(Message message) {
            if (message.getDestinationContext() == null) {
                throw new IllegalArgumentException("Message has no endpoint context");
            }
        }
    }

    private class NotificationDispatcher
    implements NotificationListener {
        private NotificationDispatcher() {
        }

        @Override
        public void onNotification(Request request, Response response) {
            for (NotificationListener notificationListener : CoapEndpoint.this.notificationListeners) {
                notificationListener.onNotification(request, response);
            }
        }
    }
}

