/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.Random;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageObserverAdapter;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.stack.AbstractLayer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReliabilityLayer
extends AbstractLayer {
    protected static final Logger LOGGER = LoggerFactory.getLogger(ReliabilityLayer.class.getCanonicalName());
    private final Random rand = new Random();
    private final int ack_timeout;
    private final float ack_random_factor;
    private final float ack_timeout_scale;
    private final int max_retransmit;

    public ReliabilityLayer(NetworkConfig config) {
        this.ack_timeout = config.getInt("ACK_TIMEOUT");
        this.ack_random_factor = config.getFloat("ACK_RANDOM_FACTOR");
        this.ack_timeout_scale = config.getFloat("ACK_TIMEOUT_SCALE");
        this.max_retransmit = config.getInt("MAX_RETRANSMIT");
        LOGGER.info("ReliabilityLayer uses ACK_TIMEOUT={}, ACK_RANDOM_FACTOR={}, and ACK_TIMEOUT_SCALE={}", this.ack_timeout, Float.valueOf(this.ack_random_factor), Float.valueOf(this.ack_timeout_scale));
    }

    @Override
    public void sendRequest(final Exchange exchange, final Request request) {
        LOGGER.debug("{} send request, failed transmissions: {}", (Object)exchange, (Object)exchange.getFailedTransmissionCount());
        if (request.getType() == null) {
            request.setType(CoAP.Type.CON);
        }
        if (request.getType() == CoAP.Type.CON) {
            LOGGER.debug("{} prepare retransmission for {}", (Object)exchange, (Object)request);
            this.prepareRetransmission(exchange, new RetransmissionTask(exchange, request){

                @Override
                public void retransmit() {
                    ReliabilityLayer.this.sendRequest(exchange, request);
                }
            });
        }
        this.lower().sendRequest(exchange, request);
    }

    @Override
    public void sendResponse(final Exchange exchange, final Response response) {
        LOGGER.debug("{} send response {}, failed transmissions: {}", exchange, response, exchange.getFailedTransmissionCount());
        CoAP.Type respType = response.getType();
        if (respType == null) {
            CoAP.Type reqType = exchange.getCurrentRequest().getType();
            if (reqType == CoAP.Type.CON) {
                if (exchange.getCurrentRequest().isAcknowledged()) {
                    response.setType(CoAP.Type.CON);
                } else {
                    exchange.getCurrentRequest().setAcknowledged(true);
                    response.setType(CoAP.Type.ACK);
                    response.setMID(exchange.getCurrentRequest().getMID());
                }
            } else {
                response.setType(CoAP.Type.NON);
            }
            LOGGER.trace("{} switched response message type from {} to {} (request was {})", new Object[]{exchange, respType, response.getType(), reqType});
        } else if (respType == CoAP.Type.ACK || respType == CoAP.Type.RST) {
            response.setMID(exchange.getCurrentRequest().getMID());
        }
        if (response.getType() == CoAP.Type.CON) {
            LOGGER.debug("{} prepare retransmission for {}", (Object)exchange, (Object)response);
            this.prepareRetransmission(exchange, new RetransmissionTask(exchange, response){

                @Override
                public void retransmit() {
                    ReliabilityLayer.this.sendResponse(exchange, response);
                }
            });
        }
        this.lower().sendResponse(exchange, response);
    }

    private void prepareRetransmission(final Exchange exchange, final RetransmissionTask task) {
        if (this.executor.isShutdown()) {
            LOGGER.info("Endpoint is being destroyed: skipping retransmission");
            return;
        }
        exchange.setRetransmissionHandle(null);
        this.updateRetransmissionTimeout(exchange);
        task.message.addMessageObserver(new MessageObserverAdapter(){

            @Override
            public void onSent() {
                task.message.removeMessageObserver(this);
                if (!exchange.isComplete()) {
                    exchange.execute(new Runnable(){

                        @Override
                        public void run() {
                            task.startTimer();
                        }
                    });
                }
            }
        });
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        if (request.isDuplicate()) {
            exchange.retransmitResponse();
            Response currentResponse = exchange.getCurrentResponse();
            if (currentResponse != null) {
                if (currentResponse.getType() == CoAP.Type.NON || currentResponse.getType() == CoAP.Type.CON) {
                    if (request.isConfirmable()) {
                        EmptyMessage ack = EmptyMessage.newACK(request);
                        this.sendEmptyMessage(exchange, ack);
                    }
                    if (currentResponse.isConfirmable()) {
                        int failedCount = exchange.getFailedTransmissionCount() + 1;
                        exchange.setFailedTransmissionCount(failedCount);
                        LOGGER.debug("{} request duplicate: retransmit response, failed: {}, response: {}", exchange, failedCount, currentResponse);
                        currentResponse.retransmitting();
                        this.sendResponse(exchange, currentResponse);
                        return;
                    }
                }
                LOGGER.debug("{} respond with the current response to the duplicate request", (Object)exchange);
                this.lower().sendResponse(exchange, currentResponse);
            } else if (exchange.getCurrentRequest().isAcknowledged()) {
                LOGGER.debug("{} duplicate request was acknowledged but no response computed yet. Retransmit ACK", (Object)exchange);
                EmptyMessage ack = EmptyMessage.newACK(request);
                this.sendEmptyMessage(exchange, ack);
            } else if (exchange.getCurrentRequest().isRejected()) {
                LOGGER.debug("{} duplicate request was rejected. Reject again", (Object)exchange);
                EmptyMessage rst = EmptyMessage.newRST(request);
                this.sendEmptyMessage(exchange, rst);
            } else {
                LOGGER.debug("{} server has not yet decided what to do with the request. We ignore the duplicate.", (Object)exchange);
            }
        } else {
            exchange.setCurrentRequest(request);
            this.upper().receiveRequest(exchange, request);
        }
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        exchange.setFailedTransmissionCount(0);
        exchange.setRetransmissionHandle(null);
        exchange.getCurrentRequest().setAcknowledged(true);
        if (response.getType() == CoAP.Type.CON && !exchange.getRequest().isCanceled()) {
            LOGGER.debug("{} acknowledging CON response", (Object)exchange);
            EmptyMessage ack = EmptyMessage.newACK(response);
            this.sendEmptyMessage(exchange, ack);
        }
        if (response.isDuplicate()) {
            LOGGER.debug("{} ignoring duplicate response", (Object)exchange);
        } else {
            this.upper().receiveResponse(exchange, response);
        }
    }

    @Override
    public void receiveEmptyMessage(Exchange exchange, EmptyMessage message) {
        Message currentMessage;
        String type;
        exchange.setFailedTransmissionCount(0);
        exchange.setRetransmissionHandle(null);
        if (exchange.isOfLocalOrigin()) {
            type = "request";
            currentMessage = exchange.getCurrentRequest();
        } else {
            type = "response";
            currentMessage = exchange.getCurrentResponse();
        }
        int observer = currentMessage.getMessageObservers().size();
        if (message.getType() == CoAP.Type.ACK) {
            LOGGER.debug("{} acknowledge {} for {} {} ({} msg observer)", exchange, message, type, currentMessage, observer);
            currentMessage.setAcknowledged(true);
        } else if (message.getType() == CoAP.Type.RST) {
            LOGGER.debug("{} reject {} for {} {} ({} msg observer)", exchange, message, type, currentMessage, observer);
            currentMessage.setRejected(true);
        } else {
            LOGGER.warn("{} received empty message that is neither ACK nor RST: {}", (Object)exchange, (Object)message);
            return;
        }
        this.upper().receiveEmptyMessage(exchange, message);
    }

    protected void updateRetransmissionTimeout(Exchange exchange) {
        int timeout = exchange.getFailedTransmissionCount() == 0 ? this.getRandomTimeout(this.ack_timeout, (int)((float)this.ack_timeout * this.ack_random_factor)) : (int)(this.ack_timeout_scale * (float)exchange.getCurrentTimeout());
        exchange.setCurrentTimeout(timeout);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int getRandomTimeout(int min, int max) {
        if (min >= max) {
            return min;
        }
        Random random = this.rand;
        synchronized (random) {
            return min + this.rand.nextInt(max - min + 1);
        }
    }

    protected abstract class RetransmissionTask
    implements Runnable {
        private final Exchange exchange;
        private final Message message;

        public RetransmissionTask(Exchange exchange, Message message) {
            this.exchange = exchange;
            this.message = message;
        }

        public void startTimer() {
            if (!this.exchange.isComplete()) {
                int timeout = this.exchange.getCurrentTimeout();
                ScheduledFuture<?> f = ReliabilityLayer.this.executor.schedule(this, (long)timeout, TimeUnit.MILLISECONDS);
                this.exchange.setRetransmissionHandle(f);
            }
        }

        @Override
        public void run() {
            this.exchange.execute(new Runnable(){

                @Override
                public void run() {
                    RetransmissionTask.this.retry();
                }
            });
        }

        private void retry() {
            try {
                this.exchange.setRetransmissionHandle(null);
                if (this.exchange.isComplete()) {
                    LOGGER.debug("Timeout: for {}, {}", (Object)this.exchange, (Object)this.message);
                    return;
                }
                int failedCount = this.exchange.getFailedTransmissionCount() + 1;
                this.exchange.setFailedTransmissionCount(failedCount);
                LOGGER.debug("Timeout: for {} retry {} of {}", this.exchange, failedCount, this.message);
                if (this.message.isAcknowledged()) {
                    LOGGER.trace("Timeout: for {} message already acknowledged, cancel retransmission of {}", (Object)this.exchange, (Object)this.message);
                    return;
                }
                if (this.message.isRejected()) {
                    LOGGER.trace("Timeout: for {} message already rejected, cancel retransmission of {}", (Object)this.exchange, (Object)this.message);
                    return;
                }
                if (this.message.isCanceled()) {
                    LOGGER.trace("Timeout: for {}, {} is canceled, do not retransmit", (Object)this.exchange, (Object)this.message);
                    return;
                }
                if (failedCount <= ReliabilityLayer.this.max_retransmit) {
                    LOGGER.debug("Timeout: for {} retransmit message, failed: {}, message: {}", this.exchange, failedCount, this.message);
                    this.message.retransmitting();
                    if (this.message.isCanceled()) {
                        LOGGER.trace("Timeout: for {}, {} got canceled, do not retransmit", (Object)this.exchange, (Object)this.message);
                        return;
                    }
                    this.retransmit();
                } else {
                    LOGGER.debug("Timeout: for {} retransmission limit reached, exchange failed, message: {}", (Object)this.exchange, (Object)this.message);
                    this.exchange.setTimedOut(this.message);
                }
            }
            catch (Exception e) {
                LOGGER.error("Exception for {} in MessageObserver: {}", this.exchange, e.getMessage(), e);
            }
        }

        public abstract void retransmit();
    }
}

