/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.eclipse.californium.core.coap.BlockOption;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.coap.Token;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.InMemoryMessageIdProvider;
import org.eclipse.californium.core.network.MessageExchangeStore;
import org.eclipse.californium.core.network.MessageIdProvider;
import org.eclipse.californium.core.network.RandomTokenGenerator;
import org.eclipse.californium.core.network.TokenGenerator;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.deduplication.Deduplicator;
import org.eclipse.californium.core.network.deduplication.DeduplicatorFactory;
import org.eclipse.californium.elements.util.ExecutorsUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryMessageExchangeStore
implements MessageExchangeStore {
    private static final Logger LOGGER = LoggerFactory.getLogger(InMemoryMessageExchangeStore.class.getName());
    private static final Logger HEALTH_LOGGER = LoggerFactory.getLogger(LOGGER.getName() + ".health");
    private final ConcurrentMap<Exchange.KeyMID, Exchange> exchangesByMID = new ConcurrentHashMap<Exchange.KeyMID, Exchange>();
    private final ConcurrentMap<Token, Exchange> exchangesByToken = new ConcurrentHashMap<Token, Exchange>();
    private volatile boolean enableStatus;
    private final NetworkConfig config;
    private final TokenGenerator tokenGenerator;
    private volatile boolean running = false;
    private volatile Deduplicator deduplicator;
    private volatile MessageIdProvider messageIdProvider;
    private ScheduledFuture<?> statusLogger;

    public InMemoryMessageExchangeStore(NetworkConfig config) {
        this(config, new RandomTokenGenerator(config));
        LOGGER.debug("using default TokenProvider {}", (Object)RandomTokenGenerator.class.getName());
    }

    public InMemoryMessageExchangeStore(NetworkConfig config, TokenGenerator tokenProvider) {
        if (config == null) {
            throw new NullPointerException("Configuration must not be null");
        }
        if (tokenProvider == null) {
            throw new NullPointerException("TokenProvider must not be null");
        }
        this.tokenGenerator = tokenProvider;
        this.config = config;
    }

    private void startStatusLogging() {
        int healthStatusInterval = this.config.getInt("HEALTH_STATUS_INTERVAL", 0);
        if (healthStatusInterval > 0 && HEALTH_LOGGER.isDebugEnabled()) {
            this.statusLogger = ExecutorsUtil.getScheduledExecutor().scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    if (InMemoryMessageExchangeStore.this.enableStatus) {
                        InMemoryMessageExchangeStore.this.dump(5);
                    }
                }
            }, healthStatusInterval, healthStatusInterval, TimeUnit.SECONDS);
        }
    }

    private String dumpCurrentLoadLevels() {
        StringBuilder b = new StringBuilder("MessageExchangeStore contents: ");
        b.append(this.exchangesByMID.size()).append(" exchanges by MID, ");
        b.append(this.exchangesByToken.size()).append(" exchanges by token, ");
        b.append(this.deduplicator.size()).append(" MIDs, ");
        return b.toString();
    }

    public synchronized void setDeduplicator(Deduplicator deduplicator) {
        if (this.running) {
            throw new IllegalStateException("Cannot set Deduplicator when store is already started");
        }
        if (deduplicator == null) {
            throw new NullPointerException("Deduplicator must not be null");
        }
        this.deduplicator = deduplicator;
    }

    public synchronized void setMessageIdProvider(MessageIdProvider provider) {
        if (this.running) {
            throw new IllegalStateException("Cannot set messageIdProvider when store is already started");
        }
        if (provider == null) {
            throw new NullPointerException("Message ID Provider must not be null");
        }
        this.messageIdProvider = provider;
    }

    @Override
    public boolean isEmpty() {
        return this.exchangesByMID.isEmpty() && this.exchangesByToken.isEmpty() && this.deduplicator.isEmpty();
    }

    public String toString() {
        return this.dumpCurrentLoadLevels();
    }

    @Override
    public int assignMessageId(Message message) {
        int mid = message.getMID();
        if (-1 == mid) {
            InetSocketAddress dest = message.getDestinationContext().getPeerAddress();
            mid = this.messageIdProvider.getNextMessageId(dest);
            if (-1 == mid) {
                LOGGER.warn("cannot send message to {}, all MIDs are in use", (Object)dest);
            } else {
                message.setMID(mid);
            }
        }
        return mid;
    }

    private int registerWithMessageId(Exchange exchange, Message message) {
        this.enableStatus = true;
        exchange.assertIncomplete(message);
        int mid = message.getMID();
        if (-1 == mid) {
            mid = this.assignMessageId(message);
            if (-1 != mid) {
                Exchange.KeyMID key = Exchange.KeyMID.fromOutboundMessage(message);
                if (this.exchangesByMID.putIfAbsent(key, exchange) != null) {
                    throw new IllegalArgumentException(String.format("generated mid [%d] already in use, cannot register %s", message.getMID(), exchange));
                }
                LOGGER.debug("{} added with generated mid {}, {}", exchange, key, message);
            }
        } else {
            Exchange.KeyMID key = Exchange.KeyMID.fromOutboundMessage(message);
            Exchange existingExchange = this.exchangesByMID.putIfAbsent(key, exchange);
            if (existingExchange != null) {
                if (existingExchange != exchange) {
                    throw new IllegalArgumentException(String.format("mid [%d] already in use, cannot register %s", message.getMID(), exchange));
                }
                if (exchange.getFailedTransmissionCount() == 0) {
                    throw new IllegalArgumentException(String.format("message with already registered mid [%d] is not a re-transmission, cannot register %s", message.getMID(), exchange));
                }
            } else {
                LOGGER.debug("{} added with {}, {}", exchange, key, message);
            }
        }
        return mid;
    }

    private void registerWithToken(Exchange exchange) {
        this.enableStatus = true;
        Request request = exchange.getCurrentRequest();
        exchange.assertIncomplete(request);
        Token token = request.getToken();
        if (token == null) {
            do {
                token = this.tokenGenerator.createToken(false);
                request.setToken(token);
            } while (this.exchangesByToken.putIfAbsent(token, exchange) != null);
            LOGGER.debug("{} added with generated token {}, {}", exchange, token, request);
        } else {
            if (token.isEmpty() && request.getCode() == null) {
                return;
            }
            Exchange previous = this.exchangesByToken.put(token, exchange);
            if (previous == null) {
                BlockOption block2 = request.getOptions().getBlock2();
                if (block2 != null) {
                    LOGGER.debug("block2 {} for block {} add with token {}", exchange, block2.getNum(), token);
                } else {
                    LOGGER.debug("{} added with token {}, {}", exchange, token, request);
                }
            } else if (previous != exchange) {
                if (!(exchange.getFailedTransmissionCount() != 0 || request.getOptions().hasBlock1() || request.getOptions().hasBlock2() || request.getOptions().hasObserve())) {
                    LOGGER.warn("{} with manual token overrides existing {} with open request: {}", exchange, previous, token);
                } else {
                    LOGGER.debug("{} replaced with token {}, {}", exchange, token, request);
                }
            } else {
                LOGGER.debug("{} keep for {}, {}", exchange, token, request);
            }
        }
    }

    @Override
    public boolean registerOutboundRequest(Exchange exchange) {
        if (exchange == null) {
            throw new NullPointerException("exchange must not be null");
        }
        if (exchange.getCurrentRequest() == null) {
            throw new IllegalArgumentException("exchange does not contain a request");
        }
        Request currentRequest = exchange.getCurrentRequest();
        int mid = this.registerWithMessageId(exchange, currentRequest);
        if (-1 != mid) {
            this.registerWithToken(exchange);
            if (exchange.getCurrentRequest() != currentRequest) {
                throw new ConcurrentModificationException("Current request modified!");
            }
            return true;
        }
        return false;
    }

    @Override
    public boolean registerOutboundRequestWithTokenOnly(Exchange exchange) {
        if (exchange == null) {
            throw new NullPointerException("exchange must not be null");
        }
        if (exchange.getCurrentRequest() == null) {
            throw new IllegalArgumentException("exchange does not contain a request");
        }
        Request currentRequest = exchange.getCurrentRequest();
        this.registerWithToken(exchange);
        if (exchange.getCurrentRequest() != currentRequest) {
            throw new ConcurrentModificationException("Current request modified!");
        }
        return true;
    }

    @Override
    public void remove(Token token, Exchange exchange) {
        boolean removed = this.exchangesByToken.remove(token, exchange);
        if (removed) {
            LOGGER.debug("removing {} for token {}", (Object)exchange, (Object)token);
        }
    }

    @Override
    public Exchange remove(Exchange.KeyMID messageId, Exchange exchange) {
        Exchange removedExchange = null == exchange ? (Exchange)this.exchangesByMID.remove(messageId) : (this.exchangesByMID.remove(messageId, exchange) ? exchange : null);
        if (null != removedExchange) {
            LOGGER.debug("removing {} for MID {}", (Object)removedExchange, (Object)messageId);
        }
        return removedExchange;
    }

    @Override
    public Exchange get(Token token) {
        if (token == null) {
            return null;
        }
        return (Exchange)this.exchangesByToken.get(token);
    }

    @Override
    public Exchange get(Exchange.KeyMID messageId) {
        if (messageId == null) {
            return null;
        }
        return (Exchange)this.exchangesByMID.get(messageId);
    }

    @Override
    public boolean registerOutboundResponse(Exchange exchange) {
        if (exchange == null) {
            throw new NullPointerException("exchange must not be null");
        }
        if (exchange.getCurrentResponse() == null) {
            throw new IllegalArgumentException("exchange does not contain a response");
        }
        Response currentResponse = exchange.getCurrentResponse();
        if (this.registerWithMessageId(exchange, currentResponse) > -1) {
            if (exchange.getCurrentResponse() != currentResponse) {
                throw new ConcurrentModificationException("Current response modified!");
            }
            return true;
        }
        return false;
    }

    @Override
    public synchronized void start() {
        if (!this.running) {
            this.startStatusLogging();
            if (this.deduplicator == null) {
                DeduplicatorFactory factory = DeduplicatorFactory.getDeduplicatorFactory();
                this.deduplicator = factory.createDeduplicator(this.config);
            }
            this.deduplicator.start();
            if (this.messageIdProvider == null) {
                LOGGER.debug("no MessageIdProvider set, using default {}", (Object)InMemoryMessageIdProvider.class.getName());
                this.messageIdProvider = new InMemoryMessageIdProvider(this.config);
            }
            this.running = true;
        }
    }

    @Override
    public synchronized void stop() {
        if (this.running) {
            this.running = false;
            for (Exchange exchange : this.exchangesByMID.values()) {
                exchange.getRequest().setCanceled(true);
            }
            if (this.statusLogger != null) {
                this.statusLogger.cancel(false);
                this.statusLogger = null;
            }
            this.deduplicator.stop();
            this.exchangesByMID.clear();
            this.exchangesByToken.clear();
        }
    }

    public void dump(int logMaxExchanges) {
        if (HEALTH_LOGGER.isDebugEnabled()) {
            HEALTH_LOGGER.debug(this.dumpCurrentLoadLevels());
            if (0 < logMaxExchanges) {
                if (!this.exchangesByMID.isEmpty()) {
                    this.dumpExchanges(logMaxExchanges, this.exchangesByMID.entrySet());
                }
                if (!this.exchangesByToken.isEmpty()) {
                    this.dumpExchanges(logMaxExchanges, this.exchangesByToken.entrySet());
                }
            }
        }
    }

    private <K> void dumpExchanges(int logMaxExchanges, Set<Map.Entry<K, Exchange>> exchangeEntries) {
        for (Map.Entry<K, Exchange> exchangeEntry : exchangeEntries) {
            String pending;
            Exchange exchange = exchangeEntry.getValue();
            Request origin = exchange.getRequest();
            Request current = exchange.getCurrentRequest();
            String string = pending = exchange.getRetransmissionHandle() == null ? "" : "/pending";
            if (origin != current && !origin.getToken().equals(current.getToken())) {
                HEALTH_LOGGER.debug("  {}, {}, retransmission {}{}, org {}, {}, {}", exchangeEntry.getKey(), exchange, exchange.getFailedTransmissionCount(), pending, origin.getToken(), current, exchange.getCurrentResponse());
            } else {
                String mark = origin == null ? "(missing origin request) " : "";
                HEALTH_LOGGER.debug("  {}, {}, retransmission {}{}, {}{}, {}", exchangeEntry.getKey(), exchange, exchange.getFailedTransmissionCount(), pending, mark, current, exchange.getCurrentResponse());
            }
            Throwable caller = exchange.getCaller();
            if (caller != null) {
                HEALTH_LOGGER.trace("  ", caller);
            }
            if (0 < --logMaxExchanges) continue;
            break;
        }
    }

    @Override
    public Exchange findPrevious(Exchange.KeyMID messageId, Exchange exchange) {
        return this.deduplicator.findPrevious(messageId, exchange);
    }

    @Override
    public Exchange find(Exchange.KeyMID messageId) {
        return this.deduplicator.find(messageId);
    }

    @Override
    public List<Exchange> findByToken(Token token) {
        ArrayList<Exchange> result = new ArrayList<Exchange>();
        if (token != null) {
            for (Map.Entry entry : this.exchangesByToken.entrySet()) {
                Request request;
                if (!((Exchange)entry.getValue()).isOfLocalOrigin() || (request = ((Exchange)entry.getValue()).getRequest()) == null || !token.equals(request.getToken())) continue;
                result.add((Exchange)entry.getValue());
            }
        }
        return result;
    }
}

