/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network.stack;

import java.util.Random;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.stack.AbstractLayer;

public class ReliabilityLayer
extends AbstractLayer {
    protected static final Logger LOGGER = Logger.getLogger(ReliabilityLayer.class.getCanonicalName());
    private Random rand = new Random();
    private NetworkConfig config;

    public ReliabilityLayer(NetworkConfig config) {
        this.config = config;
    }

    @Override
    public void sendRequest(final Exchange exchange, final Request request) {
        LOGGER.finer("Send request, failed transmissions: " + exchange.getFailedTransmissionCount());
        if (request.getType() == null) {
            request.setType(CoAP.Type.CON);
        }
        if (request.getType() == CoAP.Type.CON) {
            this.prepareRetransmission(exchange, new RetransmissionTask(exchange, request){

                @Override
                public void retransmit() {
                    ReliabilityLayer.this.sendRequest(exchange, request);
                }
            });
        }
        super.sendRequest(exchange, request);
    }

    @Override
    public void sendResponse(final Exchange exchange, final Response response) {
        LOGGER.finer("Send response, failed transmissions: " + exchange.getFailedTransmissionCount());
        CoAP.Type respType = response.getType();
        if (respType == null) {
            CoAP.Type reqType = exchange.getCurrentRequest().getType();
            if (reqType == CoAP.Type.CON) {
                if (exchange.getCurrentRequest().isAcknowledged()) {
                    response.setType(CoAP.Type.CON);
                } else {
                    exchange.getCurrentRequest().setAcknowledged(true);
                    response.setType(CoAP.Type.ACK);
                    response.setMID(exchange.getCurrentRequest().getMID());
                }
            } else {
                response.setType(CoAP.Type.NON);
            }
            LOGGER.finest("Switched response message type from " + (Object)((Object)respType) + " to " + (Object)((Object)response.getType()) + " (request was " + (Object)((Object)reqType) + ")");
        } else if (respType == CoAP.Type.ACK || respType == CoAP.Type.RST) {
            response.setMID(exchange.getCurrentRequest().getMID());
        }
        if (response.getType() == CoAP.Type.CON) {
            LOGGER.finer("Scheduling retransmission for " + response);
            this.prepareRetransmission(exchange, new RetransmissionTask(exchange, response){

                @Override
                public void retransmit() {
                    ReliabilityLayer.this.sendResponse(exchange, response);
                }
            });
        }
        super.sendResponse(exchange, response);
    }

    private void prepareRetransmission(Exchange exchange, RetransmissionTask task) {
        int timeout;
        if (exchange.getFailedTransmissionCount() == 0) {
            int ack_timeout = this.config.getInt("ACK_TIMEOUT");
            float ack_random_factor = this.config.getFloat("ACK_RANDOM_FACTOR");
            timeout = this.getRandomTimeout(ack_timeout, (int)((float)ack_timeout * ack_random_factor));
        } else {
            int ack_timeout_scale = this.config.getInt("ACK_TIMEOUT_SCALE");
            timeout = ack_timeout_scale * exchange.getCurrentTimeout();
        }
        exchange.setCurrentTimeout(timeout);
        ScheduledFuture<?> f = this.executor.schedule(task, (long)timeout, TimeUnit.MILLISECONDS);
        exchange.setRetransmissionHandle(f);
    }

    @Override
    public void receiveRequest(Exchange exchange, Request request) {
        if (request.isDuplicate()) {
            if (exchange.getCurrentResponse() != null) {
                LOGGER.fine("Respond with the current response to the duplicate request");
                super.sendResponse(exchange, exchange.getCurrentResponse());
            } else if (exchange.getCurrentRequest().isAcknowledged()) {
                LOGGER.fine("The duplicate request was acknowledged but no response computed yet. Retransmit ACK");
                EmptyMessage ack = EmptyMessage.newACK(request);
                this.sendEmptyMessage(exchange, ack);
            } else if (exchange.getCurrentRequest().isRejected()) {
                LOGGER.fine("The duplicate request was rejected. Reject again");
                EmptyMessage rst = EmptyMessage.newRST(request);
                this.sendEmptyMessage(exchange, rst);
            } else {
                LOGGER.fine("The server has not yet decided what to do with the request. We ignore the duplicate.");
            }
        } else {
            exchange.setCurrentRequest(request);
            super.receiveRequest(exchange, request);
        }
    }

    @Override
    public void receiveResponse(Exchange exchange, Response response) {
        exchange.setFailedTransmissionCount(0);
        exchange.getCurrentRequest().setAcknowledged(true);
        LOGGER.finest("Cancel any retransmission");
        exchange.setRetransmissionHandle(null);
        if (response.getType() == CoAP.Type.CON && !exchange.getRequest().isCanceled()) {
            LOGGER.finer("Response is confirmable, send ACK");
            EmptyMessage ack = EmptyMessage.newACK(response);
            this.sendEmptyMessage(exchange, ack);
        }
        if (response.isDuplicate()) {
            LOGGER.fine("Response is duplicate, ignore it");
        } else {
            super.receiveResponse(exchange, response);
        }
    }

    @Override
    public void receiveEmptyMessage(Exchange exchange, EmptyMessage message) {
        exchange.setFailedTransmissionCount(0);
        if (message.getType() == CoAP.Type.ACK) {
            if (exchange.getOrigin() == Exchange.Origin.LOCAL) {
                exchange.getCurrentRequest().setAcknowledged(true);
            } else {
                exchange.getCurrentResponse().setAcknowledged(true);
            }
        } else if (message.getType() == CoAP.Type.RST) {
            if (exchange.getOrigin() == Exchange.Origin.LOCAL) {
                exchange.getCurrentRequest().setRejected(true);
            } else {
                exchange.getCurrentResponse().setRejected(true);
            }
        } else {
            LOGGER.warning("Empty messgae was not ACK nor RST: " + message);
        }
        LOGGER.finer("Cancel retransmission");
        exchange.setRetransmissionHandle(null);
        super.receiveEmptyMessage(exchange, message);
    }

    private int getRandomTimeout(int min, int max) {
        if (min == max) {
            return min;
        }
        return min + this.rand.nextInt(max - min);
    }

    private abstract class RetransmissionTask
    implements Runnable {
        private Exchange exchange;
        private Message message;

        public RetransmissionTask(Exchange exchange, Message message) {
            this.exchange = exchange;
            this.message = message;
        }

        @Override
        public void run() {
            try {
                int failedCount = this.exchange.getFailedTransmissionCount() + 1;
                this.exchange.setFailedTransmissionCount(failedCount);
                if (this.message.isAcknowledged()) {
                    LOGGER.finest("Timeout: message already acknowledged, cancel retransmission of " + this.message);
                    return;
                }
                if (this.message.isRejected()) {
                    LOGGER.finest("Timeout: message already rejected, cancel retransmission of " + this.message);
                    return;
                }
                if (this.message.isCanceled()) {
                    LOGGER.finest("Timeout: canceled (MID=" + this.message.getMID() + "), do not retransmit");
                    return;
                }
                if (failedCount <= ReliabilityLayer.this.config.getInt("MAX_RETRANSMIT")) {
                    LOGGER.finer("Timeout: retransmit message, failed: " + failedCount + ", message: " + this.message);
                    this.message.retransmitting();
                    if (!this.message.isCanceled()) {
                        this.retransmit();
                    }
                } else {
                    LOGGER.info("Timeout: retransmission limit reached, exchange failed, message: " + this.message);
                    this.exchange.setTimedOut();
                    this.message.setTimedOut(true);
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }

        public abstract void retransmit();
    }
}

