/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.util.nat;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.SocketException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NioNatUtil
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(NioNatUtil.class);
    private static final int DATAGRAM_SIZE = 2048;
    public static final int NAT_TIMEOUT_MS = 30000;
    public static final int LB_TIMEOUT_MS = 15000;
    public static final int MAXIMUM_NAT_ENTRIES = 10000;
    private static final byte DTLS_RECORD_MINIMUM_LENGTH = 14;
    private static final byte DTLS_HANDSHAKE_RECORD = 22;
    private static final byte DTLS_1_X_MAJOR_VERSION = -2;
    private static final byte DTLS_1_0_MINOR_VERSION = -1;
    private static final byte DTLS_1_2_MINOR_VERSION = -3;
    private static final byte[] DTLS_CONTENT_TYPES = new byte[]{20, 21, 22, 23, 25};
    private static final int MESSAGE_DROPPING_LOG_INTERVAL_MS = 10000;
    private static final ThreadGroup NAT_THREAD_GROUP = new ThreadGroup("NAT");
    private static final AtomicInteger NAT_THREAD_COUNTER = new AtomicInteger();
    private final Thread proxyThread;
    private final Queue<Runnable> jobs = new ConcurrentLinkedQueue<Runnable>();
    private final String proxyName;
    private final List<NatAddress> destinations;
    private final List<NatAddress> staleDestinations;
    private final List<NatAddress> probeDestinations;
    private final List<NatAddress> pendingDestinations;
    private final ByteBuffer proxyBuffer;
    private final List<DatagramChannel> proxyChannels;
    private final ConcurrentMap<InetSocketAddress, NatEntry> nats = new ConcurrentHashMap<InetSocketAddress, NatEntry>();
    private final Selector selector = Selector.open();
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(2, new ThreadFactory(){

        @Override
        public Thread newThread(Runnable runnable) {
            Thread ret = new Thread(NAT_THREAD_GROUP, runnable, "NAT-DELAY-" + NAT_THREAD_COUNTER.getAndIncrement(), 0L);
            ret.setDaemon(true);
            return ret;
        }
    });
    private volatile boolean running = true;
    private final Random random = new Random(System.nanoTime());
    private AtomicLong messageDroppingLogTime = new AtomicLong();
    private AtomicLong forwardCounter = new AtomicLong();
    private AtomicLong backwardCounter = new AtomicLong();
    private AtomicLong wrongRoutedCounter = new AtomicLong();
    private AtomicLong droppedNoneDtls = new AtomicLong();
    private long lastWrongRoutedCounter;
    private AtomicLong timedoutEntriesCounter = new AtomicLong();
    private long lastTimedoutEntriesCounter;
    private AtomicInteger natTimeoutMillis = new AtomicInteger(30000);
    private AtomicInteger loadBalancerTimeoutMillis = new AtomicInteger(15000);
    private AtomicInteger maximumNumberOfNatEtries = new AtomicInteger(10000);
    private AtomicBoolean reverseNatUpdate = new AtomicBoolean();
    private AtomicBoolean dtlsFilter = new AtomicBoolean();
    private volatile MessageDropping forward;
    private volatile MessageDropping backward;
    private volatile MessageSizeLimit forwardSizeLimit;
    private volatile MessageSizeLimit backwardSizeLimit;
    private volatile MessageReordering reorder;

    public NioNatUtil(InetSocketAddress bindAddress, InetSocketAddress destination) throws IOException {
        this(Arrays.asList(bindAddress), destination);
    }

    public NioNatUtil(List<InetSocketAddress> bindAddresses, InetSocketAddress destination) throws IOException {
        if (bindAddresses.isEmpty()) {
            throw new IllegalArgumentException("Bind addresses must not be empty!");
        }
        this.proxyChannels = new ArrayList<DatagramChannel>();
        this.destinations = new ArrayList<NatAddress>();
        this.staleDestinations = new ArrayList<NatAddress>();
        this.probeDestinations = new ArrayList<NatAddress>();
        this.pendingDestinations = new ArrayList<NatAddress>();
        this.addDestination(destination);
        this.proxyBuffer = ByteBuffer.allocateDirect(2048);
        InetSocketAddress proxy = null;
        for (InetSocketAddress bindAddress : bindAddresses) {
            DatagramChannel proxyChannel = DatagramChannel.open();
            proxyChannel.configureBlocking(false);
            proxyChannel.bind(bindAddress);
            proxyChannel.register(this.selector, 1);
            this.proxyChannels.add(proxyChannel);
            if (proxy != null) continue;
            proxy = (InetSocketAddress)proxyChannel.getLocalAddress();
        }
        if (proxy == null) {
            proxy = bindAddresses.get(0);
        }
        this.proxyName = proxy.getHostString() + ":" + proxy.getPort();
        this.proxyThread = new Thread(NAT_THREAD_GROUP, this, "NAT-" + proxy.getPort());
        this.proxyThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addDestination(InetSocketAddress destination) {
        if (destination != null) {
            NatAddress dest = new NatAddress(destination);
            List<NatAddress> list = this.destinations;
            synchronized (list) {
                if (!this.destinations.contains(dest)) {
                    this.destinations.add(dest);
                    return true;
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeDestination(InetSocketAddress destination) {
        if (destination != null) {
            List<NatAddress> list = this.destinations;
            synchronized (list) {
                for (NatAddress address : this.destinations) {
                    if (!address.address.equals(destination)) continue;
                    this.destinations.remove(address);
                    address.remove();
                    return true;
                }
                for (NatAddress address : this.staleDestinations) {
                    if (!address.address.equals(destination)) continue;
                    this.staleDestinations.remove(address);
                    return true;
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addStaleDestinations() {
        boolean added = false;
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            for (NatAddress address : this.staleDestinations) {
                address.updateReceive();
                if (!this.destinations.add(address)) continue;
                added = true;
            }
            this.staleDestinations.clear();
        }
        return added;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        LOGGER.info("starting NAT {}.", (Object)this.proxyName);
        long lastTimeoutCheck = System.nanoTime();
        long lastLoadBalancerCheck = System.nanoTime();
        while (this.running) {
            try {
                long timeoutCheckMillis;
                Runnable job;
                if (this.messageDroppingLogTime.get() - System.nanoTime() < 0L) {
                    this.dumpMessageDroppingStatistic();
                }
                while ((job = this.jobs.poll()) != null) {
                    job.run();
                }
                long timeout = this.natTimeoutMillis.get();
                long socketTimeout = timeout > 0L ? timeout / 2L : 1000L;
                LOGGER.debug("Select {}ms, {} channels {} ready.", socketTimeout, this.selector.keys().size(), this.selector.selectedKeys().size());
                int num = this.selector.select(socketTimeout);
                if (num > 0) {
                    Set<SelectionKey> keys = this.selector.selectedKeys();
                    LOGGER.debug("Selected {} channels {} ready.", (Object)this.selector.keys().size(), (Object)keys.size());
                    for (SelectionKey key : keys) {
                        ((Buffer)this.proxyBuffer).clear();
                        Object attachment = key.attachment();
                        if (attachment != null) {
                            LOGGER.debug("backward");
                            NatEntry entry = (NatEntry)attachment;
                            if (entry.receive(this.proxyBuffer) <= 0) continue;
                            entry.backward(this.proxyBuffer);
                            continue;
                        }
                        if (this.destinations.isEmpty()) continue;
                        DatagramChannel channel = (DatagramChannel)key.channel();
                        InetSocketAddress source = (InetSocketAddress)channel.receive(this.proxyBuffer);
                        ((Buffer)this.proxyBuffer).flip();
                        if (this.dtlsFilter.get() && !this.isDtlsRecord(this.proxyBuffer)) {
                            this.droppedNoneDtls.incrementAndGet();
                            LOGGER.debug("drop none dtls {} bytes", (Object)this.proxyBuffer.limit());
                            continue;
                        }
                        NatEntry newEntry = this.getNatEntry(source, channel);
                        if (newEntry != null) {
                            MessageReordering before = this.reorder;
                            if (before != null) {
                                LOGGER.debug("reorder forward {} bytes", (Object)this.proxyBuffer.limit());
                                before.forward(source, newEntry, this.proxyBuffer);
                                continue;
                            }
                            LOGGER.debug("forward {} bytes", (Object)this.proxyBuffer.limit());
                            newEntry.forward(this.proxyBuffer);
                            continue;
                        }
                        LOGGER.debug("drop {} bytes, NAT entries exhausted (max. {})", (Object)this.proxyBuffer.limit(), (Object)this.maximumNumberOfNatEtries.get());
                    }
                    keys.clear();
                }
                long now = System.nanoTime();
                long balancerTimeout = this.loadBalancerTimeoutMillis.get();
                if (balancerTimeout > 0L && (timeoutCheckMillis = TimeUnit.NANOSECONDS.toMillis(now - lastLoadBalancerCheck)) > balancerTimeout / 4L) {
                    lastLoadBalancerCheck = now;
                    long expireNanos = now - TimeUnit.MILLISECONDS.toNanos(balancerTimeout);
                    List<NatAddress> list = this.destinations;
                    synchronized (list) {
                        this.revives(expireNanos);
                        Iterator<NatAddress> iterator = this.pendingDestinations.iterator();
                        while (iterator.hasNext()) {
                            NatAddress dest = iterator.next();
                            if (dest.getState() == NatAddressState.PENDING) continue;
                            iterator.remove();
                            this.destinations.add(dest);
                            LOGGER.warn("revived {}", (Object)dest.name);
                        }
                        this.expires(this.destinations, 1, expireNanos);
                        this.expires(this.pendingDestinations, 0, expireNanos);
                    }
                }
                if (timeout <= 0L || (timeoutCheckMillis = TimeUnit.NANOSECONDS.toMillis(now - lastTimeoutCheck)) <= timeout / 4L) continue;
                lastTimeoutCheck = now;
                long expireNanos = now - TimeUnit.MILLISECONDS.toNanos(timeout);
                Iterator iterator = this.nats.values().iterator();
                while (iterator.hasNext()) {
                    NatEntry entry = (NatEntry)iterator.next();
                    if (!entry.expires(expireNanos)) continue;
                    iterator.remove();
                    this.timedoutEntriesCounter.incrementAndGet();
                }
            }
            catch (SocketException e) {
                if (!this.running) continue;
                LOGGER.error("NAT {} to {} socket error", this.proxyName, this.getDestinationForLogging(), e);
            }
            catch (InterruptedIOException e) {
                if (!this.running) continue;
                LOGGER.error("NAT {} to {} interrupted", this.proxyName, this.getDestinationForLogging(), e);
            }
            catch (Exception e) {
                LOGGER.error("NAT {} to {} error", this.proxyName, this.getDestinationForLogging(), e);
            }
        }
    }

    private boolean isDtlsRecord(ByteBuffer packet) {
        if (packet.limit() < 14) {
            return false;
        }
        if (packet.get(1) != -2) {
            return false;
        }
        byte minorVersion = packet.get(2);
        if (minorVersion == -3) {
            byte data = packet.get(0);
            for (int index = 0; index < DTLS_CONTENT_TYPES.length; ++index) {
                byte type = DTLS_CONTENT_TYPES[index];
                if (data < type) {
                    return false;
                }
                if (data != type) continue;
                return true;
            }
        } else if (minorVersion == -1) {
            return packet.get(0) == 22;
        }
        return false;
    }

    private void expires(List<NatAddress> destinations, int minimum, long expireNanos) {
        if (destinations.size() > minimum) {
            Iterator<NatAddress> iterator = destinations.iterator();
            while (iterator.hasNext()) {
                NatAddress dest = iterator.next();
                if (!dest.expires(expireNanos)) continue;
                iterator.remove();
                dest.stale();
                this.staleDestinations.add(dest);
                LOGGER.warn("expires {}", (Object)dest.name);
                if (destinations.size() > minimum) continue;
                break;
            }
        }
    }

    private void revives(long expireNanos) {
        if (!this.staleDestinations.isEmpty()) {
            Iterator<NatAddress> iterator = this.staleDestinations.iterator();
            while (iterator.hasNext()) {
                NatAddress dest = iterator.next();
                if (!dest.expires(expireNanos)) continue;
                iterator.remove();
                dest.probe();
                this.probeDestinations.add(dest);
                LOGGER.warn("revive {}", (Object)dest.name);
            }
        }
    }

    private NatEntry getNatEntry(InetSocketAddress source, DatagramChannel proxyChannel) throws IOException {
        NatEntry previousEntry;
        NatEntry entry = (NatEntry)this.nats.get(source);
        if (entry == null && this.nats.size() < this.maximumNumberOfNatEtries.get() && (previousEntry = this.nats.putIfAbsent(source, entry = new NatEntry(source, proxyChannel, this.selector))) != null) {
            entry.stop();
            entry = previousEntry;
        }
        return entry;
    }

    private void runTask(Runnable run) {
        this.jobs.add(run);
        this.selector.wakeup();
    }

    public void stop() {
        if (this.reorder != null) {
            this.reorder.stop();
        }
        this.running = false;
        for (DatagramChannel proxyChannel : this.proxyChannels) {
            try {
                proxyChannel.close();
            }
            catch (IOException e) {
                LOGGER.error("io-error on close!", e);
            }
        }
        this.proxyChannels.clear();
        this.proxyThread.interrupt();
        this.stopAllNatEntries();
        this.scheduler.shutdownNow();
        try {
            this.proxyThread.join(1000L);
            this.scheduler.awaitTermination(1000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ex) {
            LOGGER.error("shutdown failed!", ex);
        }
        try {
            this.selector.close();
        }
        catch (IOException e) {
            LOGGER.error("io-error on close!", e);
        }
        LOGGER.warn("NAT {} stopped. {} forwarded messages, {} backwarded", this.proxyName, this.forwardCounter, this.backwardCounter);
    }

    public int stopAllNatEntries() {
        return this.stopNatEntries(this.nats.size());
    }

    public int stopNatEntries(int num) {
        int counter = 0;
        Iterator iterator = this.nats.values().iterator();
        while (num > 0 && iterator.hasNext()) {
            NatEntry entry = (NatEntry)iterator.next();
            iterator.remove();
            entry.stop();
            --num;
            ++counter;
        }
        return counter;
    }

    public void setNatTimeoutMillis(int natTimeoutMillis) {
        this.natTimeoutMillis.set(natTimeoutMillis);
    }

    public void setLoadBalancerTimeoutMillis(int loadBalancerTimeoutMillis) {
        this.loadBalancerTimeoutMillis.set(loadBalancerTimeoutMillis);
    }

    public int getNumberOfEntries() {
        return this.nats.size();
    }

    public int getNumberOfDestinations() {
        return this.destinations.size();
    }

    public int getNumberOfStaleDestinations() {
        return this.staleDestinations.size();
    }

    public int getNumberOfProbeDestinations() {
        return this.probeDestinations.size();
    }

    public int getNumberOfPendingDestinations() {
        return this.pendingDestinations.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<NatAddress> getDestinations() {
        ArrayList<NatAddress> result = new ArrayList<NatAddress>();
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            result.addAll(this.destinations);
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<NatAddress> getStaleDestinations() {
        ArrayList<NatAddress> result = new ArrayList<NatAddress>();
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            result.addAll(this.staleDestinations);
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<NatAddress> getProbeDestinations() {
        ArrayList<NatAddress> result = new ArrayList<NatAddress>();
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            result.addAll(this.probeDestinations);
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<NatAddress> getPendingDestinations() {
        ArrayList<NatAddress> result = new ArrayList<NatAddress>();
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            result.addAll(this.pendingDestinations);
        }
        return result;
    }

    public int reassignDestinationAddresses() {
        int count = 0;
        if (this.destinations.size() > 1) {
            for (NatEntry entry : this.nats.values()) {
                if (!entry.setDestination(this.getRandomDestination())) continue;
                ++count;
            }
        }
        return count;
    }

    public void reassignNewLocalAddresses() {
        if (Thread.currentThread() != this.proxyThread) {
            final CountDownLatch ready = new CountDownLatch(1);
            this.runTask(new Runnable(){

                @Override
                public void run() {
                    NioNatUtil.this.reassignNewLocalAddresses();
                    ready.countDown();
                }
            });
            try {
                ready.await();
            }
            catch (InterruptedException interruptedException) {}
        } else {
            ArrayList<NatEntry> olds = new ArrayList<NatEntry>(this.nats.size());
            ArrayList entries = new ArrayList(this.nats.entrySet());
            for (Map.Entry entry : entries) {
                InetSocketAddress incoming = (InetSocketAddress)entry.getKey();
                try {
                    NatEntry oldentry = (NatEntry)entry.getValue();
                    NatEntry newEntry = new NatEntry((InetSocketAddress)entry.getKey(), oldentry.proxyChannel, this.selector);
                    this.nats.put(incoming, newEntry);
                    oldentry.setIncoming(null);
                    olds.add(oldentry);
                    LOGGER.info("changed NAT for {} from {} to {}.", incoming, oldentry.getPort(), newEntry.getPort());
                }
                catch (IOException e) {
                    LOGGER.error("Failed to reassing NAT entry for {}.", (Object)incoming, (Object)e);
                }
            }
            for (NatEntry natEntry : olds) {
                natEntry.stop();
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int assignLocalAddress(final InetSocketAddress incoming) throws IOException {
        DatagramChannel channel;
        if (Thread.currentThread() != this.proxyThread) {
            final AtomicInteger port = new AtomicInteger();
            final AtomicReference error = new AtomicReference();
            final CountDownLatch ready = new CountDownLatch(1);
            this.runTask(new Runnable(){

                @Override
                public void run() {
                    try {
                        int p = NioNatUtil.this.assignLocalAddress(incoming);
                        port.set(p);
                    }
                    catch (IOException e) {
                        error.set(e);
                    }
                    ready.countDown();
                }
            });
            try {
                ready.await();
                if (error.get() == null) return port.get();
                throw (IOException)error.get();
            }
            catch (InterruptedException e) {
                return -1;
            }
        }
        NatEntry old = (NatEntry)this.nats.get(incoming);
        if (old == null) {
            if (this.proxyChannels.size() != 1) {
                LOGGER.error("No NAT for {}.", (Object)incoming);
                return -1;
            }
            channel = this.proxyChannels.get(0);
        } else {
            channel = old.proxyChannel;
        }
        NatEntry entry = new NatEntry(incoming, channel, this.selector);
        old = this.nats.put(incoming, entry);
        if (null != old) {
            LOGGER.info("changed NAT for {} from {} to {}.", incoming, old.getPort(), entry.getPort());
            old.stop();
            return entry.getPort();
        }
        LOGGER.info("add NAT for {} to {}.", (Object)incoming, (Object)entry.getPort());
        return entry.getPort();
    }

    public void mixLocalAddresses() {
        Random random = new Random();
        ArrayList<NatEntry> destinations = new ArrayList<NatEntry>();
        HashSet keys = new HashSet(this.nats.keySet());
        for (InetSocketAddress incoming : keys) {
            NatEntry entry = (NatEntry)this.nats.remove(incoming);
            entry.setIncoming(null);
            destinations.add(entry);
        }
        for (InetSocketAddress incoming : keys) {
            int index = random.nextInt(destinations.size());
            NatEntry entry = (NatEntry)destinations.remove(index);
            entry.setIncoming(incoming);
            NatEntry temp = this.nats.put(incoming, entry);
            if (temp == null || temp == entry) continue;
            temp.stop();
        }
    }

    public boolean removeLocalAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.remove(incoming);
        if (null != entry) {
            entry.stop();
        } else {
            LOGGER.warn("no mapping found for {}!", (Object)incoming);
        }
        return null != entry;
    }

    public int getLocalPortForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getPort();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return -1;
    }

    public InetSocketAddress getLocalAddressForAddress(InetSocketAddress incoming) {
        NatEntry entry = (NatEntry)this.nats.get(incoming);
        if (null != entry) {
            return entry.getSocketAddres();
        }
        LOGGER.warn("no mapping found for {}!", (Object)incoming);
        return null;
    }

    public InetSocketAddress getProxySocketAddress() throws IOException {
        return (InetSocketAddress)this.proxyChannels.get(0).getLocalAddress();
    }

    public void setMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null || this.backward != null) {
                this.forward = null;
                this.backward = null;
                LOGGER.info("NAT stops message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            this.backward = new MessageDropping("responses", percent);
            LOGGER.info("NAT message dropping {}%.", (Object)percent);
        }
    }

    public void setForwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forward != null) {
                this.forward = null;
                LOGGER.info("NAT stops forward message dropping.");
            }
        } else {
            this.forward = new MessageDropping("request", percent);
            LOGGER.info("NAT forward message dropping {}%.", (Object)percent);
        }
    }

    public void setBackwardMessageDropping(int percent) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message dropping " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backward != null) {
                this.backward = null;
                LOGGER.info("NAT stops backward message dropping.");
            }
        } else {
            this.backward = new MessageDropping("response", percent);
            LOGGER.info("NAT backward message dropping {}%.", (Object)percent);
        }
    }

    public void setMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null || this.backwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            this.backwardSizeLimit = new MessageSizeLimit("responses", percent, sizeLimit, drop);
            LOGGER.info("NAT message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setForwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.forwardSizeLimit != null) {
                this.forwardSizeLimit = null;
                LOGGER.info("NAT stops forward message size limit.");
            }
        } else {
            this.forwardSizeLimit = new MessageSizeLimit("request", percent, sizeLimit, drop);
            LOGGER.info("NAT forward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setBackwardMessageSizeLimit(int percent, int sizeLimit, boolean drop) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message size limit " + percent + "% out of range [0...100]!");
        }
        if (percent == 0) {
            if (this.backwardSizeLimit != null) {
                this.backwardSizeLimit = null;
                LOGGER.info("NAT stops backward message size limit.");
            }
        } else {
            this.backwardSizeLimit = new MessageSizeLimit("response", percent, sizeLimit, drop);
            LOGGER.info("NAT backward message size limit {} bytes, {}%.", (Object)sizeLimit, (Object)percent);
        }
    }

    public void setMessageReordering(int percent, int delayMillis, int randomDelayMillis) {
        if (percent < 0 || percent > 100) {
            throw new IllegalArgumentException("Message reordering " + percent + "% out of range [0...100]!");
        }
        if (this.reorder != null) {
            this.reorder.stop();
        }
        if (percent == 0) {
            if (this.reorder != null) {
                this.reorder = null;
                LOGGER.info("NAT stops message reordering.");
            }
        } else {
            this.reorder = new MessageReordering("reordering", percent, delayMillis, randomDelayMillis);
            LOGGER.info("NAT message reordering {}%.", (Object)percent);
        }
    }

    public void setReverseNatUpdate(boolean reverseUpdate) {
        this.reverseNatUpdate.set(reverseUpdate);
    }

    public boolean useReverseNatUpdate() {
        return this.reverseNatUpdate.get();
    }

    public void setMaxiumNumberOfNatEntries(int maximumNumber) {
        this.maximumNumberOfNatEtries.set(maximumNumber);
    }

    public int getMaxiumNumberOfNatEntries() {
        return this.maximumNumberOfNatEtries.get();
    }

    public void setDtlsFilter(boolean dtlsFilter) {
        this.dtlsFilter.set(dtlsFilter);
    }

    public boolean useDtlsFilter() {
        return this.dtlsFilter.get();
    }

    public void dumpMessageDroppingStatistic() {
        long current;
        this.messageDroppingLogTime.set(System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(10000L));
        TransmissionManipulation drops = this.forward;
        if (drops != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backward) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.forwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
        if ((drops = this.backwardSizeLimit) != null) {
            drops.dumpStatistic();
        }
        if (this.lastWrongRoutedCounter < (current = this.wrongRoutedCounter.get())) {
            LOGGER.warn("wrong routed messages {} (overall {}).", (Object)(current - this.lastWrongRoutedCounter), (Object)this.lastWrongRoutedCounter);
            this.lastWrongRoutedCounter = current;
        }
        if (this.lastTimedoutEntriesCounter < (current = this.timedoutEntriesCounter.get())) {
            LOGGER.warn("timed out NAT entries {} (overall {}).", (Object)(current - this.lastTimedoutEntriesCounter), (Object)this.lastTimedoutEntriesCounter);
            int stale = this.getNumberOfStaleDestinations();
            int destinations = this.getNumberOfDestinations();
            if (stale > 0) {
                LOGGER.warn("{} destinations, {} stale destinations.", (Object)destinations, (Object)stale);
            }
            this.lastTimedoutEntriesCounter = current;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NatAddress getRandomDestination() {
        if (this.destinations.isEmpty()) {
            return null;
        }
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            if (this.probeDestinations.isEmpty()) {
                int size = this.destinations.size();
                if (size == 1) {
                    return this.destinations.get(0);
                }
                int index = this.random.nextInt(size);
                return this.destinations.get(index);
            }
            NatAddress destination = this.probeDestinations.remove(0);
            destination.pending();
            this.pendingDestinations.add(destination);
            return destination;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NatAddress getDestination(InetSocketAddress destination) {
        if (destination != null) {
            List<NatAddress> list = this.destinations;
            synchronized (list) {
                for (NatAddress address : this.destinations) {
                    if (!address.address.equals(destination)) continue;
                    return address;
                }
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String getDestinationForLogging() {
        if (this.destinations.isEmpty()) {
            return "---";
        }
        List<NatAddress> list = this.destinations;
        synchronized (list) {
            int size = this.destinations.size();
            if (size == 1) {
                return this.destinations.get((int)0).name;
            }
            return this.destinations.get((int)0).name + "-" + this.destinations.get((int)(size - 1)).name;
        }
    }

    static {
        NAT_THREAD_GROUP.setDaemon(false);
    }

    private static class MessageSizeLimit
    extends TransmissionManipulation {
        private final boolean drop;
        private final int sizeLimit;

        public MessageSizeLimit(String title, int threshold, int sizeLimit, boolean drop) {
            super(title + " size limit", threshold);
            this.sizeLimit = sizeLimit;
            this.drop = drop;
        }

        public Manipulation limitMessageSize(ByteBuffer packet) {
            if (packet.position() > this.sizeLimit && this.manipulateMessage()) {
                if (this.drop) {
                    return Manipulation.DROP;
                }
                packet.position(this.sizeLimit);
                return Manipulation.LIMIT;
            }
            return Manipulation.NONE;
        }

        private static enum Manipulation {
            NONE,
            DROP,
            LIMIT;

        }
    }

    private static class MessageDropping
    extends TransmissionManipulation {
        public MessageDropping(String title, int threshold) {
            super(title + " drops", threshold);
        }

        public boolean dropMessage() {
            return this.manipulateMessage();
        }
    }

    public static class NatAddress {
        public final InetSocketAddress address;
        public final String name;
        private final AtomicInteger usageCounter;
        private long lastNanos;
        private boolean expired;
        private NatAddressState state;

        private NatAddress(InetSocketAddress address) {
            this.address = address;
            this.name = address.getHostString() + ":" + address.getPort();
            this.usageCounter = new AtomicInteger();
            this.state = NatAddressState.ACTIVE;
            this.updateReceive();
        }

        private synchronized void stale() {
            this.state = NatAddressState.STALE;
            this.expired = false;
            this.lastNanos = System.nanoTime();
        }

        private synchronized void probe() {
            this.state = NatAddressState.PROBING;
            this.expired = false;
            this.lastNanos = -1L;
        }

        private synchronized void pending() {
            this.state = NatAddressState.PENDING;
            this.expired = false;
            this.lastNanos = -1L;
        }

        private synchronized void remove() {
            this.state = NatAddressState.REMOVED;
        }

        private synchronized void updateUsage() {
            if (!this.expired) {
                this.lastNanos = System.nanoTime();
            }
        }

        private synchronized void updateSend() {
            if (!this.expired && this.lastNanos < 0L) {
                this.lastNanos = System.nanoTime();
            }
        }

        private synchronized void updateReceive() {
            this.expired = false;
            this.state = NatAddressState.ACTIVE;
            this.lastNanos = -1L;
        }

        private synchronized boolean expires(long expireNanos) {
            if (!this.expired) {
                this.expired = this.lastNanos > 0L && expireNanos - this.lastNanos > 0L;
            }
            return this.expired;
        }

        public int hashCode() {
            return this.address.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            NatAddress other = (NatAddress)obj;
            return this.address.equals(other.address);
        }

        public int usageCounter() {
            return this.usageCounter.get();
        }

        public synchronized long lastUsage() {
            long usage = -2L;
            if (!this.expired) {
                usage = this.lastNanos > 0L ? TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - this.lastNanos) : -1L;
            }
            return usage;
        }

        public synchronized NatAddressState getState() {
            return this.state;
        }

        public synchronized boolean usable() {
            if (this.state == NatAddressState.ACTIVE) {
                return !this.expired;
            }
            return false;
        }
    }

    private class NatEntry {
        private final DatagramChannel proxyChannel;
        private final DatagramChannel outgoing;
        private final String natName;
        private final InetSocketAddress local;
        private NatAddress incoming;
        private NatAddress destination;
        private boolean first;

        public NatEntry(InetSocketAddress incoming, DatagramChannel proxyChannel, Selector selector) throws IOException {
            this.setDestination(NioNatUtil.this.getRandomDestination());
            this.proxyChannel = proxyChannel;
            this.outgoing = DatagramChannel.open();
            this.outgoing.configureBlocking(false);
            this.outgoing.bind(null);
            this.local = (InetSocketAddress)this.outgoing.getLocalAddress();
            this.natName = Integer.toString(this.local.getPort());
            this.setIncoming(incoming);
            this.outgoing.register(selector, 1, this);
        }

        public synchronized boolean setDestination(NatAddress destination) {
            if (this.destination == destination) {
                return false;
            }
            if (this.destination != null) {
                if (this.destination.equals(destination)) {
                    return false;
                }
                this.destination.usageCounter.decrementAndGet();
            }
            this.destination = destination;
            if (this.destination != null) {
                this.destination.usageCounter.incrementAndGet();
                this.first = true;
            }
            return true;
        }

        public synchronized void setIncoming(InetSocketAddress incoming) {
            this.incoming = incoming != null ? new NatAddress(incoming) : null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean expires(long expireNanos) {
            NatAddress incoming;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
            }
            if (incoming == null) {
                return true;
            }
            if (incoming.expires(expireNanos)) {
                this.stop();
                LOGGER.info("expired listen on {} for incoming {}", (Object)this.natName, (Object)incoming.name);
                return true;
            }
            return false;
        }

        public void stop() {
            try {
                if (this.destination != null) {
                    this.destination.usageCounter.decrementAndGet();
                }
                this.outgoing.close();
            }
            catch (IOException e) {
                LOGGER.error("IO-error on closing", e);
            }
        }

        public InetSocketAddress getSocketAddres() {
            return this.local;
        }

        public int getPort() {
            return this.local.getPort();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public int receive(ByteBuffer packet) throws IOException {
            NatAddress destination;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                destination = this.destination;
            }
            try {
                SocketAddress source = this.outgoing.receive(packet);
                ((Buffer)packet).flip();
                if (destination.address.equals(source)) {
                    destination.updateReceive();
                } else {
                    NioNatUtil.this.wrongRoutedCounter.incrementAndGet();
                    if (NioNatUtil.this.reverseNatUpdate.get()) {
                        NatAddress newDestination = NioNatUtil.this.getDestination((InetSocketAddress)source);
                        this.setDestination(newDestination);
                    } else {
                        ((Buffer)packet).clear();
                    }
                }
                return packet.limit();
            }
            catch (IOException ex) {
                return -1;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void backward(ByteBuffer packet) throws IOException {
            NatAddress destination;
            NatAddress incoming;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
                destination = this.destination;
            }
            if (incoming == null) {
                return;
            }
            incoming.updateUsage();
            MessageDropping dropping = NioNatUtil.this.backward;
            if (dropping != null && dropping.dropMessage()) {
                LOGGER.debug("backward drops {} bytes from {} to {} via {}", packet.limit(), destination.name, incoming.name, this.natName);
            } else {
                MessageSizeLimit limit = NioNatUtil.this.backwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("backward {} bytes from {} to {} via {}", packet.limit(), destination.name, incoming.name, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("backward drops {} bytes from {} to {} via {}", packet.limit(), destination.name, incoming.name, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("backward limited {} bytes from {} to {} via {}", packet.limit(), destination.name, incoming.name, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    if (this.proxyChannel.send(packet, incoming.address) == 0) {
                        LOGGER.debug("backward overloaded {} bytes from {} to {} via {}", packet.limit(), destination.name, incoming.name, this.natName);
                    } else {
                        NioNatUtil.this.backwardCounter.incrementAndGet();
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean forward(ByteBuffer packet) throws IOException {
            MessageDropping dropping;
            boolean first;
            NatAddress destination;
            NatAddress incoming;
            NatEntry natEntry = this;
            synchronized (natEntry) {
                incoming = this.incoming;
                destination = this.destination;
                first = this.first;
                this.first = false;
            }
            if (incoming == null) {
                LOGGER.debug("forward drops {} bytes, no incoming address.", (Object)packet.limit());
                return false;
            }
            incoming.updateUsage();
            if (!first && !destination.usable()) {
                destination = NioNatUtil.this.getRandomDestination();
                this.setDestination(destination);
            }
            if ((dropping = NioNatUtil.this.forward) != null && dropping.dropMessage()) {
                LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
            } else {
                MessageSizeLimit limit = NioNatUtil.this.forwardSizeLimit;
                MessageSizeLimit.Manipulation manipulation = limit != null ? limit.limitMessageSize(packet) : MessageSizeLimit.Manipulation.NONE;
                switch (manipulation) {
                    case NONE: {
                        LOGGER.debug("forward {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
                        break;
                    }
                    case DROP: {
                        LOGGER.debug("forward drops {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
                        break;
                    }
                    case LIMIT: {
                        LOGGER.debug("forward limited {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
                    }
                }
                if (manipulation != MessageSizeLimit.Manipulation.DROP) {
                    if (this.outgoing.send(packet, destination.address) == 0) {
                        LOGGER.info("forward overloaded {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
                        return false;
                    }
                    destination.updateSend();
                    NioNatUtil.this.forwardCounter.incrementAndGet();
                    LOGGER.debug("forwarded {} bytes from {} to {} via {}", packet.limit(), incoming.name, destination.name, this.natName);
                }
            }
            return true;
        }
    }

    private class MessageReordering
    extends TransmissionManipulation {
        private final int delayMillis;
        private final int randomDelayMillis;
        private boolean reordering;

        public MessageReordering(String title, int threshold, int delayMillis, int randomDelayMillis) {
            super(title + " reorders", threshold);
            this.reordering = true;
            this.delayMillis = delayMillis;
            this.randomDelayMillis = randomDelayMillis;
        }

        public void forward(final InetSocketAddress source, NatEntry entry, ByteBuffer data) throws IOException {
            if (!this.isRunning()) {
                return;
            }
            if (this.manipulateMessage()) {
                final ByteBuffer clone = ByteBuffer.allocate(data.limit());
                clone.put(data);
                ((Buffer)clone).flip();
                final long delay = this.delayMillis + this.random.nextInt(this.randomDelayMillis);
                NioNatUtil.this.scheduler.schedule(new Runnable(){

                    @Override
                    public void run() {
                        if (MessageReordering.this.isRunning()) {
                            try {
                                LOGGER.info("deliver message {} bytes, delayed {}ms for {}", clone.limit(), delay, source);
                                NatEntry entry = (NatEntry)NioNatUtil.this.nats.get(source);
                                if (entry != null) {
                                    entry.forward(clone);
                                }
                            }
                            catch (IOException ex) {
                                LOGGER.info("delayed forward failed!", ex);
                            }
                        }
                    }
                }, delay, TimeUnit.MILLISECONDS);
            } else {
                entry.forward(data);
            }
        }

        public synchronized void stop() {
            this.reordering = false;
        }

        private synchronized boolean isRunning() {
            return this.reordering;
        }
    }

    public static enum NatAddressState {
        REMOVED,
        STALE,
        PROBING,
        PENDING,
        ACTIVE;

    }

    private static class TransmissionManipulation {
        private final String title;
        protected final Random random = new Random();
        private final int threshold;
        private final AtomicInteger sentMessages = new AtomicInteger();
        private final AtomicInteger manipulatedMessages = new AtomicInteger();

        public TransmissionManipulation(String title, int threshold) {
            this.title = title;
            this.threshold = threshold;
            this.random.setSeed(threshold);
        }

        public boolean manipulateMessage() {
            if (this.threshold == 0) {
                return false;
            }
            if (this.threshold == 100) {
                return true;
            }
            if (this.threshold > this.random.nextInt(100)) {
                this.manipulatedMessages.incrementAndGet();
                return true;
            }
            this.sentMessages.incrementAndGet();
            return false;
        }

        public void dumpStatistic() {
            int sent = this.sentMessages.get();
            int manipulated = this.manipulatedMessages.get();
            if (sent > 0) {
                LOGGER.warn("manipulated {} {}/{}%, sent {} {}.", this.title, manipulated, manipulated * 100 / (manipulated + sent), this.title, sent);
            } else if (manipulated > 0) {
                LOGGER.warn("manipulated {} {}/100%, no {} sent!.", this.title, manipulated, this.title);
            }
        }
    }
}

