/*
 * Decompiled with CFR 0.152.
 */
package org.dspace.util;

import jakarta.inject.Inject;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.Year;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;
import org.dspace.servicemanager.DSpaceKernelInit;

public class MultiFormatDateParser {
    private static final Logger log = LogManager.getLogger();
    private static final ArrayList<Rule> rules = new ArrayList();
    private static final ZoneId UTC_ZONE = ZoneOffset.UTC;

    @Inject
    public void setPatterns(Map<String, String> patterns) {
        for (Map.Entry<String, String> rule : patterns.entrySet()) {
            DateTimeFormatter format;
            Pattern pattern;
            try {
                pattern = Pattern.compile(rule.getKey(), 2);
            }
            catch (PatternSyntaxException ex) {
                Supplier[] supplierArray = new Supplier[1];
                supplierArray[0] = rule::getKey;
                log.error("Skipping format with unparsable pattern '{}'", supplierArray);
                continue;
            }
            try {
                format = DateTimeFormatter.ofPattern(rule.getValue()).withZone(UTC_ZONE);
            }
            catch (IllegalArgumentException ex) {
                Supplier[] supplierArray = new Supplier[1];
                supplierArray[0] = rule::getValue;
                log.error("Skipping uninterpretable date format '{}'", supplierArray);
                continue;
            }
            Rule.DateGranularity granularity = MultiFormatDateParser.getGranularity(rule.getValue());
            rules.add(new Rule(pattern, format, granularity));
        }
    }

    private static Rule.DateGranularity getGranularity(String datePattern) {
        Rule.DateGranularity granularity = datePattern.contains("HH") ? Rule.DateGranularity.TIME : (datePattern.contains("dd") ? Rule.DateGranularity.DAY : (datePattern.contains("MM") ? Rule.DateGranularity.MONTH : Rule.DateGranularity.YEAR));
        return granularity;
    }

    public static ZonedDateTime parse(String dateString) {
        for (Rule candidate : rules) {
            if (!candidate.pattern.matcher(dateString).matches()) continue;
            ZonedDateTime result = null;
            try {
                switch (candidate.granularity) {
                    case TIME: {
                        result = ZonedDateTime.parse(dateString, candidate.format);
                        break;
                    }
                    case DAY: {
                        result = LocalDate.parse(dateString, candidate.format).atStartOfDay(ZoneOffset.UTC);
                        break;
                    }
                    case MONTH: {
                        result = YearMonth.parse(dateString, candidate.format).atDay(1).atStartOfDay(ZoneOffset.UTC);
                        break;
                    }
                    case YEAR: {
                        result = Year.parse(dateString, candidate.format).atMonth(1).atDay(1).atStartOfDay(ZoneOffset.UTC);
                        break;
                    }
                    default: {
                        throw new DateTimeException("Could not find a valid parser for this matched pattern.");
                    }
                }
            }
            catch (DateTimeParseException ex) {
                Supplier[] supplierArray = new Supplier[3];
                supplierArray[0] = () -> dateString;
                supplierArray[1] = candidate.format::toString;
                supplierArray[2] = ex::getMessage;
                log.info("Date string '{}' matched pattern '{}' but did not parse:  {}", supplierArray);
                continue;
            }
            return result;
        }
        return null;
    }

    public static void main(String[] args) throws IOException {
        DSpaceKernelInit.getKernel(null);
        if (args.length > 0) {
            for (String arg : args) {
                MultiFormatDateParser.testDate(arg);
            }
        } else if (null == System.console()) {
            String line;
            BufferedReader input = new BufferedReader(new InputStreamReader(System.in));
            while (null != (line = input.readLine())) {
                MultiFormatDateParser.testDate(line.trim());
            }
        } else {
            String arg;
            while (null != (arg = System.console().readLine("Enter a date-time:  ", new Object[0]))) {
                MultiFormatDateParser.testDate(arg);
            }
        }
    }

    private static void testDate(String arg) {
        ZonedDateTime result = MultiFormatDateParser.parse(arg);
        if (null == result) {
            System.out.println("Did not match any pattern.");
        } else {
            System.out.println(result.format(DateTimeFormatter.ISO_INSTANT));
        }
    }

    private static class Rule {
        final Pattern pattern;
        final DateTimeFormatter format;
        final DateGranularity granularity;

        public Rule(Pattern pattern, DateTimeFormatter format, DateGranularity granularity) {
            this.pattern = pattern;
            this.format = format;
            this.granularity = granularity;
        }

        static enum DateGranularity {
            YEAR,
            MONTH,
            DAY,
            TIME;

        }
    }
}

