/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.sql;

import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * SQLTable namer that generates names like A0, B1, C0, etc.
 * Tables are prefixed by a letter based on the table-group they are in, followed by a number
 * being the number within that table-group.
 * The candidate table-group will always be prefixed A.
 */
public class SQLTableAlphaNamer implements SQLTableNamer
{
    static String[] CHARS = new String[]{"A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M",
                                         "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z"};

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.sql.SQLTableNamer#getAliasForTable(org.datanucleus.store.rdbms.sql.SQLStatement, org.datanucleus.store.mapped.DatastoreContainerObject)
     */
    public String getAliasForTable(SQLStatement stmt, DatastoreContainerObject table, String groupName)
    {
        SQLTableGroup tableGrp = stmt.tableGroups.get(groupName);
        String groupLetter = null;
        int numTablesInGroup = 0;
        if (tableGrp == null || tableGrp.getNumberOfTables() == 0)
        {
            // Take next available letter (assuming first group is A)
            groupLetter = CHARS[stmt.tableGroups.size()];
            numTablesInGroup = 0;
        }
        else
        {
            // Extract same letter from existing group
            SQLTable refSqlTbl = tableGrp.getTables()[0];
            String baseTableAlias = refSqlTbl.getAlias().toString();
            String quote = stmt.getRDBMSManager().getDatastoreAdapter().getIdentifierQuoteString();
            if (baseTableAlias.startsWith(quote))
            {
                // Omit any leading quote and take next character
                groupLetter = baseTableAlias.substring(quote.length(), quote.length()+1);
            }
            else
            {
                // Take first character
                groupLetter = baseTableAlias.substring(0, 1);
            }
            numTablesInGroup = tableGrp.getNumberOfTables();
        }

        if (stmt.parent != null)
        {
            return groupLetter + numTablesInGroup + "_SUB";
        }
        else
        {
            return groupLetter + numTablesInGroup;
        }
    }
}