/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.fieldmanager;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.Relation;
import org.datanucleus.store.fieldmanager.AbstractFieldManager;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.StatementClassMapping;
import org.datanucleus.store.mapped.StatementMappingIndex;
import org.datanucleus.store.mapped.mapping.EmbeddedPCMapping;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.mapping.SerialisedPCMapping;
import org.datanucleus.store.mapped.mapping.SerialisedReferenceMapping;
import org.datanucleus.store.query.ResultObjectFactory;

/**
 * ResultSet getter implementation of a field manager.
 */
public class ResultSetGetter extends AbstractFieldManager
{
    private final StateManager sm;
    private final ObjectManager om;
    private final Object resultSet;
    private final StatementClassMapping resultMappings;

    /**
     * Constructor
     * @param sm the StateManager
     * @param results the ResultSet
     * @param resultMappings Mappings for the results for this class
     */
    public ResultSetGetter(StateManager sm, Object results, StatementClassMapping resultMappings)
    {
        this.sm = sm;
        this.om = sm.getObjectManager();
        this.resultSet = results;
        this.resultMappings = resultMappings;
    }

    public boolean fetchBooleanField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getBoolean(om, resultSet, mapIdx.getColumnPositions());
    }

    public char fetchCharField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getChar(om, resultSet, mapIdx.getColumnPositions());
    }

    public byte fetchByteField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getByte(om, resultSet, mapIdx.getColumnPositions());
    }

    public short fetchShortField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getShort(om, resultSet, mapIdx.getColumnPositions());
    }

    public int fetchIntField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getInt(om, resultSet, mapIdx.getColumnPositions());
    }

    public long fetchLongField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getLong(om, resultSet, mapIdx.getColumnPositions());
    }

    public float fetchFloatField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getFloat(om, resultSet, mapIdx.getColumnPositions());
    }

    public double fetchDoubleField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getDouble(om, resultSet, mapIdx.getColumnPositions());
    }

    public String fetchStringField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        return mapIdx.getMapping().getString(om, resultSet, mapIdx.getColumnPositions());
    }

    public Object fetchObjectField(int fieldNumber)
    {
        StatementMappingIndex mapIdx = resultMappings.getMappingForMemberPosition(fieldNumber);
        JavaTypeMapping mapping = mapIdx.getMapping();
        Object value;
        if (mapping instanceof EmbeddedPCMapping ||
            mapping instanceof SerialisedPCMapping ||
            mapping instanceof SerialisedReferenceMapping)
        {
            value = mapping.getObject(om, resultSet, mapIdx.getColumnPositions(), sm, fieldNumber);
        }
        else
        {
            AbstractMemberMetaData mmd =
                sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
            int relationType = mmd.getRelationType(om.getClassLoaderResolver());
            if (relationType == Relation.ONE_TO_ONE_BI || relationType == Relation.ONE_TO_ONE_UNI ||
                relationType == Relation.MANY_TO_ONE_BI)
            {
                // Process fields of sub-object if available in this result set
                StatementClassMapping relationMappings =
                    resultMappings.getMappingDefinitionForMemberPosition(fieldNumber);
                if (relationMappings != null)
                {
                    ClassLoaderResolver clr = om.getClassLoaderResolver();
                    AbstractClassMetaData relatedCmd = om.getMetaDataManager().getMetaDataForClass(mmd.getType(), clr);
                    ResultObjectFactory relationROF =
                        ((MappedStoreManager)om.getStoreManager()).newResultObjectFactory(relatedCmd,
                            relationMappings, false, om.getFetchPlan(), mmd.getType());
                    value = relationROF.getObject(om, resultSet);
                }
                else
                {
                    value = mapping.getObject(om, resultSet, mapIdx.getColumnPositions());
                }
            }
            else
            {
                value = mapping.getObject(om, resultSet, mapIdx.getColumnPositions());
            }
        }

        // Return the field value (as a wrapper if wrappable)
        return sm.wrapSCOField(fieldNumber, value, false, false, false);
    }
}