/*
 * Copyright 2004-2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.compass.core.metadata.ant;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Date;
import java.util.Iterator;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.FileSet;
import org.compass.core.config.CompassSettings;
import org.compass.core.config.binding.XmlMetaDataBinding;
import org.compass.core.metadata.Alias;
import org.compass.core.metadata.MetaData;
import org.compass.core.metadata.impl.DefaultCompassMetaData;
import org.compass.core.metadata.impl.DefaultMetaDataGroup;
import org.compass.core.util.DTDEntityResolver;
import org.compass.core.util.config.ConfigurationHelper;
import org.compass.core.util.config.XmlConfigurationHelperBuilder;
import org.xml.sax.EntityResolver;

/**
 * @author kimchy
 */
public class MetaDataTask extends Task {

    private File destDir;

    private FileSet fileset;

    public void setDestDir(File destDir) {
        this.destDir = destDir;
    }

    public void addFileset(FileSet fileset) {
        this.fileset = fileset;
    }

    public void execute() throws BuildException {
        EntityResolver entityResolver = new DTDEntityResolver();

        DirectoryScanner ds = fileset.getDirectoryScanner(getProject());
        String[] files = ds.getIncludedFiles();
        File dir = fileset.getDir(getProject());
        for (int i = 0; i < files.length; i++) {
            if (!files[i].endsWith("cmd.xml")) {
                continue;
            }
            File srcFile = new File(dir, files[i]);
            ConfigurationHelper conf;
            try {
                XmlConfigurationHelperBuilder builder = new XmlConfigurationHelperBuilder();
                builder.setEntityResolver(entityResolver);
                conf = builder.buildFromFile(srcFile);
            } catch (Exception e) {
                throw new BuildException(e.getMessage(), e);
            }

            DefaultCompassMetaData compassMetaData = new DefaultCompassMetaData();
            XmlMetaDataBinding metaDataBinding = new XmlMetaDataBinding();
            metaDataBinding.setUpBinding(null, compassMetaData, new CompassSettings());
            metaDataBinding.doAddConfiguration(conf);
            for (Iterator it = compassMetaData.groupsIterator(); it.hasNext();) {
                DefaultMetaDataGroup group = (DefaultMetaDataGroup) it.next();
                String className = group.getId();
                className = className.substring(0, 1).toUpperCase() + className.substring(1, className.length());
                File javaFile = new File(destDir, files[i]);
                // create the parent dir if not exists
                if (!javaFile.getParentFile().exists()) {
                    javaFile.mkdirs();
                }
                javaFile = new File(javaFile.getParentFile(), className + ".java");
                String packageName = "";
                File tmpFile = javaFile.getParentFile();
                while (!destDir.getAbsolutePath().equals(tmpFile.getAbsolutePath())) {
                    packageName = tmpFile.getName() + "." + packageName;
                    tmpFile = tmpFile.getParentFile();
                }
                packageName = packageName.substring(0, packageName.length() - 1);
                try {
                    System.out.println("Generating common meta data for [" + srcFile.getAbsolutePath() + "]");
                    PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream(javaFile)));
                    PrintHelper printHelper = new PrintHelper(writer);
                    printHelper.print("// Auto generated by compass meta-data ant task at: " + new Date());
                    printHelper.print("package " + packageName + ";");
                    printHelper.emptyLine();
                    // writer.println("import org.compass.core.Property;");
                    printHelper.printJavaDoc(group.getDescription());
                    printHelper.startClass(false, className);
                    printHelper.printJavaDoc(group.getDescription());
                    printHelper.startClass(true, "group");
                    printHelper.printConstant("Id", group.getId());
                    printHelper.printConstant("DispayName", group.getDisplayName());
                    printHelper.printConstant("Uri", group.getUri());
                    printHelper.endClass();
                    printHelper.startClass(true, "alias");
                    for (Iterator aliasIt = group.aliasIterator(); aliasIt.hasNext();) {
                        Alias alias = (Alias) aliasIt.next();
                        printHelper.printJavaDoc(alias.getDescription());
                        printHelper.startClass(true, alias.getId());
                        printHelper.printConstant("Id", alias.getId());
                        printHelper.printConstant("Name", alias.getName());
                        printHelper.printConstant("DisplayName", alias.getDisplayName());
                        printHelper.printConstant("Uri", alias.getUri());
                        printHelper.printConstant("GroupId", alias.getGroup().getId());
                        printHelper.endClass();
                    }
                    printHelper.endClass();
                    printHelper.startClass(true, "metaData");
                    for (Iterator metaDataIt = group.metaDataIterator(); metaDataIt.hasNext();) {
                        MetaData metaData = (MetaData) metaDataIt.next();
                        printHelper.printJavaDoc(metaData.getDescription());
                        printHelper.startClass(true, metaData.getId());
                        printHelper.printConstant("Id", metaData.getId());
                        printHelper.printConstant("Name", metaData.getName());
                        printHelper.printConstant("DisplayName", metaData.getDisplayName());
                        printHelper.printConstant("Uri", metaData.getUri());
                        printHelper.printConstant("GroupId", metaData.getGroup().getId());
                        if (metaData.getFormat() != null) {
                            printHelper.printConstant("Format", metaData.getFormat());
                        }
                        printHelper.endClass();
                    }
                    printHelper.endClass();

                    printHelper.endClass();
                    printHelper.close();
                } catch (FileNotFoundException e) {
                    throw new BuildException("Failed to write to file: " + javaFile.getAbsolutePath(), e);
                }
            }
        }
    }

    private static final class PrintHelper {
        private PrintWriter writer;

        private int indent = 0;

        public PrintHelper(PrintWriter writer) {
            this.writer = writer;
        }

        public void close() {
            writer.flush();
            writer.close();
        }

        public void emptyLine() {
            writer.println();
        }

        public void startClass(boolean isStatic, String className) {
            writeIndent();
            writer.print("public ");
            if (isStatic) {
                writer.print("static ");
            }
            className = className.substring(0, 1).toUpperCase() + className.substring(1, className.length());
            writer.print("final class " + className + " {");
            emptyLine();
            indent++;
        }

        public void endClass() {
            indent--;
            writeIndent();
            writer.println("}");
            emptyLine();
        }

        public void print(String line) {
            writer.println(line);
        }

        public void printConstant(String name, String value) {
            printConstant("String", name, value);
        }

        public void printConstant(String type, String name, String value) {
            writeIndent();
            writer.print("public static final ");
            writer.print(type);
            writer.print(" ");
            writer.print(name);
            writer.print(" = ");
            writer.print("\"");
            writer.print(value);
            writer.print("\";");
            emptyLine();
        }

        public void printJavaDoc(String description) {
            writeIndent();
            writer.println("/**");
            writeIndent();
            writer.print(" * ");
            writer.println(description);
            writeIndent();
            writer.println(" */");
        }

        private void writeIndent() {
            for (int i = 0; i < indent; i++) {
                writer.print("\t");
            }
        }

    }

}
