/*
 * Copyright (c) 2014, 2015, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.graalvm.compiler.core.common.type;

import org.checkerframework.checker.nullness.qual.EnsuresNonNullIf;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.checkerframework.dataflow.qual.Pure;
import org.checkerframework.dataflow.qual.SideEffectFree;
import static jdk.vm.ci.meta.MetaUtil.getSimpleName;
import java.util.Arrays;
import java.util.Objects;
import java.util.function.Function;
import jdk.vm.ci.meta.Constant;
import jdk.vm.ci.meta.JavaKind;
import org.graalvm.compiler.core.common.calc.FloatConvert;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Add;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.And;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Div;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Mul;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.MulHigh;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Or;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Rem;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Sub;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.UMulHigh;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.BinaryOp.Xor;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.IntegerConvertOp.Narrow;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.IntegerConvertOp.SignExtend;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.IntegerConvertOp.ZeroExtend;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.ShiftOp.Shl;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.ShiftOp.Shr;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.ShiftOp.UShr;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.UnaryOp.Abs;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.UnaryOp.Neg;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.UnaryOp.Not;
import org.graalvm.compiler.core.common.type.ArithmeticOpTable.UnaryOp.Sqrt;
import org.graalvm.util.CollectionsUtil;

public final class ArithmeticOpTable {

    public static ArithmeticOpTable forStamp(Stamp s);

    public BinaryOp<?>[] getBinaryOps();

    public UnaryOp<?>[] getUnaryOps();

    public ShiftOp<?>[] getShiftOps();

    public IntegerConvertOp<?>[] getIntegerConvertOps();

    public static final ArithmeticOpTable EMPTY;

    public interface ArithmeticOpWrapper {

        <OP> UnaryOp<OP> wrapUnaryOp(UnaryOp<OP> op);

        <OP> BinaryOp<OP> wrapBinaryOp(BinaryOp<OP> op);

        <OP> ShiftOp<OP> wrapShiftOp(ShiftOp<OP> op);

        <OP> IntegerConvertOp<OP> wrapIntegerConvertOp(IntegerConvertOp<OP> op);

        FloatConvertOp wrapFloatConvertOp(FloatConvertOp op);
    }

    public static ArithmeticOpTable wrap(ArithmeticOpWrapper wrapper, ArithmeticOpTable inner);

    protected ArithmeticOpTable(UnaryOp<Neg> neg, BinaryOp<Add> add, BinaryOp<Sub> sub, BinaryOp<Mul> mul, BinaryOp<MulHigh> mulHigh, BinaryOp<UMulHigh> umulHigh, BinaryOp<Div> div, BinaryOp<Rem> rem, UnaryOp<Not> not, BinaryOp<And> and, BinaryOp<Or> or, BinaryOp<Xor> xor, ShiftOp<Shl> shl, ShiftOp<Shr> shr, ShiftOp<UShr> ushr, UnaryOp<Abs> abs, UnaryOp<Sqrt> sqrt, IntegerConvertOp<ZeroExtend> zeroExtend, IntegerConvertOp<SignExtend> signExtend, IntegerConvertOp<Narrow> narrow, FloatConvertOp... floatConvert) {
    }

    @Override
    public int hashCode();

    public UnaryOp<Neg> getNeg();

    public BinaryOp<Add> getAdd();

    public BinaryOp<Sub> getSub();

    public BinaryOp<Mul> getMul();

    public BinaryOp<MulHigh> getMulHigh();

    public BinaryOp<UMulHigh> getUMulHigh();

    public BinaryOp<Div> getDiv();

    public BinaryOp<Rem> getRem();

    public UnaryOp<Not> getNot();

    public BinaryOp<And> getAnd();

    public BinaryOp<Or> getOr();

    public BinaryOp<Xor> getXor();

    public ShiftOp<Shl> getShl();

    public ShiftOp<Shr> getShr();

    public ShiftOp<UShr> getUShr();

    public UnaryOp<Abs> getAbs();

    public UnaryOp<Sqrt> getSqrt();

    public IntegerConvertOp<ZeroExtend> getZeroExtend();

    public IntegerConvertOp<SignExtend> getSignExtend();

    public IntegerConvertOp<Narrow> getNarrow();

    public FloatConvertOp getFloatConvert(FloatConvert op);

    public static String toString(Op... ops);

    @Override
    @Pure
    @EnsuresNonNullIf(expression = "#1", result = true)
    public boolean equals(@Nullable Object obj);

    @Override
    public String toString();

    public abstract static class Op {

        protected Op(String operator) {
        }

        @Override
        public String toString();

        @Override
        public int hashCode();

        @Override
        public boolean equals(Object obj);
    }

    public abstract static class UnaryOp<T> extends Op {

        public abstract static class Neg extends UnaryOp<Neg> {

            protected Neg() {
            }
        }

        public abstract static class Not extends UnaryOp<Not> {

            protected Not() {
            }
        }

        public abstract static class Abs extends UnaryOp<Abs> {

            protected Abs() {
            }
        }

        public abstract static class Sqrt extends UnaryOp<Sqrt> {

            protected Sqrt() {
            }
        }

        protected UnaryOp(String operation) {
        }

        public abstract Constant foldConstant(Constant value);

        public abstract Stamp foldStamp(Stamp stamp);

        public UnaryOp<T> unwrap();
    }

    public abstract static class BinaryOp<T> extends Op {

        public abstract static class Add extends BinaryOp<Add> {

            protected Add(boolean associative, boolean commutative) {
            }
        }

        public abstract static class Sub extends BinaryOp<Sub> {

            protected Sub(boolean associative, boolean commutative) {
            }
        }

        public abstract static class Mul extends BinaryOp<Mul> {

            protected Mul(boolean associative, boolean commutative) {
            }
        }

        public abstract static class MulHigh extends BinaryOp<MulHigh> {

            protected MulHigh(boolean associative, boolean commutative) {
            }
        }

        public abstract static class UMulHigh extends BinaryOp<UMulHigh> {

            protected UMulHigh(boolean associative, boolean commutative) {
            }
        }

        public abstract static class Div extends BinaryOp<Div> {

            protected Div(boolean associative, boolean commutative) {
            }
        }

        public abstract static class Rem extends BinaryOp<Rem> {

            protected Rem(boolean associative, boolean commutative) {
            }
        }

        public abstract static class And extends BinaryOp<And> {

            protected And(boolean associative, boolean commutative) {
            }
        }

        public abstract static class Or extends BinaryOp<Or> {

            protected Or(boolean associative, boolean commutative) {
            }
        }

        public abstract static class Xor extends BinaryOp<Xor> {

            protected Xor(boolean associative, boolean commutative) {
            }
        }

        protected BinaryOp(String operation, boolean associative, boolean commutative) {
        }

        public abstract Constant foldConstant(Constant a, Constant b);

        public abstract Stamp foldStamp(Stamp a, Stamp b);

        public final boolean isAssociative();

        public final boolean isCommutative();

        public boolean isNeutral(Constant n);

        public Constant getZero(Stamp stamp);

        public BinaryOp<T> unwrap();

        @Override
        public int hashCode();

        @Override
        public boolean equals(Object obj);

        @Override
        public String toString();
    }

    public abstract static class ShiftOp<OP> extends Op {

        public abstract static class Shl extends ShiftOp<Shl> {

            public Shl() {
            }
        }

        public abstract static class Shr extends ShiftOp<Shr> {

            public Shr() {
            }
        }

        public abstract static class UShr extends ShiftOp<UShr> {

            public UShr() {
            }
        }

        protected ShiftOp(String operation) {
        }

        public abstract Constant foldConstant(Constant c, int amount);

        public abstract Stamp foldStamp(Stamp s, IntegerStamp amount);

        public abstract int getShiftAmountMask(Stamp s);
    }

    public abstract static class FloatConvertOp extends UnaryOp<FloatConvertOp> {

        protected FloatConvertOp(FloatConvert op) {
        }

        public FloatConvert getFloatConvert();

        @Override
        public FloatConvertOp unwrap();

        @Override
        public int hashCode();

        @Override
        public boolean equals(Object obj);
    }

    public abstract static class IntegerConvertOp<T> extends Op {

        public abstract static class ZeroExtend extends IntegerConvertOp<ZeroExtend> {

            protected ZeroExtend() {
            }
        }

        public abstract static class SignExtend extends IntegerConvertOp<SignExtend> {

            protected SignExtend() {
            }
        }

        public abstract static class Narrow extends IntegerConvertOp<Narrow> {

            protected Narrow() {
            }

            @Override
            public Stamp invertStamp(int inputBits, int resultBits, Stamp outStamp);
        }

        protected IntegerConvertOp(String op) {
        }

        public abstract Constant foldConstant(int inputBits, int resultBits, Constant value);

        public abstract Stamp foldStamp(int inputBits, int resultBits, Stamp stamp);

        public IntegerConvertOp<T> unwrap();

        public abstract Stamp invertStamp(int inputBits, int resultBits, Stamp outStamp);
    }
}
