/* Simple JavaScript Inheritance
 * By John Resig http://ejohn.org/
 * MIT Licensed.
 */
// Inspired by base2 and Prototype
(function(){
    var initializing = false, fnTest = /xyz/.test(function(){xyz;}) ? /\b_super\b/ : /.*/;

    // The base Class implementation (does nothing)
    this.Class = function(){};

    // Create a new Class that inherits from this class
    Class.extend = function(prop) {
        var _super = this.prototype;

        // Instantiate a base class (but only create the instance,
        // don't run the init constructor)
        initializing = true;
        var prototype = new this();
        initializing = false;

        // Copy the properties over onto the new prototype
        for (var name in prop) {
            // Check if we're overwriting an existing function
            prototype[name] = typeof prop[name] == "function" &&
            typeof _super[name] == "function" && fnTest.test(prop[name]) ?
                    (function(name, fn){
                        return function() {
                            var tmp = this._super;

                            // Add a new ._super() method that is the same method
                            // but on the super-class
                            this._super = _super[name];

                            // The method only need to be bound temporarily, so we
                            // remove it when we're done executing
                            var ret = fn.apply(this, arguments);
                            this._super = tmp;

                            return ret;
                        };
                    })(name, prop[name]) :
                    prop[name];
        }

        // The dummy class constructor
        function Class() {
            // All construction is actually done in the init method
            if ( !initializing && this.init )
                this.init.apply(this, arguments);
        }

        // Populate our constructed prototype object
        Class.prototype = prototype;

        // Enforce the constructor to be what we expect
        Class.prototype.constructor = Class;

        // And make this class extendable
        Class.extend = arguments.callee;

        return Class;
    };
})();
/**
 * jQuery-Plugin "butterHandleAutoComplete" for text autocomplete tag. Initializes auto complete functionality to
 * text component.
 *
 * How to use:
 * jQuery("#selector")._butterAutoCompleteInit();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------

    $.fn._butterAutoCompleteInit = function () {
        return this.each(function () {
            new AutocompleteList(this);
        });
    };

    // define objects --------------------------------------------------------------------

    var AutocompleteList = Class.extend({
        init: function (rootElement) {
            this.SEARCH_REQUEST_DELAY = 300;// in ms

            var $autocompleteTmp = $(rootElement);
            this.$input = $autocompleteTmp.prev();
            this.$input.parent().css({position: "relative"});
            this.autocompleteId = $autocompleteTmp.attr("id");
            this.$selectedOption = null;
            this.ignoreKeyupEvent = false;
            this.requestDelayTimerId = null;
            this.isRequestRunning = false;
            this.areChangesMadeWhileRequestWasRunning = false;

            this._keyCodes = {
                //backspace: 8,
                tab: 9,
                enter: 13,
                shift: 16,
                ctrl: 17,
                alt: 18,
                pause: 19,
                caps_lock: 20,
                escape: 27,
                page_up: 33,
                page_down: 34,
                end: 35,
                home: 36,
                arrow_left: 37,
                arrow_up: 38,
                arrow_right: 39,
                arrow_down: 40,
                insert: 45,
                // 'delete' is a reserved key word
                delete_key: 46,
                left_window_key: 91,
                right_window_key: 92,
                select_key: 93,
                num_lock: 144,
                scroll_lock: 145
            };

            var self = this;
            self.$input
                    .on("keydown", function (event) {
                        if (event.which === self._keyCodes.enter) {
                            self._handleEnterKeyDown(event);
                        } else if (event.which === self._keyCodes.arrow_up
                                || event.which === self._keyCodes.arrow_down) {
                            self._handleArrowUpAndDownKeyDown(event);
                        } else if (event.which === self._keyCodes.escape) {
                            self._handleEscapeKeyDown(event);
                        }
                    })
                    .on("keyup", function (event) {
                        // don't handle other keys than character keys
                        for (keyName in self._keyCodes) {
                            if (self._keyCodes[keyName] === event.which) {
                                self._stopEvent(event);
                                return;
                            }
                        }

                        if (self.ignoreKeyupEvent) {
                            self._stopEvent(event);
                            self.ignoreKeyupEvent = false;
                            return;
                        }

                        if (self.$input.val().length === 0) {
                            self._stopEvent(event);
                            self._hideAutocompleteResultList();
                            return;
                        }

                        self._trySendJsfAjaxRequest();
                    })
                    .on("blur", function (event) {
                        window.setTimeout(function () {
                            self._hideAutocompleteResultList();
                        }, 100);
                    });
        },

        _handleEnterKeyDown: function (event) {
            if (this.$selectedOption !== null) {
                this._stopEvent(event);
                this._setSelectedValue();
            }
        },

        _handleArrowUpAndDownKeyDown: function (event) {
            this._stopEvent(event);
            var $autocomplete = this._getAutocompleteElement();
            if (!$autocomplete.is(":visible") && this.$input.val().length > 0) {
                this._trySendJsfAjaxRequest();
            } else if ($autocomplete.is(":visible") && $autocomplete.find("li").length > 0) {
                if (this.$selectedOption === null) {
                    this._selectResultOptionElement($autocomplete.find("li")[0]);
                } else {
                    this._moveResultOptionElementSelectionCursor(
                            $autocomplete, event.which === this._keyCodes.arrow_up ? -1 : 1);
                }
            }
        },

        _handleEscapeKeyDown: function (event) {
            this._stopEvent(event);
            this._hideAutocompleteResultList();
        },

        _trySendJsfAjaxRequest: function () {
            var self = this;

            if (self.isRequestRunning) {
                // console.log("request is active, so remember that changes has been made while request was running");
                self.areChangesMadeWhileRequestWasRunning = true;
            }

            if (self.requestDelayTimerId !== null) {
                window.clearTimeout(self.requestDelayTimerId)
            }

            self.requestDelayTimerId = window.setTimeout(function () {
                self.requestDelayTimerId = null;
                self._sendJsfAjaxRequest();
            }, self.SEARCH_REQUEST_DELAY);
        },

        _sendJsfAjaxRequest: function () {
            var self = this;

            if (self.isRequestRunning) {
                // console.log("request is running, abort");
                return;
            }
            self.isRequestRunning = true;

            self.areChangesMadeWhileRequestWasRunning = false;
            self._showLoadingSpinner();

            // console.log("starting request");

            var id = self.$input.parent().parent().attr('id');
            var params = self.$input.val();

            jsf.ajax.request(id, "autocomplete", {
                "javax.faces.behavior.event": "autocomplete",
                render: self.autocompleteId,
                params: params,
                "butterfaces.params": params,
                onevent: function (data) {
                    if (data.status === "success") {
                        // console.log("request finished");

                        // only show result if input field still has focus
                        if (self.$input.is(":focus")) {
                            self._handleAutocompleteResultListVisibility();
                        }
                        self._hideLoadingSpinner();
                        self.isRequestRunning = false;

                        if (self.areChangesMadeWhileRequestWasRunning) {
                            // console.log("changes made while request was running, start new request automatically");
                            self._sendJsfAjaxRequest();
                        }
                    }
                }
            });
        },

        _handleAutocompleteResultListVisibility: function () {
            var self = this;
            var $autocomplete = self._getAutocompleteElement();

            if ($autocomplete.find("li").length > 0) {
                self._initAndShowAutocompleteResultList();
            } else {
                self._hideAutocompleteResultList();
            }
        },

        _showLoadingSpinner: function () {
            $('<div class="butter-dropdownlist-spinner-container"><div class="butter-dropdownlist-spinner"></div></div>')
                    .appendTo(this.$input.parent());
        },

        _hideLoadingSpinner: function () {
            this.$input.parent().find(".butter-dropdownlist-spinner").remove();
        },

        _initAndShowAutocompleteResultList: function () {
            var self = this;
            var $autocomplete = self._getAutocompleteElement();
            $autocomplete
                    .show()
                    .css({
                        width: self.$input.innerWidth()
                    })
                    .highlight(self.$input.val());

            $autocomplete.find("ul")
                    .on("mouseleave", function () {
                        self._clearResultOptionSelection();
                    });

            $autocomplete.find("li")
                    .on("mousedown", function () {
                        self._setSelectedValue();
                    })
                    .on("mouseenter", function () {
                        self._selectResultOptionElement(this);
                    });
        },

        _selectResultOptionElement: function (optionElement) {
            this._clearResultOptionSelection();
            var $selectedOptionElement = $(optionElement);
            $selectedOptionElement.addClass("butter-dropdownlist-resultItem-selected");
            this.$selectedOption = $selectedOptionElement;
        },

        _clearResultOptionSelection: function () {
            this.$selectedOption = null;
            this._getAutocompleteElement()
                    .find(".butter-dropdownlist-resultItem-selected")
                    .removeClass("butter-dropdownlist-resultItem-selected");
        },

        _moveResultOptionElementSelectionCursor: function ($autocomplete, direction) {
            if (direction > 0) {
                var $next = this.$selectedOption.next();
                if ($next.length > 0) {
                    this._selectResultOptionElement($next[0]);
                } else {
                    //there is no next
                    this._selectResultOptionElement($autocomplete.find("li")[0]);
                }
            } else {
                var $prev = this.$selectedOption.prev();
                if ($prev.length > 0) {
                    this._selectResultOptionElement($prev[0]);
                } else {
                    //there is no previous
                    var resultListOptions = $autocomplete.find("li");
                    this._selectResultOptionElement(resultListOptions[resultListOptions.length - 1]);
                }
            }
        },

        _setSelectedValue: function () {
            if (this.$selectedOption !== null) {
                this.ignoreKeyupEvent = true;
                this.$input
                        .val(this.$selectedOption.attr("data-select-value"))
                        .change()
                        .focus()
                        .keyup();
                this._hideAutocompleteResultList();
            }
        },

        _hideAutocompleteResultList: function () {
            if (this.requestDelayTimerId !== null) {
                window.clearTimeout(this.requestDelayTimerId)
            }
            this.$selectedOption = null;
            this._getAutocompleteElement().hide();
        },

        _getAutocompleteElement: function () {
            return $(document.getElementById(this.autocompleteId));
        },

        _stopEvent: function (event) {
            event.stopPropagation();
            event.preventDefault();
        }
    });

}(jQuery));
/**
 * jQuery-Plugin to handle bootstrap fixes.
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("#someTreeSelector").fixBootstrapDropDown();
 */
(function ($) {
    $.fn.fixBootstrapDropDown = function () {

        return this.each(function () {
            $('.dropdown-menu').on('click', function(e) {
                if($(this).hasClass('dropdown-menu-form')) {
                    e.stopPropagation();
                }
            });
        });

    };
}(jQuery));
/**
 * butterItemFilterField is a jQuery plugin that filters html element with the css class <code>filterable-item</code>.
 * It is applied to the search field.<br/>
 * If no filter text is entered, then all filterable-items are displayed. Else the search field value is matched against <b>all</b> text contained by a filterable-item.
 *
 * How to use:
 * jQuery("#someInputSelector").butterItemFilterField();
 *
 * Author: Yann Massard
 */
(function ($) {
    var delay = (function () {
        var timer = 0;
        return function (callback, ms) {
            clearTimeout(timer);
            timer = setTimeout(callback, ms);
        };
    })();

    // extend jQuery --------------------------------------------------------------------
    $.fn.butterItemFilterField = function (filterableItemContainerSelector) {
        return this.each(function () {
            var $this = $(this);
            $this.keyup(function () {
                delay(function () {
                    var filterValue = $this.val();

                    // find container again every time, because it could have been rerendered.
                    var $filterableItemContainer;
                    if (filterableItemContainerSelector) {
                        $filterableItemContainer = $(filterableItemContainerSelector);
                    } else {
                        var containerSelector = $this.attr('data-filterable-item-container');
                        $filterableItemContainer = $(containerSelector);
                    }

                    var filterableItems = $filterableItemContainer.find('.filterable-item');

                    filterableItems.each(function (i, elem) {
                        var $filterableItem = $(elem);
                        if ($filterableItem.is(':containsIgnoreCase(' + filterValue + ')')) {
                            $filterableItem.removeAttr("hidden");
                            $filterableItem.highlight(filterValue);
                        } else {
                            $filterableItem.attr("hidden", "hidden");
                        }

                    });
                }, 300);
            });
        });
    };
}(jQuery));

(function ($) {
    $.expr[":"].containsIgnoreCase = $.expr.createPseudo(function (arg) {
        return function (elem) {
            return !arg || $(elem).text().toUpperCase().indexOf(arg.toUpperCase()) >= 0;
        };
    });
}(jQuery));
/**
 * jQuery-Plugin "Animate dots" for tree animation. Animates a waiting dot line (4 dots) in an interval of 200 millis
 * as html string in given component. Note: existing html code will be cleared.
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("#selector").startDots();
 * jQuery("#selector").stopDots();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------

    var intervalTrigger = null;

    $.fn.startDots = function () {
        return this.each(function () {
            var $originalElement = $(this);

            $originalElement.html('');

            intervalTrigger = setInterval(function () {
                $originalElement.append('.');

                if ($originalElement.html().length > 5) {
                    $originalElement.html('');
                }
            }, 200);
        });
    };

    $.fn.stopDots = function () {
        return this.each(function () {
            var $originalElement = $(this);

            $originalElement.html('');
            window.clearInterval(intervalTrigger);
        });
    };

}(jQuery));
(function ($) {
    $.fn.highlight = function (searchString) {
        var highlightClassName = "search-highlighted";
        var regex = new RegExp(searchString, "gi");

        var elements = this.find('*');
        this.each(function () {
            elements.push(this);
            $(this).find('.' + highlightClassName).contents().unwrap();
            this.normalize();
        });

        return elements.each(function () {
            var $this = $(this);

            if (searchString && searchString !== '') {
                $this.contents()
                    .filter(function () {
                        return this.nodeType === 3 && regex.test(this.nodeValue);
                    })
                    .replaceWith(function () {
                        return (this.nodeValue || "").replace(regex, function (match) {
                            return "<span class=\"" + highlightClassName + "\">" + match + "</span>";
                        });
                    });
            }
        });
    };
}(jQuery));
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.markdownReadonly = function () {
        var root = $(this);

        var $readonlyMarkdown = root.find('.butter-component-value-readonly-wrapper');
        var markdownText = $readonlyMarkdown.html()
            .replace('&amp;', '&')
            .replace('&lt;', '<')
            .replace('&gt;', '>');
        var markdownTextToHtml = new showdown.Converter().makeHtml(markdownText);

        $readonlyMarkdown.empty();
        $readonlyMarkdown.append(markdownTextToHtml);
    };
}(jQuery));
if (typeof butter === 'undefined') {
    butter = {};
}
butter.modal = {};

butter.modal.open = function (/* string */ modalPanelId) {
    // console.log('Opening modal panel with data-modal-id ' + modalPanelId);
    $('.butter-modal[data-modal-id=' + modalPanelId + ']').modal({show: true, keyboard: false, backdrop: 'static'})
};

butter.modal.close = function (/* string */ modalPanelId) {
    // console.log('Closing modal panel with data-modal-id ' + modalPanelId);
    $('.butter-modal[data-modal-id=' + modalPanelId + ']').modal('hide');
};
/**
 * jQuery-Plugin "Number Spinner" for input fields.
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("someButterComponentSelector").butterNumberSpinner();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.butterNumberSpinner = function (options) {
        return this.each(function () {
            new NumberSpinner(this, options);
        });
    };

    // define classes --------------------------------------------------------------------
    var NumberSpinner = Class.extend({
        init: function (element, options) {
            this.$input = $(element).find("input");
            this._initInput();
            this._initOptions(options);
            this._initCounter();
            this._initButtons();
            this._initArrowKeys();
            this._initMouseWheel();
        },

        _initInput: function () {
            var self = this;
            this.$input
                .addClass("butter-component-number-input")
                .blur(function () {
                    self._setValueOnBlur();
                })
                .parent().addClass("input-group");
        },

        _initOptions: function (options) {
            var defaultOptions = {
                step: 1,
                disabled: false
            };
            this._options = $.extend({}, defaultOptions, options);

            // ensure that this values are numbers
            if (this._options.step !== undefined) {
                this._options.step = this._options.step * 1;
            }
            if (this._options.min !== undefined) {
                this._options.min = this._options.min * 1;
            }
            if (this._options.max !== undefined) {
                this._options.max = this._options.max * 1;
            }
        },

        _initButtons: function () {
            var $outerAddon = $("<div>")
                .addClass("input-group-append")
                .insertAfter(this.$input);

            var $addon = $("<span>")
                .addClass("input-group-text")
                .addClass("butter-component-number-buttons")
                .appendTo($outerAddon);

            var self = this;
            $("<span>")
                .addClass("glyphicon glyphicon-chevron-up")
                .addClass("butter-component-number-button")
                .addClass(function () {
                    return self._options.disabled ? "disabled" : "";
                })
                .click(function () {
                    if (!self._options.disabled) {
                        self.increaseCounter();
                    }
                })
                .appendTo($addon);

            $("<span>")
                .addClass("glyphicon glyphicon-chevron-down")
                .addClass("butter-component-number-button")
                .addClass(function () {
                    return self._options.disabled ? "disabled" : "";
                })
                .click(function () {
                    if (!self._options.disabled) {
                        self.decreaseCounter();
                    }
                })
                .appendTo($addon);
        },

        _initArrowKeys: function () {
            var self = this;
            this.$input.keydown(function (event) {
                if (event.which === 38) {
                    event.stopPropagation();
                    event.preventDefault();
                    self.increaseCounter();
                } else if (event.which === 40) {
                    event.stopPropagation();
                    event.preventDefault();
                    self.decreaseCounter();
                }
            });
        },

        _initMouseWheel: function () {
            var self = this;
            this.$input.on("mousewheel DOMMouseScroll", function (event) {
                if (!self.$input.is(':focus')) {
                    return;
                }

                var delta = event.originalEvent.wheelDelta || -event.originalEvent.deltaY || -event.originalEvent.detail;

                event.stopPropagation();
                event.preventDefault();

                if (delta < 0) {
                    self.decreaseCounter();
                } else {
                    self.increaseCounter();
                }
            });
        },

        _initCounter: function() {
            this._counter = null;
            this._isValueSet = false;

            var value = this.$input.val();
            if (!this._isStringEmpty(value)) {
                var parsedInt = parseInt(value);
                if (!isNaN(parsedInt)) {
                    this.setCounter(parsedInt);
                    this._isValueSet = true;
                }
            }

            if (!this._isValueSet) {
                this.setCounter(0);
            }
        },

        _setValueOnBlur: function () {
            var value = this.$input.val();
            if (this._isStringEmpty(value)) {
                this.$input.val("");
                this.setCounter(0);
                this._isValueSet = false;
            } else {
                var parsedInt = parseInt(value);
                if (isNaN(parsedInt)) {
                    if (this._isValueSet) {
                        this.$input.val(this._counter);
                    } else {
                        this.$input.val("");
                        this.setCounter(0);
                        this._isValueSet = false;
                    }
                } else {
                    this.setCounter(parsedInt);
                    this.$input.val(this._counter);
                    this._isValueSet = true;
                }
            }
        },

        increaseCounter: function () {
            if (this._isValueSet) {
                this.setCounter(this._counter + this._options.step);
            } else {
                this._isValueSet = true;
            }
            this.$input.val(this._counter);
            this.$input.change();
        },

        decreaseCounter: function () {
            if (this._isValueSet) {
                this.setCounter(this._counter - this._options.step);
            } else {
                this._isValueSet = true;
            }
            this.$input.val(this._counter);
            this.$input.change();
        },

        _isStringEmpty: function (value) {
            return (value.length === 0 || !value.trim());
        },

        setCounter: function (value) {
            if (this._options.min !== undefined && value < this._options.min) {
                this._counter = this._options.min;
            } else if (this._options.max !== undefined && value > this._options.max) {
                this._counter = this._options.max;
            } else {
                this._counter = value;
            }
        }
    });
}(jQuery));
if (typeof butter === 'undefined') {
    butter = {};
}
butter.prettyprint = {};

/**
 * calls pretty print javascript framework and removes first and last empty children.
 */
butter.prettyprint.cleanup = function() {
    prettyPrint();

    jQuery('.butter-component-prettyprint').each(function () {
        var $firstPreChild = jQuery(this).find("pre span").first();
        var $lastPreChild = jQuery(this).find("pre span").last();


        if (!(typeof $firstPreChild.html() === "undefined")) {
            if (!$firstPreChild.html().trim()) {
                $firstPreChild.remove();
            }
        }
        if (!(typeof $lastPreChild.html() === "undefined")) {
            if (!$lastPreChild.html().trim()) {
                $lastPreChild.remove();
            }
        }
    });
};
/**
 * How to use:
 * jQuery("#someSelector").scrollToFirst() or jQuery("#someSelector").scrollToLast();
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.butterScrollToFirst = function (/* int */ offset) {
        void 0;
        var _offset = offset === undefined ? $(this).first().offset().top : $(this).first().offset().top + offset;
        $('html, body').animate({scrollTop: _offset + 'px'}, 'fast');
    };

    $.fn.butterScrollToLast = function (/* int */ offset) {
        void 0;
        var _offset = offset === undefined ? $(this).last().offset().top : $(this).last().offset().top + offset;
        $('html, body').animate({scrollTop: _offset + 'px'}, 'fast');
    };
}(jQuery));
/**
 * jQuery-Plugin to handle selection style classes on JSF-Component "b:table".
 * Works with at least jQuery 1.3.2.
 *
 * How to use:
 * jQuery("#someTreeSelector").selectRow( {rowIndex: '6'} );
 */
(function ($) {
    // extend jQuery --------------------------------------------------------------------
    $.fn.toggleColumnVisibilty = function (renderIds, disableRenderIds) {

        return this.each(function () {
            var $toolbar = $(this);

            var json = JSON.stringify(createColumnVisibilty($toolbar));
            ButterFaces.Ajax.sendRequest($toolbar.attr('id'), 'toggle', renderIds, json, disableRenderIds);
        });

        function createColumnVisibilty($toolbar) {
            var columns = [];

            $toolbar.find('.butter-table-toolbar-column-option input[type=checkbox]').each(function (index, checkbox) {
                var $checkbox = $(checkbox).parent('.butter-table-toolbar-column-option');
                columns.push({
                    identifier: $checkbox.attr('data-column-model-identifier'),
                    visible: $checkbox.find('input[type=checkbox]').is(':checked')
                });
            });

            return columns;
        }
    };

    $.fn.orderColumn = function (renderIds, disableRenderIds, toLeft, columnNumber) {

        return this.each(function () {
            var $toolbar = $(this);

            if (toLeft) {
                //console.log('order column ' + columnNumber + ' to left');
                orderColumnLeft($toolbar, columnNumber);
            } else {
                //console.log('order column ' + columnNumber + ' to right');
                orderColumnRight($toolbar, columnNumber);
            }

            var json = JSON.stringify(createColumnOrder($toolbar));
            ButterFaces.Ajax.sendRequest($toolbar.attr('id'), 'order', renderIds, json, disableRenderIds);
        });

        function createColumnOrder($toolbar) {
            var columns = [];

            $toolbar.find('.butter-table-toolbar-column-option input[type=checkbox]').each(function (index, checkbox) {
                var $checkbox = $(checkbox).parent('.butter-table-toolbar-column-option');
                columns.push({
                    identifier: $checkbox.attr('data-column-model-identifier'),
                    position: index
                });
            });

            return columns;
        }

        function orderColumnLeft(/* jquery toolbar */ $toolbar, columnNumber) {
            //console.log($toolbar);

            var $column = $toolbar.find('li[data-original-column="' + columnNumber + '"]');
            var $nextColumn = $column.prev();

            //console.log($column);
            //console.log($nextColumn);

            var $detachtedColumn = $column.detach();
            $nextColumn.before($detachtedColumn);
        }

        function orderColumnRight(/* jquery toolbar */ $toolbar, columnNumber) {
            //console.log($toolbar);

            var $column = $toolbar.find('li[data-original-column="' + columnNumber + '"]');
            var $nextColumn = $column.next();

            //console.log($column);
            //console.log($nextColumn);

            var $detachtedColumn = $column.detach();
            $nextColumn.after($detachtedColumn);
        }
    };
}(jQuery));
(function ($) {

    $.fn._closePopoverOnBlur = function (/* object */ data) {
        return this.each(function () {
            var root = $(this);

            root.find('.butter-input-component').on('blur', function () {
                root.popover('hide')
            });
        });
    };

    $.fn._butterTooltip = function (/* object */ data) {
        var root = $(this);

        //console.log(data);

        var content = jQuery('[name=' + data.contentByName + ']');

        var newData = {
            trigger: data.trigger,
            title: data.title,
            placement: data.placement,
            placementFunction: data.placementFunction,
            minVerticalOffset: data.minVerticalOffset,
            minHorizontalOffset: data.minHorizontalOffset,
            content: content.html().trim()
        };

        //console.log(newData);

        content.remove();

        root.butterTooltip(newData);
    };

    $.fn.butterTooltip = function (/* object */ data) {
        return this.each(function () {
            var root = $(this);
            var trigger = data.trigger ? data.trigger : 'hover';

            //console.log(data);
            //console.log('placement: ' + placement);
            //console.log('trigger: ' + trigger);
            //console.log('viewport: ' + data.viewport);

            if (data.minVerticalOffset) {
                root.attr('data-tooltip-min-vertical-offset', data.minVerticalOffset);
            }
            if (data.minHorizontalOffset) {
                root.attr('data-tooltip-min-horizontal-offset', data.minHorizontalOffset);
            }
            if (root.next().hasClass('popover')) {
                root.next().remove();
            }


            root.popover({
                trigger: trigger,
                placement: function (popover, source) {
                    return data.placement ? data.placement : (data.placementFunction ? data.placementFunction : new ButterFaces.Tooltip().calculateTooltipPosition(popover, source));
                },
                title: data.title,
                html: true,
                content: data.content,
                viewport: data.viewport
            });
        });
    };
}(jQuery));
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
