/*
 * Decompiled with CFR 0.152.
 */
package org.bouncycastle.crypto.fips;

import java.math.BigInteger;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.sec.SECObjectIdentifiers;
import org.bouncycastle.asn1.x9.ECNamedCurveTable;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.Algorithm;
import org.bouncycastle.crypto.AsymmetricPrivateKey;
import org.bouncycastle.crypto.AsymmetricPublicKey;
import org.bouncycastle.crypto.CryptoServicesRegistrar;
import org.bouncycastle.crypto.InvalidSignatureException;
import org.bouncycastle.crypto.asymmetric.AsymmetricECPrivateKey;
import org.bouncycastle.crypto.asymmetric.AsymmetricECPublicKey;
import org.bouncycastle.crypto.asymmetric.AsymmetricKeyPair;
import org.bouncycastle.crypto.asymmetric.ECDomainParameters;
import org.bouncycastle.crypto.asymmetric.ECDomainParametersID;
import org.bouncycastle.crypto.asymmetric.ECDomainParametersIndex;
import org.bouncycastle.crypto.asymmetric.NamedECDomainParameters;
import org.bouncycastle.crypto.fips.DSAOutputSigner;
import org.bouncycastle.crypto.fips.DSAOutputValidator;
import org.bouncycastle.crypto.fips.DSAOutputVerifier;
import org.bouncycastle.crypto.fips.EcDHAgreement;
import org.bouncycastle.crypto.fips.EcDHUAgreement;
import org.bouncycastle.crypto.fips.EcDhBasicAgreement;
import org.bouncycastle.crypto.fips.EcDhcBasicAgreement;
import org.bouncycastle.crypto.fips.EcDhcuBasicAgreement;
import org.bouncycastle.crypto.fips.EcDsaSigner;
import org.bouncycastle.crypto.fips.EcKeyGenerationParameters;
import org.bouncycastle.crypto.fips.EcKeyPairGenerator;
import org.bouncycastle.crypto.fips.EcMqvBasicAgreement;
import org.bouncycastle.crypto.fips.FipsAgreement;
import org.bouncycastle.crypto.fips.FipsAgreementFactory;
import org.bouncycastle.crypto.fips.FipsAgreementParameters;
import org.bouncycastle.crypto.fips.FipsAlgorithm;
import org.bouncycastle.crypto.fips.FipsAsymmetricKeyPairGenerator;
import org.bouncycastle.crypto.fips.FipsDigestAlgorithm;
import org.bouncycastle.crypto.fips.FipsEngineProvider;
import org.bouncycastle.crypto.fips.FipsKDF;
import org.bouncycastle.crypto.fips.FipsOutputSignerUsingSecureRandom;
import org.bouncycastle.crypto.fips.FipsOutputValidator;
import org.bouncycastle.crypto.fips.FipsOutputVerifier;
import org.bouncycastle.crypto.fips.FipsParameters;
import org.bouncycastle.crypto.fips.FipsSHS;
import org.bouncycastle.crypto.fips.FipsSignatureOperatorFactory;
import org.bouncycastle.crypto.fips.FipsUnapprovedOperationError;
import org.bouncycastle.crypto.fips.HMacDsaKCalculator;
import org.bouncycastle.crypto.fips.NullDigest;
import org.bouncycastle.crypto.fips.PrivilegedUtils;
import org.bouncycastle.crypto.fips.SelfTestExecutor;
import org.bouncycastle.crypto.fips.Utils;
import org.bouncycastle.crypto.fips.VariantInternalKatTest;
import org.bouncycastle.crypto.fips.VariantKatTest;
import org.bouncycastle.crypto.internal.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.internal.DSA;
import org.bouncycastle.crypto.internal.Digest;
import org.bouncycastle.crypto.internal.ExtendedDigest;
import org.bouncycastle.crypto.internal.Permissions;
import org.bouncycastle.crypto.internal.params.EcDhuPrivateParameters;
import org.bouncycastle.crypto.internal.params.EcDhuPublicParameters;
import org.bouncycastle.crypto.internal.params.EcDomainParameters;
import org.bouncycastle.crypto.internal.params.EcMqvPrivateParameters;
import org.bouncycastle.crypto.internal.params.EcMqvPublicParameters;
import org.bouncycastle.crypto.internal.params.EcNamedDomainParameters;
import org.bouncycastle.crypto.internal.params.EcPrivateKeyParameters;
import org.bouncycastle.crypto.internal.params.EcPublicKeyParameters;
import org.bouncycastle.crypto.internal.params.ParametersWithRandom;
import org.bouncycastle.crypto.internal.test.ConsistencyTest;
import org.bouncycastle.math.ec.ECConstants;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.Properties;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.TestRandomBigInteger;
import org.bouncycastle.util.test.TestRandomData;

/*
 * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
 */
public final class FipsEC {
    private static final Logger LOG = Logger.getLogger(FipsEC.class.getName());
    private static final int MIN_FIPS_FIELD_SIZE = 224;
    private static final BigInteger TEST_D_OFFSET = new BigInteger("deadbeef", 16);
    private static final AsymmetricCipherKeyPair katKeyPair = FipsEC.getKATKeyPair();
    private static final AsymmetricCipherKeyPair f2mKatKeyPair = FipsEC.getF2mKATKeyPair();
    public static final FipsAlgorithm ALGORITHM = new FipsAlgorithm("EC");
    private static final FipsAlgorithm ALGORITHM_MQV = new FipsAlgorithm("ECMQV", (Enum)Variations.ECMQV);
    private static final FipsAlgorithm ALGORITHM_DHU = new FipsAlgorithm("ECCDHU", (Enum)Variations.ECCDHU);
    public static final DSAParameters DSA = new DSAParameters(new FipsAlgorithm("ECDSA", (Enum)Variations.ECDSA), FipsSHS.Algorithm.SHA1);
    public static final DSAParameters DDSA = new DSAParameters(new FipsAlgorithm(ALGORITHM.getName(), (Enum)Variations.ECDDSA), FipsSHS.Algorithm.SHA1);
    public static final AgreementParameters DH = new AgreementParameters(new FipsAlgorithm("ECDH", (Enum)Variations.ECDH));
    public static final AgreementParameters CDH = new AgreementParameters(new FipsAlgorithm("ECCDH", (Enum)Variations.ECCDH));
    public static final MQVAgreementParametersBuilder MQV = new MQVAgreementParametersBuilder();
    public static final DHUAgreementParametersBuilder CDHU = new DHUAgreementParametersBuilder();
    private static final DsaProvider DSA_PROVIDER = new DsaProvider();
    private static final FipsEngineProvider<EcDhBasicAgreement> DH_PROVIDER = new DhProvider();
    private static final FipsEngineProvider<EcDhcBasicAgreement> CDH_PROVIDER = new DhcProvider();
    private static final FipsEngineProvider<EcMqvBasicAgreement> MQV_PROVIDER = new MqvProvider();
    private static final FipsEngineProvider<EcDhcuBasicAgreement> DHU_PROVIDER = new DhuProvider();
    private static Set<ASN1ObjectIdentifier> approvedKeySet;

    private FipsEC() {
    }

    private static void validateDomainParameters(ECDomainParameters domainParameters) {
        ASN1ObjectIdentifier paramOid = domainParameters instanceof NamedECDomainParameters ? ((NamedECDomainParameters)domainParameters).getID() : ECDomainParametersIndex.lookupOID(domainParameters);
        if (paramOid == null || !approvedKeySet.contains(paramOid)) {
            throw new FipsUnapprovedOperationError("EC domain parameters not on approved list");
        }
    }

    private static void checkEnabled() {
        if (Properties.isOverrideSet("org.bouncycastle.ec.disable")) {
            throw new UnsupportedOperationException("EC has been disabled by setting \"org.bouncycastle.ec.disable\"");
        }
    }

    private static void validateKeyPair(FipsAlgorithm algorithm, AsymmetricCipherKeyPair kp) {
        Variations variation = algorithm == ALGORITHM ? Variations.ECDSA : (Variations)algorithm.basicVariation();
        switch (variation) {
            case ECDSA: {
                SelfTestExecutor.validate(algorithm, kp, new ConsistencyTest<AsymmetricCipherKeyPair>(){

                    @Override
                    public boolean hasTestPassed(AsymmetricCipherKeyPair kp) throws Exception {
                        EcDsaSigner signer = new EcDsaSigner();
                        signer.init(true, new ParametersWithRandom(kp.getPrivate(), Utils.testRandom));
                        byte[] message = new byte[32];
                        message[1] = 1;
                        BigInteger[] rs = signer.generateSignature(message);
                        signer.init(false, kp.getPublic());
                        return signer.verifySignature(message, rs[0], rs[1]);
                    }
                });
                break;
            }
            case ECDH: {
                SelfTestExecutor.validate(algorithm, kp, new ConsistencyTest<AsymmetricCipherKeyPair>(){

                    @Override
                    public boolean hasTestPassed(AsymmetricCipherKeyPair kp) throws Exception {
                        EcDhBasicAgreement agreement = new EcDhBasicAgreement();
                        agreement.init(kp.getPrivate());
                        BigInteger agree1 = agreement.calculateAgreement(kp.getPublic());
                        AsymmetricCipherKeyPair testKP = FipsEC.getTestKeyPair(kp);
                        agreement.init(testKP.getPrivate());
                        BigInteger agree2 = agreement.calculateAgreement(testKP.getPublic());
                        agreement.init(kp.getPrivate());
                        BigInteger agree3 = agreement.calculateAgreement(testKP.getPublic());
                        agreement.init(testKP.getPrivate());
                        BigInteger agree4 = agreement.calculateAgreement(kp.getPublic());
                        return !agree1.equals(agree2) && !agree1.equals(agree3) && agree3.equals(agree4);
                    }
                });
                break;
            }
            case ECCDH: {
                SelfTestExecutor.validate(algorithm, kp, new ConsistencyTest<AsymmetricCipherKeyPair>(){

                    @Override
                    public boolean hasTestPassed(AsymmetricCipherKeyPair kp) throws Exception {
                        EcDhcBasicAgreement agreement = new EcDhcBasicAgreement();
                        agreement.init(kp.getPrivate());
                        BigInteger agree1 = agreement.calculateAgreement(kp.getPublic());
                        AsymmetricCipherKeyPair testKP = FipsEC.getTestKeyPair(kp);
                        agreement.init(testKP.getPrivate());
                        BigInteger agree2 = agreement.calculateAgreement(testKP.getPublic());
                        agreement.init(kp.getPrivate());
                        BigInteger agree3 = agreement.calculateAgreement(testKP.getPublic());
                        agreement.init(testKP.getPrivate());
                        BigInteger agree4 = agreement.calculateAgreement(kp.getPublic());
                        return !agree1.equals(agree2) && !agree1.equals(agree3) && agree3.equals(agree4);
                    }
                });
                break;
            }
            case ECMQV: {
                SelfTestExecutor.validate(algorithm, kp, new ConsistencyTest<AsymmetricCipherKeyPair>(){

                    @Override
                    public boolean hasTestPassed(AsymmetricCipherKeyPair kp) throws Exception {
                        EcMqvBasicAgreement agreement = new EcMqvBasicAgreement();
                        agreement.init(new EcMqvPrivateParameters((EcPrivateKeyParameters)kp.getPrivate(), (EcPrivateKeyParameters)kp.getPrivate()));
                        BigInteger agree1 = agreement.calculateAgreement(new EcMqvPublicParameters((EcPublicKeyParameters)kp.getPublic(), (EcPublicKeyParameters)kp.getPublic()));
                        AsymmetricCipherKeyPair testSKP = FipsEC.getTestKeyPair(kp);
                        AsymmetricCipherKeyPair testEKP = FipsEC.getTestKeyPair(kp);
                        agreement.init(new EcMqvPrivateParameters((EcPrivateKeyParameters)kp.getPrivate(), (EcPrivateKeyParameters)kp.getPrivate()));
                        BigInteger agree2 = agreement.calculateAgreement(new EcMqvPublicParameters((EcPublicKeyParameters)testSKP.getPublic(), (EcPublicKeyParameters)testEKP.getPublic()));
                        agreement.init(new EcMqvPrivateParameters((EcPrivateKeyParameters)testSKP.getPrivate(), (EcPrivateKeyParameters)testEKP.getPrivate()));
                        BigInteger agree3 = agreement.calculateAgreement(new EcMqvPublicParameters((EcPublicKeyParameters)kp.getPublic(), (EcPublicKeyParameters)kp.getPublic()));
                        return !agree1.equals(agree2) && agree2.equals(agree3);
                    }
                });
                break;
            }
            case ECCDHU: {
                SelfTestExecutor.validate(algorithm, kp, new ConsistencyTest<AsymmetricCipherKeyPair>(){

                    @Override
                    public boolean hasTestPassed(AsymmetricCipherKeyPair kp) throws Exception {
                        EcDhcuBasicAgreement agreement = new EcDhcuBasicAgreement();
                        agreement.init(new EcDhuPrivateParameters((EcPrivateKeyParameters)kp.getPrivate(), (EcPrivateKeyParameters)kp.getPrivate()));
                        byte[] agree1 = agreement.calculateAgreement(new EcDhuPublicParameters((EcPublicKeyParameters)kp.getPublic(), (EcPublicKeyParameters)kp.getPublic()));
                        AsymmetricCipherKeyPair testSKP = FipsEC.getTestKeyPair(kp);
                        AsymmetricCipherKeyPair testEKP = FipsEC.getTestKeyPair(kp);
                        agreement.init(new EcDhuPrivateParameters((EcPrivateKeyParameters)kp.getPrivate(), (EcPrivateKeyParameters)kp.getPrivate()));
                        byte[] agree2 = agreement.calculateAgreement(new EcDhuPublicParameters((EcPublicKeyParameters)testSKP.getPublic(), (EcPublicKeyParameters)testEKP.getPublic()));
                        agreement.init(new EcDhuPrivateParameters((EcPrivateKeyParameters)testSKP.getPrivate(), (EcPrivateKeyParameters)testEKP.getPrivate()));
                        byte[] agree3 = agreement.calculateAgreement(new EcDhuPublicParameters((EcPublicKeyParameters)kp.getPublic(), (EcPublicKeyParameters)kp.getPublic()));
                        return !Arrays.areEqual(agree1, agree2) && Arrays.areEqual(agree2, agree3);
                    }
                });
                break;
            }
            default: {
                throw new IllegalStateException("Unhandled EC algorithm: " + algorithm.getName());
            }
        }
    }

    private static AsymmetricCipherKeyPair getKATKeyPair() {
        NamedECDomainParameters p = ECDomainParametersIndex.lookupDomainParameters(SECObjectIdentifiers.secp256r1);
        EcDomainParameters params = new EcDomainParameters(new ECDomainParameters(p.getCurve(), p.getG(), p.getN(), p.getH(), p.getSeed()));
        EcPrivateKeyParameters priKey = new EcPrivateKeyParameters(new BigInteger("20186677036482506117540275567393538695075300175221296989956723148347484984008"), params);
        EcPublicKeyParameters pubKey = new EcPublicKeyParameters(params.getCurve().decodePoint(Hex.decode("03596375E6CE57E0F20294FC46BDFCFD19A39F8161B58695B3EC5B3D16427C274D")), params);
        return new AsymmetricCipherKeyPair(pubKey, priKey);
    }

    private static AsymmetricCipherKeyPair getF2mKATKeyPair() {
        X9ECParameters p = NISTNamedCurves.getByName("B-233");
        EcDomainParameters params = new EcDomainParameters(new ECDomainParameters(p.getCurve(), p.getG(), p.getN(), p.getH(), p.getSeed()));
        EcPrivateKeyParameters priKey = new EcPrivateKeyParameters(new BigInteger("20186677036482506115567393538695075300175221296989956723148347484984008"), params);
        EcPublicKeyParameters pubKey = new EcPublicKeyParameters(params.getCurve().decodePoint(Hex.decode("03000518bce3b1b492c23094dcd7674c8ea6a3bcb7861bd2fb11be1999b796")), params);
        return new AsymmetricCipherKeyPair(pubKey, priKey);
    }

    private static void f2mDsaTest(EcDsaSigner signer) {
        SelfTestExecutor.validate(ALGORITHM, signer, new VariantKatTest<EcDsaSigner>(){

            @Override
            void evaluate(EcDsaSigner dsa) throws Exception {
                BigInteger f2mR = new BigInteger(1, Hex.decode("d001312179360f7a557d4686e2faf9740fd3289edbafb5e551402cf1b0"));
                BigInteger f2mS = new BigInteger(1, Hex.decode("9d4c2f24b50ce6b9ac725c7833c495fe703296c038dab05ea7af06cafe"));
                AsymmetricCipherKeyPair kp = f2mKatKeyPair;
                TestRandomBigInteger k = new TestRandomBigInteger(233, Hex.decode("640d4957f27d091ab1aebc69949d96e5ac2bb283ed5284a5674758b12f"));
                byte[] M = Hex.decode("1BD4ED430B0F384B4E8D458EFF1A8A553286D7AC21CB2F6806172EF5F94A06AD");
                dsa.init(true, new ParametersWithRandom(kp.getPrivate(), k));
                BigInteger[] sig = dsa.generateSignature(M);
                if (!sig[0].equals(f2mR) || !sig[1].equals(f2mS)) {
                    this.fail("F2m signature incorrect");
                }
                dsa.init(false, kp.getPublic());
                if (!dsa.verifySignature(M, sig[0], sig[1])) {
                    this.fail("F2m signature fails");
                }
            }
        });
    }

    private static void ecPrimitiveZTest() {
        SelfTestExecutor.validate(ALGORITHM, new VariantInternalKatTest(ALGORITHM){

            @Override
            void evaluate() throws Exception {
                X9ECParameters p = NISTNamedCurves.getByName("P-256");
                ECDomainParameters params = new ECDomainParameters(p.getCurve(), p.getG(), p.getN(), p.getH(), p.getSeed());
                BigInteger dValue = new BigInteger("20186677036482506117540275567393538695075300175221296989956723148347484984008");
                ECPoint Q = params.getCurve().decodePoint(Hex.decode("03596375E6CE57E0F20294FC46BDFCFD19A39F8161B58695B3EC5B3D16427C274D"));
                if (!Q.equals(params.getG().multiply(dValue))) {
                    this.fail("EC primitive 'Z' computation failed");
                }
            }
        });
    }

    private static void ecF2mPrimitiveZTest() {
        SelfTestExecutor.validate(ALGORITHM, new VariantInternalKatTest(ALGORITHM){

            @Override
            void evaluate() throws Exception {
                X9ECParameters p = NISTNamedCurves.getByName("B-233");
                ECDomainParameters params = new ECDomainParameters(p.getCurve(), p.getG(), p.getN(), p.getH(), p.getSeed());
                BigInteger dValue = new BigInteger("20186677036482506115567393538695075300175221296989956723148347484984008");
                ECPoint Q = params.getCurve().decodePoint(Hex.decode("03000518bce3b1b492c23094dcd7674c8ea6a3bcb7861bd2fb11be1999b796"));
                if (!Q.equals(params.getG().multiply(dValue))) {
                    this.fail("EC primitive F2m 'Z' computation failed");
                }
            }
        });
    }

    private static AsymmetricCipherKeyPair getTestKeyPair(AsymmetricCipherKeyPair kp) {
        EcPrivateKeyParameters privKey = (EcPrivateKeyParameters)kp.getPrivate();
        EcDomainParameters ecDomainParameters = privKey.getParameters();
        BigInteger testD = privKey.getD().add(TEST_D_OFFSET).mod(ecDomainParameters.getN());
        if (testD.compareTo(ECConstants.TWO) < 0) {
            testD = testD.add(TEST_D_OFFSET);
        }
        EcPrivateKeyParameters testPriv = new EcPrivateKeyParameters(testD, ecDomainParameters);
        EcPublicKeyParameters testPub = new EcPublicKeyParameters(ecDomainParameters.getG().multiply(testD), ecDomainParameters);
        return new AsymmetricCipherKeyPair(testPub, testPriv);
    }

    private static void validateCurveSize(Algorithm algorithm, ECDomainParameters domainParameters) {
        if (domainParameters.getCurve().getFieldSize() < 224) {
            throw new FipsUnapprovedOperationError("Attempt to use curve with field size less than 224 bits", algorithm);
        }
    }

    private static EcDomainParameters getDomainParams(ECDomainParameters curveParams) {
        if (curveParams instanceof NamedECDomainParameters) {
            return new EcNamedDomainParameters((NamedECDomainParameters)curveParams);
        }
        return new EcDomainParameters(curveParams);
    }

    private static EcPrivateKeyParameters getLwKey(final AsymmetricECPrivateKey privKey) {
        return AccessController.doPrivileged(new PrivilegedAction<EcPrivateKeyParameters>(){

            @Override
            public EcPrivateKeyParameters run() {
                return new EcPrivateKeyParameters(privKey.getS(), FipsEC.getDomainParams(privKey.getDomainParameters()));
            }
        });
    }

    private static EcDomainParameters getDomainParamsWithInv(ECDomainParameters curveParams) {
        if (curveParams instanceof NamedECDomainParameters) {
            return new EcNamedDomainParameters((NamedECDomainParameters)curveParams, curveParams.getInverseH());
        }
        return new EcDomainParameters(curveParams, curveParams.getInverseH());
    }

    private static EcPrivateKeyParameters getLwKeyWithInv(final AsymmetricECPrivateKey privKey) {
        return AccessController.doPrivileged(new PrivilegedAction<EcPrivateKeyParameters>(){

            @Override
            public EcPrivateKeyParameters run() {
                return new EcPrivateKeyParameters(privKey.getS(), FipsEC.getDomainParamsWithInv(privKey.getDomainParameters()));
            }
        });
    }

    static {
        FipsEC.ecPrimitiveZTest();
        FipsEC.ecF2mPrimitiveZTest();
        EcDsaSigner signer = DSA_PROVIDER.createEngine(true);
        DSA_PROVIDER.createEngine(false);
        FipsEC.f2mDsaTest(signer);
        approvedKeySet = new HashSet<ASN1ObjectIdentifier>();
        DomainParameterID[] ids = DomainParameterID.values();
        for (int i = 0; i != ids.length; ++i) {
            String name = ids[i].curveName;
            if (!Properties.isOverrideSet("org.bouncycastle.ec.enable_f2m") && (name.charAt(0) == 'B' || name.charAt(0) == 'K')) continue;
            approvedKeySet.add(ECNamedCurveTable.getOID(name));
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    private static class DhuProvider
    extends FipsEngineProvider<EcDhcuBasicAgreement> {
        static final byte[] expected = Hex.decode("cad5c428ea0645794bc5634549e08a3ed563bd0cf32e909862e08b41d4b6fc17cad5c428ea0645794bc5634549e08a3ed563bd0cf32e909862e08b41d4b6fc17");

        private DhuProvider() {
        }

        @Override
        public EcDhcuBasicAgreement createEngine() {
            return SelfTestExecutor.validate(ALGORITHM, new EcDhcuBasicAgreement(), new VariantKatTest<EcDhcuBasicAgreement>(){

                @Override
                void evaluate(EcDhcuBasicAgreement agreement) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    AsymmetricCipherKeyPair testSKP = FipsEC.getTestKeyPair(kp);
                    AsymmetricCipherKeyPair testEKP = FipsEC.getTestKeyPair(kp);
                    agreement.init(new EcDhuPrivateParameters((EcPrivateKeyParameters)kp.getPrivate(), (EcPrivateKeyParameters)kp.getPrivate()));
                    byte[] calculated = agreement.calculateAgreement(new EcDhuPublicParameters((EcPublicKeyParameters)testSKP.getPublic(), (EcPublicKeyParameters)testEKP.getPublic()));
                    if (!Arrays.areEqual(expected, calculated)) {
                        this.fail("KAT ECCDHU agreement not verified");
                    }
                }
            });
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    private static class MqvProvider
    extends FipsEngineProvider<EcMqvBasicAgreement> {
        static final BigInteger expected = new BigInteger("8cae3483c0d3dac87d1c1d32be8e7b7a3c1558bd01cb7e7bb37c1c81126b0f98", 16);

        private MqvProvider() {
        }

        @Override
        public EcMqvBasicAgreement createEngine() {
            return SelfTestExecutor.validate(ALGORITHM, new EcMqvBasicAgreement(), new VariantKatTest<EcMqvBasicAgreement>(){

                @Override
                void evaluate(EcMqvBasicAgreement agreement) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    AsymmetricCipherKeyPair testSKP = FipsEC.getTestKeyPair(kp);
                    AsymmetricCipherKeyPair testEKP = FipsEC.getTestKeyPair(kp);
                    agreement.init(new EcMqvPrivateParameters((EcPrivateKeyParameters)kp.getPrivate(), (EcPrivateKeyParameters)kp.getPrivate()));
                    BigInteger calculated = agreement.calculateAgreement(new EcMqvPublicParameters((EcPublicKeyParameters)testSKP.getPublic(), (EcPublicKeyParameters)testEKP.getPublic()));
                    if (!expected.equals(calculated)) {
                        this.fail("KAT ECMQV agreement not verified");
                    }
                }
            });
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    private static class DhcProvider
    extends FipsEngineProvider<EcDhcBasicAgreement> {
        static final BigInteger expected = new BigInteger("cad5c428ea0645794bc5634549e08a3ed563bd0cf32e909862e08b41d4b6fc17", 16);

        private DhcProvider() {
        }

        @Override
        public EcDhcBasicAgreement createEngine() {
            return SelfTestExecutor.validate(ALGORITHM, new EcDhcBasicAgreement(), new VariantKatTest<EcDhcBasicAgreement>(){

                @Override
                void evaluate(EcDhcBasicAgreement agreement) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    AsymmetricCipherKeyPair testOther = FipsEC.getTestKeyPair(kp);
                    agreement.init(kp.getPrivate());
                    if (!expected.equals(agreement.calculateAgreement(testOther.getPublic()))) {
                        this.fail("KAT ECDHC agreement not verified");
                    }
                }
            });
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    private static class DhProvider
    extends FipsEngineProvider<EcDhBasicAgreement> {
        static BigInteger expected = new BigInteger("cad5c428ea0645794bc5634549e08a3ed563bd0cf32e909862e08b41d4b6fc17", 16);

        private DhProvider() {
        }

        @Override
        public EcDhBasicAgreement createEngine() {
            return SelfTestExecutor.validate(ALGORITHM, new EcDhBasicAgreement(), new VariantKatTest<EcDhBasicAgreement>(){

                @Override
                void evaluate(EcDhBasicAgreement agreement) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    AsymmetricCipherKeyPair testOther = FipsEC.getTestKeyPair(kp);
                    agreement.init(kp.getPrivate());
                    if (!expected.equals(agreement.calculateAgreement(testOther.getPublic()))) {
                        this.fail("KAT ECDH agreement not verified");
                    }
                }
            });
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    private static class DsaProvider
    extends FipsEngineProvider<EcDsaSigner> {
        private static final BigInteger r = new BigInteger("d73cd3722bae6cc0b39065bb4003d8ece1ef2f7a8a55bfd677234b0b3b902650", 16);
        private static final BigInteger s = new BigInteger("d9c88297fefed8441e08dda69554a6452b8a0bd4a0ea1ddb750499f0c2298c2f", 16);
        private static final Map<FipsDigestAlgorithm, BigInteger[]> sigKats = new HashMap<FipsDigestAlgorithm, BigInteger[]>();
        private static final BigInteger rDetSha1 = new BigInteger("9390526f81255b33c950a4419878801be7bf3e5823468bae22bc6e4e86734ffb", 16);
        private static final BigInteger sDetSha1 = new BigInteger("5a4841d6927d9a3bc5c3b37413d798d6d01fba107411ed4b02986d15becf9670", 16);
        private static final BigInteger rDetSha224 = new BigInteger("b2f31c147179b5c27161b2cc7dcbd0f6a2d74dd5ee3b433928a0fe81a0d77d38", 16);
        private static final BigInteger sDetSha224 = new BigInteger("fb6f8c56b1b66bafc4d3294c0df57bc6ec22d12fd22da0bcb4c4741a5d10a2fd", 16);
        private static final BigInteger rDetSha256 = new BigInteger("522bb76fa0ab8f043b6be30f099b56b12bcc69f247f5a5a39cec4b02fd08b87b", 16);
        private static final BigInteger sDetSha256 = new BigInteger("32fc666aa042b991aa86a1c5f4697418484f81f91485bf8ffa51590131a7a78f", 16);
        private static final BigInteger rDetSha384 = new BigInteger("6d6b76dbf798f27e8950bd0fe6c228d78852d513613423a3ee085cca56bea0ba", 16);
        private static final BigInteger sDetSha384 = new BigInteger("71547e92702e2b6c770dc542144c8dfc5b31e6c8a8c5a8929915d0ec12cfff5", 16);
        private static final BigInteger rDetSha512 = new BigInteger("caa9a9f66852c1744fb40f1b59835337eabb0148f97f1f867ad8246cd282d0a2", 16);
        private static final BigInteger sDetSha512 = new BigInteger("946d6b64469b25d059ec8083e5cda36e4443dc98d2289a890587f49590360814", 16);
        private static final BigInteger rDetSha512_224 = new BigInteger("b92d4c176762df6027431e2b08686eeb3395cde69cf3872ff1ba864caa7f32cd", 16);
        private static final BigInteger sDetSha512_224 = new BigInteger("ab7af8d8646e917744ea5bda468e28d0d413c207135b6604862412a2ef704c40", 16);
        private static final BigInteger rDetSha512_256 = new BigInteger("57e55a88f7ea9f036ea64bd247029c821b10fd167ba6442ba2a9d89660f6c605", 16);
        private static final BigInteger sDetSha512_256 = new BigInteger("763419c86554203db3de94642877b4d74d6ac37f6cf09b58978a7b945243928e", 16);
        private static final BigInteger rDetSha3_224 = new BigInteger("5330797301a8abe67a44cc890f160fabf41dea224d7f9b139f1b3ca48b113fae", 16);
        private static final BigInteger sDetSha3_224 = new BigInteger("a95a916fec76fbbe67fa4f3903d22c20069b46e90aa15bb49e7ff27b1f7a6505", 16);
        private static final BigInteger rDetSha3_256 = new BigInteger("c79d5244c610b6f0f8ca6d848ea77e67c2cc7087b9617e9c93d52a08ee03d70", 16);
        private static final BigInteger sDetSha3_256 = new BigInteger("1ac50fbb370c75ad97d81ca98018e3422209d57fecab4427f236b67f3a148fa7", 16);
        private static final BigInteger rDetSha3_384 = new BigInteger("9f4a4e59dc8ba7dcc35ff3542c37e5d5f3a9df697573cc38156e103731181977", 16);
        private static final BigInteger sDetSha3_384 = new BigInteger("9743b86945e035447613bc3ea86885bb4855b8ff638e02bb841b4685dcc04605", 16);
        private static final BigInteger rDetSha3_512 = new BigInteger("597d41ed723bc9827cab2b011a9a540aa4082024f347d3d168166e29df5029d4", 16);
        private static final BigInteger sDetSha3_512 = new BigInteger("627f032d2252d942fabf533df4f11817fac3fd1a53559d2689a0cc368b22824e", 16);
        private static final BigInteger rDetShake128 = new BigInteger("cacabf72ff5250ff193e201b65a9c0d25ae52a4d02f67f3ad6c817fbd962c254", 16);
        private static final BigInteger sDetShake128 = new BigInteger("96ec4a5b73603df896adc6965e18be02c326cc922b3dd399c1309b0bf14aa0e4", 16);
        private static final BigInteger rDetShake256 = new BigInteger("680d40574d06cbf3c2502b76f91ebe16d4eca4607554d0036b50446526118111", 16);
        private static final BigInteger sDetShake256 = new BigInteger("bedaf371aa2de86dd3d201db52c351f6f0c5fe132a355db0e26f3922e691439e", 16);

        private DsaProvider() {
        }

        @Override
        public EcDsaSigner createEngine() {
            return SelfTestExecutor.validate(ALGORITHM, new EcDsaSigner(), new VariantKatTest<EcDsaSigner>(){

                @Override
                void evaluate(EcDsaSigner dsa) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    byte[] M = Hex.decode("1BD4ED430B0F384B4E8D458EFF1A8A553286D7AC21CB2F6806172EF5F94A06AD");
                    TestRandomData k = new TestRandomData(Hex.decode("a0640d4957f27d091ab1aebc69949d96e5ac2bb283ed5284a5674758b12f08df"));
                    dsa.init(true, new ParametersWithRandom(kp.getPrivate(), k));
                    BigInteger[] sig = dsa.generateSignature(M);
                    if (!sig[0].equals(r) || !sig[1].equals(s)) {
                        this.fail("signature incorrect");
                    }
                    dsa.init(false, kp.getPublic());
                    if (!dsa.verifySignature(M, r, s)) {
                        this.fail("signature fails");
                    }
                }
            });
        }

        public EcDsaSigner createEngine(final boolean forSigning) {
            return SelfTestExecutor.validate(ALGORITHM, new EcDsaSigner(), new VariantKatTest<EcDsaSigner>(){

                @Override
                void evaluate(EcDsaSigner dsa) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    byte[] M = Hex.decode("1BD4ED430B0F384B4E8D458EFF1A8A553286D7AC21CB2F6806172EF5F94A06AD");
                    if (forSigning) {
                        TestRandomData k = new TestRandomData(Hex.decode("a0640d4957f27d091ab1aebc69949d96e5ac2bb283ed5284a5674758b12f08df"));
                        dsa.init(true, new ParametersWithRandom(kp.getPrivate(), k));
                        BigInteger[] sig = dsa.generateSignature(M);
                        if (!sig[0].equals(r) || !sig[1].equals(s)) {
                            this.fail("signature incorrect");
                        }
                    } else {
                        dsa.init(false, kp.getPublic());
                        if (!dsa.verifySignature(M, r, s)) {
                            this.fail("signature fails");
                        }
                    }
                }
            });
        }

        public EcDsaSigner createEngine(final FipsDigestAlgorithm digest, HMacDsaKCalculator calculator) {
            return SelfTestExecutor.validate(ALGORITHM, new EcDsaSigner(calculator), new VariantKatTest<EcDsaSigner>(){

                @Override
                void evaluate(EcDsaSigner dsa) throws Exception {
                    AsymmetricCipherKeyPair kp = katKeyPair;
                    byte[] M = Hex.decode("1BD4ED430B0F384B4E8D458EFF1A8A553286D7AC21CB2F6806172EF5F94A06AD");
                    dsa.init(true, kp.getPrivate());
                    BigInteger[] sig = dsa.generateSignature(M);
                    BigInteger[] katSig = (BigInteger[])sigKats.get(digest);
                    if (katSig != null) {
                        if (!sig[0].equals(katSig[0]) || !sig[1].equals(katSig[1])) {
                            this.fail("signature incorrect");
                        }
                    } else {
                        dsa.init(false, kp.getPublic());
                        if (!dsa.verifySignature(M, sig[0], sig[1])) {
                            this.fail("signature check failed");
                        }
                    }
                }
            });
        }

        static {
            sigKats.put(FipsSHS.Algorithm.SHA1, new BigInteger[]{rDetSha1, sDetSha1});
            sigKats.put(FipsSHS.Algorithm.SHA224, new BigInteger[]{rDetSha224, sDetSha224});
            sigKats.put(FipsSHS.Algorithm.SHA256, new BigInteger[]{rDetSha256, sDetSha256});
            sigKats.put(FipsSHS.Algorithm.SHA384, new BigInteger[]{rDetSha384, sDetSha384});
            sigKats.put(FipsSHS.Algorithm.SHA512, new BigInteger[]{rDetSha512, sDetSha512});
            sigKats.put(FipsSHS.Algorithm.SHA512_224, new BigInteger[]{rDetSha512_224, sDetSha512_224});
            sigKats.put(FipsSHS.Algorithm.SHA512_256, new BigInteger[]{rDetSha512_256, sDetSha512_256});
            sigKats.put(FipsSHS.Algorithm.SHA3_224, new BigInteger[]{rDetSha3_224, sDetSha3_224});
            sigKats.put(FipsSHS.Algorithm.SHA3_256, new BigInteger[]{rDetSha3_256, sDetSha3_256});
            sigKats.put(FipsSHS.Algorithm.SHA3_384, new BigInteger[]{rDetSha3_384, sDetSha3_384});
            sigKats.put(FipsSHS.Algorithm.SHA3_512, new BigInteger[]{rDetSha3_512, sDetSha3_512});
            sigKats.put(FipsSHS.Algorithm.SHAKE128, new BigInteger[]{rDetShake128, sDetShake128});
            sigKats.put(FipsSHS.Algorithm.SHAKE256, new BigInteger[]{rDetShake256, sDetShake256});
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class DSAOperatorFactory
    extends FipsSignatureOperatorFactory<DSAParameters> {
        public DSAOperatorFactory() {
            FipsEC.checkEnabled();
        }

        @Override
        public FipsOutputSignerUsingSecureRandom<DSAParameters> createSigner(AsymmetricPrivateKey key, DSAParameters parameters) {
            EcDsaSigner ecdsaSigner;
            Digest digest;
            AsymmetricECPrivateKey k = (AsymmetricECPrivateKey)key;
            if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                FipsEC.validateDomainParameters(k.getDomainParameters());
                FipsEC.validateCurveSize(key.getAlgorithm(), k.getDomainParameters());
                Utils.checkDigestAlgorithm(LOG, parameters.getDigestAlgorithm(), "org.bouncycastle.ec.allow_sha1_sig");
            }
            Digest digest2 = digest = parameters.digestAlgorithm != null ? FipsSHS.createDigest(parameters.digestAlgorithm) : new NullDigest();
            if (parameters.getAlgorithm().basicVariation() == Variations.ECDSA) {
                ecdsaSigner = DSA_PROVIDER.createEngine(true);
            } else {
                ExtendedDigest macDigest = FipsSHS.createDigest(parameters.digestAlgorithm);
                if (macDigest == null) {
                    throw new IllegalArgumentException("no HMAC support for chosen digest: " + parameters.digestAlgorithm.getName());
                }
                ecdsaSigner = DSA_PROVIDER.createEngine(parameters.digestAlgorithm, new HMacDsaKCalculator(macDigest));
            }
            final EcPrivateKeyParameters privateKeyParameters = FipsEC.getLwKey(k);
            return new DSAOutputSigner<DSAParameters>(ecdsaSigner, digest, parameters, new DSAOutputSigner.Initializer(){

                @Override
                public void initialize(DSA signer, SecureRandom random) {
                    signer.init(true, new ParametersWithRandom(privateKeyParameters, random));
                }
            });
        }

        @Override
        public FipsOutputVerifier<DSAParameters> createVerifier(AsymmetricPublicKey key, DSAParameters parameters) {
            EcDsaSigner ecdsaSigner = DSA_PROVIDER.createEngine(false);
            Digest digest = parameters.digestAlgorithm != null ? FipsSHS.createDigest(parameters.digestAlgorithm) : new NullDigest();
            AsymmetricECPublicKey k = (AsymmetricECPublicKey)key;
            EcPublicKeyParameters publicKeyParameters = new EcPublicKeyParameters(k.getW(), FipsEC.getDomainParams(k.getDomainParameters()));
            ecdsaSigner.init(false, publicKeyParameters);
            return new DSAOutputVerifier<DSAParameters>(ecdsaSigner, digest, parameters);
        }

        @Override
        public FipsOutputValidator<DSAParameters> createValidator(AsymmetricPublicKey key, DSAParameters parameters, byte[] signature) throws InvalidSignatureException {
            EcDsaSigner ecdsaSigner = DSA_PROVIDER.createEngine(false);
            Digest digest = parameters.digestAlgorithm != null ? FipsSHS.createDigest(parameters.digestAlgorithm) : new NullDigest();
            AsymmetricECPublicKey k = (AsymmetricECPublicKey)key;
            EcPublicKeyParameters publicKeyParameters = new EcPublicKeyParameters(k.getW(), FipsEC.getDomainParams(k.getDomainParameters()));
            ecdsaSigner.init(false, publicKeyParameters);
            return new DSAOutputValidator<DSAParameters>(ecdsaSigner, digest, parameters, signature);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class DHUAgreementFactory
    extends FipsAgreementFactory<DHUAgreementParameters> {
        public DHUAgreementFactory() {
            FipsEC.checkEnabled();
        }

        @Override
        public FipsAgreement<DHUAgreementParameters> createAgreement(AsymmetricPrivateKey key, DHUAgreementParameters parameters) {
            AsymmetricECPrivateKey ecKey = (AsymmetricECPrivateKey)key;
            if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                FipsEC.validateDomainParameters(ecKey.getDomainParameters());
                FipsEC.validateCurveSize(key.getAlgorithm(), ecKey.getDomainParameters());
            }
            EcPrivateKeyParameters lwECKey = FipsEC.getLwKey(ecKey);
            EcDhcuBasicAgreement ecdh = (EcDhcuBasicAgreement)DHU_PROVIDER.createEngine();
            ecdh.init(new EcDhuPrivateParameters(lwECKey, parameters.ephemeralPrivateKey == null ? lwECKey : FipsEC.getLwKey(parameters.ephemeralPrivateKey)));
            return new EcDHUAgreement<DHUAgreementParameters>(ecdh, parameters);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class MQVAgreementFactory
    extends FipsAgreementFactory<MQVAgreementParameters> {
        public MQVAgreementFactory() {
            FipsEC.checkEnabled();
            if (Properties.isOverrideSet("org.bouncycastle.ec.disable_mqv")) {
                throw new UnsupportedOperationException("EC MQV has been disabled by setting \"org.bouncycastle.ec.disable_mqv\"");
            }
        }

        @Override
        public FipsAgreement<MQVAgreementParameters> createAgreement(AsymmetricPrivateKey key, MQVAgreementParameters parameters) {
            AsymmetricECPrivateKey ecKey = (AsymmetricECPrivateKey)key;
            if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                FipsEC.validateDomainParameters(ecKey.getDomainParameters());
                FipsEC.validateCurveSize(key.getAlgorithm(), ecKey.getDomainParameters());
            }
            EcPrivateKeyParameters lwECKey = FipsEC.getLwKey(ecKey);
            EcMqvBasicAgreement ecdh = (EcMqvBasicAgreement)MQV_PROVIDER.createEngine();
            ecdh.init(new EcMqvPrivateParameters(lwECKey, parameters.ephemeralPrivateKey == null ? lwECKey : FipsEC.getLwKey(parameters.ephemeralPrivateKey)));
            return new EcDHAgreement<MQVAgreementParameters>(ecdh, parameters);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class DHAgreementFactory
    extends FipsAgreementFactory<AgreementParameters> {
        public DHAgreementFactory() {
            FipsEC.checkEnabled();
        }

        @Override
        public FipsAgreement<AgreementParameters> createAgreement(AsymmetricPrivateKey key, AgreementParameters parameters) {
            if (parameters.getAlgorithm() == DH.getAlgorithm()) {
                AsymmetricECPrivateKey ecKey = (AsymmetricECPrivateKey)key;
                if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                    if (!BigInteger.ONE.equals(ecKey.getDomainParameters().getH())) {
                        throw new FipsUnapprovedOperationError("ECDH can only be executed on curves with a co-factor of 1 in approved mode", key.getAlgorithm());
                    }
                    FipsEC.validateDomainParameters(ecKey.getDomainParameters());
                    FipsEC.validateCurveSize(key.getAlgorithm(), ecKey.getDomainParameters());
                }
                EcPrivateKeyParameters lwECKey = !BigInteger.ONE.equals(ecKey.getDomainParameters().getH()) ? FipsEC.getLwKeyWithInv(ecKey) : FipsEC.getLwKey(ecKey);
                EcDhBasicAgreement ecdh = (EcDhBasicAgreement)DH_PROVIDER.createEngine();
                ecdh.init(lwECKey);
                return new EcDHAgreement<AgreementParameters>(ecdh, parameters);
            }
            if (parameters.getAlgorithm() == CDH.getAlgorithm()) {
                AsymmetricECPrivateKey ecKey = (AsymmetricECPrivateKey)key;
                if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                    FipsEC.validateDomainParameters(ecKey.getDomainParameters());
                    FipsEC.validateCurveSize(key.getAlgorithm(), ecKey.getDomainParameters());
                }
                EcPrivateKeyParameters lwECKey = FipsEC.getLwKey(ecKey);
                EcDhcBasicAgreement ecdh = (EcDhcBasicAgreement)CDH_PROVIDER.createEngine();
                ecdh.init(lwECKey);
                return new EcDHAgreement<AgreementParameters>(ecdh, parameters);
            }
            throw new IllegalArgumentException("Incorrect algorithm in parameters for EC DH: " + parameters.getAlgorithm().getName());
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class KeyPairGenerator
    extends FipsAsymmetricKeyPairGenerator<KeyGenParameters, AsymmetricECPublicKey, AsymmetricECPrivateKey> {
        private final EcKeyPairGenerator engine = new EcKeyPairGenerator();
        private final ECDomainParameters domainParameters;
        private final EcKeyGenerationParameters param;

        public KeyPairGenerator(KeyGenParameters keyGenParameters, SecureRandom random) {
            super(keyGenParameters);
            FipsEC.checkEnabled();
            if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                FipsEC.validateDomainParameters(keyGenParameters.getDomainParameters());
                FipsEC.validateCurveSize(keyGenParameters.getAlgorithm(), keyGenParameters.getDomainParameters());
                Utils.validateKeyPairGenRandom(random, Utils.getECCurveSecurityStrength(keyGenParameters.getDomainParameters().getCurve()), ALGORITHM);
            }
            this.param = ((KeyGenParameters)this.getParameters()).getAlgorithm().equals(DH.getAlgorithm()) && !ECConstants.ONE.equals(keyGenParameters.domainParameters.getH()) ? new EcKeyGenerationParameters(FipsEC.getDomainParamsWithInv(keyGenParameters.getDomainParameters()), random) : new EcKeyGenerationParameters(FipsEC.getDomainParams(keyGenParameters.getDomainParameters()), random);
            this.domainParameters = keyGenParameters.getDomainParameters();
            this.engine.init(this.param);
        }

        @Override
        public AsymmetricKeyPair<AsymmetricECPublicKey, AsymmetricECPrivateKey> generateKeyPair() {
            AsymmetricCipherKeyPair kp = this.engine.generateKeyPair();
            EcPublicKeyParameters pubKey = (EcPublicKeyParameters)kp.getPublic();
            EcPrivateKeyParameters prvKey = (EcPrivateKeyParameters)kp.getPrivate();
            FipsAlgorithm algorithm = ((KeyGenParameters)this.getParameters()).getAlgorithm();
            FipsEC.validateKeyPair(algorithm, kp);
            return new AsymmetricKeyPair<AsymmetricECPublicKey, AsymmetricECPrivateKey>(new AsymmetricECPublicKey((Algorithm)algorithm, this.domainParameters, pubKey.getQ()), new AsymmetricECPrivateKey((Algorithm)algorithm, this.domainParameters, prvKey.getD(), pubKey.getQ()));
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class DHUAgreementParameters
    extends FipsAgreementParameters {
        private final AsymmetricECPublicKey ephemeralPublicKey;
        private final AsymmetricECPrivateKey ephemeralPrivateKey;
        private final AsymmetricECPublicKey otherPartyEphemeralKey;

        private DHUAgreementParameters(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey, FipsAlgorithm digestAlgorithm) {
            super(ALGORITHM_DHU, digestAlgorithm);
            this.ephemeralPublicKey = ephemeralPublicKey;
            this.ephemeralPrivateKey = ephemeralPrivateKey;
            this.otherPartyEphemeralKey = otherPartyEphemeralKey;
        }

        private DHUAgreementParameters(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey, FipsKDF.PRF prfAlgorithm, byte[] salt) {
            super(ALGORITHM_DHU, prfAlgorithm, salt);
            this.ephemeralPublicKey = ephemeralPublicKey;
            this.ephemeralPrivateKey = ephemeralPrivateKey;
            this.otherPartyEphemeralKey = otherPartyEphemeralKey;
        }

        private DHUAgreementParameters(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey, FipsKDF.AgreementKDFParametersBuilder kdfType, byte[] iv, int outputSize) {
            super(ALGORITHM_DHU, kdfType, iv, outputSize);
            this.ephemeralPublicKey = ephemeralPublicKey;
            this.ephemeralPrivateKey = ephemeralPrivateKey;
            this.otherPartyEphemeralKey = otherPartyEphemeralKey;
        }

        public AsymmetricECPublicKey getEphemeralPublicKey() {
            return this.ephemeralPublicKey;
        }

        public AsymmetricECPrivateKey getEphemeralPrivateKey() {
            return this.ephemeralPrivateKey;
        }

        public AsymmetricECPublicKey getOtherPartyEphemeralKey() {
            return this.otherPartyEphemeralKey;
        }

        public DHUAgreementParameters withDigest(FipsAlgorithm digestAlgorithm) {
            return new DHUAgreementParameters(this.ephemeralPublicKey, this.ephemeralPrivateKey, this.otherPartyEphemeralKey, digestAlgorithm);
        }

        public DHUAgreementParameters withPRF(FipsKDF.PRF prfAlgorithm, byte[] salt) {
            return new DHUAgreementParameters(this.ephemeralPublicKey, this.ephemeralPrivateKey, this.otherPartyEphemeralKey, prfAlgorithm, salt);
        }

        public DHUAgreementParameters withKDF(FipsKDF.AgreementKDFParametersBuilder kdfType, byte[] iv, int outputSize) {
            return new DHUAgreementParameters(this.ephemeralPublicKey, this.ephemeralPrivateKey, this.otherPartyEphemeralKey, kdfType, iv, outputSize);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class DHUAgreementParametersBuilder
    extends FipsParameters {
        DHUAgreementParametersBuilder() {
            super(ALGORITHM_DHU);
        }

        public DHUAgreementParameters using(AsymmetricKeyPair ephemeralKeyPair, AsymmetricECPublicKey otherPartyEphemeralKey) {
            return new DHUAgreementParameters((AsymmetricECPublicKey)ephemeralKeyPair.getPublicKey(), (AsymmetricECPrivateKey)ephemeralKeyPair.getPrivateKey(), otherPartyEphemeralKey, null);
        }

        public DHUAgreementParameters using(AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey) {
            return new DHUAgreementParameters(null, ephemeralPrivateKey, otherPartyEphemeralKey, null);
        }

        public DHUAgreementParameters using(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey) {
            return new DHUAgreementParameters(ephemeralPublicKey, ephemeralPrivateKey, otherPartyEphemeralKey, null);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class MQVAgreementParameters
    extends FipsAgreementParameters {
        private final AsymmetricECPublicKey ephemeralPublicKey;
        private final AsymmetricECPrivateKey ephemeralPrivateKey;
        private final AsymmetricECPublicKey otherPartyEphemeralKey;

        private MQVAgreementParameters(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey, FipsAlgorithm digestAlgorithm) {
            super(ALGORITHM_MQV, digestAlgorithm);
            this.ephemeralPublicKey = ephemeralPublicKey;
            this.ephemeralPrivateKey = ephemeralPrivateKey;
            this.otherPartyEphemeralKey = otherPartyEphemeralKey;
        }

        private MQVAgreementParameters(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey, FipsKDF.PRF prfAlgorithm, byte[] salt) {
            super(ALGORITHM_MQV, prfAlgorithm, salt);
            this.ephemeralPublicKey = ephemeralPublicKey;
            this.ephemeralPrivateKey = ephemeralPrivateKey;
            this.otherPartyEphemeralKey = otherPartyEphemeralKey;
        }

        private MQVAgreementParameters(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey, FipsKDF.AgreementKDFParametersBuilder kdfType, byte[] iv, int outputSize) {
            super(ALGORITHM_MQV, kdfType, iv, outputSize);
            this.ephemeralPublicKey = ephemeralPublicKey;
            this.ephemeralPrivateKey = ephemeralPrivateKey;
            this.otherPartyEphemeralKey = otherPartyEphemeralKey;
        }

        public AsymmetricECPublicKey getEphemeralPublicKey() {
            return this.ephemeralPublicKey;
        }

        public AsymmetricECPrivateKey getEphemeralPrivateKey() {
            return this.ephemeralPrivateKey;
        }

        public AsymmetricECPublicKey getOtherPartyEphemeralKey() {
            return this.otherPartyEphemeralKey;
        }

        public MQVAgreementParameters withDigest(FipsAlgorithm digestAlgorithm) {
            return new MQVAgreementParameters(this.ephemeralPublicKey, this.ephemeralPrivateKey, this.otherPartyEphemeralKey, digestAlgorithm);
        }

        public MQVAgreementParameters withPRF(FipsKDF.PRF prfAlgorithm, byte[] salt) {
            return new MQVAgreementParameters(this.ephemeralPublicKey, this.ephemeralPrivateKey, this.otherPartyEphemeralKey, prfAlgorithm, salt);
        }

        public MQVAgreementParameters withKDF(FipsKDF.AgreementKDFParametersBuilder kdfType, byte[] iv, int outputSize) {
            return new MQVAgreementParameters(this.ephemeralPublicKey, this.ephemeralPrivateKey, this.otherPartyEphemeralKey, kdfType, iv, outputSize);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class MQVAgreementParametersBuilder
    extends FipsParameters {
        MQVAgreementParametersBuilder() {
            super(ALGORITHM_MQV);
        }

        public MQVAgreementParameters using(AsymmetricKeyPair ephemeralKeyPair, AsymmetricECPublicKey otherPartyEphemeralKey) {
            return new MQVAgreementParameters((AsymmetricECPublicKey)ephemeralKeyPair.getPublicKey(), (AsymmetricECPrivateKey)ephemeralKeyPair.getPrivateKey(), otherPartyEphemeralKey, null);
        }

        public MQVAgreementParameters using(AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey) {
            return new MQVAgreementParameters(null, ephemeralPrivateKey, otherPartyEphemeralKey, null);
        }

        public MQVAgreementParameters using(AsymmetricECPublicKey ephemeralPublicKey, AsymmetricECPrivateKey ephemeralPrivateKey, AsymmetricECPublicKey otherPartyEphemeralKey) {
            return new MQVAgreementParameters(ephemeralPublicKey, ephemeralPrivateKey, otherPartyEphemeralKey, null);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class DSAParameters
    extends FipsParameters {
        private final FipsDigestAlgorithm digestAlgorithm;

        DSAParameters(FipsAlgorithm algorithm, FipsDigestAlgorithm digestAlgorithm) {
            super(algorithm);
            if (digestAlgorithm == null && CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                PrivilegedUtils.checkPermission(Permissions.TlsNullDigestEnabled);
            }
            if (algorithm.basicVariation() == Variations.ECDDSA && digestAlgorithm == null) {
                throw new IllegalArgumentException("ECDDSA cannot be used with a NULL digest");
            }
            this.digestAlgorithm = digestAlgorithm;
        }

        public FipsDigestAlgorithm getDigestAlgorithm() {
            return this.digestAlgorithm;
        }

        public DSAParameters withDigestAlgorithm(FipsDigestAlgorithm digestAlgorithm) {
            return new DSAParameters(this.getAlgorithm(), digestAlgorithm);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class AgreementParameters
    extends FipsAgreementParameters {
        AgreementParameters(FipsAlgorithm agreementAlgorithm) {
            this(agreementAlgorithm, null);
        }

        private AgreementParameters(FipsAlgorithm agreementAlgorithm, FipsAlgorithm digestAlgorithm) {
            super(agreementAlgorithm, digestAlgorithm);
        }

        private AgreementParameters(FipsAlgorithm agreementAlgorithm, FipsKDF.PRF prfAlgorithm, byte[] salt) {
            super(agreementAlgorithm, prfAlgorithm, salt);
        }

        private AgreementParameters(FipsAlgorithm agreementAlgorithm, FipsKDF.AgreementKDFParametersBuilder kdfType, byte[] iv, int outputSize) {
            super(agreementAlgorithm, kdfType, iv, outputSize);
        }

        public AgreementParameters withDigest(FipsDigestAlgorithm digestAlgorithm) {
            return new AgreementParameters(this.getAlgorithm(), digestAlgorithm);
        }

        public AgreementParameters withPRF(FipsKDF.PRF prfAlgorithm, byte[] salt) {
            return new AgreementParameters(this.getAlgorithm(), prfAlgorithm, salt);
        }

        public AgreementParameters withKDF(FipsKDF.AgreementKDFParametersBuilder kdfType, byte[] iv, int outputSize) {
            return new AgreementParameters(this.getAlgorithm(), kdfType, iv, outputSize);
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static final class KeyGenParameters
    extends FipsParameters {
        private final ECDomainParameters domainParameters;

        public KeyGenParameters(ECDomainParameters domainParameters) {
            this(ALGORITHM, domainParameters);
        }

        public KeyGenParameters(DSAParameters parameters, ECDomainParameters domainParameters) {
            this(parameters.getAlgorithm(), domainParameters);
        }

        public KeyGenParameters(AgreementParameters parameters, ECDomainParameters domainParameters) {
            this(parameters.getAlgorithm(), domainParameters);
        }

        public KeyGenParameters(MQVAgreementParametersBuilder builder, ECDomainParameters domainParameters) {
            this(ALGORITHM_MQV, domainParameters);
        }

        public KeyGenParameters(DHUAgreementParametersBuilder builder, ECDomainParameters domainParameters) {
            this(ALGORITHM_DHU, domainParameters);
        }

        KeyGenParameters(FipsAlgorithm algorithm, ECDomainParameters domainParameters) {
            super(algorithm);
            if (CryptoServicesRegistrar.isInApprovedOnlyMode()) {
                FipsEC.validateDomainParameters(domainParameters);
            }
            this.domainParameters = domainParameters;
        }

        public ECDomainParameters getDomainParameters() {
            return this.domainParameters;
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    public static enum DomainParameterID implements ECDomainParametersID
    {
        B571("B-571"),
        B409("B-409"),
        B283("B-283"),
        B233("B-233"),
        K571("K-571"),
        K409("K-409"),
        K283("K-283"),
        K233("K-233"),
        P521("P-521"),
        P384("P-384"),
        P256("P-256"),
        P224("P-224"),
        brainpoolP224r1("brainpoolP224r1"),
        brainpoolP224t1("brainpoolP224t1"),
        brainpoolP256r1("brainpoolP256r1"),
        brainpoolP256t1("brainpoolP256t1"),
        brainpoolP320r1("brainpoolP320r1"),
        brainpoolP320t1("brainpoolP320t1"),
        brainpoolP384r1("brainpoolP384r1"),
        brainpoolP384t1("brainpoolP384t1"),
        brainpoolP512r1("brainpoolP512r1"),
        brainpoolP512t1("brainpoolP512t1"),
        secp256k1("secp256k1");

        private final String curveName;

        private DomainParameterID(String curveName) {
            this.curveName = curveName;
        }

        @Override
        public String getCurveName() {
            return this.curveName;
        }
    }

    /*
     * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
     */
    private static enum Variations {
        ECDSA,
        ECDH,
        ECCDH,
        ECMQV,
        ECCDHU,
        ECDDSA;

    }
}

