/*
 * Decompiled with CFR 0.152.
 */
package com.subgraph.orchid.circuits;

import com.subgraph.orchid.Cell;
import com.subgraph.orchid.CircuitNode;
import com.subgraph.orchid.Connection;
import com.subgraph.orchid.ConnectionIOException;
import com.subgraph.orchid.RelayCell;
import com.subgraph.orchid.Stream;
import com.subgraph.orchid.Threading;
import com.subgraph.orchid.TorException;
import com.subgraph.orchid.circuits.CircuitImpl;
import com.subgraph.orchid.circuits.CircuitStatus;
import com.subgraph.orchid.circuits.StreamImpl;
import com.subgraph.orchid.circuits.cells.CellImpl;
import com.subgraph.orchid.circuits.cells.RelayCellImpl;
import com.subgraph.orchid.dashboard.DashboardRenderable;
import com.subgraph.orchid.dashboard.DashboardRenderer;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public class CircuitIO
implements DashboardRenderable {
    private static final Logger logger = Logger.getLogger(CircuitIO.class.getName());
    private static final long CIRCUIT_BUILD_TIMEOUT_MS = 30000L;
    private static final long CIRCUIT_RELAY_RESPONSE_TIMEOUT = 20000L;
    private final CircuitImpl circuit;
    private final Connection connection;
    private final int circuitId;
    private final BlockingQueue<RelayCell> relayCellResponseQueue;
    private final BlockingQueue<Cell> controlCellResponseQueue;
    private final Map<Integer, StreamImpl> streamMap;
    private final ReentrantLock streamLock = Threading.lock("stream");
    private final ReentrantLock relaySendLock = Threading.lock("relaySend");
    private boolean isMarkedForClose;
    private boolean isClosed;

    CircuitIO(CircuitImpl circuit, Connection connection, int circuitId) {
        this.circuit = circuit;
        this.connection = connection;
        this.circuitId = circuitId;
        this.relayCellResponseQueue = new LinkedBlockingQueue<RelayCell>();
        this.controlCellResponseQueue = new LinkedBlockingQueue<Cell>();
        this.streamMap = new HashMap<Integer, StreamImpl>();
    }

    Connection getConnection() {
        return this.connection;
    }

    int getCircuitId() {
        return this.circuitId;
    }

    RelayCell dequeueRelayResponseCell() {
        try {
            long timeout = this.getReceiveTimeout();
            return this.relayCellResponseQueue.poll(timeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return null;
        }
    }

    private RelayCell decryptRelayCell(Cell cell) {
        for (CircuitNode node : this.circuit.getNodeList()) {
            if (!node.decryptBackwardCell(cell)) continue;
            return RelayCellImpl.createFromCell(node, cell);
        }
        this.destroyCircuit();
        throw new TorException("Could not decrypt relay cell");
    }

    Cell receiveControlCellResponse() {
        try {
            long timeout = this.getReceiveTimeout();
            return this.controlCellResponseQueue.poll(timeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return null;
        }
    }

    private long getReceiveTimeout() {
        if (this.circuit.getStatus().isBuilding()) {
            return this.remainingBuildTime();
        }
        return 20000L;
    }

    private long remainingBuildTime() {
        long elapsed = this.circuit.getStatus().getMillisecondsElapsedSinceCreated();
        if (elapsed == 0L || elapsed >= 30000L) {
            return 0L;
        }
        return 30000L - elapsed;
    }

    void deliverControlCell(Cell cell) {
        if (cell.getCommand() == 4) {
            this.processDestroyCell(cell.getByte());
        } else {
            this.controlCellResponseQueue.add(cell);
        }
    }

    private void processDestroyCell(int reason) {
        logger.fine("DESTROY cell received (" + CellImpl.errorToDescription(reason) + ") on " + this.circuit);
        this.destroyCircuit();
    }

    void deliverRelayCell(Cell cell) {
        this.circuit.getStatus().updateDirtyTimestamp();
        RelayCell relayCell = this.decryptRelayCell(cell);
        this.logRelayCell("Dispatching: ", relayCell);
        switch (relayCell.getRelayCommand()) {
            case 7: 
            case 9: 
            case 12: 
            case 15: 
            case 37: 
            case 39: 
            case 40: {
                this.relayCellResponseQueue.add(relayCell);
                break;
            }
            case 2: 
            case 3: 
            case 4: {
                this.processRelayDataCell(relayCell);
                break;
            }
            case 5: {
                if (relayCell.getStreamId() != 0) {
                    this.processRelayDataCell(relayCell);
                    break;
                }
                this.processCircuitSendme(relayCell);
                break;
            }
            case 1: 
            case 6: 
            case 8: 
            case 11: 
            case 13: {
                this.destroyCircuit();
                throw new TorException("Unexpected 'forward' direction relay cell type: " + relayCell.getRelayCommand());
            }
        }
    }

    private void processRelayDataCell(RelayCell cell) {
        if (cell.getRelayCommand() == 2) {
            cell.getCircuitNode().decrementDeliverWindow();
            if (cell.getCircuitNode().considerSendingSendme()) {
                RelayCell sendme = this.createRelayCell(5, 0, cell.getCircuitNode());
                this.sendRelayCellTo(sendme, sendme.getCircuitNode());
            }
        }
        this.streamLock.lock();
        try {
            StreamImpl stream = this.streamMap.get(cell.getStreamId());
            if (stream != null) {
                stream.addInputCell(cell);
            }
        }
        finally {
            this.streamLock.unlock();
        }
    }

    RelayCell createRelayCell(int relayCommand, int streamId, CircuitNode targetNode) {
        return new RelayCellImpl(targetNode, this.circuitId, streamId, relayCommand);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void sendRelayCellTo(RelayCell cell, CircuitNode targetNode) {
        this.relaySendLock.lock();
        try {
            this.logRelayCell("Sending:     ", cell);
            cell.setLength();
            targetNode.updateForwardDigest(cell);
            cell.setDigest(targetNode.getForwardDigestBytes());
            for (CircuitNode node = targetNode; node != null; node = node.getPreviousNode()) {
                node.encryptForwardCell(cell);
            }
            if (cell.getRelayCommand() == 2) {
                targetNode.waitForSendWindowAndDecrement();
            }
            this.sendCell(cell);
        }
        finally {
            this.relaySendLock.unlock();
        }
    }

    private void logRelayCell(String message, RelayCell cell) {
        Level level = this.getLogLevelForCell(cell);
        if (!logger.isLoggable(level)) {
            return;
        }
        logger.log(level, message + cell);
    }

    private Level getLogLevelForCell(RelayCell cell) {
        switch (cell.getRelayCommand()) {
            case 2: 
            case 5: {
                return Level.FINEST;
            }
        }
        return Level.FINER;
    }

    void sendCell(Cell cell) {
        CircuitStatus status = this.circuit.getStatus();
        if (!status.isConnected() && !status.isBuilding()) {
            return;
        }
        try {
            status.updateDirtyTimestamp();
            this.connection.sendCell(cell);
        }
        catch (ConnectionIOException e) {
            this.destroyCircuit();
        }
    }

    void markForClose() {
        boolean shouldClose;
        this.streamLock.lock();
        try {
            if (this.isMarkedForClose) {
                return;
            }
            this.isMarkedForClose = true;
            shouldClose = this.streamMap.isEmpty();
        }
        finally {
            this.streamLock.unlock();
        }
        if (shouldClose) {
            this.closeCircuit();
        }
    }

    boolean isMarkedForClose() {
        this.streamLock.lock();
        try {
            boolean bl = this.isMarkedForClose;
            return bl;
        }
        finally {
            this.streamLock.unlock();
        }
    }

    private void closeCircuit() {
        logger.fine("Closing circuit " + this.circuit);
        this.sendDestroyCell(0);
        this.connection.removeCircuit(this.circuit);
        this.circuit.setStateDestroyed();
        this.isClosed = true;
    }

    void sendDestroyCell(int reason) {
        CellImpl destroy = CellImpl.createCell(this.circuitId, 4);
        destroy.putByte(reason);
        try {
            this.connection.sendCell(destroy);
        }
        catch (ConnectionIOException e) {
            logger.warning("Connection IO error sending DESTROY cell: " + e.getMessage());
        }
    }

    private void processCircuitSendme(RelayCell cell) {
        cell.getCircuitNode().incrementSendWindow();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void destroyCircuit() {
        this.streamLock.lock();
        try {
            if (this.isClosed) {
                return;
            }
            this.circuit.setStateDestroyed();
            this.connection.removeCircuit(this.circuit);
            ArrayList<StreamImpl> tmpList = new ArrayList<StreamImpl>(this.streamMap.values());
            for (StreamImpl s : tmpList) {
                s.close();
            }
            this.isClosed = true;
        }
        finally {
            this.streamLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    StreamImpl createNewStream(boolean autoclose) {
        this.streamLock.lock();
        try {
            int streamId = this.circuit.getStatus().nextStreamId();
            StreamImpl stream = new StreamImpl(this.circuit, this.circuit.getFinalCircuitNode(), streamId, autoclose);
            this.streamMap.put(streamId, stream);
            StreamImpl streamImpl = stream;
            return streamImpl;
        }
        finally {
            this.streamLock.unlock();
        }
    }

    void removeStream(StreamImpl stream) {
        boolean shouldClose;
        this.streamLock.lock();
        try {
            this.streamMap.remove(stream.getStreamId());
            shouldClose = this.streamMap.isEmpty() && this.isMarkedForClose;
        }
        finally {
            this.streamLock.unlock();
        }
        if (shouldClose) {
            this.closeCircuit();
        }
    }

    List<Stream> getActiveStreams() {
        this.streamLock.lock();
        try {
            ArrayList<Stream> arrayList = new ArrayList<Stream>(this.streamMap.values());
            return arrayList;
        }
        finally {
            this.streamLock.unlock();
        }
    }

    @Override
    public void dashboardRender(DashboardRenderer renderer, PrintWriter writer, int flags) throws IOException {
        if ((flags & 0x10) == 0) {
            return;
        }
        for (Stream s : this.getActiveStreams()) {
            renderer.renderComponent(writer, flags, s);
        }
    }
}

