/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tika.language.translate.impl;

import jakarta.websocket.ClientEndpoint;
import jakarta.websocket.ContainerProvider;
import jakarta.websocket.DeploymentException;
import jakarta.websocket.OnClose;
import jakarta.websocket.OnMessage;
import jakarta.websocket.OnOpen;
import jakarta.websocket.Session;
import jakarta.websocket.WebSocketContainer;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import org.apache.commons.io.FileUtils;
import org.apache.tika.exception.TikaException;
import org.apache.tika.language.translate.impl.AbstractTranslator;
import org.apache.tika.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MarianTranslator
extends AbstractTranslator {
    private static final Logger LOG = LoggerFactory.getLogger(MarianTranslator.class);
    private static final String DEFAULT_PATH = "dummy-path";
    private static final String INPUT_TMP_NAME = "tika.marian.input";
    private static final String OUTPUT_TMP_NAME = "tika.marian.translation";
    private final String marianPath;
    private final Properties config = new Properties();
    private long maxWaitForMarianServerResponse = 120000L;
    private long pulseCheckMarianServerResponse = 1000L;

    public MarianTranslator() {
        try {
            this.config.load(MarianTranslator.class.getResourceAsStream("translator.marian.properties"));
            this.marianPath = this.config.getProperty("translator.marian.path", DEFAULT_PATH);
            if (this.config.containsKey("translator.marian.server.responseTimeout")) {
                this.maxWaitForMarianServerResponse = Long.parseLong(this.config.getProperty("translator.marian.server.responseTimeout"));
            }
            if (this.config.containsKey("translator.marian.server.responsePulse")) {
                this.pulseCheckMarianServerResponse = Long.parseLong(this.config.getProperty("translator.marian.server.responsePulse"));
            }
        }
        catch (IOException e) {
            throw new AssertionError((Object)"Failed to read translator.marian.properties.");
        }
    }

    public String translate(String text, String targetLanguage) throws TikaException, IOException {
        String sourceLanguage = this.detectLanguage(text).getLanguage();
        return this.translate(text, sourceLanguage, targetLanguage);
    }

    public String translate(String text, String sourceLanguage, String targetLanguage) throws TikaException, IOException {
        String configPath = this.config.getProperty("translator.marian." + sourceLanguage + "_" + targetLanguage + ".config");
        String serverSocket = this.config.getProperty("translator.marian." + sourceLanguage + "_" + targetLanguage + ".server");
        if (!this.isAvailable(sourceLanguage, targetLanguage)) {
            return text;
        }
        if (!StringUtils.isEmpty((CharSequence)configPath) && !StringUtils.isEmpty((CharSequence)serverSocket)) {
            LOG.info("Both local and server configurations exist for " + sourceLanguage + " to " + targetLanguage + "\nDefaulting to use local engine.");
        }
        StringBuilder translation = new StringBuilder();
        File tmpFile = Files.createTempFile(INPUT_TMP_NAME, ".tmp", new FileAttribute[0]).toFile();
        tmpFile.deleteOnExit();
        try (OutputStreamWriter out = new OutputStreamWriter((OutputStream)new FileOutputStream(tmpFile), Charset.defaultCharset());){
            out.append(text).append('\n').close();
        }
        File tmpTranslatedFile = Files.createTempFile(OUTPUT_TMP_NAME, ".tmp", new FileAttribute[0]).toFile();
        tmpTranslatedFile.deleteOnExit();
        try {
            String preProcessScript = this.config.getProperty("translator.marian.preprocess");
            this.executeScript(preProcessScript, tmpFile);
            if (!StringUtils.isEmpty((CharSequence)configPath)) {
                this.processWithLocalMarian(configPath, tmpFile, tmpTranslatedFile);
            } else {
                this.processWithMarianServer(serverSocket, tmpFile, tmpTranslatedFile);
            }
            String postProcessScript = this.config.getProperty("translator.marian.postprocess");
            this.executeScript(postProcessScript, tmpTranslatedFile);
            try (BufferedReader fileReader = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(tmpTranslatedFile), Charset.defaultCharset()));){
                fileReader.lines().forEach(translation::append);
            }
        }
        catch (InterruptedException e) {
            throw new TikaException("Failed perform translation", (Throwable)e);
        }
        if (!tmpFile.delete() || !tmpTranslatedFile.delete()) {
            throw new IOException("Failed to delete temporary files.");
        }
        return translation.toString();
    }

    private void processWithLocalMarian(String configPath, File sourceFile, File targetFile) throws IOException, InterruptedException {
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        String device = this.config.getProperty("translator.marian.device", "cpu");
        builder.command(this.buildMarianCommand(configPath, sourceFile, targetFile, device));
        builder.directory(new File(configPath).getParentFile());
        builder.redirectErrorStream(true);
        Process process = builder.start();
        try (BufferedReader stdOutReader = new BufferedReader(new InputStreamReader(process.getInputStream(), Charset.defaultCharset()));){
            stdOutReader.lines().forEach(arg_0 -> ((Logger)LOG).debug(arg_0));
        }
        process.waitFor();
    }

    private void processWithMarianServer(String serverURI, File sourceFile, File targetFile) throws TikaException {
        try {
            MarianServerClient clientEndpoint = new MarianServerClient(new URI(serverURI), targetFile);
            clientEndpoint.translate(FileUtils.readFileToString((File)sourceFile, (Charset)Charset.defaultCharset()));
            long start = System.currentTimeMillis();
            long elapsed = System.currentTimeMillis() - start;
            while (!clientEndpoint.receivedResponse && elapsed < this.maxWaitForMarianServerResponse) {
                try {
                    Thread.sleep(this.pulseCheckMarianServerResponse);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                elapsed = System.currentTimeMillis() - start;
            }
            clientEndpoint.close();
        }
        catch (IOException | URISyntaxException e) {
            throw new TikaException("Failed perform translation", (Throwable)e);
        }
    }

    private void executeScript(String script, File file) throws IOException, InterruptedException {
        if (!StringUtils.isEmpty((CharSequence)script) && !script.equals("no-script")) {
            Path scriptPath = Paths.get(script, new String[0]);
            if (!Files.exists(scriptPath, new LinkOption[0]) || !Files.isExecutable(scriptPath)) {
                throw new IOException("Cannot execute configured script at " + String.valueOf(scriptPath));
            }
            ProcessBuilder postProcess = new ProcessBuilder(new String[0]);
            postProcess.command(script, file.getAbsolutePath());
            postProcess.directory(new File(script).getParentFile());
            postProcess.redirectErrorStream(true);
            Process processProc = postProcess.start();
            processProc.waitFor();
        }
    }

    private List<String> buildMarianCommand(String configPath, File input, File output, String device) {
        ArrayList<String> command = new ArrayList<String>();
        command.add(Paths.get(this.marianPath, new String[0]).toString());
        command.add("-c");
        command.add(configPath);
        command.add("-i");
        command.add(input.getAbsolutePath());
        command.add("-o");
        command.add(output.getAbsolutePath());
        if (device.equalsIgnoreCase("cpu")) {
            command.add("--cpu-threads");
            command.add(this.config.getProperty("translator.marian.device.cpuThreads", "1"));
        }
        return command;
    }

    public boolean isAvailable() {
        return !this.marianPath.equals(DEFAULT_PATH);
    }

    public boolean isAvailable(String sourceLanguage, String targetLanguage) {
        String configPath = this.config.getProperty("translator.marian." + sourceLanguage + "_" + targetLanguage + ".config");
        String serverSocket = this.config.getProperty("translator.marian." + sourceLanguage + "_" + targetLanguage + ".server");
        return !this.marianPath.equals(DEFAULT_PATH) && !StringUtils.isEmpty((CharSequence)configPath) || !StringUtils.isEmpty((CharSequence)serverSocket);
    }

    @ClientEndpoint
    public static class MarianServerClient {
        Session session;
        File translationResult;
        volatile boolean receivedResponse = false;

        public MarianServerClient(URI endpointURI, File file) throws TikaException {
            try {
                WebSocketContainer container = ContainerProvider.getWebSocketContainer();
                container.connectToServer((Object)this, endpointURI);
                this.translationResult = file;
            }
            catch (DeploymentException | IOException e) {
                throw new TikaException("Failed to create connection to Marian NMT Server", e);
            }
        }

        @OnOpen
        public void onOpen(Session session) {
            LOG.debug("Opened connection, Session ID: " + session.getId());
            this.session = session;
        }

        @OnMessage
        public void processMessage(String message) throws IOException {
            LOG.debug("Message received: " + message);
            FileUtils.writeStringToFile((File)this.translationResult, (String)message, (Charset)Charset.defaultCharset());
            this.receivedResponse = true;
        }

        @OnClose
        public void onClose(Session session) {
            LOG.debug("Closed connection, Session ID: " + session.getId());
            this.receivedResponse = true;
        }

        public void translate(String sourceText) throws IOException {
            this.session.getBasicRemote().sendText(sourceText);
        }

        public void close() throws IOException {
            this.session.close();
        }
    }
}

