/*
 * Decompiled with CFR 0.152.
 */
package org.apache.seatunnel.format.csv;

import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQueries;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import lombok.NonNull;
import org.apache.commons.lang3.StringUtils;
import org.apache.seatunnel.api.table.catalog.CatalogTable;
import org.apache.seatunnel.api.table.catalog.TablePath;
import org.apache.seatunnel.api.table.type.ArrayType;
import org.apache.seatunnel.api.table.type.MapType;
import org.apache.seatunnel.api.table.type.SeaTunnelDataType;
import org.apache.seatunnel.api.table.type.SeaTunnelRow;
import org.apache.seatunnel.api.table.type.SeaTunnelRowType;
import org.apache.seatunnel.common.exception.CommonError;
import org.apache.seatunnel.common.exception.CommonErrorCode;
import org.apache.seatunnel.common.exception.SeaTunnelErrorCode;
import org.apache.seatunnel.common.utils.DateTimeUtils;
import org.apache.seatunnel.common.utils.DateUtils;
import org.apache.seatunnel.common.utils.EncodingUtils;
import org.apache.seatunnel.common.utils.TimeUtils;
import org.apache.seatunnel.format.csv.constant.CsvFormatConstant;
import org.apache.seatunnel.format.csv.exception.SeaTunnelCsvFormatException;
import org.apache.seatunnel.format.csv.processor.CsvLineProcessor;
import org.apache.seatunnel.format.csv.processor.DefaultCsvLineProcessor;

public class CsvDeserializationSchema
implements Serializable {
    private final SeaTunnelRowType seaTunnelRowType;
    private final String[] separators;
    private final String encoding;
    private final String nullFormat;
    private final CsvLineProcessor processor;
    private final CatalogTable catalogTable;
    public static final DateTimeFormatter TIME_FORMAT = new DateTimeFormatterBuilder().appendPattern("HH:mm:ss").appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).toFormatter();
    public Map<String, DateTimeFormatter> fieldFormatterMap = new HashMap<String, DateTimeFormatter>();

    private CsvDeserializationSchema(@NonNull SeaTunnelRowType seaTunnelRowType, String[] separators, String encoding, String nullFormat, CsvLineProcessor processor, CatalogTable catalogTable) {
        if (seaTunnelRowType == null) {
            throw new NullPointerException("seaTunnelRowType is marked non-null but is null");
        }
        this.seaTunnelRowType = seaTunnelRowType;
        this.separators = separators;
        this.encoding = encoding;
        this.nullFormat = nullFormat;
        this.processor = processor;
        this.catalogTable = catalogTable;
    }

    public static Builder builder() {
        return new Builder();
    }

    protected SeaTunnelRow deserialize(byte[] message) throws IOException {
        if (message == null || message.length == 0) {
            return null;
        }
        String content = new String(message, EncodingUtils.tryParseCharset((String)this.encoding));
        Map<Integer, String> splitsMap = this.splitLineBySeaTunnelRowType(content, this.seaTunnelRowType, 0);
        SeaTunnelRow seaTunnelRow = this.getSeaTunnelRow(splitsMap);
        return seaTunnelRow;
    }

    public SeaTunnelRow getSeaTunnelRow(Map<Integer, String> splitsMap) {
        Object[] objects = new Object[this.seaTunnelRowType.getTotalFields()];
        for (int i = 0; i < objects.length; ++i) {
            String fieldValue = splitsMap.get(i);
            if (StringUtils.isBlank(fieldValue) || StringUtils.equals(fieldValue, this.nullFormat)) continue;
            objects[i] = this.convert(fieldValue, this.seaTunnelRowType.getFieldType(i), 0, this.seaTunnelRowType.getFieldNames()[i]);
        }
        SeaTunnelRow seaTunnelRow = new SeaTunnelRow(objects);
        Optional<TablePath> tablePath = Optional.ofNullable(this.catalogTable).map(CatalogTable::getTablePath);
        if (tablePath.isPresent()) {
            seaTunnelRow.setTableId(tablePath.toString());
        }
        return seaTunnelRow;
    }

    public SeaTunnelDataType<SeaTunnelRow> getProducedType() {
        return this.seaTunnelRowType;
    }

    protected Map<Integer, String> splitLineBySeaTunnelRowType(String line, SeaTunnelRowType seaTunnelRowType, int level) {
        int i;
        String[] splits = this.processor.splitLine(line, this.separators[level]);
        LinkedHashMap<Integer, String> splitsMap = new LinkedHashMap<Integer, String>();
        SeaTunnelDataType[] fieldTypes = seaTunnelRowType.getFieldTypes();
        for (i = 0; i < splits.length; ++i) {
            splitsMap.put(i, splits[i]);
        }
        if (fieldTypes.length > splits.length) {
            for (i = splits.length; i < fieldTypes.length; ++i) {
                splitsMap.put(i, null);
            }
        }
        return splitsMap;
    }

    private Object convert(String field, SeaTunnelDataType<?> fieldType, int level, String fieldName) {
        if (StringUtils.isBlank(field)) {
            return null;
        }
        switch (fieldType.getSqlType()) {
            case ARRAY: {
                SeaTunnelDataType elementType = ((ArrayType)fieldType).getElementType();
                String[] elements = field.split(this.separators[level + 1]);
                ArrayList<Object> objectArrayList = new ArrayList<Object>();
                for (String element : elements) {
                    objectArrayList.add(this.convert(element, elementType, level + 1, fieldName));
                }
                switch (elementType.getSqlType()) {
                    case STRING: {
                        return objectArrayList.toArray(new String[0]);
                    }
                    case BOOLEAN: {
                        return objectArrayList.toArray(new Boolean[0]);
                    }
                    case TINYINT: {
                        return objectArrayList.toArray(new Byte[0]);
                    }
                    case SMALLINT: {
                        return objectArrayList.toArray(new Short[0]);
                    }
                    case INT: {
                        return objectArrayList.toArray(new Integer[0]);
                    }
                    case BIGINT: {
                        return objectArrayList.toArray(new Long[0]);
                    }
                    case FLOAT: {
                        return objectArrayList.toArray(new Float[0]);
                    }
                    case DOUBLE: {
                        return objectArrayList.toArray(new Double[0]);
                    }
                    case DECIMAL: {
                        return objectArrayList.toArray(new BigDecimal[0]);
                    }
                    case DATE: {
                        return objectArrayList.toArray(new LocalDate[0]);
                    }
                    case TIME: {
                        return objectArrayList.toArray(new LocalTime[0]);
                    }
                    case TIMESTAMP: {
                        return objectArrayList.toArray(new LocalDateTime[0]);
                    }
                }
                throw new SeaTunnelCsvFormatException((SeaTunnelErrorCode)CommonErrorCode.UNSUPPORTED_DATA_TYPE, String.format("SeaTunnel array not support this data type [%s]", elementType.getSqlType()));
            }
            case MAP: {
                String[] kvs;
                SeaTunnelDataType keyType = ((MapType)fieldType).getKeyType();
                SeaTunnelDataType valueType = ((MapType)fieldType).getValueType();
                LinkedHashMap<Object, Object> objectMap = new LinkedHashMap<Object, Object>();
                for (String kv : kvs = field.split(this.separators[level + 1])) {
                    String[] splits = kv.split(this.separators[level + 2]);
                    if (splits.length < 2) {
                        objectMap.put(this.convert(splits[0], keyType, level + 1, fieldName), null);
                        continue;
                    }
                    objectMap.put(this.convert(splits[0], keyType, level + 1, fieldName), this.convert(splits[1], valueType, level + 1, fieldName));
                }
                return objectMap;
            }
            case STRING: {
                return field;
            }
            case BOOLEAN: {
                return Boolean.parseBoolean(field);
            }
            case TINYINT: {
                return Byte.parseByte(field);
            }
            case SMALLINT: {
                return Short.parseShort(field);
            }
            case INT: {
                return Integer.parseInt(field);
            }
            case BIGINT: {
                return Long.parseLong(field);
            }
            case FLOAT: {
                return Float.valueOf(Float.parseFloat(field));
            }
            case DOUBLE: {
                return Double.parseDouble(field);
            }
            case DECIMAL: {
                return new BigDecimal(field);
            }
            case NULL: {
                return null;
            }
            case BYTES: {
                return field.getBytes(StandardCharsets.UTF_8);
            }
            case DATE: {
                return this.parseDate(field, fieldName);
            }
            case TIME: {
                return this.parseTime(field);
            }
            case TIMESTAMP: {
                return this.parseTimestamp(field, fieldName);
            }
            case ROW: {
                Map<Integer, String> splitsMap = this.splitLineBySeaTunnelRowType(field, (SeaTunnelRowType)fieldType, level + 1);
                Object[] objects = new Object[splitsMap.size()];
                String[] eleFieldNames = ((SeaTunnelRowType)fieldType).getFieldNames();
                for (int i = 0; i < objects.length; ++i) {
                    objects[i] = this.convert(splitsMap.get(i), ((SeaTunnelRowType)fieldType).getFieldType(i), level + 1, fieldName + "." + eleFieldNames[i]);
                }
                return new SeaTunnelRow(objects);
            }
        }
        throw CommonError.unsupportedDataType((String)"SeaTunnel", (String)fieldType.getSqlType().toString(), (String)fieldName);
    }

    private LocalDate parseDate(String field, String fieldName) {
        DateTimeFormatter dateFormatter = this.fieldFormatterMap.get(fieldName);
        if (dateFormatter == null) {
            dateFormatter = DateUtils.matchDateFormatter((String)field);
            this.fieldFormatterMap.put(fieldName, dateFormatter);
        }
        if (dateFormatter == null) {
            throw CommonError.formatDateError((String)field, (String)fieldName);
        }
        return dateFormatter.parse(field).query(TemporalQueries.localDate());
    }

    private LocalTime parseTime(String field) {
        try {
            TemporalAccessor parsedTime = TIME_FORMAT.parse(field);
            return parsedTime.query(TemporalQueries.localTime());
        }
        catch (DateTimeParseException e) {
            throw new SeaTunnelCsvFormatException((SeaTunnelErrorCode)CommonErrorCode.UNSUPPORTED_DATA_TYPE, "Invalid time format: " + field, e);
        }
    }

    private LocalDateTime parseTimestamp(String field, String fieldName) {
        DateTimeFormatter dateTimeFormatter = this.fieldFormatterMap.computeIfAbsent(fieldName, f -> DateTimeUtils.matchDateTimeFormatter((String)field));
        if (dateTimeFormatter == null) {
            throw new SeaTunnelCsvFormatException((SeaTunnelErrorCode)CommonErrorCode.UNSUPPORTED_DATA_TYPE, String.format("SeaTunnel can not parse this date format [%s] of field [%s]", field, fieldName));
        }
        TemporalAccessor parsedTimestamp = dateTimeFormatter.parse(field);
        return LocalDateTime.of(parsedTimestamp.query(TemporalQueries.localDate()), parsedTimestamp.query(TemporalQueries.localTime()));
    }

    public static class Builder {
        private SeaTunnelRowType seaTunnelRowType;
        private CatalogTable catalogTable;
        private String[] separators = (String[])CsvFormatConstant.SEPARATOR.clone();
        private DateUtils.Formatter dateFormatter = DateUtils.Formatter.YYYY_MM_DD;
        private DateTimeUtils.Formatter dateTimeFormatter = DateTimeUtils.Formatter.YYYY_MM_DD_HH_MM_SS;
        private TimeUtils.Formatter timeFormatter = TimeUtils.Formatter.HH_MM_SS;
        private String encoding = StandardCharsets.UTF_8.name();
        private String nullFormat;
        private CsvLineProcessor csvLineProcessor = new DefaultCsvLineProcessor();

        private Builder() {
        }

        public Builder setCatalogTable(CatalogTable catalogTable) {
            this.catalogTable = catalogTable;
            return this;
        }

        public Builder seaTunnelRowType(SeaTunnelRowType seaTunnelRowType) {
            this.seaTunnelRowType = seaTunnelRowType;
            return this;
        }

        public Builder delimiter(String delimiter) {
            this.separators[0] = delimiter;
            return this;
        }

        public Builder separators(String[] separators) {
            this.separators = separators;
            return this;
        }

        public Builder dateFormatter(DateUtils.Formatter dateFormatter) {
            this.dateFormatter = dateFormatter;
            return this;
        }

        public Builder dateTimeFormatter(DateTimeUtils.Formatter dateTimeFormatter) {
            this.dateTimeFormatter = dateTimeFormatter;
            return this;
        }

        public Builder timeFormatter(TimeUtils.Formatter timeFormatter) {
            this.timeFormatter = timeFormatter;
            return this;
        }

        public Builder encoding(String encoding) {
            this.encoding = encoding;
            return this;
        }

        public Builder nullFormat(String nullFormat) {
            this.nullFormat = nullFormat;
            return this;
        }

        public Builder csvLineProcessor(CsvLineProcessor csvLineProcessor) {
            this.csvLineProcessor = csvLineProcessor;
            return this;
        }

        public CsvDeserializationSchema build() {
            return new CsvDeserializationSchema(this.seaTunnelRowType, this.separators, this.encoding, this.nullFormat, this.csvLineProcessor, this.catalogTable);
        }
    }
}

