/*
 * Decompiled with CFR 0.152.
 */
package org.apache.seatunnel.connectors.seatunnel.file.hadoop;

import java.io.Closeable;
import java.io.IOException;
import java.io.Serializable;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.List;
import lombok.NonNull;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.seatunnel.common.exception.CommonError;
import org.apache.seatunnel.common.exception.SeaTunnelRuntimeException;
import org.apache.seatunnel.connectors.seatunnel.file.config.HadoopConf;
import org.apache.seatunnel.connectors.seatunnel.file.hadoop.HadoopLoginFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HadoopFileSystemProxy
implements Serializable,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(HadoopFileSystemProxy.class);
    private transient UserGroupInformation userGroupInformation;
    private transient FileSystem fileSystem;
    private transient Configuration configuration;
    private final HadoopConf hadoopConf;
    private boolean isAuthTypeKerberos;

    public HadoopFileSystemProxy(@NonNull HadoopConf hadoopConf) {
        if (hadoopConf == null) {
            throw new NullPointerException("hadoopConf is marked non-null but is null");
        }
        this.hadoopConf = hadoopConf;
        this.initialize();
    }

    public boolean fileExist(@NonNull String filePath) throws IOException {
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        return this.execute(() -> this.getFileSystem().exists(new Path(filePath)));
    }

    public boolean isFile(@NonNull String filePath) throws IOException {
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        return this.execute(() -> this.getFileSystem().getFileStatus(new Path(filePath)).isFile());
    }

    public void createFile(@NonNull String filePath) throws IOException {
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        this.execute(() -> {
            if (!this.getFileSystem().createNewFile(new Path(filePath))) {
                throw CommonError.fileOperationFailed((String)"SeaTunnel", (String)"create", (String)filePath);
            }
            return Void.class;
        });
    }

    public void deleteFile(@NonNull String filePath) throws IOException {
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        this.execute(() -> {
            Path path = new Path(filePath);
            if (this.getFileSystem().exists(path) && !this.getFileSystem().delete(path, true)) {
                throw CommonError.fileOperationFailed((String)"SeaTunnel", (String)"delete", (String)filePath);
            }
            return Void.class;
        });
    }

    public void renameFile(@NonNull String oldFilePath, @NonNull String newFilePath, boolean removeWhenNewFilePathExist) throws IOException {
        if (oldFilePath == null) {
            throw new NullPointerException("oldFilePath is marked non-null but is null");
        }
        if (newFilePath == null) {
            throw new NullPointerException("newFilePath is marked non-null but is null");
        }
        this.execute(() -> {
            Path oldPath = new Path(oldFilePath);
            Path newPath = new Path(newFilePath);
            if (!this.fileExist(oldPath.toString())) {
                log.warn("rename file :[" + oldPath + "] to [" + newPath + "] already finished in the last commit, skip");
                return Void.class;
            }
            if (removeWhenNewFilePathExist && this.fileExist(newFilePath)) {
                this.getFileSystem().delete(newPath, true);
                log.info("Delete already file: {}", (Object)newPath);
            }
            if (!this.fileExist(newPath.getParent().toString())) {
                this.createDir(newPath.getParent().toString());
            }
            if (!this.getFileSystem().rename(oldPath, newPath)) {
                throw CommonError.fileOperationFailed((String)"SeaTunnel", (String)"rename", (String)(oldFilePath + " -> " + newFilePath));
            }
            log.info("rename file :[" + oldPath + "] to [" + newPath + "] finish");
            return Void.class;
        });
    }

    public void createDir(@NonNull String filePath) throws IOException {
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        this.execute(() -> {
            Path dfs = new Path(filePath);
            if (!this.getFileSystem().mkdirs(dfs)) {
                throw CommonError.fileOperationFailed((String)"SeaTunnel", (String)"create", (String)filePath);
            }
            return Void.class;
        });
    }

    public List<LocatedFileStatus> listFile(String path) throws IOException {
        return this.execute(() -> {
            ArrayList<LocatedFileStatus> fileList = new ArrayList<LocatedFileStatus>();
            if (!this.fileExist(path)) {
                return fileList;
            }
            Path fileName = new Path(path);
            RemoteIterator locatedFileStatusRemoteIterator = this.getFileSystem().listFiles(fileName, false);
            while (locatedFileStatusRemoteIterator.hasNext()) {
                fileList.add((LocatedFileStatus)locatedFileStatusRemoteIterator.next());
            }
            return fileList;
        });
    }

    public List<Path> getAllSubFiles(@NonNull String filePath) throws IOException {
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        return this.execute(() -> {
            ArrayList<Path> pathList = new ArrayList<Path>();
            if (!this.fileExist(filePath)) {
                return pathList;
            }
            Path fileName = new Path(filePath);
            FileStatus[] status = this.getFileSystem().listStatus(fileName);
            if (status != null) {
                for (FileStatus fileStatus : status) {
                    if (!fileStatus.isDirectory()) continue;
                    pathList.add(fileStatus.getPath());
                }
            }
            return pathList;
        });
    }

    public FileStatus[] listStatus(String filePath) throws IOException {
        return this.execute(() -> this.getFileSystem().listStatus(new Path(filePath)));
    }

    public FileStatus getFileStatus(String filePath) throws IOException {
        return this.execute(() -> this.getFileSystem().getFileStatus(new Path(filePath)));
    }

    public FSDataOutputStream getOutputStream(String filePath) throws IOException {
        return this.execute(() -> this.getFileSystem().create(new Path(filePath), true));
    }

    public FSDataInputStream getInputStream(String filePath) throws IOException {
        return this.execute(() -> this.getFileSystem().open(new Path(filePath)));
    }

    public FileSystem getFileSystem() {
        if (this.fileSystem == null) {
            this.initialize();
        }
        return this.fileSystem;
    }

    public <T> T doWithHadoopAuth(HadoopLoginFactory.LoginFunction<T> loginFunction) {
        if (this.configuration == null) {
            this.configuration = this.createConfiguration();
        }
        if (this.enableKerberos()) {
            this.configuration.set("hadoop.security.authentication", "kerberos");
            return HadoopLoginFactory.loginWithKerberos(this.configuration, this.hadoopConf.getKrb5Path(), this.hadoopConf.getKerberosPrincipal(), this.hadoopConf.getKerberosKeytabPath(), loginFunction);
        }
        if (this.enableRemoteUser()) {
            return HadoopLoginFactory.loginWithRemoteUser(this.configuration, this.hadoopConf.getRemoteUser(), loginFunction);
        }
        return loginFunction.run(this.configuration, UserGroupInformation.getCurrentUser());
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.userGroupInformation != null && this.isAuthTypeKerberos) {
                this.userGroupInformation.logoutUserFromKeytab();
            }
        }
        finally {
            if (this.fileSystem != null) {
                this.fileSystem.close();
            }
        }
    }

    private void initialize() {
        this.configuration = this.createConfiguration();
        if (this.enableKerberos()) {
            this.configuration.set("hadoop.security.authentication", "kerberos");
            this.initializeWithKerberosLogin();
            this.isAuthTypeKerberos = true;
            return;
        }
        if (this.enableRemoteUser()) {
            this.initializeWithRemoteUserLogin();
            this.isAuthTypeKerberos = true;
            return;
        }
        this.fileSystem = FileSystem.get((Configuration)this.configuration);
        this.fileSystem.setWriteChecksum(false);
        this.isAuthTypeKerberos = false;
    }

    private Configuration createConfiguration() {
        Configuration configuration = this.hadoopConf.toConfiguration();
        this.hadoopConf.setExtraOptionsForConfiguration(configuration);
        return configuration;
    }

    private boolean enableKerberos() {
        boolean kerberosPrincipalEmpty = StringUtils.isBlank(this.hadoopConf.getKerberosPrincipal());
        boolean kerberosKeytabPathEmpty = StringUtils.isBlank(this.hadoopConf.getKerberosKeytabPath());
        if (kerberosKeytabPathEmpty && kerberosPrincipalEmpty) {
            return false;
        }
        if (!kerberosPrincipalEmpty && !kerberosKeytabPathEmpty) {
            return true;
        }
        if (kerberosPrincipalEmpty) {
            throw new IllegalArgumentException("Please set kerberosPrincipal");
        }
        throw new IllegalArgumentException("Please set kerberosKeytabPath");
    }

    private void initializeWithKerberosLogin() throws IOException, InterruptedException {
        Pair pair = HadoopLoginFactory.loginWithKerberos(this.configuration, this.hadoopConf.getKrb5Path(), this.hadoopConf.getKerberosPrincipal(), this.hadoopConf.getKerberosKeytabPath(), (configuration, userGroupInformation) -> {
            this.userGroupInformation = userGroupInformation;
            this.fileSystem = FileSystem.get((Configuration)configuration);
            return Pair.of(userGroupInformation, this.fileSystem);
        });
        this.userGroupInformation = (UserGroupInformation)pair.getKey();
        this.fileSystem = (FileSystem)pair.getValue();
        this.fileSystem.setWriteChecksum(false);
        log.info("Create FileSystem success with Kerberos: {}.", (Object)this.hadoopConf.getKerberosPrincipal());
    }

    private boolean enableRemoteUser() {
        return StringUtils.isNotBlank(this.hadoopConf.getRemoteUser());
    }

    private void initializeWithRemoteUserLogin() throws Exception {
        Pair pair = HadoopLoginFactory.loginWithRemoteUser(this.configuration, this.hadoopConf.getRemoteUser(), (configuration, userGroupInformation) -> {
            this.userGroupInformation = userGroupInformation;
            this.fileSystem = FileSystem.get((Configuration)configuration);
            return Pair.of(userGroupInformation, this.fileSystem);
        });
        log.info("Create FileSystem success with RemoteUser: {}.", (Object)this.hadoopConf.getRemoteUser());
        this.userGroupInformation = (UserGroupInformation)pair.getKey();
        this.fileSystem = (FileSystem)pair.getValue();
        this.fileSystem.setWriteChecksum(false);
    }

    private <T> T execute(PrivilegedExceptionAction<T> action) throws IOException {
        if (this.isAuthTypeKerberos) {
            return this.doAsPrivileged(action);
        }
        try {
            return action.run();
        }
        catch (IOException | SeaTunnelRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private <T> T doAsPrivileged(PrivilegedExceptionAction<T> action) throws IOException {
        if (this.fileSystem == null || this.userGroupInformation == null) {
            this.initialize();
        }
        try {
            return (T)this.userGroupInformation.doAs(action);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(e);
        }
    }
}

