/*
 * Decompiled with CFR 0.152.
 */
package org.tikv.common.catalog;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tikv.common.Snapshot;
import org.tikv.common.catalog.CatalogTransaction;
import org.tikv.common.meta.TiDBInfo;
import org.tikv.common.meta.TiTableInfo;
import org.tikv.shade.com.google.common.annotations.VisibleForTesting;
import org.tikv.shade.com.google.common.collect.ImmutableList;
import org.tikv.shade.com.google.common.collect.ImmutableMap;

public class Catalog
implements AutoCloseable {
    private final boolean showRowId;
    private final String dbPrefix;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Supplier<Snapshot> snapshotProvider;
    private CatalogCache metaCache;

    public Catalog(Supplier<Snapshot> snapshotProvider, boolean showRowId, String dbPrefix) {
        this.snapshotProvider = Objects.requireNonNull(snapshotProvider, "Snapshot Provider is null");
        this.showRowId = showRowId;
        this.dbPrefix = dbPrefix;
        this.metaCache = new CatalogCache(new CatalogTransaction(snapshotProvider.get()), dbPrefix, false);
    }

    @Override
    public void close() {
    }

    private synchronized void reloadCache(boolean loadTables) {
        Snapshot snapshot = this.snapshotProvider.get();
        CatalogTransaction newTrx = new CatalogTransaction(snapshot);
        long latestVersion = newTrx.getLatestSchemaVersion();
        if (latestVersion > this.metaCache.getVersion()) {
            this.metaCache = new CatalogCache(newTrx, this.dbPrefix, loadTables);
        }
    }

    private void reloadCache() {
        this.reloadCache(false);
    }

    public List<TiDBInfo> listDatabases() {
        this.reloadCache();
        return this.metaCache.listDatabases();
    }

    public List<TiTableInfo> listTables(TiDBInfo database) {
        Objects.requireNonNull(database, "database is null");
        this.reloadCache(true);
        if (this.showRowId) {
            return this.metaCache.listTables(database).stream().map(TiTableInfo::copyTableWithRowId).collect(Collectors.toList());
        }
        return this.metaCache.listTables(database);
    }

    public TiDBInfo getDatabase(String dbName) {
        Objects.requireNonNull(dbName, "dbName is null");
        this.reloadCache();
        return this.metaCache.getDatabase(dbName);
    }

    public TiTableInfo getTable(String dbName, String tableName) {
        TiDBInfo database = this.getDatabase(dbName);
        if (database == null) {
            return null;
        }
        return this.getTable(database, tableName);
    }

    public TiTableInfo getTable(TiDBInfo database, String tableName) {
        Objects.requireNonNull(database, "database is null");
        Objects.requireNonNull(tableName, "tableName is null");
        this.reloadCache(true);
        TiTableInfo table = this.metaCache.getTable(database, tableName);
        if (this.showRowId && table != null) {
            return table.copyTableWithRowId();
        }
        return table;
    }

    @VisibleForTesting
    public TiTableInfo getTable(TiDBInfo database, long tableId) {
        Objects.requireNonNull(database, "database is null");
        List<TiTableInfo> tables = this.listTables(database);
        for (TiTableInfo table : tables) {
            if (table.getId() != tableId) continue;
            if (this.showRowId) {
                return table.copyTableWithRowId();
            }
            return table;
        }
        return null;
    }

    private static class CatalogCache {
        private final Map<String, TiDBInfo> dbCache;
        private final ConcurrentHashMap<TiDBInfo, Map<String, TiTableInfo>> tableCache;
        private final String dbPrefix;
        private final CatalogTransaction transaction;
        private final long currentVersion;

        private CatalogCache(CatalogTransaction transaction, String dbPrefix, boolean loadTables) {
            this.transaction = transaction;
            this.dbPrefix = dbPrefix;
            this.tableCache = new ConcurrentHashMap();
            this.dbCache = this.loadDatabases(loadTables);
            this.currentVersion = transaction.getLatestSchemaVersion();
        }

        public CatalogTransaction getTransaction() {
            return this.transaction;
        }

        public long getVersion() {
            return this.currentVersion;
        }

        public TiDBInfo getDatabase(String name) {
            Objects.requireNonNull(name, "name is null");
            return this.dbCache.get(name.toLowerCase());
        }

        public List<TiDBInfo> listDatabases() {
            return ImmutableList.copyOf(this.dbCache.values());
        }

        public List<TiTableInfo> listTables(TiDBInfo db) {
            Map<String, TiTableInfo> tableMap = this.tableCache.get(db);
            if (tableMap == null) {
                tableMap = this.loadTables(db);
            }
            Collection<TiTableInfo> tables = tableMap.values();
            return tables.stream().filter(tbl -> !tbl.isView() || !tbl.isSequence()).collect(Collectors.toList());
        }

        public TiTableInfo getTable(TiDBInfo db, String tableName) {
            TiTableInfo tbl;
            Map<String, TiTableInfo> tableMap = this.tableCache.get(db);
            if (tableMap == null) {
                tableMap = this.loadTables(db);
            }
            if ((tbl = tableMap.get(tableName.toLowerCase())) != null && (tbl.isView() || tbl.isSequence())) {
                return null;
            }
            return tbl;
        }

        private Map<String, TiTableInfo> loadTables(TiDBInfo db) {
            List<TiTableInfo> tables = this.transaction.getTables(db.getId());
            ImmutableMap.Builder<String, TiTableInfo> builder = ImmutableMap.builder();
            for (TiTableInfo table : tables) {
                builder.put(table.getName().toLowerCase(), table);
            }
            ImmutableMap<String, TiTableInfo> tableMap = builder.build();
            this.tableCache.put(db, tableMap);
            return tableMap;
        }

        private Map<String, TiDBInfo> loadDatabases(boolean loadTables) {
            HashMap<String, TiDBInfo> newDBCache = new HashMap<String, TiDBInfo>();
            List<TiDBInfo> databases = this.transaction.getDatabases();
            databases.forEach(db -> {
                TiDBInfo newDBInfo = db.rename(this.dbPrefix + db.getName());
                newDBCache.put(newDBInfo.getName().toLowerCase(), newDBInfo);
                if (loadTables) {
                    this.loadTables(newDBInfo);
                }
            });
            return newDBCache;
        }
    }
}

