/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.kafka.connect.source.schema;

import com.mongodb.kafka.connect.util.BsonDocumentFieldLookup;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import org.apache.kafka.connect.data.Field;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaAndValue;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.data.Values;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.errors.DataException;
import org.bson.BsonBinaryWriter;
import org.bson.BsonDocument;
import org.bson.BsonNumber;
import org.bson.BsonValue;
import org.bson.RawBsonDocument;
import org.bson.codecs.BsonValueCodec;
import org.bson.codecs.Codec;
import org.bson.codecs.EncoderContext;
import org.bson.io.BasicOutputBuffer;
import org.bson.json.JsonWriterSettings;

public class BsonValueToSchemaAndValue {
    private static final Codec<BsonValue> BSON_VALUE_CODEC = new BsonValueCodec();
    private final JsonWriterSettings jsonWriterSettings;

    public BsonValueToSchemaAndValue(JsonWriterSettings jsonWriterSettings) {
        this.jsonWriterSettings = jsonWriterSettings;
    }

    public SchemaAndValue toSchemaAndValue(Schema schema, BsonValue bsonValue) {
        SchemaAndValue schemaAndValue;
        if (schema.isOptional() && bsonValue.isNull()) {
            return new SchemaAndValue(schema, null);
        }
        switch (schema.type()) {
            case INT8: 
            case INT16: 
            case INT32: 
            case INT64: 
            case FLOAT32: 
            case FLOAT64: {
                schemaAndValue = this.numberToSchemaAndValue(schema, bsonValue);
                break;
            }
            case BOOLEAN: {
                schemaAndValue = this.booleanToSchemaAndValue(schema, bsonValue);
                break;
            }
            case STRING: {
                schemaAndValue = this.stringToSchemaAndValue(schema, bsonValue);
                break;
            }
            case BYTES: {
                schemaAndValue = this.bytesToSchemaAndValue(schema, bsonValue);
                break;
            }
            case ARRAY: {
                schemaAndValue = this.arrayToSchemaAndValue(schema, bsonValue);
                break;
            }
            case MAP: {
                schemaAndValue = this.mapToSchemaAndValue(schema, bsonValue);
                break;
            }
            case STRUCT: {
                schemaAndValue = this.recordToSchemaAndValue(schema, bsonValue);
                break;
            }
            default: {
                throw this.unsupportedSchemaType(schema);
            }
        }
        return schemaAndValue;
    }

    public static byte[] documentToByteArray(BsonDocument document) {
        if (document instanceof RawBsonDocument) {
            RawBsonDocument rawBsonDocument = (RawBsonDocument)document;
            ByteBuffer byteBuffer = rawBsonDocument.getByteBuffer().asNIO();
            int startPosition = byteBuffer.position();
            int length = byteBuffer.limit() - startPosition;
            byte[] byteArray = new byte[length];
            System.arraycopy(byteBuffer.array(), startPosition, byteArray, 0, length);
            return byteArray;
        }
        BasicOutputBuffer buffer = new BasicOutputBuffer();
        try (BsonBinaryWriter writer = new BsonBinaryWriter(buffer);){
            BSON_VALUE_CODEC.encode(writer, document, EncoderContext.builder().build());
        }
        return buffer.toByteArray();
    }

    private SchemaAndValue numberToSchemaAndValue(Schema schema, BsonValue bsonValue) {
        Comparable<Integer> value = null;
        if (bsonValue.isNumber()) {
            BsonNumber bsonNumber = bsonValue.asNumber();
            if (bsonNumber.isInt32()) {
                value = bsonNumber.intValue();
            } else if (bsonNumber.isInt64()) {
                value = bsonNumber.longValue();
            } else if (bsonNumber.isDouble()) {
                value = bsonNumber.doubleValue();
            }
        } else if (bsonValue.isTimestamp()) {
            value = (long)bsonValue.asTimestamp().getTime() * 1000L;
        } else if (bsonValue.isDateTime()) {
            value = bsonValue.asDateTime().getValue();
        } else {
            throw this.unexpectedBsonValueType(schema.type(), bsonValue);
        }
        switch (schema.type()) {
            case INT8: {
                value = Values.convertToByte(schema, value);
                break;
            }
            case INT16: {
                value = Values.convertToShort(schema, value);
                break;
            }
            case INT32: {
                value = Values.convertToInteger(schema, value);
                break;
            }
            case INT64: {
                value = Values.convertToLong(schema, value);
                break;
            }
            case FLOAT32: {
                value = Values.convertToFloat(schema, value);
                break;
            }
            case FLOAT64: {
                value = Values.convertToDouble(schema, value);
                break;
            }
            default: {
                throw this.unexpectedBsonValueType(schema.type(), bsonValue);
            }
        }
        if (schema.name() != null) {
            switch (schema.name()) {
                case "org.apache.kafka.connect.data.Time": {
                    value = Values.convertToTime(schema, value);
                    break;
                }
                case "org.apache.kafka.connect.data.Date": {
                    value = Values.convertToDate(schema, value);
                    break;
                }
                case "org.apache.kafka.connect.data.Timestamp": {
                    value = Values.convertToTimestamp(schema, value);
                    break;
                }
            }
        }
        return new SchemaAndValue(schema, value);
    }

    private SchemaAndValue stringToSchemaAndValue(Schema schema, BsonValue bsonValue) {
        String value;
        if (bsonValue.isString()) {
            value = bsonValue.asString().getValue();
        } else if (bsonValue.isDocument()) {
            value = bsonValue.asDocument().toJson(this.jsonWriterSettings);
        } else {
            value = new BsonDocument("v", bsonValue).toJson(this.jsonWriterSettings);
            if ((value = value.substring(6, value.length() - 1)).startsWith("\"") && value.endsWith("\"")) {
                value = value.substring(1, value.length() - 1);
            }
        }
        return new SchemaAndValue(schema, value);
    }

    private SchemaAndValue bytesToSchemaAndValue(Schema schema, BsonValue bsonValue) {
        Object value = null;
        switch (bsonValue.getBsonType()) {
            case STRING: {
                value = bsonValue.asString().getValue().getBytes(StandardCharsets.UTF_8);
                break;
            }
            case BINARY: {
                value = bsonValue.asBinary().getData();
                break;
            }
            case DECIMAL128: {
                if (!Objects.equals(schema.name(), "org.apache.kafka.connect.data.Decimal")) break;
                value = bsonValue.asDecimal128().getValue().bigDecimalValue();
                break;
            }
            case DOCUMENT: {
                value = BsonValueToSchemaAndValue.documentToByteArray(bsonValue.asDocument());
                break;
            }
        }
        if (value == null) {
            throw this.unexpectedBsonValueType(Schema.Type.BYTES, bsonValue);
        }
        return new SchemaAndValue(schema, value);
    }

    private SchemaAndValue arrayToSchemaAndValue(Schema schema, BsonValue value) {
        if (!value.isArray()) {
            throw this.unexpectedBsonValueType(Schema.Type.ARRAY, value);
        }
        ArrayList values = new ArrayList();
        value.asArray().forEach(v -> values.add(this.toSchemaAndValue(schema.valueSchema(), (BsonValue)v).value()));
        return new SchemaAndValue(schema, values);
    }

    private SchemaAndValue mapToSchemaAndValue(Schema schema, BsonValue value) {
        if (!value.isDocument()) {
            throw this.unexpectedBsonValueType(schema.type(), value);
        }
        if (schema.keySchema() != Schema.STRING_SCHEMA) {
            throw this.unsupportedSchemaType(schema, ". Unexpected key schema type.");
        }
        BsonDocument document = value.asDocument();
        LinkedHashMap mapValue = new LinkedHashMap();
        document.forEach((k, v) -> mapValue.put(k, this.toSchemaAndValue(schema.valueSchema(), (BsonValue)v).value()));
        return new SchemaAndValue(schema, mapValue);
    }

    private SchemaAndValue recordToSchemaAndValue(Schema schema, BsonValue value) {
        if (!value.isDocument()) {
            throw this.unexpectedBsonValueType(schema.type(), value);
        }
        BsonDocument document = value.asDocument();
        Struct structValue = new Struct(schema);
        schema.fields().forEach(f -> {
            Optional<BsonValue> fieldValue = BsonDocumentFieldLookup.fieldLookup(f.name(), document);
            if (fieldValue.isPresent()) {
                SchemaAndValue schemaAndValue = this.toSchemaAndValue(f.schema(), fieldValue.get());
                structValue.put((Field)f, schemaAndValue.value());
            } else {
                boolean optionalField = f.schema().isOptional();
                Object defaultValue = f.schema().defaultValue();
                if (optionalField || defaultValue != null) {
                    structValue.put((Field)f, defaultValue);
                } else {
                    throw this.missingFieldException((Field)f, document);
                }
            }
        });
        return new SchemaAndValue(schema, structValue);
    }

    private SchemaAndValue booleanToSchemaAndValue(Schema schema, BsonValue bsonValue) {
        if (!bsonValue.isBoolean()) {
            throw this.unexpectedBsonValueType(Schema.Type.BOOLEAN, bsonValue);
        }
        return new SchemaAndValue(schema, bsonValue.asBoolean().getValue());
    }

    private ConnectException unsupportedSchemaType(Schema schema) {
        return this.unsupportedSchemaType(schema, "");
    }

    private ConnectException unsupportedSchemaType(Schema schema, String extra) {
        return new ConnectException(String.format("Unsupported Schema type: %s %s", new Object[]{schema.type(), extra}));
    }

    private DataException unexpectedBsonValueType(Schema.Type type, BsonValue value) {
        return new DataException(String.format("Schema type of %s but value was of type: %s", type.getName(), value.getBsonType().toString().toLowerCase(Locale.ROOT)));
    }

    private DataException missingFieldException(Field field, BsonDocument value) {
        return new DataException(String.format("Missing field '%s' in: '%s'", field.name(), value.toJson(this.jsonWriterSettings)));
    }
}

