/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.recon.tasks;

import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.RDBBatchOperation;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.helpers.BucketLayout;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfoGroup;
import org.apache.hadoop.ozone.recon.ReconConstants;
import org.apache.hadoop.ozone.recon.api.types.ContainerKeyPrefix;
import org.apache.hadoop.ozone.recon.api.types.KeyPrefixContainer;
import org.apache.hadoop.ozone.recon.spi.ReconContainerMetadataManager;
import org.apache.hadoop.ozone.recon.tasks.OMDBUpdateEvent;
import org.apache.hadoop.ozone.recon.tasks.OMUpdateEventBatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ContainerKeyMapperHelper {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerKeyMapperHelper.class);
    private static final Object TRUNCATE_LOCK = new Object();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void truncateTablesIfNeeded(ReconContainerMetadataManager reconContainerMetadataManager, String taskName) {
        Object object = TRUNCATE_LOCK;
        synchronized (object) {
            if (ReconConstants.CONTAINER_KEY_TABLES_TRUNCATED.compareAndSet(false, true)) {
                try {
                    reconContainerMetadataManager.reinitWithNewContainerDataFromOm(Collections.emptyMap());
                    LOG.debug("Successfully truncated container key tables.");
                }
                catch (Exception e) {
                    ReconConstants.CONTAINER_KEY_TABLES_TRUNCATED.set(false);
                    LOG.error("Error while truncating container key tables for task {}. Resetting flag.", (Object)taskName, (Object)e);
                    throw new RuntimeException("Table truncation failed", e);
                }
            } else {
                LOG.debug("Container key tables already truncated by another task.");
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean reprocess(OMMetadataManager omMetadataManager, ReconContainerMetadataManager reconContainerMetadataManager, BucketLayout bucketLayout, String taskName, long containerKeyFlushToDBMaxThreshold) {
        long omKeyCount = 0L;
        HashMap<ContainerKeyPrefix, Integer> containerKeyMap = new HashMap<ContainerKeyPrefix, Integer>();
        HashMap<Long, Long> containerKeyCountMap = new HashMap<Long, Long>();
        try {
            LOG.debug("Starting a 'reprocess' run for {}.", (Object)taskName);
            Instant start = Instant.now();
            ContainerKeyMapperHelper.truncateTablesIfNeeded(reconContainerMetadataManager, taskName);
            Table omKeyInfoTable = omMetadataManager.getKeyTable(bucketLayout);
            try (TableIterator keyIter = omKeyInfoTable.iterator();){
                while (keyIter.hasNext()) {
                    Table.KeyValue kv = (Table.KeyValue)keyIter.next();
                    ContainerKeyMapperHelper.handleKeyReprocess((String)kv.getKey(), (OmKeyInfo)kv.getValue(), containerKeyMap, containerKeyCountMap, reconContainerMetadataManager);
                    ++omKeyCount;
                    if (ContainerKeyMapperHelper.checkAndCallFlushToDB(containerKeyMap, containerKeyFlushToDBMaxThreshold, reconContainerMetadataManager)) continue;
                    LOG.error("Failed to flush container key data for {}", (Object)taskName);
                    boolean bl = false;
                    return bl;
                }
            }
            if (!ContainerKeyMapperHelper.flushAndCommitContainerKeyInfoToDB(containerKeyMap, containerKeyCountMap, reconContainerMetadataManager)) {
                LOG.error("Failed to flush Container Key data to DB for {}", (Object)taskName);
                return false;
            }
            Instant end = Instant.now();
            long durationMillis = Duration.between(start, end).toMillis();
            double durationSeconds = (double)durationMillis / 1000.0;
            LOG.debug("Completed 'reprocess' for {}. Processed {} keys in {} ms ({} seconds).", new Object[]{taskName, omKeyCount, durationMillis, durationSeconds});
            return true;
        }
        catch (IOException ioEx) {
            LOG.error("Error populating Container Key data for {} in Recon DB.", (Object)taskName, (Object)ioEx);
            return false;
        }
    }

    private static boolean checkAndCallFlushToDB(Map<ContainerKeyPrefix, Integer> containerKeyMap, long containerKeyFlushToDBMaxThreshold, ReconContainerMetadataManager reconContainerMetadataManager) {
        if ((long)containerKeyMap.size() >= containerKeyFlushToDBMaxThreshold) {
            return ContainerKeyMapperHelper.flushAndCommitContainerKeyInfoToDB(containerKeyMap, Collections.emptyMap(), reconContainerMetadataManager);
        }
        return true;
    }

    public static boolean process(OMUpdateEventBatch events, String tableName, ReconContainerMetadataManager reconContainerMetadataManager, String taskName) {
        Iterator<OMDBUpdateEvent> eventIterator = events.getIterator();
        int eventCount = 0;
        HashMap<ContainerKeyPrefix, Integer> containerKeyMap = new HashMap<ContainerKeyPrefix, Integer>();
        HashMap<Long, Long> containerKeyCountMap = new HashMap<Long, Long>();
        ArrayList<ContainerKeyPrefix> deletedKeyCountList = new ArrayList<ContainerKeyPrefix>();
        long startTime = System.currentTimeMillis();
        while (eventIterator.hasNext()) {
            OMDBUpdateEvent omdbUpdateEvent = eventIterator.next();
            if (!tableName.equals(omdbUpdateEvent.getTable())) continue;
            String updatedKey = (String)omdbUpdateEvent.getKey();
            OmKeyInfo updatedKeyValue = (OmKeyInfo)omdbUpdateEvent.getValue();
            try {
                switch (omdbUpdateEvent.getAction()) {
                    case PUT: {
                        ContainerKeyMapperHelper.handlePutOMKeyEvent(updatedKey, updatedKeyValue, containerKeyMap, containerKeyCountMap, deletedKeyCountList, reconContainerMetadataManager);
                        break;
                    }
                    case DELETE: {
                        ContainerKeyMapperHelper.handleDeleteOMKeyEvent(updatedKey, containerKeyMap, containerKeyCountMap, deletedKeyCountList, reconContainerMetadataManager);
                        break;
                    }
                    case UPDATE: {
                        if (omdbUpdateEvent.getOldValue() != null) {
                            ContainerKeyMapperHelper.handleDeleteOMKeyEvent(((OmKeyInfo)omdbUpdateEvent.getOldValue()).getKeyName(), containerKeyMap, containerKeyCountMap, deletedKeyCountList, reconContainerMetadataManager);
                        } else {
                            LOG.warn("Update event does not have the old Key Info for {}.", (Object)updatedKey);
                        }
                        ContainerKeyMapperHelper.handlePutOMKeyEvent(updatedKey, updatedKeyValue, containerKeyMap, containerKeyCountMap, deletedKeyCountList, reconContainerMetadataManager);
                        break;
                    }
                    default: {
                        LOG.info("Skipping DB update event: {}", (Object)omdbUpdateEvent.getAction());
                    }
                }
                ++eventCount;
            }
            catch (IOException e) {
                LOG.error("Unexpected exception while updating key data: {} ", (Object)updatedKey, (Object)e);
                return false;
            }
        }
        try {
            ContainerKeyMapperHelper.writeToTheDB(containerKeyMap, containerKeyCountMap, deletedKeyCountList, reconContainerMetadataManager);
        }
        catch (IOException e) {
            LOG.error("Unable to write Container Key Prefix data in Recon DB.", (Throwable)e);
            return false;
        }
        LOG.debug("{} successfully processed {} OM DB update event(s) in {} milliseconds.", new Object[]{taskName, eventCount, System.currentTimeMillis() - startTime});
        return true;
    }

    private static void handlePutOMKeyEvent(String key, OmKeyInfo omKeyInfo, Map<ContainerKeyPrefix, Integer> containerKeyMap, Map<Long, Long> containerKeyCountMap, List<ContainerKeyPrefix> deletedContainerKeyList, ReconContainerMetadataManager reconContainerMetadataManager) throws IOException {
        long containerCountToIncrement = 0L;
        for (OmKeyLocationInfoGroup omKeyLocationInfoGroup : omKeyInfo.getKeyLocationVersions()) {
            long keyVersion = omKeyLocationInfoGroup.getVersion();
            for (OmKeyLocationInfo omKeyLocationInfo : omKeyLocationInfoGroup.getLocationList()) {
                long containerId = omKeyLocationInfo.getContainerID();
                ContainerKeyPrefix containerKeyPrefix = ContainerKeyPrefix.get(containerId, key, keyVersion);
                if (reconContainerMetadataManager.getCountForContainerKeyPrefix(containerKeyPrefix) != 0 || containerKeyMap.containsKey(containerKeyPrefix)) continue;
                containerKeyMap.put(containerKeyPrefix, 1);
                deletedContainerKeyList.remove(containerKeyPrefix);
                if (!reconContainerMetadataManager.doesContainerExists(containerId) && !containerKeyCountMap.containsKey(containerId)) {
                    ++containerCountToIncrement;
                }
                long keyCount = containerKeyCountMap.containsKey(containerId) ? containerKeyCountMap.get(containerId).longValue() : reconContainerMetadataManager.getKeyCountForContainer(containerId);
                containerKeyCountMap.put(containerId, ++keyCount);
            }
        }
        if (containerCountToIncrement > 0L) {
            reconContainerMetadataManager.incrementContainerCountBy(containerCountToIncrement);
        }
    }

    private static void handleDeleteOMKeyEvent(String key, Map<ContainerKeyPrefix, Integer> containerKeyMap, Map<Long, Long> containerKeyCountMap, List<ContainerKeyPrefix> deletedContainerKeyList, ReconContainerMetadataManager reconContainerMetadataManager) throws IOException {
        HashSet<ContainerKeyPrefix> keysToBeDeleted = new HashSet<ContainerKeyPrefix>();
        try (TableIterator keyContainerIterator = reconContainerMetadataManager.getKeyContainerTableIterator();){
            Table.KeyValue keyValue;
            String keyPrefix;
            keyContainerIterator.seek((Object)KeyPrefixContainer.get(key));
            while (keyContainerIterator.hasNext() && (keyPrefix = ((KeyPrefixContainer)(keyValue = (Table.KeyValue)keyContainerIterator.next()).getKey()).getKeyPrefix()).equals(key)) {
                if (((KeyPrefixContainer)keyValue.getKey()).getContainerId() == -1L) continue;
                keysToBeDeleted.add(((KeyPrefixContainer)keyValue.getKey()).toContainerKeyPrefix());
            }
        }
        containerKeyMap.keySet().forEach(containerKeyPrefix -> {
            String keyPrefix = containerKeyPrefix.getKeyPrefix();
            if (keyPrefix.equals(key)) {
                keysToBeDeleted.add((ContainerKeyPrefix)containerKeyPrefix);
            }
        });
        for (ContainerKeyPrefix containerKeyPrefix2 : keysToBeDeleted) {
            deletedContainerKeyList.add(containerKeyPrefix2);
            containerKeyMap.remove(containerKeyPrefix2);
            Long containerID = containerKeyPrefix2.getContainerId();
            long keyCount = containerKeyCountMap.containsKey(containerID) ? containerKeyCountMap.get(containerID).longValue() : reconContainerMetadataManager.getKeyCountForContainer(containerID);
            if (keyCount <= 0L) continue;
            containerKeyCountMap.put(containerID, --keyCount);
        }
    }

    private static void writeToTheDB(Map<ContainerKeyPrefix, Integer> containerKeyMap, Map<Long, Long> containerKeyCountMap, List<ContainerKeyPrefix> deletedContainerKeyList, ReconContainerMetadataManager reconContainerMetadataManager) throws IOException {
        try (RDBBatchOperation rdbBatchOperation = new RDBBatchOperation();){
            containerKeyMap.keySet().forEach(key -> {
                try {
                    reconContainerMetadataManager.batchStoreContainerKeyMapping((BatchOperation)rdbBatchOperation, (ContainerKeyPrefix)key, (Integer)containerKeyMap.get(key));
                }
                catch (IOException e) {
                    LOG.error("Unable to write Container Key Prefix data in Recon DB.", (Throwable)e);
                }
            });
            containerKeyCountMap.keySet().forEach(key -> {
                try {
                    reconContainerMetadataManager.batchStoreContainerKeyCounts((BatchOperation)rdbBatchOperation, (Long)key, (Long)containerKeyCountMap.get(key));
                }
                catch (IOException e) {
                    LOG.error("Unable to write Container Key Count data in Recon DB.", (Throwable)e);
                }
            });
            deletedContainerKeyList.forEach(key -> {
                try {
                    reconContainerMetadataManager.batchDeleteContainerMapping((BatchOperation)rdbBatchOperation, (ContainerKeyPrefix)key);
                }
                catch (IOException e) {
                    LOG.error("Unable to delete Container Key Prefix data in Recon DB.", (Throwable)e);
                }
            });
            reconContainerMetadataManager.commitBatchOperation(rdbBatchOperation);
        }
    }

    public static void handleKeyReprocess(String key, OmKeyInfo omKeyInfo, Map<ContainerKeyPrefix, Integer> containerKeyMap, Map<Long, Long> containerKeyCountMap, ReconContainerMetadataManager reconContainerMetadataManager) throws IOException {
        long containerCountToIncrement = 0L;
        for (OmKeyLocationInfoGroup omKeyLocationInfoGroup : omKeyInfo.getKeyLocationVersions()) {
            long keyVersion = omKeyLocationInfoGroup.getVersion();
            for (OmKeyLocationInfo omKeyLocationInfo : omKeyLocationInfoGroup.getLocationList()) {
                long containerId = omKeyLocationInfo.getContainerID();
                ContainerKeyPrefix containerKeyPrefix = ContainerKeyPrefix.get(containerId, key, keyVersion);
                if (reconContainerMetadataManager.getCountForContainerKeyPrefix(containerKeyPrefix) != 0 || containerKeyMap.containsKey(containerKeyPrefix)) continue;
                containerKeyMap.put(containerKeyPrefix, 1);
                if (!reconContainerMetadataManager.doesContainerExists(containerId) && !containerKeyCountMap.containsKey(containerId)) {
                    ++containerCountToIncrement;
                }
                long keyCount = containerKeyCountMap.getOrDefault(containerId, reconContainerMetadataManager.getKeyCountForContainer(containerId));
                containerKeyCountMap.put(containerId, keyCount + 1L);
            }
        }
        if (containerCountToIncrement > 0L) {
            reconContainerMetadataManager.incrementContainerCountBy(containerCountToIncrement);
        }
    }

    public static boolean flushAndCommitContainerKeyInfoToDB(Map<ContainerKeyPrefix, Integer> containerKeyMap, Map<Long, Long> containerKeyCountMap, ReconContainerMetadataManager reconContainerMetadataManager) {
        try {
            ContainerKeyMapperHelper.writeToTheDB(containerKeyMap, containerKeyCountMap, Collections.emptyList(), reconContainerMetadataManager);
            containerKeyMap.clear();
            containerKeyCountMap.clear();
        }
        catch (IOException e) {
            LOG.error("Unable to write Container Key and Container Key Count data in Recon DB.", (Throwable)e);
            return false;
        }
        return true;
    }
}

