/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.symmetric;

import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.crypto.KeyGenerator;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.security.symmetric.ManagedSecretKey;
import org.apache.hadoop.hdds.security.symmetric.SecretKeyClient;
import org.apache.hadoop.hdds.security.symmetric.SecretKeyConfig;
import org.apache.hadoop.hdds.security.symmetric.SecretKeyState;
import org.apache.hadoop.hdds.security.symmetric.SecretKeyStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecretKeyManager
implements SecretKeyClient {
    private static final Logger LOG = LoggerFactory.getLogger(SecretKeyManager.class);
    private final SecretKeyState state;
    private final Duration rotationDuration;
    private final Duration validityDuration;
    private final SecretKeyStore keyStore;
    private final KeyGenerator keyGenerator;

    public SecretKeyManager(SecretKeyState state, SecretKeyStore keyStore, Duration rotationDuration, Duration validityDuration, String algorithm) {
        this.state = Objects.requireNonNull(state);
        this.rotationDuration = Objects.requireNonNull(rotationDuration);
        this.validityDuration = Objects.requireNonNull(validityDuration);
        this.keyStore = Objects.requireNonNull(keyStore);
        this.keyGenerator = this.createKeyGenerator(algorithm);
    }

    public SecretKeyManager(SecretKeyState state, SecretKeyStore keyStore, SecretKeyConfig config) {
        this(state, keyStore, config.getRotateDuration(), config.getExpiryDuration(), config.getAlgorithm());
    }

    public synchronized void checkAndInitialize() throws SCMException {
        if (this.isInitialized()) {
            return;
        }
        LOG.info("Initializing SecretKeys.");
        List<ManagedSecretKey> allKeys = this.keyStore.load().stream().filter(x -> !x.isExpired()).collect(Collectors.toList());
        if (allKeys.isEmpty()) {
            ManagedSecretKey newKey = this.generateSecretKey();
            allKeys.add(newKey);
            LOG.info("No valid key has been loaded. A new key is generated: {}", (Object)newKey);
        } else {
            LOG.info("Keys reloaded: {}", allKeys);
        }
        this.state.updateKeys(allKeys);
    }

    public boolean isInitialized() {
        return this.state.getCurrentKey() != null;
    }

    public synchronized boolean checkAndRotate(boolean force) throws SCMException {
        this.checkAndInitialize();
        ManagedSecretKey currentKey = this.state.getCurrentKey();
        if (force || this.shouldRotate(currentKey)) {
            ManagedSecretKey newCurrentKey = this.generateSecretKey();
            List<ManagedSecretKey> updatedKeys = this.state.getSortedKeys().stream().filter(x -> !x.isExpired()).collect(Collectors.toList());
            updatedKeys.add(newCurrentKey);
            LOG.info((force ? "Forced " : "") + "SecretKey rotation is happening, new key generated {}", (Object)newCurrentKey);
            this.state.updateKeys(updatedKeys);
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("The latest key was created at: " + currentKey.getCreationTime() + " which does not pass the rotation duration");
        }
        return false;
    }

    @Override
    public ManagedSecretKey getCurrentSecretKey() {
        return this.state.getCurrentKey();
    }

    @Override
    public ManagedSecretKey getSecretKey(UUID id) {
        return this.state.getKey(id);
    }

    public List<ManagedSecretKey> getSortedKeys() {
        return this.state.getSortedKeys();
    }

    public void reinitialize(List<ManagedSecretKey> secretKeys) {
        this.state.reinitialize(secretKeys);
    }

    private boolean shouldRotate(ManagedSecretKey currentKey) {
        Duration established = Duration.between(currentKey.getCreationTime(), Instant.now());
        return established.compareTo(this.rotationDuration) >= 0;
    }

    private ManagedSecretKey generateSecretKey() {
        Instant now = Instant.now();
        return new ManagedSecretKey(UUID.randomUUID(), now, now.plus(this.validityDuration), this.keyGenerator.generateKey());
    }

    private KeyGenerator createKeyGenerator(String algorithm) {
        try {
            return KeyGenerator.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("Error creating KeyGenerator for algorithm " + algorithm, e);
        }
    }
}

