/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.ozoneimpl;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.impl.ContainerDataYaml;
import org.apache.hadoop.ozone.container.common.impl.ContainerSet;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.MutableVolumeSet;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerReader
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerReader.class);
    private HddsVolume hddsVolume;
    private final ContainerSet containerSet;
    private final ConfigurationSource config;
    private final File hddsVolumeDir;
    private final MutableVolumeSet volumeSet;
    private final boolean shouldDelete;

    public ContainerReader(MutableVolumeSet volSet, HddsVolume volume, ContainerSet cset, ConfigurationSource conf, boolean shouldDelete) {
        Preconditions.checkNotNull((Object)volume);
        this.hddsVolume = volume;
        this.hddsVolumeDir = this.hddsVolume.getHddsRootDir();
        this.containerSet = cset;
        this.config = conf;
        this.volumeSet = volSet;
        this.shouldDelete = shouldDelete;
    }

    @Override
    public void run() {
        try {
            this.readVolume(this.hddsVolumeDir);
        }
        catch (Throwable t) {
            LOG.error("Caught an exception during reading container files from Volume {} {}", (Object)this.hddsVolumeDir, (Object)t);
            this.volumeSet.failVolume(this.hddsVolumeDir.getPath());
        }
    }

    public void readVolume(File hddsVolumeRootDir) {
        Preconditions.checkNotNull((Object)hddsVolumeRootDir, (Object)"hddsVolumeRootDircannot be null");
        File[] storageDirs = hddsVolumeRootDir.listFiles(File::isDirectory);
        if (storageDirs == null) {
            LOG.error("IO error for the volume {}, skipped loading", (Object)hddsVolumeRootDir);
            this.volumeSet.failVolume(hddsVolumeRootDir.getPath());
            return;
        }
        if (storageDirs.length > 0) {
            File clusterIDDir;
            File idDir = clusterIDDir = new File(hddsVolumeRootDir, this.hddsVolume.getClusterID());
            if (storageDirs.length == 1 && !clusterIDDir.exists()) {
                idDir = storageDirs[0];
            } else if (!clusterIDDir.exists()) {
                LOG.error("Volume {} is in an inconsistent state. Expected clusterID directory {} not found.", (Object)hddsVolumeRootDir, (Object)clusterIDDir);
                this.volumeSet.failVolume(hddsVolumeRootDir.getPath());
                return;
            }
            LOG.info("Start to verify containers on volume {}", (Object)hddsVolumeRootDir);
            File currentDir = new File(idDir, "current");
            File[] containerTopDirs = currentDir.listFiles();
            if (containerTopDirs != null && containerTopDirs.length > 0) {
                for (File containerTopDir : containerTopDirs) {
                    File[] containerDirs;
                    if (!containerTopDir.isDirectory() || (containerDirs = containerTopDir.listFiles()) == null) continue;
                    for (File containerDir : containerDirs) {
                        try {
                            File containerFile = ContainerUtils.getContainerFile(containerDir);
                            long containerID = ContainerUtils.getContainerID(containerDir);
                            if (containerFile.exists()) {
                                this.verifyContainerFile(containerID, containerFile);
                                continue;
                            }
                            LOG.error("Missing .container file for ContainerID: {}", (Object)containerDir.getName());
                        }
                        catch (Throwable e) {
                            LOG.error("Failed to load container from {}", (Object)containerDir.getAbsolutePath(), (Object)e);
                        }
                    }
                }
            }
        }
        LOG.info("Finish verifying containers on volume {}", (Object)hddsVolumeRootDir);
    }

    private void verifyContainerFile(long containerID, File containerFile) {
        try {
            ContainerData containerData = ContainerDataYaml.readContainerFile(containerFile);
            if (containerID != containerData.getContainerID()) {
                LOG.error("Invalid ContainerID in file {}. Skipping loading of this container.", (Object)containerFile);
                return;
            }
            this.verifyAndFixupContainerData(containerData);
        }
        catch (IOException ex) {
            LOG.error("Failed to parse ContainerFile for ContainerID: {}", (Object)containerID, (Object)ex);
        }
    }

    public void verifyAndFixupContainerData(ContainerData containerData) throws IOException {
        switch (containerData.getContainerType()) {
            case KeyValueContainer: {
                if (containerData instanceof KeyValueContainerData) {
                    KeyValueContainerData kvContainerData = (KeyValueContainerData)containerData;
                    containerData.setVolume(this.hddsVolume);
                    KeyValueContainerUtil.parseKVContainerData(kvContainerData, this.config);
                    KeyValueContainer kvContainer = new KeyValueContainer(kvContainerData, this.config);
                    if (kvContainer.getContainerState() == ContainerProtos.ContainerDataProto.State.RECOVERING) {
                        if (this.shouldDelete) {
                            kvContainer.markContainerUnhealthy();
                            LOG.info("Stale recovering container {} marked UNHEALTHY", (Object)kvContainerData.getContainerID());
                            this.containerSet.addContainer(kvContainer);
                        }
                        return;
                    }
                    if (kvContainer.getContainerState() == ContainerProtos.ContainerDataProto.State.DELETED) {
                        if (this.shouldDelete) {
                            this.cleanupContainer(this.hddsVolume, kvContainer);
                        }
                        return;
                    }
                    try {
                        this.containerSet.addContainer(kvContainer);
                    }
                    catch (StorageContainerException e) {
                        if (e.getResult() != ContainerProtos.Result.CONTAINER_EXISTS) {
                            throw e;
                        }
                        if (!this.shouldDelete) break;
                        this.resolveDuplicate((KeyValueContainer)this.containerSet.getContainer(kvContainer.getContainerData().getContainerID()), kvContainer);
                    }
                    break;
                }
                throw new StorageContainerException("Container File is corrupted. ContainerType is KeyValueContainer but cast to KeyValueContainerData failed. ", ContainerProtos.Result.CONTAINER_METADATA_ERROR);
            }
            default: {
                throw new StorageContainerException("Unrecognized ContainerType " + containerData.getContainerType(), ContainerProtos.Result.UNKNOWN_CONTAINER_TYPE);
            }
        }
    }

    private void resolveDuplicate(KeyValueContainer existing, KeyValueContainer toAdd) throws IOException {
        if (existing.getContainerData().getReplicaIndex() != 0 || toAdd.getContainerData().getReplicaIndex() != 0) {
            LOG.warn("Container {} is present at {} and at {}. Both are EC containers. Leaving both containers on disk.", new Object[]{existing.getContainerData().getContainerID(), existing.getContainerData().getContainerPath(), toAdd.getContainerData().getContainerPath()});
            return;
        }
        long existingBCSID = existing.getBlockCommitSequenceId();
        ContainerProtos.ContainerDataProto.State existingState = existing.getContainerState();
        long toAddBCSID = toAdd.getBlockCommitSequenceId();
        ContainerProtos.ContainerDataProto.State toAddState = toAdd.getContainerState();
        if (existingState != toAddState) {
            if (existingState == ContainerProtos.ContainerDataProto.State.CLOSED) {
                LOG.warn("Container {} is present at {} with state CLOSED and at {} with state {}. Removing the latter container.", new Object[]{existing.getContainerData().getContainerID(), existing.getContainerData().getContainerPath(), toAdd.getContainerData().getContainerPath(), toAddState});
                KeyValueContainerUtil.removeContainer(toAdd.getContainerData(), this.hddsVolume.getConf());
                return;
            }
            if (toAddState == ContainerProtos.ContainerDataProto.State.CLOSED) {
                LOG.warn("Container {} is present at {} with state CLOSED and at {} with state {}. Removing the latter container.", new Object[]{toAdd.getContainerData().getContainerID(), toAdd.getContainerData().getContainerPath(), existing.getContainerData().getContainerPath(), existingState});
                this.swapAndRemoveContainer(existing, toAdd);
                return;
            }
        }
        if (existingBCSID >= toAddBCSID) {
            LOG.warn("Container {} is present at {} with a newer or equal BCSID than at {}. Removing the latter container.", new Object[]{existing.getContainerData().getContainerID(), existing.getContainerData().getContainerPath(), toAdd.getContainerData().getContainerPath()});
            KeyValueContainerUtil.removeContainer(toAdd.getContainerData(), this.hddsVolume.getConf());
        } else {
            LOG.warn("Container {} is present at {} with a lesser BCSID than at {}. Removing the former container.", new Object[]{existing.getContainerData().getContainerID(), existing.getContainerData().getContainerPath(), toAdd.getContainerData().getContainerPath()});
            this.swapAndRemoveContainer(existing, toAdd);
        }
    }

    private void swapAndRemoveContainer(KeyValueContainer existing, KeyValueContainer toAdd) throws IOException {
        this.containerSet.removeContainerOnlyFromMemory(existing.getContainerData().getContainerID());
        this.containerSet.addContainer(toAdd);
        KeyValueContainerUtil.removeContainer(existing.getContainerData(), this.hddsVolume.getConf());
    }

    private void cleanupContainer(HddsVolume volume, KeyValueContainer kvContainer) {
        try {
            LOG.info("Finishing delete of container {}.", (Object)kvContainer.getContainerData().getContainerID());
            KeyValueContainerUtil.removeContainer(kvContainer.getContainerData(), volume.getConf());
            kvContainer.delete();
        }
        catch (IOException ex) {
            LOG.warn("Failed to remove deleted container {}.", (Object)kvContainer.getContainerData().getContainerID(), (Object)ex);
        }
    }
}

