/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.ozoneimpl;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Instant;
import java.util.Iterator;
import java.util.Optional;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.ozoneimpl.AbstractBackgroundContainerScanner;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerController;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerDataScannerMetrics;
import org.apache.hadoop.ozone.container.ozoneimpl.ContainerScannerConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackgroundContainerDataScanner
extends AbstractBackgroundContainerScanner {
    public static final Logger LOG = LoggerFactory.getLogger(BackgroundContainerDataScanner.class);
    private final HddsVolume volume;
    private final ContainerController controller;
    private final DataTransferThrottler throttler;
    private final Canceler canceler;
    private static final String NAME_FORMAT = "ContainerDataScanner(%s)";
    private final ContainerDataScannerMetrics metrics;
    private final long minScanGap;

    public BackgroundContainerDataScanner(ContainerScannerConfiguration conf, ContainerController controller, HddsVolume volume) {
        super(String.format(NAME_FORMAT, volume), conf.getDataScanInterval());
        this.controller = controller;
        this.volume = volume;
        this.throttler = new HddsDataTransferThrottler(conf.getBandwidthPerVolume());
        this.canceler = new Canceler();
        this.metrics = ContainerDataScannerMetrics.create(volume.toString());
        this.metrics.setStorageDirectory(volume.toString());
        this.minScanGap = conf.getContainerScanMinGap();
    }

    private boolean shouldScan(Container<?> container) {
        return container.shouldScanData() && !ContainerUtils.recentlyScanned(container, this.minScanGap, LOG);
    }

    @Override
    public void scanContainer(Container<?> c) throws IOException, InterruptedException {
        if (this.volume.isFailed()) {
            this.shutdown("The volume has failed.");
            return;
        }
        if (!this.shouldScan(c)) {
            return;
        }
        Object containerData = c.getContainerData();
        long containerId = ((ContainerData)containerData).getContainerID();
        BackgroundContainerDataScanner.logScanStart(containerData);
        Container.ScanResult result = c.scanData(this.throttler, this.canceler);
        if (result.getFailureType() == Container.ScanResult.FailureType.DELETED_CONTAINER) {
            LOG.error("Container [{}] has been deleted.", (Object)containerId, (Object)result.getException());
            return;
        }
        if (!result.isHealthy()) {
            LOG.error("Corruption detected in container [{}]. Marking it UNHEALTHY.", (Object)containerId, (Object)result.getException());
            this.metrics.incNumUnHealthyContainers();
            this.controller.markContainerUnhealthy(containerId, result);
        }
        this.metrics.incNumContainersScanned();
        Instant now = Instant.now();
        BackgroundContainerDataScanner.logScanCompleted(containerData, now);
        this.controller.updateDataScanTimestamp(containerId, now);
    }

    @Override
    public Iterator<Container<?>> getContainerIterator() {
        return this.controller.getContainers(this.volume);
    }

    private static void logScanStart(ContainerData containerData) {
        if (LOG.isDebugEnabled()) {
            Optional<Instant> scanTimestamp = containerData.lastDataScanTime();
            String lastScanTime = scanTimestamp.map(ts -> "at " + ts).orElse("never");
            LOG.debug("Scanning container {}, last scanned {}", (Object)containerData.getContainerID(), (Object)lastScanTime);
        }
    }

    private static void logScanCompleted(ContainerData containerData, Instant timestamp) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Completed scan of container {} at {}", (Object)containerData.getContainerID(), (Object)timestamp);
        }
    }

    @Override
    public synchronized void shutdown() {
        this.shutdown("");
    }

    private synchronized void shutdown(String reason) {
        String shutdownMessage = String.format(NAME_FORMAT, this.volume) + " is shutting down. " + reason;
        LOG.info(shutdownMessage);
        this.canceler.cancel(shutdownMessage);
        super.shutdown();
    }

    @Override
    @VisibleForTesting
    public ContainerDataScannerMetrics getMetrics() {
        return this.metrics;
    }

    public String toString() {
        return String.format(NAME_FORMAT, this.volume + ", " + this.volume.getStorageID());
    }

    private class HddsDataTransferThrottler
    extends DataTransferThrottler {
        HddsDataTransferThrottler(long bandwidthPerSec) {
            super(bandwidthPerSec);
        }

        public synchronized void throttle(long numOfBytes) {
            BackgroundContainerDataScanner.this.metrics.incNumBytesScanned(numOfBytes);
            super.throttle(numOfBytes);
        }

        public synchronized void throttle(long numOfBytes, Canceler c) {
            BackgroundContainerDataScanner.this.metrics.incNumBytesScanned(numOfBytes);
            super.throttle(numOfBytes, c);
        }
    }
}

