/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.flowanalysis.rules;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.flow.VersionedConnection;
import org.apache.nifi.flow.VersionedProcessGroup;
import org.apache.nifi.flowanalysis.AbstractFlowAnalysisRule;
import org.apache.nifi.flowanalysis.FlowAnalysisRuleContext;
import org.apache.nifi.flowanalysis.GroupAnalysisResult;
import org.apache.nifi.flowanalysis.rules.util.ConnectionViolation;
import org.apache.nifi.flowanalysis.rules.util.FlowAnalysisRuleUtils;
import org.apache.nifi.flowanalysis.rules.util.ViolationType;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.util.FormatUtils;

@Tags(value={"connection", "expiration", "age"})
@CapabilityDescription(value="This rule will generate a violation if FlowFile expiration settings of a connection exceed configured thresholds. Improper configuration of FlowFile expiration settings can cause files to be deleted unexpectedly and can cause the content repository to fill up.")
public class RestrictFlowFileExpiration
extends AbstractFlowAnalysisRule {
    public static final PropertyDescriptor ALLOW_ZERO = new PropertyDescriptor.Builder().name("Allow Zero Expiration").description("If set to true, a 0 second FlowFile Expiration on connections is allowed despite other configured restrictions. If set to false, a 0 second FlowFile Expiration will be compared against the other configured restrictions. This can be used to prevent a user from setting a value of 0 seconds which could fill up the content repository if files accumulate in front of stopped processors.").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").addValidator(StandardValidators.BOOLEAN_VALIDATOR).build();
    public static final PropertyDescriptor MIN_FLOWFILE_EXPIRATION = new PropertyDescriptor.Builder().name("Minimum FlowFile Expiration").description("This is the minimum value that should be set for the FlowFile Expiration setting on connections. This can be used to prevent a user from setting a very small expiration which can cause files to be deleted unexpectedly.").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue("1 min").build();
    public static final PropertyDescriptor MAX_FLOWFILE_EXPIRATION = new PropertyDescriptor.Builder().name("Maximum FlowFile Expiration").description("This is the maximum value that should be set for the FlowFile Expiration setting on connections. This can be used to prevent a user from setting a large expiration which could fill up the content repository if files accumulate in front of stopped processors.").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue("30 days").build();
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(ALLOW_ZERO, MIN_FLOWFILE_EXPIRATION, MAX_FLOWFILE_EXPIRATION);

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        long maxSize;
        ArrayList<ValidationResult> results = new ArrayList<ValidationResult>();
        long minSize = validationContext.getProperty(MIN_FLOWFILE_EXPIRATION).asTimePeriod(TimeUnit.MILLISECONDS);
        if (minSize > (maxSize = validationContext.getProperty(MAX_FLOWFILE_EXPIRATION).asTimePeriod(TimeUnit.MILLISECONDS).longValue())) {
            results.add(new ValidationResult.Builder().subject(MIN_FLOWFILE_EXPIRATION.getName()).valid(false).explanation("Value of '" + MIN_FLOWFILE_EXPIRATION.getName() + "' cannot be greater than '" + MAX_FLOWFILE_EXPIRATION.getName() + "'").build());
        }
        return results;
    }

    public Collection<GroupAnalysisResult> analyzeProcessGroup(VersionedProcessGroup pg, FlowAnalysisRuleContext context) {
        ArrayList<ConnectionViolation> violations = new ArrayList<ConnectionViolation>();
        boolean allowZero = context.getProperty(ALLOW_ZERO).asBoolean();
        long minSize = context.getProperty(MIN_FLOWFILE_EXPIRATION).asTimePeriod(TimeUnit.MILLISECONDS);
        long maxSize = context.getProperty(MAX_FLOWFILE_EXPIRATION).asTimePeriod(TimeUnit.MILLISECONDS);
        pg.getConnections().forEach(connection -> {
            long connectionExpiration = FormatUtils.getTimeDuration((String)connection.getFlowFileExpiration(), (TimeUnit)TimeUnit.MILLISECONDS);
            if (connectionExpiration != 0L || !allowZero) {
                if (connectionExpiration < minSize) {
                    violations.add(new ConnectionViolation((VersionedConnection)connection, ExpirationViolationType.EXPIRATION_BELOW_MINIMUM, ((Object)((Object)this)).getClass().getSimpleName(), connection.getFlowFileExpiration(), context.getProperty(MIN_FLOWFILE_EXPIRATION).getValue()));
                }
                if (connectionExpiration > maxSize) {
                    violations.add(new ConnectionViolation((VersionedConnection)connection, ExpirationViolationType.EXPIRATION_ABOVE_MAXIMUM, ((Object)((Object)this)).getClass().getSimpleName(), connection.getFlowFileExpiration(), context.getProperty(MAX_FLOWFILE_EXPIRATION).getValue()));
                }
            }
        });
        return FlowAnalysisRuleUtils.convertToGroupAnalysisResults(pg, violations);
    }

    private static enum ExpirationViolationType implements ViolationType
    {
        EXPIRATION_BELOW_MINIMUM("FlowFileExpirationTooLow", "FlowFile Expiration", "cannot be less than"),
        EXPIRATION_ABOVE_MAXIMUM("FlowFileExpirationTooHigh", "FlowFile Expiration", "cannot be greater than");

        private final String id;
        private final String configurationItem;
        private final String violationMessage;

        private ExpirationViolationType(String id, String configurationItem, String violationMessage) {
            this.id = id;
            this.configurationItem = configurationItem;
            this.violationMessage = violationMessage;
        }

        @Override
        public String getId() {
            return this.id;
        }

        @Override
        public String getConfigurationItem() {
            return this.configurationItem;
        }

        @Override
        public String getViolationMessage() {
            return this.violationMessage;
        }
    }
}

