/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileOwnerAttributeView;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.Restricted;
import org.apache.nifi.annotation.behavior.Restriction;
import org.apache.nifi.annotation.behavior.TriggerWhenEmpty;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.RequiredPermission;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.standard.FetchFile;
import org.apache.nifi.processors.standard.PutFile;

@TriggerWhenEmpty
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@Tags(value={"local", "files", "filesystem", "ingest", "ingress", "get", "source", "input"})
@CapabilityDescription(value="    Reads files from a directory and streams them into the contents of new FlowFiles.\n    NiFi will ignore files it doesn't have at least read permissions for.\n")
@WritesAttributes(value={@WritesAttribute(attribute="filename", description="The filename is set to the name of the file on disk"), @WritesAttribute(attribute="path", description="The path is set to the relative path of the file's directory on disk. For example, if the <Input Directory> property is set to /tmp, files picked up from /tmp will have the path attribute set to ./. If the <Recurse Subdirectories> property is set to true and a file is picked up from /tmp/abc/1/2/3, then the path attribute will be set to abc/1/2/3"), @WritesAttribute(attribute="file.creationTime", description="The date and time that the file was created. May not work on all file systems"), @WritesAttribute(attribute="file.lastModifiedTime", description="The date and time that the file was last modified. May not work on all file systems"), @WritesAttribute(attribute="file.lastAccessTime", description="The date and time that the file was last accessed. May not work on all file systems"), @WritesAttribute(attribute="file.owner", description="The owner of the file. May not work on all file systems"), @WritesAttribute(attribute="file.group", description="The group owner of the file. May not work on all file systems"), @WritesAttribute(attribute="file.permissions", description="The read/write/execute permissions of the file. May not work on all file systems"), @WritesAttribute(attribute="absolute.path", description="The full/absolute path from where a file was picked up. The current 'path' attribute is still populated, but may be a relative path")})
@SeeAlso(value={PutFile.class, FetchFile.class})
@Restricted(restrictions={@Restriction(requiredPermission=RequiredPermission.READ_FILESYSTEM, explanation="Provides operator the ability to read from any file that NiFi has access to."), @Restriction(requiredPermission=RequiredPermission.WRITE_FILESYSTEM, explanation="Provides operator the ability to delete any file that NiFi has access to.")})
public class GetFile
extends AbstractProcessor {
    public static final PropertyDescriptor DIRECTORY = new PropertyDescriptor.Builder().name("Input Directory").description("The input directory from which to pull files").required(true).addValidator(StandardValidators.createDirectoryExistsValidator((boolean)true, (boolean)false)).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).build();
    public static final PropertyDescriptor RECURSE = new PropertyDescriptor.Builder().name("Recurse Subdirectories").description("Indicates whether or not to pull files from subdirectories").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").build();
    public static final PropertyDescriptor KEEP_SOURCE_FILE = new PropertyDescriptor.Builder().name("Keep Source File").description("If true, the file is not deleted after it has been copied to the Content Repository;\nthis causes the file to be picked up continually and is useful for testing purposes.\nIf not keeping the source file, NiFi will need write permissions on the directory\nwhere the file is located in order to delete it; otherwise the file will be ignored.\n").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("false").build();
    public static final PropertyDescriptor FILE_FILTER = new PropertyDescriptor.Builder().name("File Filter").description("Only files whose names match the given regular expression will be picked up").required(true).defaultValue("[^\\.].*").addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).build();
    public static final PropertyDescriptor PATH_FILTER = new PropertyDescriptor.Builder().name("Path Filter").description("When " + RECURSE.getName() + " is true, then only subdirectories whose path matches the given regular expression will be scanned").required(false).addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).build();
    public static final PropertyDescriptor MIN_AGE = new PropertyDescriptor.Builder().name("Minimum File Age").description("The minimum age that a file must be in order to be pulled; any file younger than this amount of time (according to last modification date) will be ignored").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue("0 sec").build();
    public static final PropertyDescriptor MAX_AGE = new PropertyDescriptor.Builder().name("Maximum File Age").description("The maximum age that a file must be in order to be pulled; any file older than this amount of time (according to last modification date) will be ignored").required(false).addValidator(StandardValidators.createTimePeriodValidator((long)100L, (TimeUnit)TimeUnit.MILLISECONDS, (long)Long.MAX_VALUE, (TimeUnit)TimeUnit.NANOSECONDS)).build();
    public static final PropertyDescriptor MIN_SIZE = new PropertyDescriptor.Builder().name("Minimum File Size").description("The minimum size that a file must be in order to be pulled").required(true).addValidator(StandardValidators.DATA_SIZE_VALIDATOR).defaultValue("0 B").build();
    public static final PropertyDescriptor MAX_SIZE = new PropertyDescriptor.Builder().name("Maximum File Size").description("The maximum size that a file can be in order to be pulled").required(false).addValidator(StandardValidators.DATA_SIZE_VALIDATOR).build();
    public static final PropertyDescriptor IGNORE_HIDDEN_FILES = new PropertyDescriptor.Builder().name("Ignore Hidden Files").description("Indicates whether or not hidden files should be ignored").allowableValues(new String[]{"true", "false"}).defaultValue("true").required(true).build();
    public static final PropertyDescriptor POLLING_INTERVAL = new PropertyDescriptor.Builder().name("Polling Interval").description("Indicates the amount of time between performing directory listings to find new files\nthat appear in the Input Directory").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue("30 sec").build();
    public static final PropertyDescriptor BATCH_SIZE = new PropertyDescriptor.Builder().name("Batch Size").description("The maximum number of files to pull in each invocation of the processor").required(true).addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).defaultValue("10").build();
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(DIRECTORY, FILE_FILTER, PATH_FILTER, BATCH_SIZE, KEEP_SOURCE_FILE, RECURSE, POLLING_INTERVAL, IGNORE_HIDDEN_FILES, MIN_AGE, MAX_AGE, MIN_SIZE, MAX_SIZE);
    public static final String FILE_CREATION_TIME_ATTRIBUTE = "file.creationTime";
    public static final String FILE_LAST_MODIFY_TIME_ATTRIBUTE = "file.lastModifiedTime";
    public static final String FILE_LAST_ACCESS_TIME_ATTRIBUTE = "file.lastAccessTime";
    public static final String FILE_OWNER_ATTRIBUTE = "file.owner";
    public static final String FILE_GROUP_ATTRIBUTE = "file.group";
    public static final String FILE_PERMISSIONS_ATTRIBUTE = "file.permissions";
    public static final String FILE_MODIFY_DATE_ATTR_FORMAT = "yyyy-MM-dd'T'HH:mm:ssZ";
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("All files are routed to success").build();
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS);
    private final BlockingQueue<File> fileQueue = new LinkedBlockingQueue<File>();
    private final Set<File> inProcess = new HashSet<File>();
    private final Set<File> recentlyProcessed = new HashSet<File>();
    private final Lock queueLock = new ReentrantLock();
    private final Lock listingLock = new ReentrantLock();
    private final AtomicLong queueLastUpdated = new AtomicLong(0L);

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        this.fileQueue.clear();
    }

    private FileFilter createFileFilter(ProcessContext context, final Path inputDirectory) {
        final long minSize = context.getProperty(MIN_SIZE).asDataSize(DataUnit.B).longValue();
        final Double maxSize = context.getProperty(MAX_SIZE).asDataSize(DataUnit.B);
        final long minAge = context.getProperty(MIN_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        final Long maxAge = context.getProperty(MAX_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        final boolean ignoreHidden = context.getProperty(IGNORE_HIDDEN_FILES).asBoolean();
        final Pattern filePattern = Pattern.compile(context.getProperty(FILE_FILTER).getValue());
        boolean recurseDirs = context.getProperty(RECURSE).asBoolean();
        String pathPatternStr = context.getProperty(PATH_FILTER).getValue();
        final Pattern pathPattern = !recurseDirs || pathPatternStr == null ? null : Pattern.compile(pathPatternStr);
        final boolean keepOriginal = context.getProperty(KEEP_SOURCE_FILE).asBoolean();
        return new FileFilter(){

            @Override
            public boolean accept(File file) {
                if (minSize > file.length()) {
                    return false;
                }
                if (maxSize != null && maxSize < (double)file.length()) {
                    return false;
                }
                long fileAge = System.currentTimeMillis() - file.lastModified();
                if (minAge > fileAge) {
                    return false;
                }
                if (maxAge != null && maxAge < fileAge) {
                    return false;
                }
                if (ignoreHidden && file.isHidden()) {
                    return false;
                }
                if (pathPattern != null) {
                    Path relativePath = inputDirectory.relativize(file.toPath()).getParent();
                    if (relativePath == null || relativePath.toString().isEmpty()) {
                        return false;
                    }
                    if (!pathPattern.matcher(relativePath.toString()).matches()) {
                        return false;
                    }
                }
                if (!Files.isReadable(file.toPath())) {
                    return false;
                }
                if (!keepOriginal && !Files.isWritable(file.toPath().getParent())) {
                    return false;
                }
                return filePattern.matcher(file.getName()).matches();
            }
        };
    }

    private Set<File> performListing(File directory, FileFilter filter, boolean recurseSubdirectories) {
        Path p = directory.toPath();
        if (!Files.isWritable(p) || !Files.isReadable(p)) {
            throw new IllegalStateException("Directory '" + String.valueOf(directory) + "' does not have sufficient permissions (i.e., not writable and readable)");
        }
        HashSet<File> queue = new HashSet<File>();
        if (!directory.exists()) {
            return queue;
        }
        File[] children = directory.listFiles();
        if (children == null) {
            return queue;
        }
        for (File child : children) {
            if (child.isDirectory()) {
                if (!recurseSubdirectories) continue;
                queue.addAll(this.performListing(child, filter, recurseSubdirectories));
                continue;
            }
            if (!filter.accept(child)) continue;
            queue.add(child);
        }
        return queue;
    }

    protected Map<String, String> getAttributesFromFile(Path file) {
        HashMap<String, String> attributes = new HashMap<String, String>();
        try {
            FileOwnerAttributeView view3;
            FileStore store = Files.getFileStore(file);
            if (store.supportsFileAttributeView("basic")) {
                try {
                    DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(FILE_MODIFY_DATE_ATTR_FORMAT, Locale.US);
                    BasicFileAttributeView view2 = Files.getFileAttributeView(file, BasicFileAttributeView.class, new LinkOption[0]);
                    BasicFileAttributes attrs = view2.readAttributes();
                    attributes.put(FILE_LAST_MODIFY_TIME_ATTRIBUTE, dateTimeFormatter.format(attrs.lastModifiedTime().toInstant().atZone(ZoneId.systemDefault())));
                    attributes.put(FILE_CREATION_TIME_ATTRIBUTE, dateTimeFormatter.format(attrs.creationTime().toInstant().atZone(ZoneId.systemDefault())));
                    attributes.put(FILE_LAST_ACCESS_TIME_ATTRIBUTE, dateTimeFormatter.format(attrs.lastAccessTime().toInstant().atZone(ZoneId.systemDefault())));
                }
                catch (Exception dateTimeFormatter) {
                    // empty catch block
                }
            }
            if (store.supportsFileAttributeView("owner")) {
                try {
                    view3 = Files.getFileAttributeView(file, FileOwnerAttributeView.class, new LinkOption[0]);
                    attributes.put(FILE_OWNER_ATTRIBUTE, view3.getOwner().getName());
                }
                catch (Exception view3) {
                    // empty catch block
                }
            }
            if (store.supportsFileAttributeView("posix")) {
                try {
                    view3 = Files.getFileAttributeView(file, PosixFileAttributeView.class, new LinkOption[0]);
                    attributes.put(FILE_PERMISSIONS_ATTRIBUTE, PosixFilePermissions.toString(view3.readAttributes().permissions()));
                    attributes.put(FILE_GROUP_ATTRIBUTE, view3.readAttributes().group().getName());
                }
                catch (Exception exception) {}
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return attributes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        File directory = new File(context.getProperty(DIRECTORY).evaluateAttributeExpressions().getValue());
        boolean keepingSourceFile = context.getProperty(KEEP_SOURCE_FILE).asBoolean();
        ComponentLog logger = this.getLogger();
        FileFilter fileFilter = this.createFileFilter(context, directory.toPath());
        if (this.fileQueue.size() < 100) {
            long pollingMillis = context.getProperty(POLLING_INTERVAL).asTimePeriod(TimeUnit.MILLISECONDS);
            if (this.queueLastUpdated.get() < System.currentTimeMillis() - pollingMillis && this.listingLock.tryLock()) {
                try {
                    Set<File> listing = this.performListing(directory, fileFilter, context.getProperty(RECURSE).asBoolean());
                    this.queueLock.lock();
                    try {
                        listing.removeAll(this.inProcess);
                        if (!keepingSourceFile) {
                            listing.removeAll(this.recentlyProcessed);
                        }
                        this.fileQueue.clear();
                        this.fileQueue.addAll(listing);
                        this.queueLastUpdated.set(System.currentTimeMillis());
                        this.recentlyProcessed.clear();
                        if (listing.isEmpty()) {
                            context.yield();
                        }
                    }
                    finally {
                        this.queueLock.unlock();
                    }
                }
                finally {
                    this.listingLock.unlock();
                }
            }
        }
        int batchSize = context.getProperty(BATCH_SIZE).asInteger();
        ArrayList files = new ArrayList(batchSize);
        this.queueLock.lock();
        try {
            this.fileQueue.drainTo(files, batchSize);
            if (files.isEmpty()) {
                return;
            }
            this.inProcess.addAll(files);
        }
        finally {
            this.queueLock.unlock();
        }
        ListIterator itr = files.listIterator();
        FlowFile flowFile = null;
        try {
            Path directoryPath = directory.toPath();
            while (itr.hasNext()) {
                File file = (File)itr.next();
                Path filePath = file.toPath();
                Path relativePath = directoryPath.relativize(filePath.getParent());
                String relativePathString = relativePath.toString() + "/";
                Path absPath = filePath.toAbsolutePath();
                String absPathString = absPath.getParent().toString() + "/";
                flowFile = session.create();
                long importStart = System.nanoTime();
                flowFile = session.importFrom(filePath, keepingSourceFile, flowFile);
                long importNanos = System.nanoTime() - importStart;
                long importMillis = TimeUnit.MILLISECONDS.convert(importNanos, TimeUnit.NANOSECONDS);
                flowFile = session.putAttribute(flowFile, CoreAttributes.FILENAME.key(), file.getName());
                flowFile = session.putAttribute(flowFile, CoreAttributes.PATH.key(), relativePathString);
                flowFile = session.putAttribute(flowFile, CoreAttributes.ABSOLUTE_PATH.key(), absPathString);
                Map<String, String> attributes = this.getAttributesFromFile(filePath);
                if (!attributes.isEmpty()) {
                    flowFile = session.putAllAttributes(flowFile, attributes);
                }
                session.getProvenanceReporter().receive(flowFile, file.toURI().toString(), importMillis);
                session.transfer(flowFile, REL_SUCCESS);
                logger.info("added {} to flow", new Object[]{flowFile});
                if (this.isScheduled()) continue;
                this.queueLock.lock();
                try {
                    while (itr.hasNext()) {
                        File nextFile = (File)itr.next();
                        this.fileQueue.add(nextFile);
                        this.inProcess.remove(nextFile);
                    }
                }
                finally {
                    this.queueLock.unlock();
                }
            }
        }
        catch (Exception e) {
            logger.error("Failed to retrieve files due to {}", (Throwable)e);
            if (flowFile != null) {
                session.remove(flowFile);
            }
        }
        finally {
            this.queueLock.lock();
            try {
                this.inProcess.removeAll(files);
                this.recentlyProcessed.addAll(files);
            }
            finally {
                this.queueLock.unlock();
            }
        }
    }
}

