/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.snmp.processors;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.Validator;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.migration.PropertyConfiguration;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.snmp.dto.SNMPSingleResponse;
import org.apache.nifi.snmp.dto.SNMPTreeResponse;
import org.apache.nifi.snmp.exception.SNMPWalkException;
import org.apache.nifi.snmp.operations.GetSNMPHandler;
import org.apache.nifi.snmp.processors.AbstractSNMPProcessor;
import org.apache.nifi.snmp.processors.properties.BasicProperties;
import org.apache.nifi.snmp.processors.properties.V3SecurityProperties;
import org.apache.nifi.snmp.validators.OIDValidator;
import org.snmp4j.Target;

@Tags(value={"snmp", "get", "oid", "walk"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_ALLOWED)
@CapabilityDescription(value="Retrieves information from SNMP Agent with SNMP Get request and outputs a FlowFile with information in attributes and without any content")
@WritesAttributes(value={@WritesAttribute(attribute="snmp$<OID>", description="Response variable binding: OID (e.g. 1.3.6.1.4.1.343) and its value."), @WritesAttribute(attribute="snmp$errorIndex", description="Denotes the variable binding in which the error occured."), @WritesAttribute(attribute="snmp$errorStatus", description="The snmp4j error status of the PDU."), @WritesAttribute(attribute="snmp$errorStatusText", description="The description of error status."), @WritesAttribute(attribute="snmp$nonRepeaters", description="The number of non repeater variable bindings in a GETBULK PDU (currently not supported)."), @WritesAttribute(attribute="snmp$requestID", description="The request ID associated with the PDU."), @WritesAttribute(attribute="snmp$type", description="The snmp4j numeric representation of the type of the PDU."), @WritesAttribute(attribute="snmp$typeString", description="The name of the PDU type."), @WritesAttribute(attribute="snmp$textualOid", description="This attribute will exist if and only if the strategy is GET and will be equal to the value given in Textual Oid property.")})
public class GetSNMP
extends AbstractSNMPProcessor {
    public static final AllowableValue GET = new AllowableValue("GET", "GET", "A manager-to-agent request to retrieve the value of a variable. A response with the current value returned.");
    public static final AllowableValue WALK = new AllowableValue("WALK", "WALK", "A manager-to-agent request to retrieve the value of multiple variables. Snmp WALK also traverses all subnodes under the specified OID.");
    public static final PropertyDescriptor OID = new PropertyDescriptor.Builder().name("OID").description("Each OID (object identifier) identifies a variable that can be read or set via SNMP. This value is not taken into account for an input flowfile and will be omitted. Can be set to emptystring when the OIDs are provided through flowfile.").addValidator((Validator)new OIDValidator()).build();
    public static final PropertyDescriptor SNMP_STRATEGY = new PropertyDescriptor.Builder().name("SNMP Strategy").description("SNMP strategy to use (SNMP Get or SNMP Walk)").required(true).allowableValues(new DescribedValue[]{GET, WALK}).defaultValue((DescribedValue)GET).build();
    public static final PropertyDescriptor TEXTUAL_OID = new PropertyDescriptor.Builder().name("Textual OID").description("The textual form of the numeric OID to request. This property is user defined, not processed and appended to the outgoing flowfile.").required(false).addValidator(StandardValidators.NON_BLANK_VALIDATOR).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("All FlowFiles that are received from the SNMP agent are routed to this relationship.").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("All FlowFiles that cannot received from the SNMP agent are routed to this relationship.").build();
    protected static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(AGENT_HOST, AGENT_PORT, BasicProperties.SNMP_VERSION, BasicProperties.SNMP_COMMUNITY, V3SecurityProperties.SNMP_SECURITY_LEVEL, V3SecurityProperties.SNMP_SECURITY_NAME, V3SecurityProperties.SNMP_AUTH_PROTOCOL, V3SecurityProperties.SNMP_AUTH_PASSWORD, V3SecurityProperties.SNMP_PRIVACY_PROTOCOL, V3SecurityProperties.SNMP_PRIVACY_PASSWORD, BasicProperties.SNMP_RETRIES, BasicProperties.SNMP_TIMEOUT, OID, TEXTUAL_OID, SNMP_STRATEGY);
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS, REL_FAILURE);
    private volatile GetSNMPHandler snmpHandler;

    @OnScheduled
    public void init(ProcessContext context) {
        this.initSnmpManager(context);
        this.snmpHandler = new GetSNMPHandler(this.snmpManager);
    }

    public void onTrigger(ProcessContext context, ProcessSession processSession) {
        boolean isNewFlowFileCreated;
        SNMPStrategy snmpStrategy = SNMPStrategy.valueOf(context.getProperty(SNMP_STRATEGY).getValue());
        String oid = context.getProperty(OID).getValue();
        FlowFile flowfile = processSession.get();
        if (flowfile == null) {
            isNewFlowFileCreated = true;
            flowfile = processSession.create();
        } else {
            isNewFlowFileCreated = false;
        }
        Target target = this.factory.createTargetInstance(this.getTargetConfiguration(context, flowfile));
        if (SNMPStrategy.GET == snmpStrategy) {
            this.performSnmpGet(context, processSession, oid, target, flowfile, isNewFlowFileCreated);
        } else if (SNMPStrategy.WALK == snmpStrategy) {
            this.performSnmpWalk(context, processSession, oid, target, flowfile, isNewFlowFileCreated);
        }
    }

    @Override
    public void migrateProperties(PropertyConfiguration config) {
        super.migrateProperties(config);
        config.renameProperty("snmp-oid", OID.getName());
        config.renameProperty("snmp-strategy", SNMP_STRATEGY.getName());
        config.renameProperty("snmp-textual-oid", TEXTUAL_OID.getName());
        config.renameProperty("snmp-version", BasicProperties.SNMP_VERSION.getName());
        config.renameProperty("snmp-community", BasicProperties.SNMP_COMMUNITY.getName());
        config.renameProperty("snmp-retries", BasicProperties.SNMP_RETRIES.getName());
        BasicProperties.OLD_SNMP_TIMEOUT_PROPERTY_NAMES.forEach(oldPropertyName -> config.renameProperty(oldPropertyName, BasicProperties.SNMP_TIMEOUT.getName()));
        config.renameProperty("snmp-security-level", V3SecurityProperties.SNMP_SECURITY_LEVEL.getName());
        config.renameProperty("snmp-security-name", V3SecurityProperties.SNMP_SECURITY_NAME.getName());
        config.renameProperty("snmp-authentication-protocol", V3SecurityProperties.SNMP_AUTH_PROTOCOL.getName());
        config.renameProperty("snmp-authentication-passphrase", V3SecurityProperties.SNMP_AUTH_PASSWORD.getName());
        config.renameProperty("snmp-private-protocol", V3SecurityProperties.SNMP_PRIVACY_PROTOCOL.getName());
        config.renameProperty("snmp-private-protocol-passphrase", V3SecurityProperties.SNMP_PRIVACY_PASSWORD.getName());
    }

    void performSnmpWalk(ProcessContext context, ProcessSession processSession, String oid, Target target, FlowFile flowFile, boolean isNewFlowFileCreated) {
        if (oid != null) {
            String prefixedOid = "snmp$" + oid;
            flowFile = processSession.putAttribute(flowFile, prefixedOid, "");
        }
        try {
            Optional<SNMPTreeResponse> optionalResponse = this.snmpHandler.walk(flowFile.getAttributes(), target);
            if (optionalResponse.isPresent()) {
                SNMPTreeResponse response = optionalResponse.get();
                response.logErrors(this.getLogger());
                flowFile = processSession.putAllAttributes(flowFile, response.getAttributes());
                if (isNewFlowFileCreated) {
                    processSession.getProvenanceReporter().receive(flowFile, "/walk");
                } else {
                    processSession.getProvenanceReporter().fetch(flowFile, "/walk");
                }
                processSession.transfer(flowFile, response.isError() ? REL_FAILURE : REL_SUCCESS);
            } else {
                this.getLogger().warn("No SNMP specific attributes found in flowfile.");
                processSession.transfer(flowFile, REL_FAILURE);
            }
        }
        catch (SNMPWalkException e) {
            this.getLogger().error(e.getMessage());
            context.yield();
        }
    }

    void performSnmpGet(ProcessContext context, ProcessSession processSession, String oid, Target target, FlowFile flowFile, boolean isNewFlowFileCreated) {
        String textualOidKey = "snmp$textualOid";
        Map<String, String> textualOidMap = Collections.singletonMap("snmp$textualOid", context.getProperty(TEXTUAL_OID).getValue());
        if (oid != null) {
            String prefixedOid = "snmp$" + oid;
            flowFile = processSession.putAttribute(flowFile, prefixedOid, "");
        }
        try {
            Optional<SNMPSingleResponse> optionalResponse = this.snmpHandler.get(flowFile.getAttributes(), target);
            if (optionalResponse.isPresent()) {
                SNMPSingleResponse response = optionalResponse.get();
                flowFile = processSession.putAllAttributes(flowFile, textualOidMap);
                this.handleResponse(context, processSession, flowFile, response, REL_SUCCESS, REL_FAILURE, "/get", isNewFlowFileCreated);
            } else {
                this.getLogger().warn("No SNMP specific attributes found in flowfile.");
                processSession.transfer(flowFile, REL_FAILURE);
                context.yield();
            }
        }
        catch (IOException e) {
            this.getLogger().error("Failed to send request to the agent. Check if the agent supports the used version.", (Throwable)e);
            context.yield();
        }
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    @Override
    protected String getTargetHost(ProcessContext processContext, FlowFile flowFile) {
        return processContext.getProperty(AGENT_HOST).evaluateAttributeExpressions(flowFile).getValue();
    }

    @Override
    protected String getTargetPort(ProcessContext processContext, FlowFile flowFile) {
        return processContext.getProperty(AGENT_PORT).evaluateAttributeExpressions(flowFile).getValue();
    }

    private static enum SNMPStrategy {
        GET,
        WALK;

    }
}

