/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.services.protobuf;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.squareup.wire.schema.Location;
import com.squareup.wire.schema.Schema;
import com.squareup.wire.schema.SchemaLoader;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.schemaregistry.services.SchemaDefinition;
import org.apache.nifi.serialization.record.SchemaIdentifier;
import org.apache.nifi.services.protobuf.ProtobufSchemaValidator;
import org.apache.nifi.services.protobuf.SchemaCompilationException;

final class ProtobufSchemaCompiler {
    private static final List<Location> STANDARD_PROTOBUF_LOCATIONS = Arrays.asList(Location.get((String)"wire-runtime.jar", (String)"google/protobuf/any.proto"), Location.get((String)"wire-runtime.jar", (String)"google/protobuf/duration.proto"), Location.get((String)"wire-runtime.jar", (String)"google/protobuf/empty.proto"), Location.get((String)"wire-runtime.jar", (String)"google/protobuf/struct.proto"), Location.get((String)"wire-runtime.jar", (String)"google/protobuf/timestamp.proto"), Location.get((String)"wire-runtime.jar", (String)"google/protobuf/wrappers.proto"));
    private static final int CACHE_EXPIRE_HOURS = 1;
    private static final int COMPILED_SCHEMAS_CACHE_SIZE = 200;
    private static final String PROTO_EXTENSION = ".proto";
    private final Cache<SchemaIdentifier, Schema> compiledSchemaCache;
    private final ComponentLog logger;
    private final String tempDirectorySuffix;

    public ProtobufSchemaCompiler(String tempDirectorySuffix, ComponentLog logger) {
        this.tempDirectorySuffix = Objects.requireNonNull(tempDirectorySuffix, "Temporary directory suffix cannot be null");
        this.logger = logger;
        this.compiledSchemaCache = Caffeine.newBuilder().expireAfterAccess(1L, TimeUnit.HOURS).maximumSize(200L).build();
    }

    public Schema compileOrGetFromCache(SchemaDefinition schemaDefinition) {
        return (Schema)this.compiledSchemaCache.get((Object)schemaDefinition.getIdentifier(), identifier -> {
            try {
                return this.compileSchemaDefinition(schemaDefinition);
            }
            catch (IOException e) {
                throw new UncheckedIOException("Could not compile schema for identifier: " + String.valueOf(identifier), e);
            }
        });
    }

    private Schema compileSchemaDefinition(SchemaDefinition schemaDefinition) throws IOException {
        this.logger.debug("Starting schema compilation for identifier: {}", new Object[]{schemaDefinition.getIdentifier()});
        ProtobufSchemaValidator.validateSchemaDefinitionIdentifiers(schemaDefinition, true);
        return this.executeWithTemporaryDirectory(tempDir -> {
            try {
                this.writeSchemaToTempDirectory(tempDir, schemaDefinition);
                this.processSchemaReferences(tempDir, schemaDefinition.getReferences());
                Schema compiledSchema = this.createAndLoadSchema(tempDir);
                this.logger.debug("Successfully compiled schema for identifier: {}", new Object[]{schemaDefinition.getIdentifier()});
                return compiledSchema;
            }
            catch (IllegalStateException e) {
                throw new SchemaCompilationException("Could not compile schema: %s".formatted(schemaDefinition.toString()), e);
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to compile Protobuf schema for identifier: " + String.valueOf(schemaDefinition.getIdentifier()), e);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T executeWithTemporaryDirectory(WithTemporaryDirectory<T> function) throws IOException {
        Path tempDir = Files.createTempDirectory(this.tempDirectorySuffix + "_protobuf_schema_compiler", new FileAttribute[0]);
        this.logger.debug("Created temporary directory for schema compilation: {}", new Object[]{tempDir});
        try {
            T t = function.apply(tempDir);
            return t;
        }
        finally {
            this.safeDeleteDirectory(tempDir);
        }
    }

    private Schema createAndLoadSchema(Path tempDir) {
        SchemaLoader schemaLoader = new SchemaLoader(FileSystems.getDefault());
        ArrayList<Location> roots = new ArrayList<Location>();
        roots.add(Location.get((String)tempDir.toString()));
        roots.addAll(STANDARD_PROTOBUF_LOCATIONS);
        schemaLoader.initRoots(roots, Collections.emptyList());
        return schemaLoader.loadSchema();
    }

    private void safeDeleteDirectory(Path directory) {
        if (Files.exists(directory, new LinkOption[0])) {
            try {
                FileUtils.deleteDirectory((File)directory.toFile());
            }
            catch (IOException | IllegalArgumentException e) {
                this.logger.warn("Failed to delete temporary directory: {}", new Object[]{directory, e});
            }
        }
    }

    private void writeSchemaToTempDirectory(Path tempDir, SchemaDefinition schemaDefinition) throws IOException {
        this.logger.debug("Writing schema definition to temporary directory. Identifier: {}", new Object[]{schemaDefinition.getIdentifier()});
        String schemaFileName = this.generateSchemaFileName(schemaDefinition);
        Path schemaFile = tempDir.resolve(schemaFileName);
        Files.write(schemaFile, schemaDefinition.getText().getBytes(), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);
        this.logger.debug("Successfully wrote schema to file: {} (string length: {})", new Object[]{schemaFile, schemaDefinition.getText().length()});
    }

    private String generateSchemaFileName(SchemaDefinition schemaDefinition) {
        Object schemaFileName = schemaDefinition.getIdentifier().getName().orElseGet(() -> String.valueOf(schemaDefinition.getIdentifier().getSchemaVersionId().orElse(0L)));
        if (!((String)schemaFileName).endsWith(PROTO_EXTENSION)) {
            schemaFileName = (String)schemaFileName + PROTO_EXTENSION;
        }
        return schemaFileName;
    }

    private void processSchemaReferences(Path tempDir, Map<String, SchemaDefinition> references) throws IOException {
        this.logger.debug("Processing [{}] schema references in [{}]", new Object[]{references.size(), tempDir});
        for (Map.Entry<String, SchemaDefinition> entry : references.entrySet()) {
            String referenceKey = entry.getKey();
            SchemaDefinition referencedSchema = entry.getValue();
            this.logger.debug("Processing schema reference [{}] Identifier [{}]", new Object[]{referenceKey, referencedSchema.getIdentifier()});
            this.writeSchemaToTempDirectory(tempDir, referencedSchema);
            if (!referencedSchema.getReferences().isEmpty()) {
                this.logger.debug("Processing {} nested references for schema reference: {}", new Object[]{referencedSchema.getReferences().size(), referenceKey});
                this.processSchemaReferences(tempDir, referencedSchema.getReferences());
                continue;
            }
            this.logger.debug("No nested references found for schema reference: {}", new Object[]{referenceKey});
        }
    }

    @FunctionalInterface
    private static interface WithTemporaryDirectory<T> {
        public T apply(Path var1);
    }
}

