/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.kafka.processors;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.ConfigVerificationResult;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.PropertyValue;
import org.apache.nifi.components.Validator;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.kafka.processors.PublishKafka;
import org.apache.nifi.kafka.processors.common.KafkaUtils;
import org.apache.nifi.kafka.processors.consumer.OffsetTracker;
import org.apache.nifi.kafka.processors.consumer.ProcessingStrategy;
import org.apache.nifi.kafka.processors.consumer.bundle.ByteRecordBundler;
import org.apache.nifi.kafka.processors.consumer.convert.AbstractRecordStreamKafkaMessageConverter;
import org.apache.nifi.kafka.processors.consumer.convert.FlowFileStreamKafkaMessageConverter;
import org.apache.nifi.kafka.processors.consumer.convert.InjectOffsetRecordStreamKafkaMessageConverter;
import org.apache.nifi.kafka.processors.consumer.convert.RecordStreamKafkaMessageConverter;
import org.apache.nifi.kafka.processors.consumer.convert.WrapperRecordStreamKafkaMessageConverter;
import org.apache.nifi.kafka.service.api.KafkaConnectionService;
import org.apache.nifi.kafka.service.api.consumer.AutoOffsetReset;
import org.apache.nifi.kafka.service.api.consumer.KafkaConsumerService;
import org.apache.nifi.kafka.service.api.consumer.PollingContext;
import org.apache.nifi.kafka.service.api.record.ByteRecord;
import org.apache.nifi.kafka.shared.property.KeyEncoding;
import org.apache.nifi.kafka.shared.property.KeyFormat;
import org.apache.nifi.kafka.shared.property.OutputStrategy;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.VerifiableProcessor;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.serialization.RecordReaderFactory;
import org.apache.nifi.serialization.RecordSetWriterFactory;
import org.apache.nifi.util.StringUtils;

@CapabilityDescription(value="Consumes messages from Apache Kafka Consumer API. The complementary NiFi processor for sending messages is PublishKafka. The Processor supports consumption of Kafka messages, optionally interpreted as NiFi records. Please note that, at this time (in read record mode), the Processor assumes that all records that are retrieved from a given partition have the same schema. For this mode, if any of the Kafka messages are pulled but cannot be parsed or written with the configured Record Reader or Record Writer, the contents of the message will be written to a separate FlowFile, and that FlowFile will be transferred to the 'parse.failure' relationship. Otherwise, each FlowFile is sent to the 'success' relationship and may contain many individual messages within the single FlowFile. A 'record.count' attribute is added to indicate how many messages are contained in the FlowFile. No two Kafka messages will be placed into the same FlowFile if they have different schemas, or if they have different values for a message header that is included by the <Headers to Add as Attributes> property.")
@Tags(value={"Kafka", "Get", "Record", "csv", "avro", "json", "Ingest", "Ingress", "Topic", "PubSub", "Consume"})
@WritesAttributes(value={@WritesAttribute(attribute="record.count", description="The number of records received"), @WritesAttribute(attribute="mime.type", description="The MIME Type that is provided by the configured Record Writer"), @WritesAttribute(attribute="kafka.count", description="The number of records in the FlowFile for a batch of records"), @WritesAttribute(attribute="kafka.key", description="The key of message if present and if single message. How the key is encoded depends on the value of the 'Key Attribute Encoding' property."), @WritesAttribute(attribute="kafka.offset", description="The offset of the record in the partition or the minimum value of the offset in a batch of records"), @WritesAttribute(attribute="kafka.timestamp", description="The timestamp of the message in the partition of the topic."), @WritesAttribute(attribute="kafka.partition", description="The partition of the topic for a record or batch of records"), @WritesAttribute(attribute="kafka.topic", description="The topic the for a record or batch of records"), @WritesAttribute(attribute="kafka.tombstone", description="Set to true if the consumed message is a tombstone message"), @WritesAttribute(attribute="kafka.max.offset", description="The maximum value of the Kafka offset in batch of records")})
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@SeeAlso(value={PublishKafka.class})
public class ConsumeKafka
extends AbstractProcessor
implements VerifiableProcessor {
    static final AllowableValue TOPIC_NAME = new AllowableValue("names", "names", "Topic is a full topic name or comma separated list of names");
    static final AllowableValue TOPIC_PATTERN = new AllowableValue("pattern", "pattern", "Topic is a regular expression according to the Java Pattern syntax");
    static final PropertyDescriptor CONNECTION_SERVICE = new PropertyDescriptor.Builder().name("Kafka Connection Service").description("Provides connections to Kafka Broker for publishing Kafka Records").identifiesControllerService(KafkaConnectionService.class).expressionLanguageSupported(ExpressionLanguageScope.NONE).required(true).build();
    static final PropertyDescriptor GROUP_ID = new PropertyDescriptor.Builder().name("Group ID").description("Kafka Consumer Group Identifier corresponding to Kafka group.id property").required(true).addValidator(StandardValidators.NON_BLANK_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    static final PropertyDescriptor TOPIC_FORMAT = new PropertyDescriptor.Builder().name("Topic Format").description("Specifies whether the Topics provided are a comma separated list of names or a single regular expression").required(true).allowableValues(new DescribedValue[]{TOPIC_NAME, TOPIC_PATTERN}).defaultValue((DescribedValue)TOPIC_NAME).build();
    static final PropertyDescriptor TOPICS = new PropertyDescriptor.Builder().name("Topics").description("The name or pattern of the Kafka Topics from which the Processor consumes Kafka Records. More than one can be supplied if comma separated.").required(true).addValidator(StandardValidators.NON_BLANK_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).build();
    static final PropertyDescriptor AUTO_OFFSET_RESET = new PropertyDescriptor.Builder().name("auto.offset.reset").displayName("Auto Offset Reset").description("Automatic offset configuration applied when no previous consumer offset found corresponding to Kafka auto.offset.reset property").required(true).allowableValues(AutoOffsetReset.class).defaultValue((DescribedValue)AutoOffsetReset.LATEST).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    static final PropertyDescriptor COMMIT_OFFSETS = new PropertyDescriptor.Builder().name("Commit Offsets").description("Specifies whether this Processor should commit the offsets to Kafka after receiving messages. Typically, this value should be set to true so that messages that are received are not duplicated. However, in certain scenarios, we may want to avoid committing the offsets, that the data can be processed and later acknowledged by PublishKafka in order to provide Exactly Once semantics.").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").build();
    static final PropertyDescriptor MAX_UNCOMMITTED_SIZE = new PropertyDescriptor.Builder().name("Max Uncommitted Size").description("Maximum total size of records to consume from Kafka before transferring FlowFiles to an output\nrelationship. Evaluated when specified based on the size of serialized keys and values from each\nKafka record, before reaching the [Max Uncommitted Time].\n").required(false).addValidator(StandardValidators.DATA_SIZE_VALIDATOR).build();
    static final PropertyDescriptor MAX_UNCOMMITTED_TIME = new PropertyDescriptor.Builder().name("Max Uncommitted Time").description("Maximum amount of time to spend consuming records from Kafka before transferring FlowFiles to an\noutput relationship. Longer amounts of time may produce larger FlowFiles and increase processing\nlatency for individual records.\n").required(true).defaultValue("100 millis").addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).build();
    static final PropertyDescriptor HEADER_ENCODING = new PropertyDescriptor.Builder().name("Header Encoding").description("Character encoding applied when reading Kafka Record Header values and writing FlowFile attributes").addValidator(StandardValidators.CHARACTER_SET_VALIDATOR).defaultValue(StandardCharsets.UTF_8.name()).required(true).build();
    static final PropertyDescriptor HEADER_NAME_PATTERN = new PropertyDescriptor.Builder().name("Header Name Pattern").description("Regular Expression Pattern applied to Kafka Record Header Names for selecting Header Values to be written as FlowFile attributes").addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).required(false).build();
    static final PropertyDescriptor PROCESSING_STRATEGY = new PropertyDescriptor.Builder().name("Processing Strategy").description("Strategy for processing Kafka Records and writing serialized output to FlowFiles").required(true).allowableValues(ProcessingStrategy.class).defaultValue((DescribedValue)ProcessingStrategy.FLOW_FILE).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    static final PropertyDescriptor RECORD_READER = new PropertyDescriptor.Builder().name("Record Reader").description("The Record Reader to use for incoming Kafka messages").identifiesControllerService(RecordReaderFactory.class).required(true).dependsOn(PROCESSING_STRATEGY, (DescribedValue)ProcessingStrategy.RECORD, new DescribedValue[0]).build();
    static final PropertyDescriptor RECORD_WRITER = new PropertyDescriptor.Builder().name("Record Writer").description("The Record Writer to use in order to serialize the outgoing FlowFiles").identifiesControllerService(RecordSetWriterFactory.class).required(true).dependsOn(PROCESSING_STRATEGY, (DescribedValue)ProcessingStrategy.RECORD, new DescribedValue[0]).build();
    static final PropertyDescriptor OUTPUT_STRATEGY = new PropertyDescriptor.Builder().name("Output Strategy").description("The format used to output the Kafka Record into a FlowFile Record.").required(true).defaultValue((DescribedValue)OutputStrategy.USE_VALUE).allowableValues(OutputStrategy.class).dependsOn(PROCESSING_STRATEGY, (DescribedValue)ProcessingStrategy.RECORD, new DescribedValue[0]).build();
    static final PropertyDescriptor KEY_ATTRIBUTE_ENCODING = new PropertyDescriptor.Builder().name("Key Attribute Encoding").description("Encoding for value of configured FlowFile attribute containing Kafka Record Key.").required(true).defaultValue((DescribedValue)KeyEncoding.UTF8).allowableValues(KeyEncoding.class).build();
    static final PropertyDescriptor KEY_FORMAT = new PropertyDescriptor.Builder().name("Key Format").description("Specifies how to represent the Kafka Record Key in the output FlowFile").required(true).defaultValue((DescribedValue)KeyFormat.BYTE_ARRAY).allowableValues(KeyFormat.class).dependsOn(OUTPUT_STRATEGY, (DescribedValue)OutputStrategy.USE_WRAPPER, new DescribedValue[]{OutputStrategy.INJECT_METADATA}).build();
    static final PropertyDescriptor KEY_RECORD_READER = new PropertyDescriptor.Builder().name("Key Record Reader").description("The Record Reader to use for parsing the Kafka Record Key into a Record").identifiesControllerService(RecordReaderFactory.class).expressionLanguageSupported(ExpressionLanguageScope.NONE).required(true).dependsOn(KEY_FORMAT, (DescribedValue)KeyFormat.RECORD, new DescribedValue[0]).build();
    static final PropertyDescriptor MESSAGE_DEMARCATOR = new PropertyDescriptor.Builder().name("Message Demarcator").required(true).addValidator(Validator.VALID).description("Since KafkaConsumer receives messages in batches, this Processor has an option to output FlowFiles which contains all Kafka messages in a single batch for a given topic and partition and this property allows you to provide a string (interpreted as UTF-8) to use for demarcating apart multiple Kafka messages. This is an optional property and if not provided each Kafka message received will result in a single FlowFile which  time it is triggered. To enter special character such as 'new line' use CTRL+Enter or Shift+Enter depending on the OS").dependsOn(PROCESSING_STRATEGY, (DescribedValue)ProcessingStrategy.DEMARCATOR, new DescribedValue[0]).build();
    static final PropertyDescriptor SEPARATE_BY_KEY = new PropertyDescriptor.Builder().name("Separate By Key").description("When this property is enabled, two messages will only be added to the same FlowFile if both of the Kafka Messages have identical keys.").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("false").dependsOn(MESSAGE_DEMARCATOR, new AllowableValue[0]).build();
    public static final Relationship SUCCESS = new Relationship.Builder().name("success").description("FlowFiles containing one or more serialized Kafka Records").build();
    public static final Relationship PARSE_FAILURE = new Relationship.Builder().name("parse failure").description("If configured to use a Record Reader, a Kafka message that cannot be parsed using the configured Record Reader will be routed to this relationship").build();
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(CONNECTION_SERVICE, GROUP_ID, TOPIC_FORMAT, TOPICS, AUTO_OFFSET_RESET, COMMIT_OFFSETS, MAX_UNCOMMITTED_SIZE, MAX_UNCOMMITTED_TIME, HEADER_NAME_PATTERN, HEADER_ENCODING, PROCESSING_STRATEGY, RECORD_READER, RECORD_WRITER, OUTPUT_STRATEGY, KEY_ATTRIBUTE_ENCODING, KEY_FORMAT, KEY_RECORD_READER, MESSAGE_DEMARCATOR, SEPARATE_BY_KEY);
    private static final Set<Relationship> SUCCESS_RELATIONSHIP = Set.of(SUCCESS);
    private static final Set<Relationship> SUCCESS_FAILURE_RELATIONSHIPS = Set.of(SUCCESS, PARSE_FAILURE);
    private volatile Charset headerEncoding;
    private volatile Pattern headerNamePattern;
    private volatile ProcessingStrategy processingStrategy;
    private volatile KeyEncoding keyEncoding;
    private volatile OutputStrategy outputStrategy;
    private volatile KeyFormat keyFormat;
    private volatile boolean commitOffsets;
    private volatile boolean useReader;
    private volatile String brokerUri;
    private volatile PollingContext pollingContext;
    private volatile int maxConsumerCount;
    private volatile boolean maxUncommittedSizeConfigured;
    private volatile long maxUncommittedSize;
    private final Queue<KafkaConsumerService> consumerServices = new LinkedBlockingQueue<KafkaConsumerService>();
    private final AtomicInteger activeConsumerCount = new AtomicInteger();

    public List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return this.useReader ? SUCCESS_FAILURE_RELATIONSHIPS : SUCCESS_RELATIONSHIP;
    }

    public void onPropertyModified(PropertyDescriptor descriptor, String oldValue, String newValue) {
        if (descriptor.equals((Object)RECORD_READER)) {
            this.useReader = newValue != null;
        }
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        this.pollingContext = this.createPollingContext(context);
        this.headerEncoding = Charset.forName(context.getProperty(HEADER_ENCODING).getValue());
        String headerNamePatternProperty = context.getProperty(HEADER_NAME_PATTERN).getValue();
        this.headerNamePattern = StringUtils.isNotBlank((String)headerNamePatternProperty) ? Pattern.compile(headerNamePatternProperty) : null;
        this.keyEncoding = (KeyEncoding)context.getProperty(KEY_ATTRIBUTE_ENCODING).asAllowableValue(KeyEncoding.class);
        this.commitOffsets = context.getProperty(COMMIT_OFFSETS).asBoolean();
        this.processingStrategy = (ProcessingStrategy)context.getProperty(PROCESSING_STRATEGY).asAllowableValue(ProcessingStrategy.class);
        this.outputStrategy = this.processingStrategy == ProcessingStrategy.RECORD ? (OutputStrategy)context.getProperty(OUTPUT_STRATEGY).asAllowableValue(OutputStrategy.class) : null;
        this.keyFormat = this.outputStrategy == OutputStrategy.USE_WRAPPER || this.outputStrategy == OutputStrategy.INJECT_METADATA ? (KeyFormat)context.getProperty(KEY_FORMAT).asAllowableValue(KeyFormat.class) : KeyFormat.BYTE_ARRAY;
        this.brokerUri = ((KafkaConnectionService)context.getProperty(CONNECTION_SERVICE).asControllerService(KafkaConnectionService.class)).getBrokerUri();
        this.maxConsumerCount = context.getMaxConcurrentTasks();
        this.activeConsumerCount.set(0);
        PropertyValue maxUncommittedSizeProperty = context.getProperty(MAX_UNCOMMITTED_SIZE);
        this.maxUncommittedSizeConfigured = maxUncommittedSizeProperty.isSet();
        if (this.maxUncommittedSizeConfigured) {
            this.maxUncommittedSize = maxUncommittedSizeProperty.asDataSize(DataUnit.B).longValue();
        }
    }

    @OnStopped
    public void onStopped() {
        KafkaConsumerService service;
        while ((service = this.consumerServices.poll()) != null) {
            this.close(service, "Processor stopped");
        }
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        KafkaConsumerService consumerService = this.getConsumerService(context);
        if (consumerService == null) {
            this.getLogger().debug("No Kafka Consumer Service available; will yield and return immediately");
            context.yield();
            return;
        }
        long maxUncommittedMillis = context.getProperty(MAX_UNCOMMITTED_TIME).asTimePeriod(TimeUnit.MILLISECONDS);
        long stopTime = System.currentTimeMillis() + maxUncommittedMillis;
        OffsetTracker offsetTracker = new OffsetTracker();
        boolean recordsReceived = false;
        while (System.currentTimeMillis() < stopTime) {
            try {
                long totalRecordSize;
                Duration maxWaitDuration = Duration.ofMillis(stopTime - System.currentTimeMillis());
                if (maxWaitDuration.toMillis() <= 0L) break;
                Iterator<ByteRecord> consumerRecords = consumerService.poll(maxWaitDuration).iterator();
                if (!consumerRecords.hasNext()) {
                    this.getLogger().trace("No Kafka Records consumed: {}", new Object[]{this.pollingContext});
                    continue;
                }
                recordsReceived = true;
                this.processConsumerRecords(context, session, offsetTracker, consumerRecords);
                if (!this.maxUncommittedSizeConfigured || (totalRecordSize = offsetTracker.getTotalRecordSize()) <= this.maxUncommittedSize) continue;
                break;
            }
            catch (Exception e) {
                this.getLogger().error("Failed to consume Kafka Records", (Throwable)e);
                consumerService.rollback();
                this.close(consumerService, "Encountered Exception while consuming or writing out Kafka Records");
                context.yield();
                session.rollback();
                return;
            }
        }
        if (!recordsReceived) {
            this.getLogger().trace("No Kafka Records consumed, re-queuing consumer");
            this.consumerServices.offer(consumerService);
            return;
        }
        session.commitAsync(() -> this.commitOffsets(consumerService, offsetTracker, this.pollingContext, session), throwable -> {
            this.getLogger().error("Failed to commit session; will roll back any uncommitted records", throwable);
            this.rollback(consumerService, offsetTracker, session);
            context.yield();
        });
    }

    private void commitOffsets(KafkaConsumerService consumerService, OffsetTracker offsetTracker, PollingContext pollingContext, ProcessSession session) {
        try {
            if (this.commitOffsets) {
                consumerService.commit(offsetTracker.getPollingSummary(pollingContext));
                offsetTracker.getRecordCounts().forEach((topic, count) -> session.adjustCounter("Records Acknowledged for " + topic, count.longValue(), true));
            }
            this.consumerServices.offer(consumerService);
            this.getLogger().debug("Committed offsets for Kafka Consumer Service");
        }
        catch (Exception e) {
            this.getLogger().error("Failed to commit offsets for Kafka Consumer Service; will attempt to rollback to latest committed offsets", (Throwable)e);
            this.rollback(consumerService, offsetTracker, session);
        }
    }

    private void rollback(KafkaConsumerService consumerService, OffsetTracker offsetTracker, ProcessSession session) {
        if (!consumerService.isClosed()) {
            try {
                consumerService.rollback();
                this.consumerServices.offer(consumerService);
                this.getLogger().debug("Rolled back offsets for Kafka Consumer Service");
            }
            catch (Exception e) {
                this.getLogger().warn("Failed to rollback offsets for Kafka Consumer", (Throwable)e);
                this.close(consumerService, "Failed to rollback offsets");
            }
            offsetTracker.getRecordCounts().forEach((topic, count) -> session.adjustCounter("Records Rolled Back for " + topic, count.longValue(), true));
        }
    }

    private void close(KafkaConsumerService consumerService, String reason) {
        if (consumerService.isClosed()) {
            this.getLogger().debug("Asked to close Kafka Consumer Service but consumer already closed");
            return;
        }
        this.getLogger().info("Closing Kafka Consumer due to: {}", new Object[]{reason});
        try {
            consumerService.close();
            this.activeConsumerCount.decrementAndGet();
        }
        catch (IOException ioe) {
            this.getLogger().warn("Failed to close Kafka Consumer Service", (Throwable)ioe);
        }
    }

    public List<ConfigVerificationResult> verify(ProcessContext context, ComponentLog verificationLogger, Map<String, String> attributes) {
        ArrayList<ConfigVerificationResult> verificationResults = new ArrayList<ConfigVerificationResult>();
        KafkaConnectionService connectionService = (KafkaConnectionService)context.getProperty(CONNECTION_SERVICE).asControllerService(KafkaConnectionService.class);
        PollingContext pollingContext = this.createPollingContext(context);
        KafkaConsumerService consumerService = connectionService.getConsumerService(pollingContext);
        ConfigVerificationResult.Builder verificationPartitions = new ConfigVerificationResult.Builder().verificationStepName("Verify Topic Partitions");
        try {
            List partitionStates = consumerService.getPartitionStates();
            verificationPartitions.outcome(ConfigVerificationResult.Outcome.SUCCESSFUL).explanation(String.format("Partitions [%d] found for Topics %s", partitionStates.size(), pollingContext.getTopics()));
        }
        catch (Exception e) {
            this.getLogger().error("Topics {} Partition verification failed", new Object[]{pollingContext.getTopics(), e});
            verificationPartitions.outcome(ConfigVerificationResult.Outcome.FAILED).explanation(String.format("Topics %s Partition access failed: %s", pollingContext.getTopics(), e));
        }
        verificationResults.add(verificationPartitions.build());
        return verificationResults;
    }

    private KafkaConsumerService getConsumerService(ProcessContext context) {
        KafkaConsumerService consumerService = this.consumerServices.poll();
        if (consumerService != null) {
            return consumerService;
        }
        int activeCount = this.activeConsumerCount.incrementAndGet();
        if (activeCount > this.getMaxConsumerCount()) {
            this.getLogger().trace("No Kafka Consumer Service available; have already reached max count of {} so will not create a new one", new Object[]{this.getMaxConsumerCount()});
            this.activeConsumerCount.decrementAndGet();
            return null;
        }
        this.getLogger().info("No Kafka Consumer Service available; creating a new one. Active count: {}", new Object[]{activeCount});
        KafkaConnectionService connectionService = (KafkaConnectionService)context.getProperty(CONNECTION_SERVICE).asControllerService(KafkaConnectionService.class);
        return connectionService.getConsumerService(this.pollingContext);
    }

    private int getMaxConsumerCount() {
        return this.maxConsumerCount;
    }

    private void processConsumerRecords(ProcessContext context, ProcessSession session, OffsetTracker offsetTracker, Iterator<ByteRecord> consumerRecords) {
        switch (this.processingStrategy) {
            case RECORD: {
                this.processInputRecords(context, session, offsetTracker, consumerRecords);
                break;
            }
            case FLOW_FILE: {
                this.processInputFlowFile(session, offsetTracker, consumerRecords);
                break;
            }
            case DEMARCATOR: {
                Iterator<ByteRecord> demarcatedRecords = this.transformDemarcator(context, consumerRecords);
                this.processInputFlowFile(session, offsetTracker, demarcatedRecords);
            }
        }
    }

    private Iterator<ByteRecord> transformDemarcator(ProcessContext context, Iterator<ByteRecord> consumerRecords) {
        String demarcatorValue = context.getProperty(MESSAGE_DEMARCATOR).getValue();
        if (demarcatorValue == null) {
            return consumerRecords;
        }
        byte[] demarcator = demarcatorValue.getBytes(StandardCharsets.UTF_8);
        boolean separateByKey = context.getProperty(SEPARATE_BY_KEY).asBoolean();
        return new ByteRecordBundler(demarcator, separateByKey, this.keyEncoding, this.headerNamePattern, this.headerEncoding, this.commitOffsets).bundle(consumerRecords);
    }

    private void processInputRecords(ProcessContext context, ProcessSession session, OffsetTracker offsetTracker, Iterator<ByteRecord> consumerRecords) {
        AbstractRecordStreamKafkaMessageConverter converter;
        RecordReaderFactory readerFactory = (RecordReaderFactory)context.getProperty(RECORD_READER).asControllerService(RecordReaderFactory.class);
        RecordSetWriterFactory writerFactory = (RecordSetWriterFactory)context.getProperty(RECORD_WRITER).asControllerService(RecordSetWriterFactory.class);
        if (this.outputStrategy == OutputStrategy.USE_VALUE) {
            converter = new RecordStreamKafkaMessageConverter(readerFactory, writerFactory, this.headerEncoding, this.headerNamePattern, this.keyEncoding, this.commitOffsets, offsetTracker, this.getLogger(), this.brokerUri);
        } else if (this.outputStrategy == OutputStrategy.INJECT_OFFSET) {
            converter = new InjectOffsetRecordStreamKafkaMessageConverter(readerFactory, writerFactory, this.headerEncoding, this.headerNamePattern, this.keyEncoding, this.commitOffsets, offsetTracker, this.getLogger(), this.brokerUri);
        } else {
            RecordReaderFactory keyReaderFactory = this.keyFormat == KeyFormat.RECORD ? (RecordReaderFactory)context.getProperty(KEY_RECORD_READER).asControllerService(RecordReaderFactory.class) : null;
            converter = new WrapperRecordStreamKafkaMessageConverter(readerFactory, writerFactory, keyReaderFactory, this.headerEncoding, this.headerNamePattern, this.keyFormat, this.keyEncoding, this.commitOffsets, offsetTracker, this.getLogger(), this.brokerUri, this.outputStrategy);
        }
        converter.toFlowFiles(session, consumerRecords);
    }

    private void processInputFlowFile(ProcessSession session, OffsetTracker offsetTracker, Iterator<ByteRecord> consumerRecords) {
        FlowFileStreamKafkaMessageConverter converter = new FlowFileStreamKafkaMessageConverter(this.headerEncoding, this.headerNamePattern, this.keyEncoding, this.commitOffsets, offsetTracker, this.brokerUri);
        converter.toFlowFiles(session, consumerRecords);
    }

    private PollingContext createPollingContext(ProcessContext context) {
        PollingContext pollingContext;
        String groupId = context.getProperty(GROUP_ID).getValue();
        String offsetReset = context.getProperty(AUTO_OFFSET_RESET).getValue();
        AutoOffsetReset autoOffsetReset = AutoOffsetReset.valueOf((String)offsetReset.toUpperCase());
        String topics = context.getProperty(TOPICS).evaluateAttributeExpressions().getValue();
        String topicFormat = context.getProperty(TOPIC_FORMAT).getValue();
        if (topicFormat.equals(TOPIC_PATTERN.getValue())) {
            Pattern topicPattern = Pattern.compile(topics.trim());
            pollingContext = new PollingContext(groupId, topicPattern, autoOffsetReset);
        } else if (topicFormat.equals(TOPIC_NAME.getValue())) {
            Collection<String> topicList = KafkaUtils.toTopicList(topics);
            pollingContext = new PollingContext(groupId, topicList, autoOffsetReset);
        } else {
            throw new ProcessException(String.format("Topic Format [%s] not supported", topicFormat));
        }
        return pollingContext;
    }
}

