/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.kubernetes.leader.election;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.nifi.controller.leader.election.LeaderElectionRole;
import org.apache.nifi.controller.leader.election.LeaderElectionStateChangeListener;
import org.apache.nifi.controller.leader.election.TrackedLeaderElectionManager;
import org.apache.nifi.kubernetes.client.KubernetesClientProvider;
import org.apache.nifi.kubernetes.client.ServiceAccountNamespaceProvider;
import org.apache.nifi.kubernetes.client.StandardKubernetesClientProvider;
import org.apache.nifi.kubernetes.leader.election.command.LeaderElectionCommandProvider;
import org.apache.nifi.kubernetes.leader.election.command.StandardLeaderElectionCommandProvider;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KubernetesLeaderElectionManager
extends TrackedLeaderElectionManager {
    private static final boolean INTERRUPT_ENABLED = true;
    private static final int SERVICE_THREADS = 4;
    private static final Logger logger = LoggerFactory.getLogger(KubernetesLeaderElectionManager.class);
    private static final Map<String, String> ROLE_NAMES;
    private final ExecutorService executorService;
    private final AtomicBoolean started = new AtomicBoolean();
    private final Map<String, Future<?>> roleCommands = new ConcurrentHashMap();
    private final Map<String, ParticipantRegistration> roleRegistrations = new ConcurrentHashMap<String, ParticipantRegistration>();
    private final Map<String, String> roleLeaders = new ConcurrentHashMap<String, String>();
    private final LeaderElectionCommandProvider leaderElectionCommandProvider;
    private final String roleIdPrefix;

    public KubernetesLeaderElectionManager(NiFiProperties nifiProperties) {
        String leasePrefix = nifiProperties.getProperty("nifi.cluster.leader.election.kubernetes.lease.prefix");
        this.roleIdPrefix = leasePrefix == null || leasePrefix.isBlank() ? null : leasePrefix;
        this.executorService = this.createExecutorService();
        this.leaderElectionCommandProvider = this.createLeaderElectionCommandProvider();
    }

    public void start() {
        if (this.started.get()) {
            logger.debug("Start requested when running");
        } else {
            this.started.getAndSet(true);
            logger.debug("Started");
            for (ParticipantRegistration roleRegistration : this.roleRegistrations.values()) {
                this.register(roleRegistration.roleName, roleRegistration.listener, roleRegistration.participantId);
            }
        }
    }

    public void stop() {
        try {
            this.leaderElectionCommandProvider.close();
        }
        catch (IOException e) {
            logger.warn("Leader Election Command Factory close failed", (Throwable)e);
        }
        this.roleLeaders.clear();
        this.executorService.shutdown();
        this.started.getAndSet(false);
        logger.debug("Stopped");
    }

    public synchronized void register(String roleName, LeaderElectionStateChangeListener listener, String participantId) {
        this.requireRoleName(roleName);
        Objects.requireNonNull(listener, "Change Listener required");
        ParticipantRegistration roleRegistration = new ParticipantRegistration(roleName, participantId, listener);
        this.roleRegistrations.put(roleName, roleRegistration);
        boolean participating = this.isParticipating(participantId);
        if (participating) {
            logger.debug("Registered Participation for Election Role [{}] ID [{}]", (Object)roleName, (Object)participantId);
            if (this.started.get()) {
                this.registerLeaderElectionCommand(roleName, listener, participantId);
            }
        } else {
            logger.info("Registered Observation for Election Role [{}]", (Object)roleName);
        }
    }

    public synchronized void unregister(String roleName) {
        this.requireRoleName(roleName);
        this.roleLeaders.remove(roleName);
        ParticipantRegistration roleRegistration = this.roleRegistrations.remove(roleName);
        if (roleRegistration == null) {
            logger.info("Not registered for Election Role [{}]", (Object)roleName);
        } else {
            Future<?> roleCommand = this.roleCommands.remove(roleName);
            if (roleCommand == null) {
                logger.warn("Leader Election Command not found Role [{}] ID [{}]", (Object)roleName, (Object)roleRegistration.participantId);
            } else {
                roleCommand.cancel(true);
            }
            logger.info("Unregistered for Election Role [{}] ID [{}]", (Object)roleName, (Object)roleRegistration.participantId);
        }
    }

    public boolean isActiveParticipant(String roleName) {
        this.requireRoleName(roleName);
        String participantId = this.getParticipantId(roleName);
        return this.isParticipating(participantId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<String> getLeader(String roleName) {
        this.requireRoleName(roleName);
        String roleId = this.getRoleId(roleName);
        long pollStarted = System.nanoTime();
        try {
            Optional<String> leader = this.leaderElectionCommandProvider.findLeader(roleId);
            leader.ifPresent(leaderId -> this.setRoleLeader(roleName, (String)leaderId));
            Optional<String> optional = leader;
            return optional;
        }
        finally {
            long elapsed = System.nanoTime() - pollStarted;
            this.registerPollTime(elapsed);
        }
    }

    public boolean isLeader(String roleName) {
        boolean leader;
        this.requireRoleName(roleName);
        String participantId = this.getParticipantId(roleName);
        if (participantId == null) {
            logger.debug("Role [{}] not participating in Leader election", (Object)roleName);
            leader = false;
        } else {
            Optional<String> leaderAddress = this.getLeader(roleName);
            String leaderId = leaderAddress.orElse(null);
            leader = participantId.equals(leaderId);
            if (leader) {
                logger.debug("Role [{}] Participant ID [{}] is Leader", (Object)roleName, (Object)participantId);
            } else {
                logger.debug("Role [{}] Participant ID [{}] not Leader", (Object)roleName, (Object)leaderId);
            }
        }
        return leader;
    }

    protected ExecutorService createExecutorService() {
        return Executors.newFixedThreadPool(4, new NamedThreadFactory());
    }

    protected LeaderElectionCommandProvider createLeaderElectionCommandProvider() {
        ServiceAccountNamespaceProvider namespaceProvider = new ServiceAccountNamespaceProvider();
        String namespace = namespaceProvider.getNamespace();
        StandardKubernetesClientProvider kubernetesClientProvider = new StandardKubernetesClientProvider();
        return new StandardLeaderElectionCommandProvider((KubernetesClientProvider)kubernetesClientProvider, namespace);
    }

    private synchronized void registerLeaderElectionCommand(String roleName, LeaderElectionStateChangeListener listener, String participantId) {
        Future<?> currentRoleCommand = this.roleCommands.get(roleName);
        if (currentRoleCommand == null) {
            String roleId = this.getRoleId(roleName);
            Runnable leaderElectionCommand = this.leaderElectionCommandProvider.getCommand(roleId, participantId, () -> ((LeaderElectionStateChangeListener)listener).onStartLeading(), () -> ((LeaderElectionStateChangeListener)listener).onStopLeading(), leaderId -> this.setRoleLeader(roleName, (String)leaderId));
            Future<?> roleCommand = this.executorService.submit(leaderElectionCommand);
            this.roleCommands.put(roleName, roleCommand);
            logger.info("Registered command for Election Role [{}] ID [{}]", (Object)roleName, (Object)participantId);
        }
    }

    private void setRoleLeader(String roleName, String leaderId) {
        String previousLeaderId = this.roleLeaders.put(roleName, leaderId);
        if (leaderId.equals(previousLeaderId)) {
            logger.debug("Role [{}] Leader [{}] not changed", (Object)roleName, (Object)leaderId);
        } else {
            logger.debug("Role [{}] Leader [{}] Previous [{}] changed", new Object[]{roleName, leaderId, previousLeaderId});
            this.onLeaderChanged(roleName);
        }
    }

    private String getParticipantId(String roleName) {
        ParticipantRegistration roleRegistration = this.roleRegistrations.get(roleName);
        return roleRegistration == null ? null : roleRegistration.participantId;
    }

    private void requireRoleName(String roleName) {
        if (roleName == null || roleName.isEmpty()) {
            throw new IllegalArgumentException("Role Name required");
        }
    }

    private String getRoleId(String roleName) {
        String roleId = ROLE_NAMES.get(roleName);
        if (roleId == null) {
            throw new IllegalArgumentException(String.format("Role Name [%s] not supported", roleName));
        }
        return this.roleIdPrefix == null ? roleId : String.format("%s-%s", this.roleIdPrefix, roleId);
    }

    static {
        LinkedHashMap<String, String> roleNames = new LinkedHashMap<String, String>();
        for (LeaderElectionRole leaderElectionRole : LeaderElectionRole.values()) {
            roleNames.put(leaderElectionRole.getRoleName(), leaderElectionRole.getRoleId());
        }
        ROLE_NAMES = Collections.unmodifiableMap(roleNames);
    }

    private static class ParticipantRegistration {
        private final String roleName;
        private final String participantId;
        private final LeaderElectionStateChangeListener listener;

        private ParticipantRegistration(String roleName, String participantId, LeaderElectionStateChangeListener listener) {
            this.roleName = roleName;
            this.participantId = participantId;
            this.listener = listener;
        }
    }

    private static class NamedThreadFactory
    implements ThreadFactory {
        private final ThreadFactory defaultFactory = Executors.defaultThreadFactory();

        private NamedThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = this.defaultFactory.newThread(runnable);
            thread.setName(KubernetesLeaderElectionManager.class.getSimpleName());
            thread.setDaemon(true);
            return thread;
        }
    }
}

