/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.atlassian.bitbucket;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import org.apache.nifi.atlassian.bitbucket.BitbucketAuthenticationType;
import org.apache.nifi.atlassian.bitbucket.BitbucketFormFactor;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.oauth2.OAuth2AccessTokenProvider;
import org.apache.nifi.registry.flow.FlowRegistryException;
import org.apache.nifi.registry.flow.git.client.GitCommit;
import org.apache.nifi.registry.flow.git.client.GitCreateContentRequest;
import org.apache.nifi.registry.flow.git.client.GitRepositoryClient;
import org.apache.nifi.stream.io.StreamUtils;
import org.apache.nifi.web.client.api.HttpContentType;
import org.apache.nifi.web.client.api.HttpResponseEntity;
import org.apache.nifi.web.client.api.HttpUriBuilder;
import org.apache.nifi.web.client.api.StandardHttpContentType;
import org.apache.nifi.web.client.api.StandardMultipartFormDataStreamBuilder;
import org.apache.nifi.web.client.provider.api.WebClientServiceProvider;

public class BitbucketRepositoryClient
implements GitRepositoryClient {
    private final ComponentLog logger;
    public static final String CLOUD_API_VERSION = "2.0";
    public static final String DATA_CENTER_API_VERSION = "1.0";
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String CONTENT_TYPE_HEADER = "Content-Type";
    private static final String BASIC = "Basic";
    private static final String BEARER = "Bearer";
    private static final String FIELD_BRANCH = "branch";
    private static final String FIELD_MESSAGE = "message";
    private static final String FIELD_SOURCE_COMMIT_ID = "sourceCommitId";
    private static final String FIELD_CONTENT = "content";
    private static final String FIELD_FILES = "files";
    private static final String FIELD_CHILDREN = "children";
    private static final String FIELD_VALUES = "values";
    private static final String FIELD_NEXT = "next";
    private static final String FIELD_NEXT_PAGE_START = "nextPageStart";
    private static final String FIELD_IS_LAST_PAGE = "isLastPage";
    private static final String FIELD_PERMISSION = "permission";
    private static final String FIELD_TYPE = "type";
    private static final String FIELD_PATH = "path";
    private static final String FIELD_TO_STRING = "toString";
    private static final String FIELD_DISPLAY_NAME = "displayName";
    private static final String FIELD_NAME = "name";
    private static final String FIELD_EMAIL_ADDRESS = "emailAddress";
    private static final String FIELD_DISPLAY_ID = "displayId";
    private static final String FIELD_ID = "id";
    private static final String FIELD_HASH = "hash";
    private static final String FIELD_MESSAGE_TEXT = "message";
    private static final String FIELD_AUTHOR = "author";
    private static final String FIELD_AUTHOR_TIMESTAMP = "authorTimestamp";
    private static final String FIELD_ERROR_MESSAGE = "error";
    private static final String FIELD_ERRORS = "errors";
    private static final String FIELD_RAW = "raw";
    private static final String EMPTY_STRING = "";
    private static final String ENTRY_DIRECTORY_DATA_CENTER = "DIRECTORY";
    private static final String ENTRY_DIRECTORY_CLOUD = "commit_directory";
    private static final String ENTRY_FILE_DATA_CENTER = "FILE";
    private static final String ENTRY_FILE_CLOUD = "commit_file";
    private final ObjectMapper objectMapper = JsonMapper.builder().build();
    private final String apiUrl;
    private final String apiVersion;
    private final BitbucketFormFactor formFactor;
    private final String projectKey;
    private final String apiScheme;
    private final String apiHost;
    private final int apiPort;
    private final List<String> apiBasePathSegments;
    private final String clientId;
    private final String workspace;
    private final String repoName;
    private final String repoPath;
    private WebClientServiceProvider webClient;
    private BitbucketToken<String> authToken;
    private final boolean canRead;
    private final boolean canWrite;
    private boolean hasCommits = false;

    public static String getDefaultApiVersion(BitbucketFormFactor formFactor) {
        return formFactor == BitbucketFormFactor.DATA_CENTER ? DATA_CENTER_API_VERSION : CLOUD_API_VERSION;
    }

    private BitbucketRepositoryClient(Builder builder) throws FlowRegistryException {
        String permission;
        this.webClient = Objects.requireNonNull(builder.webClient, "Web Client is required");
        this.logger = Objects.requireNonNull(builder.logger, "ComponentLog required");
        this.formFactor = builder.formFactor == null ? BitbucketFormFactor.CLOUD : builder.formFactor;
        this.apiUrl = Objects.requireNonNull(builder.apiUrl, "API Instance is required");
        ParsedApiUrl parsedApiUrl = BitbucketRepositoryClient.parseApiUrl(this.apiUrl);
        this.apiScheme = parsedApiUrl.scheme();
        this.apiHost = parsedApiUrl.host();
        this.apiPort = parsedApiUrl.port();
        this.apiBasePathSegments = parsedApiUrl.pathSegments();
        if (this.formFactor == BitbucketFormFactor.CLOUD) {
            this.workspace = Objects.requireNonNull(builder.workspace, "Workspace is required for Bitbucket Cloud");
            this.projectKey = null;
        } else {
            this.projectKey = Objects.requireNonNull(builder.projectKey, "Project Key is required for Bitbucket Data Center");
            this.workspace = builder.workspace;
        }
        this.repoName = Objects.requireNonNull(builder.repoName, "Repository Name is required");
        this.clientId = Objects.requireNonNull(builder.clientId, "Client ID is required");
        this.repoPath = builder.repoPath;
        this.apiVersion = BitbucketRepositoryClient.getDefaultApiVersion(this.formFactor);
        BitbucketAuthenticationType authenticationType = Objects.requireNonNull(builder.authenticationType, "Authentication type is required");
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER && authenticationType != BitbucketAuthenticationType.ACCESS_TOKEN) {
            throw new FlowRegistryException("Bitbucket Data Center only supports Access Token authentication");
        }
        switch (authenticationType) {
            case ACCESS_TOKEN: {
                Objects.requireNonNull(builder.accessToken, "Access Token is required");
                this.authToken = new AccessToken(this, builder.accessToken);
                break;
            }
            case BASIC_AUTH: {
                Objects.requireNonNull(builder.username, "Username is required");
                Objects.requireNonNull(builder.appPassword, "App Password is required");
                this.authToken = new BasicAuthToken(this, builder.username, builder.appPassword);
                break;
            }
            case OAUTH2: {
                Objects.requireNonNull(builder.oauthService, "OAuth 2.0 Token Provider is required");
                this.authToken = new OAuthToken(this, builder.oauthService);
            }
        }
        switch (permission = this.checkRepoPermissions(authenticationType)) {
            case "admin": 
            case "write": {
                this.canRead = true;
                this.canWrite = true;
                break;
            }
            case "read": {
                this.canRead = true;
                this.canWrite = false;
                break;
            }
            case "none": {
                this.canRead = false;
                this.canWrite = false;
                break;
            }
            default: {
                this.canRead = false;
                this.canWrite = false;
            }
        }
        this.logger.info("Created {} for clientId = [{}], repository [{}]", new Object[]{this.getClass().getSimpleName(), this.clientId, this.repoName});
    }

    public boolean hasReadPermission() {
        return this.canRead;
    }

    public boolean hasWritePermission() {
        return this.canWrite;
    }

    public String getWorkspace() {
        return this.workspace;
    }

    public String getProjectKey() {
        return this.projectKey;
    }

    public String getRepoName() {
        return this.repoName;
    }

    public BitbucketFormFactor getFormFactor() {
        return this.formFactor;
    }

    public String getApiHost() {
        return this.apiHost;
    }

    public Set<String> getBranches() throws FlowRegistryException {
        this.logger.debug("Getting branches for repository [{}]", new Object[]{this.repoName});
        return this.formFactor == BitbucketFormFactor.DATA_CENTER ? this.getBranchesDataCenter() : this.getBranchesCloud();
    }

    private Set<String> getBranchesCloud() throws FlowRegistryException {
        HashSet<String> hashSet;
        block10: {
            URI uri = this.getRepositoryUriBuilder().addPathSegment("refs").addPathSegment("branches").build();
            HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
            try {
                this.verifyStatusCode(response, "Error while listing branches for repository [%s]".formatted(this.repoName), 200);
                JsonNode jsonResponse = this.parseResponseBody(response, uri);
                JsonNode values = jsonResponse.get(FIELD_VALUES);
                HashSet<String> result = new HashSet<String>();
                if (values != null && values.isArray()) {
                    for (JsonNode branch : values) {
                        String branchName = branch.path(FIELD_NAME).asText(EMPTY_STRING);
                        if (branchName.isEmpty()) continue;
                        result.add(branchName);
                    }
                }
                hashSet = result;
                if (response == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (response != null) {
                        try {
                            response.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new FlowRegistryException("Failed closing Bitbucket branch listing response", (Throwable)e);
                }
            }
            response.close();
        }
        return hashSet;
    }

    private Set<String> getBranchesDataCenter() throws FlowRegistryException {
        HashSet<String> hashSet;
        block10: {
            URI uri = this.getRepositoryUriBuilder().addPathSegment("branches").build();
            HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
            try {
                this.verifyStatusCode(response, "Error while listing branches for repository [%s]".formatted(this.repoName), 200);
                JsonNode jsonResponse = this.parseResponseBody(response, uri);
                JsonNode values = jsonResponse.get(FIELD_VALUES);
                HashSet<String> result = new HashSet<String>();
                if (values != null && values.isArray()) {
                    for (JsonNode branch : values) {
                        String displayId = branch.path(FIELD_DISPLAY_ID).asText(EMPTY_STRING);
                        if (displayId.isEmpty()) continue;
                        result.add(displayId);
                    }
                }
                hashSet = result;
                if (response == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (response != null) {
                        try {
                            response.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new FlowRegistryException("Failed closing Bitbucket branch listing response", (Throwable)e);
                }
            }
            response.close();
        }
        return hashSet;
    }

    public Set<String> getTopLevelDirectoryNames(String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(EMPTY_STRING);
        this.logger.debug("Getting top-level directories for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Iterator<JsonNode> files = this.getFiles(branch, resolvedPath);
        HashSet<String> result = new HashSet<String>();
        while (files.hasNext()) {
            String entryPath;
            JsonNode file = files.next();
            if (!this.isDirectoryEntry(file) || (entryPath = this.getEntryPath(file)).isEmpty()) continue;
            Path fullPath = Paths.get(entryPath, new String[0]);
            result.add(fullPath.getFileName().toString());
        }
        return result;
    }

    public Set<String> getFileNames(String directory, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(directory);
        this.logger.debug("Getting filenames for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Iterator<JsonNode> files = this.getFiles(branch, resolvedPath);
        HashSet<String> result = new HashSet<String>();
        while (files.hasNext()) {
            String entryPath;
            JsonNode file = files.next();
            if (!this.isFileEntry(file) || (entryPath = this.getEntryPath(file)).isEmpty()) continue;
            Path fullPath = Paths.get(entryPath, new String[0]);
            result.add(fullPath.getFileName().toString());
        }
        return result;
    }

    public List<GitCommit> getCommits(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting commits for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Iterator<JsonNode> commits = this.getListCommits(branch, resolvedPath);
        ArrayList<GitCommit> result = new ArrayList<GitCommit>();
        while (commits.hasNext()) {
            JsonNode commit = commits.next();
            result.add(this.toGitCommit(commit));
        }
        return result;
    }

    public InputStream getContentFromBranch(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting content for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Optional<String> lastCommit = this.getLatestCommit(branch, resolvedPath);
        if (lastCommit.isEmpty()) {
            throw new FlowRegistryException(String.format("Could not find committed files at %s on branch %s response from Bitbucket API", resolvedPath, branch));
        }
        return this.getContentFromCommit(path, lastCommit.get());
    }

    public InputStream getContentFromCommit(String path, String commitSha) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting content for path [{}] from commit [{}] in repository [{}]", new Object[]{resolvedPath, commitSha, this.repoName});
        return this.formFactor == BitbucketFormFactor.DATA_CENTER ? this.getContentFromCommitDataCenter(resolvedPath, commitSha) : this.getContentFromCommitCloud(resolvedPath, commitSha);
    }

    private InputStream getContentFromCommitCloud(String resolvedPath, String commitSha) throws FlowRegistryException {
        HttpUriBuilder builder = this.getRepositoryUriBuilder().addPathSegment("src").addPathSegment(commitSha);
        this.addPathSegments(builder, resolvedPath);
        URI uri = builder.build();
        String errorMessage = "Error while retrieving content for repository [%s] at path %s".formatted(this.repoName, resolvedPath);
        HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
        return this.getResponseBody(response, errorMessage, 200);
    }

    private InputStream getContentFromCommitDataCenter(String resolvedPath, String commitSha) throws FlowRegistryException {
        URI uri = this.buildRawUri(resolvedPath, commitSha);
        String errorMessage = "Error while retrieving content for repository [%s] at path %s".formatted(this.repoName, resolvedPath);
        HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
        return this.getResponseBody(response, errorMessage, 200);
    }

    public Optional<String> getContentSha(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting content SHA for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        return this.getLatestCommit(branch, resolvedPath);
    }

    public String createContent(GitCreateContentRequest request) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(request.getPath());
        String branch = request.getBranch();
        this.logger.debug("Creating content at path [{}] on branch [{}] in repository [{}] ", new Object[]{resolvedPath, branch, this.repoName});
        return this.formFactor == BitbucketFormFactor.DATA_CENTER ? this.createContentDataCenter(request, resolvedPath, branch) : this.createContentCloud(request, resolvedPath, branch);
    }

    private String createContentCloud(GitCreateContentRequest request, String resolvedPath, String branch) throws FlowRegistryException {
        StandardMultipartFormDataStreamBuilder multipartBuilder = new StandardMultipartFormDataStreamBuilder();
        multipartBuilder.addPart(resolvedPath, (HttpContentType)StandardHttpContentType.APPLICATION_JSON, request.getContent().getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart("message", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, request.getMessage().getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart(FIELD_BRANCH, (HttpContentType)StandardHttpContentType.TEXT_PLAIN, branch.getBytes(StandardCharsets.UTF_8));
        URI uri = this.getRepositoryUriBuilder().addPathSegment("src").build();
        String errorMessage = "Error while committing content for repository [%s] on branch %s at path %s".formatted(this.repoName, branch, resolvedPath);
        try (HttpResponseEntity response = this.webClient.getWebClientService().post().uri(uri).body(multipartBuilder.build(), OptionalLong.empty()).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).header(CONTENT_TYPE_HEADER, multipartBuilder.getHttpContentType().getContentType()).retrieve();){
            this.verifyStatusCode(response, errorMessage, 201);
        }
        catch (IOException e) {
            throw new FlowRegistryException("Failed closing Bitbucket create content response", (Throwable)e);
        }
        return this.getRequiredLatestCommit(branch, resolvedPath);
    }

    private String createContentDataCenter(GitCreateContentRequest request, String resolvedPath, String branch) throws FlowRegistryException {
        String existingContentSha;
        boolean existingContentProvided;
        StandardMultipartFormDataStreamBuilder multipartBuilder = new StandardMultipartFormDataStreamBuilder();
        String fileName = this.getFileName(resolvedPath);
        multipartBuilder.addPart(FIELD_CONTENT, fileName, (HttpContentType)StandardHttpContentType.APPLICATION_OCTET_STREAM, request.getContent().getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart(FIELD_BRANCH, (HttpContentType)StandardHttpContentType.TEXT_PLAIN, branch.getBytes(StandardCharsets.UTF_8));
        String message = request.getMessage();
        if (message != null && !message.isEmpty()) {
            multipartBuilder.addPart("message", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, message.getBytes(StandardCharsets.UTF_8));
        }
        boolean bl = existingContentProvided = (existingContentSha = request.getExistingContentSha()) != null && !existingContentSha.isBlank();
        if (existingContentProvided) {
            multipartBuilder.addPart(FIELD_SOURCE_COMMIT_ID, (HttpContentType)StandardHttpContentType.TEXT_PLAIN, existingContentSha.getBytes(StandardCharsets.UTF_8));
        }
        HttpUriBuilder uriBuilder = this.getRepositoryUriBuilder().addPathSegment("browse");
        this.addPathSegments(uriBuilder, resolvedPath);
        URI uri = uriBuilder.build();
        byte[] requestBody = this.toByteArray(multipartBuilder.build());
        String errorMessage = "Error while committing content for repository [%s] on branch %s at path %s".formatted(this.repoName, branch, resolvedPath);
        try (HttpResponseEntity response = this.webClient.getWebClientService().put().uri(uri).body((InputStream)new ByteArrayInputStream(requestBody), OptionalLong.of(requestBody.length)).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).header(CONTENT_TYPE_HEADER, multipartBuilder.getHttpContentType().getContentType()).retrieve();){
            this.verifyStatusCode(response, errorMessage, 201, 200);
        }
        catch (IOException e) {
            throw new FlowRegistryException("Failed closing Bitbucket create content response", (Throwable)e);
        }
        return this.getRequiredLatestCommit(branch, resolvedPath);
    }

    public InputStream deleteContent(String filePath, String commitMessage, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(filePath);
        this.logger.debug("Deleting content at path [{}] on branch [{}] in repository [{}] ", new Object[]{resolvedPath, branch, this.repoName});
        InputStream fileToBeDeleted = this.getContentFromBranch(filePath, branch);
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            this.deleteContentDataCenter(resolvedPath, commitMessage, branch);
        } else {
            this.deleteContentCloud(resolvedPath, commitMessage, branch);
        }
        return fileToBeDeleted;
    }

    private void deleteContentCloud(String resolvedPath, String commitMessage, String branch) throws FlowRegistryException {
        StandardMultipartFormDataStreamBuilder multipartBuilder = new StandardMultipartFormDataStreamBuilder();
        multipartBuilder.addPart(FIELD_FILES, (HttpContentType)StandardHttpContentType.TEXT_PLAIN, resolvedPath.getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart("message", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, commitMessage.getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart(FIELD_BRANCH, (HttpContentType)StandardHttpContentType.TEXT_PLAIN, branch.getBytes(StandardCharsets.UTF_8));
        URI uri = this.getRepositoryUriBuilder().addPathSegment("src").build();
        String errorMessage = "Error while deleting content for repository [%s] on branch %s at path %s".formatted(this.repoName, branch, resolvedPath);
        try (HttpResponseEntity response = this.webClient.getWebClientService().post().uri(uri).body(multipartBuilder.build(), OptionalLong.empty()).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).header(CONTENT_TYPE_HEADER, multipartBuilder.getHttpContentType().getContentType()).retrieve();){
            this.verifyStatusCode(response, errorMessage, 201);
        }
        catch (IOException e) {
            throw new FlowRegistryException("Failed closing Bitbucket delete content response", (Throwable)e);
        }
    }

    private void deleteContentDataCenter(String resolvedPath, String commitMessage, String branch) throws FlowRegistryException {
        Optional<String> latestCommit = this.getLatestCommit(branch, resolvedPath);
        HttpUriBuilder uriBuilder = this.getRepositoryUriBuilder().addPathSegment("browse");
        this.addPathSegments(uriBuilder, resolvedPath);
        uriBuilder.addQueryParameter(FIELD_BRANCH, branch);
        if (commitMessage != null) {
            uriBuilder.addQueryParameter("message", commitMessage);
        }
        latestCommit.ifPresent(commit -> uriBuilder.addQueryParameter(FIELD_SOURCE_COMMIT_ID, commit));
        URI uri = uriBuilder.build();
        String errorMessage = "Error while deleting content for repository [%s] on branch %s at path %s".formatted(this.repoName, branch, resolvedPath);
        try (HttpResponseEntity response = this.webClient.getWebClientService().delete().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();){
            this.verifyStatusCode(response, errorMessage, 200, 202, 204, 201);
        }
        catch (IOException e) {
            throw new FlowRegistryException("Failed closing Bitbucket delete content response", (Throwable)e);
        }
    }

    private Iterator<JsonNode> getFiles(String branch, String resolvedPath) throws FlowRegistryException {
        Optional<String> lastCommit = this.getLatestCommit(branch, resolvedPath);
        if (lastCommit.isEmpty()) {
            return Collections.emptyIterator();
        }
        return this.formFactor == BitbucketFormFactor.DATA_CENTER ? this.getFilesDataCenter(branch, resolvedPath, lastCommit.get()) : this.getFilesCloud(branch, resolvedPath, lastCommit.get());
    }

    private Iterator<JsonNode> getFilesCloud(String branch, String resolvedPath, String commit) throws FlowRegistryException {
        URI uri = this.getRepositoryUriBuilder().addPathSegment("src").addPathSegment(commit).addPathSegment(resolvedPath).build();
        String errorMessage = String.format("Error while listing content for repository [%s] on branch %s at path %s", this.repoName, branch, resolvedPath);
        return this.getPagedResponseValues(uri, errorMessage);
    }

    private Iterator<JsonNode> getFilesDataCenter(String branch, String resolvedPath, String commit) throws FlowRegistryException {
        ArrayList allValues = new ArrayList();
        Integer nextPageStart = null;
        do {
            URI uri = this.buildBrowseUri(resolvedPath, commit, nextPageStart, false);
            String errorMessage = "Error while listing content for repository [%s] on branch %s at path %s".formatted(this.repoName, branch, resolvedPath);
            try (HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();){
                JsonNode nextPageStartNode;
                this.verifyStatusCode(response, errorMessage, 200);
                JsonNode root = this.parseResponseBody(response, uri);
                JsonNode children = root.path(FIELD_CHILDREN);
                JsonNode values = children.path(FIELD_VALUES);
                if (values.isArray()) {
                    values.forEach(allValues::add);
                }
                nextPageStart = children.path(FIELD_IS_LAST_PAGE).asBoolean(true) ? null : ((nextPageStartNode = children.get(FIELD_NEXT_PAGE_START)) != null && nextPageStartNode.isInt() ? Integer.valueOf(nextPageStartNode.intValue()) : null);
            }
            catch (IOException e) {
                throw new FlowRegistryException("Failed closing Bitbucket browse response", (Throwable)e);
            }
        } while (nextPageStart != null);
        return allValues.iterator();
    }

    private URI buildBrowseUri(String resolvedPath, String commit, Integer start, boolean rawContent) {
        HttpUriBuilder builder = this.getRepositoryUriBuilder().addPathSegment("browse");
        this.addPathSegments(builder, resolvedPath);
        builder.addQueryParameter("at", commit);
        if (start != null) {
            builder.addQueryParameter("start", String.valueOf(start));
        }
        return builder.build();
    }

    private URI buildRawUri(String resolvedPath, String commit) {
        HttpUriBuilder builder = this.getRepositoryUriBuilder().addPathSegment(FIELD_RAW);
        this.addPathSegments(builder, resolvedPath);
        builder.addQueryParameter("at", commit);
        return builder.build();
    }

    private Iterator<JsonNode> getListCommits(String branch, String path) throws FlowRegistryException {
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            return this.getListCommitsDataCenter(branch, path);
        }
        return this.getListCommitsCloud(branch, path);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Iterator<JsonNode> getListCommitsCloud(String branch, String path) throws FlowRegistryException {
        String errorMessage;
        URI uri;
        if (!this.hasCommits) {
            uri = this.getRepositoryUriBuilder().addPathSegment("commits").build();
            errorMessage = "Error while listing commits for repository [%s] on branch %s".formatted(this.repoName, branch);
            try (HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();){
                this.verifyStatusCode(response, errorMessage, 200);
                JsonNode root = this.parseResponseBody(response, uri);
                JsonNode values = root.path(FIELD_VALUES);
                if (values.isArray() && values.isEmpty()) {
                    Iterator<JsonNode> iterator = Collections.emptyIterator();
                    return iterator;
                }
                this.hasCommits = true;
            }
            catch (IOException e) {
                throw new FlowRegistryException("Failed closing Bitbucket commits response", (Throwable)e);
            }
        }
        uri = this.getRepositoryUriBuilder().addPathSegment("commits").addPathSegment(branch).addQueryParameter(FIELD_PATH, path).build();
        errorMessage = String.format("Error while listing commits for repository [%s] on branch %s", this.repoName, branch);
        return this.getPagedResponseValues(uri, errorMessage);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Iterator<JsonNode> getListCommitsDataCenter(String branch, String path) throws FlowRegistryException {
        ArrayList allValues = new ArrayList();
        Integer nextPageStart = null;
        do {
            JsonNode values;
            HttpResponseEntity response;
            String errorMessage;
            URI uri;
            if (!this.hasCommits) {
                uri = this.buildCommitsUri(null, null, null);
                errorMessage = "Error while listing commits for repository [%s] on branch %s".formatted(this.repoName, branch);
                try {
                    response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
                    try {
                        this.verifyStatusCode(response, errorMessage, 200);
                        JsonNode root2 = this.parseResponseBody(response, uri);
                        values = root2.path(FIELD_VALUES);
                        if (values.isArray() && values.isEmpty()) {
                            Iterator<JsonNode> iterator = Collections.emptyIterator();
                            return iterator;
                        }
                        this.hasCommits = true;
                    }
                    finally {
                        if (response != null) {
                            response.close();
                        }
                    }
                }
                catch (IOException e) {
                    throw new FlowRegistryException("Failed closing Bitbucket commits response", (Throwable)e);
                }
            }
            uri = this.buildCommitsUri(branch, path, nextPageStart);
            errorMessage = "Error while listing commits for repository [%s] on branch %s".formatted(this.repoName, branch);
            try {
                response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
                try {
                    this.verifyStatusCode(response, errorMessage, 200);
                    JsonNode root = this.parseResponseBody(response, uri);
                    values = root.path(FIELD_VALUES);
                    if (values.isArray()) {
                        values.forEach(allValues::add);
                    }
                    if (root.path(FIELD_IS_LAST_PAGE).asBoolean(true)) {
                        nextPageStart = null;
                        continue;
                    }
                    JsonNode nextPageStartNode = root.get(FIELD_NEXT_PAGE_START);
                    nextPageStart = nextPageStartNode != null && nextPageStartNode.isInt() ? Integer.valueOf(nextPageStartNode.intValue()) : null;
                }
                finally {
                    if (response != null) {
                        response.close();
                    }
                }
            }
            catch (IOException e) {
                throw new FlowRegistryException("Failed closing Bitbucket commits response", (Throwable)e);
            }
        } while (nextPageStart != null);
        return allValues.iterator();
    }

    private URI buildCommitsUri(String branch, String path, Integer start) {
        HttpUriBuilder builder = this.getRepositoryUriBuilder().addPathSegment("commits");
        if (path != null && !path.isBlank()) {
            builder.addQueryParameter(FIELD_PATH, path);
        }
        if (branch != null && !branch.isBlank()) {
            builder.addQueryParameter("until", branch);
        }
        if (start != null) {
            builder.addQueryParameter("start", String.valueOf(start));
        }
        return builder.build();
    }

    private Iterator<JsonNode> getPagedResponseValues(URI uri, String errorMessage) throws FlowRegistryException {
        ArrayList allValues = new ArrayList();
        URI nextUri = uri;
        while (nextUri != null) {
            try {
                HttpResponseEntity response = this.webClient.getWebClientService().get().uri(nextUri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
                try {
                    JsonNode next;
                    this.verifyStatusCode(response, errorMessage, 200);
                    JsonNode root = this.parseResponseBody(response, nextUri);
                    JsonNode values = root.get(FIELD_VALUES);
                    if (values != null && values.isArray()) {
                        values.forEach(allValues::add);
                    }
                    nextUri = (next = root.get(FIELD_NEXT)) != null && next.isTextual() ? URI.create(next.asText()) : null;
                }
                finally {
                    if (response == null) continue;
                    response.close();
                }
            }
            catch (IOException e) {
                throw new FlowRegistryException("Failed closing Bitbucket paged response", (Throwable)e);
            }
        }
        return allValues.iterator();
    }

    private Optional<String> getLatestCommit(String branch, String path) throws FlowRegistryException {
        Iterator<JsonNode> commits = this.getListCommits(branch, path);
        if (commits.hasNext()) {
            return Optional.ofNullable(this.getCommitHash(commits.next())).filter(hash -> !hash.isEmpty());
        }
        return Optional.empty();
    }

    private String getRequiredLatestCommit(String branch, String resolvedPath) throws FlowRegistryException {
        Optional<String> lastCommit = this.getLatestCommit(branch, resolvedPath);
        if (lastCommit.isEmpty()) {
            throw new FlowRegistryException(String.format("Could not find commit for the file %s we just tried to commit on branch %s", resolvedPath, branch));
        }
        return lastCommit.get();
    }

    private String getCommitHash(JsonNode commit) {
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            return commit.path(FIELD_ID).asText(EMPTY_STRING);
        }
        return commit.path(FIELD_HASH).asText(EMPTY_STRING);
    }

    private String checkRepoPermissions(BitbucketAuthenticationType authenticationType) throws FlowRegistryException {
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            return this.checkReadByListingBranches();
        }
        return switch (authenticationType) {
            default -> throw new MatchException(null, null);
            case BitbucketAuthenticationType.OAUTH2 -> this.checkOAuthPermissions();
            case BitbucketAuthenticationType.ACCESS_TOKEN, BitbucketAuthenticationType.BASIC_AUTH -> this.checkReadByListingBranches();
        };
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String checkOAuthPermissions() throws FlowRegistryException {
        this.logger.debug("Retrieving information about current user");
        HttpUriBuilder builder = this.webClient.getHttpUriBuilder().scheme(this.apiScheme).host(this.apiHost);
        if (this.apiPort != -1) {
            builder.port(this.apiPort);
        }
        this.apiBasePathSegments.forEach(arg_0 -> ((HttpUriBuilder)builder).addPathSegment(arg_0));
        URI uri = builder.addPathSegment(this.apiVersion).addPathSegment("user").addPathSegment("permissions").addPathSegment("repositories").addQueryParameter("q", "repository.name=\"" + this.repoName + "\"").build();
        String errorMessage = "Error while retrieving permission metadata for specified repo";
        try (HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();){
            this.verifyStatusCode(response, "Error while retrieving permission metadata for specified repo", 200);
            JsonNode jsonResponse = this.parseResponseBody(response, uri);
            JsonNode values = jsonResponse.get(FIELD_VALUES);
            if (values == null) return "none";
            if (!values.isArray()) return "none";
            if (!values.elements().hasNext()) return "none";
            JsonNode permissionNode = ((JsonNode)values.elements().next()).get(FIELD_PERMISSION);
            String string = permissionNode == null ? "none" : permissionNode.asText();
            return string;
        }
        catch (IOException e) {
            throw new FlowRegistryException("Failed closing Bitbucket permission response", (Throwable)e);
        }
    }

    private String checkReadByListingBranches() {
        try {
            this.getBranches();
            return "admin";
        }
        catch (FlowRegistryException e) {
            return "none";
        }
    }

    private GitCommit toGitCommit(JsonNode commit) {
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            String hash = commit.path(FIELD_ID).asText();
            JsonNode authorNode = commit.path(FIELD_AUTHOR);
            String authorName = authorNode.path(FIELD_DISPLAY_NAME).asText(authorNode.path(FIELD_NAME).asText(EMPTY_STRING));
            String authorEmail = authorNode.path(FIELD_EMAIL_ADDRESS).asText(EMPTY_STRING);
            String author = authorEmail.isBlank() ? authorName : authorName + " <" + authorEmail + ">";
            String message = commit.path("message").asText();
            long timestamp = commit.path(FIELD_AUTHOR_TIMESTAMP).asLong(0L);
            Instant date = timestamp == 0L ? Instant.now() : Instant.ofEpochMilli(timestamp);
            return new GitCommit(hash, author, message, date);
        }
        JsonNode authorNode = commit.path(FIELD_AUTHOR);
        String author = authorNode.path(FIELD_RAW).asText();
        String message = commit.path("message").asText();
        String dateText = commit.path("date").asText();
        Instant date = dateText == null || dateText.isEmpty() ? Instant.now() : Instant.parse(dateText);
        return new GitCommit(commit.path(FIELD_HASH).asText(), author, message, date);
    }

    private InputStream getResponseBody(HttpResponseEntity response, String errorMessage, int ... expectedStatusCodes) throws FlowRegistryException {
        try {
            this.verifyStatusCode(response, errorMessage, expectedStatusCodes);
            return new HttpResponseBodyInputStream(response);
        }
        catch (FlowRegistryException e) {
            this.closeQuietly(response);
            throw e;
        }
    }

    private void closeQuietly(HttpResponseEntity response) {
        try {
            response.close();
        }
        catch (IOException ioe) {
            this.logger.warn("Failed closing Bitbucket HTTP response", (Throwable)ioe);
        }
    }

    private String getErrorMessage(HttpResponseEntity response) throws FlowRegistryException {
        JsonNode jsonResponse;
        try {
            jsonResponse = this.objectMapper.readTree(response.body());
        }
        catch (IOException e) {
            throw new FlowRegistryException("Could not parse response from Bitbucket API", (Throwable)e);
        }
        if (jsonResponse == null) {
            return "Unknown error";
        }
        JsonNode errorNode = jsonResponse.get(FIELD_ERROR_MESSAGE);
        if (errorNode != null) {
            String type = jsonResponse.path(FIELD_TYPE).asText("Error");
            String message = errorNode.path("message").asText(errorNode.toString());
            return String.format("[%s] - %s", type, message);
        }
        JsonNode errorsNode = jsonResponse.get(FIELD_ERRORS);
        if (errorsNode != null && errorsNode.isArray() && errorsNode.size() > 0) {
            JsonNode firstError = errorsNode.get(0);
            return firstError.path("message").asText(firstError.toString());
        }
        JsonNode messageNode = jsonResponse.get("message");
        if (messageNode != null) {
            return messageNode.asText();
        }
        return jsonResponse.toString();
    }

    private JsonNode parseResponseBody(HttpResponseEntity response, URI uri) throws FlowRegistryException {
        try {
            return this.objectMapper.readTree(response.body());
        }
        catch (IOException e) {
            throw new FlowRegistryException(String.format("Could not parse Bitbucket API response at %s", uri), (Throwable)e);
        }
    }

    private void verifyStatusCode(HttpResponseEntity response, String errorMessage, int ... expectedStatusCodes) throws FlowRegistryException {
        int statusCode = response.statusCode();
        for (int expectedStatusCode : expectedStatusCodes) {
            if (statusCode != expectedStatusCode) continue;
            return;
        }
        throw new FlowRegistryException("%s: %s".formatted(errorMessage, this.getErrorMessage(response)));
    }

    private String getResolvedPath(String path) {
        return this.repoPath == null ? path : this.repoPath + "/" + path;
    }

    private void addPathSegments(HttpUriBuilder builder, String path) {
        String[] segments;
        if (path == null || path.isBlank()) {
            return;
        }
        String normalizedPath = path.startsWith("/") ? path.substring(1) : path;
        for (String segment : segments = normalizedPath.split("/")) {
            if (segment.isBlank()) continue;
            builder.addPathSegment(segment);
        }
    }

    private boolean isDirectoryEntry(JsonNode entry) {
        JsonNode typeNode = entry.get(FIELD_TYPE);
        if (typeNode == null) {
            return false;
        }
        String type = typeNode.asText();
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            return ENTRY_DIRECTORY_DATA_CENTER.equalsIgnoreCase(type);
        }
        return ENTRY_DIRECTORY_CLOUD.equals(type);
    }

    private boolean isFileEntry(JsonNode entry) {
        JsonNode typeNode = entry.get(FIELD_TYPE);
        if (typeNode == null) {
            return false;
        }
        String type = typeNode.asText();
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            return ENTRY_FILE_DATA_CENTER.equalsIgnoreCase(type);
        }
        return ENTRY_FILE_CLOUD.equals(type);
    }

    private String getEntryPath(JsonNode entry) {
        if (this.formFactor == BitbucketFormFactor.DATA_CENTER) {
            JsonNode pathNode = entry.get(FIELD_PATH);
            if (pathNode == null) {
                return EMPTY_STRING;
            }
            JsonNode toStringNode = pathNode.get(FIELD_TO_STRING);
            if (toStringNode != null && toStringNode.isTextual()) {
                return toStringNode.asText();
            }
            return pathNode.asText(EMPTY_STRING);
        }
        JsonNode pathNode = entry.get(FIELD_PATH);
        return pathNode == null ? EMPTY_STRING : pathNode.asText();
    }

    private HttpUriBuilder getRepositoryUriBuilder() {
        HttpUriBuilder builder = this.webClient.getHttpUriBuilder().scheme(this.apiScheme).host(this.apiHost);
        if (this.apiPort != -1) {
            builder.port(this.apiPort);
        }
        this.apiBasePathSegments.forEach(arg_0 -> ((HttpUriBuilder)builder).addPathSegment(arg_0));
        if (this.formFactor == BitbucketFormFactor.CLOUD) {
            builder.addPathSegment(this.apiVersion).addPathSegment("repositories").addPathSegment(this.workspace).addPathSegment(this.repoName);
        } else {
            builder.addPathSegment("rest").addPathSegment("api").addPathSegment(this.apiVersion).addPathSegment("projects").addPathSegment(this.projectKey).addPathSegment("repos").addPathSegment(this.repoName);
        }
        return builder;
    }

    private static ParsedApiUrl parseApiUrl(String apiUrl) throws FlowRegistryException {
        String trimmedApiUrl;
        String string = trimmedApiUrl = apiUrl == null ? null : apiUrl.trim();
        if (trimmedApiUrl == null || trimmedApiUrl.isEmpty()) {
            throw new FlowRegistryException("API Instance is required");
        }
        if (!trimmedApiUrl.contains("://")) {
            return new ParsedApiUrl("https", trimmedApiUrl, -1, List.of());
        }
        URI uri = URI.create(trimmedApiUrl);
        String scheme = uri.getScheme() == null ? "https" : uri.getScheme();
        String host = uri.getHost();
        if (host == null || host.isBlank()) {
            throw new FlowRegistryException("API Instance must include a host");
        }
        ArrayList<String> segments = new ArrayList<String>();
        String path = uri.getPath();
        if (path != null && !path.isBlank()) {
            String[] rawSegments;
            for (String segment : rawSegments = path.split("/")) {
                if (segment.isBlank()) continue;
                segments.add(segment);
            }
        }
        return new ParsedApiUrl(scheme, host, uri.getPort(), List.copyOf(segments));
    }

    private String getFileName(String resolvedPath) {
        if (resolvedPath == null || resolvedPath.isBlank()) {
            return FIELD_CONTENT;
        }
        Path path = Paths.get(resolvedPath, new String[0]);
        Path fileName = path.getFileName();
        return fileName == null ? resolvedPath : fileName.toString();
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private byte[] toByteArray(InputStream inputStream) throws FlowRegistryException {
        try (InputStream inputStream2 = inputStream;){
            byte[] byArray;
            try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
                StreamUtils.copy((InputStream)inputStream, (OutputStream)outputStream);
                byArray = outputStream.toByteArray();
            }
            return byArray;
        }
        catch (IOException e) {
            throw new FlowRegistryException("Failed to prepare multipart request", (Throwable)e);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String clientId;
        private String apiUrl;
        private BitbucketFormFactor formFactor;
        private BitbucketAuthenticationType authenticationType;
        private String accessToken;
        private String username;
        private String appPassword;
        private OAuth2AccessTokenProvider oauthService;
        private WebClientServiceProvider webClient;
        private String workspace;
        private String projectKey;
        private String repoName;
        private String repoPath;
        private ComponentLog logger;

        public Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public Builder apiUrl(String apiUrl) {
            this.apiUrl = apiUrl;
            return this;
        }

        public Builder formFactor(BitbucketFormFactor formFactor) {
            this.formFactor = formFactor;
            return this;
        }

        public Builder authenticationType(BitbucketAuthenticationType authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        public Builder accessToken(String accessToken) {
            this.accessToken = accessToken;
            return this;
        }

        public Builder username(String username) {
            this.username = username;
            return this;
        }

        public Builder appPassword(String appPassword) {
            this.appPassword = appPassword;
            return this;
        }

        public Builder oauthService(OAuth2AccessTokenProvider oauthService) {
            this.oauthService = oauthService;
            return this;
        }

        public Builder webClient(WebClientServiceProvider webClient) {
            this.webClient = webClient;
            return this;
        }

        public Builder workspace(String workspace) {
            this.workspace = workspace;
            return this;
        }

        public Builder projectKey(String projectKey) {
            this.projectKey = projectKey;
            return this;
        }

        public Builder repoName(String repoName) {
            this.repoName = repoName;
            return this;
        }

        public Builder repoPath(String repoPath) {
            this.repoPath = repoPath;
            return this;
        }

        public Builder logger(ComponentLog logger) {
            this.logger = logger;
            return this;
        }

        public BitbucketRepositoryClient build() throws FlowRegistryException {
            return new BitbucketRepositoryClient(this);
        }
    }

    private record ParsedApiUrl(String scheme, String host, int port, List<String> pathSegments) {
    }

    private class AccessToken
    implements BitbucketToken<String> {
        private String token;

        public AccessToken(BitbucketRepositoryClient bitbucketRepositoryClient, String token) {
            this.token = token;
        }

        @Override
        public String getAuthzHeaderValue() {
            return "Bearer " + this.token;
        }
    }

    private static interface BitbucketToken<T> {
        public T getAuthzHeaderValue();
    }

    private class BasicAuthToken
    implements BitbucketToken<String> {
        private String token;

        public BasicAuthToken(BitbucketRepositoryClient bitbucketRepositoryClient, String username, String appPassword) {
            String basicCreds = username + ":" + appPassword;
            byte[] basicCredsBytes = basicCreds.getBytes(StandardCharsets.UTF_8);
            Base64.Encoder encoder = Base64.getEncoder();
            this.token = encoder.encodeToString(basicCredsBytes);
        }

        @Override
        public String getAuthzHeaderValue() {
            return "Basic " + this.token;
        }
    }

    private class OAuthToken
    implements BitbucketToken<String> {
        private OAuth2AccessTokenProvider oauthService;

        public OAuthToken(BitbucketRepositoryClient bitbucketRepositoryClient, OAuth2AccessTokenProvider oauthService) {
            this.oauthService = oauthService;
        }

        @Override
        public String getAuthzHeaderValue() {
            return "Bearer " + this.oauthService.getAccessDetails().getAccessToken();
        }
    }

    private static class HttpResponseBodyInputStream
    extends FilterInputStream {
        private final HttpResponseEntity response;
        private boolean closed;

        protected HttpResponseBodyInputStream(HttpResponseEntity response) {
            super(response.body());
            this.response = response;
        }

        @Override
        public void close() throws IOException {
            if (!this.closed) {
                this.closed = true;
                IOException suppressed = null;
                try {
                    super.close();
                }
                catch (IOException closeException) {
                    suppressed = closeException;
                }
                try {
                    this.response.close();
                }
                catch (IOException responseCloseException) {
                    if (suppressed != null) {
                        responseCloseException.addSuppressed(suppressed);
                    }
                    throw responseCloseException;
                }
                if (suppressed != null) {
                    throw suppressed;
                }
            }
        }
    }
}

