/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.atlassian.bitbucket;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import org.apache.nifi.atlassian.bitbucket.BitbucketAuthenticationType;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.oauth2.OAuth2AccessTokenProvider;
import org.apache.nifi.registry.flow.FlowRegistryException;
import org.apache.nifi.registry.flow.git.client.GitCommit;
import org.apache.nifi.registry.flow.git.client.GitCreateContentRequest;
import org.apache.nifi.registry.flow.git.client.GitRepositoryClient;
import org.apache.nifi.web.client.api.HttpContentType;
import org.apache.nifi.web.client.api.HttpResponseEntity;
import org.apache.nifi.web.client.api.HttpUriBuilder;
import org.apache.nifi.web.client.api.StandardHttpContentType;
import org.apache.nifi.web.client.api.StandardMultipartFormDataStreamBuilder;
import org.apache.nifi.web.client.provider.api.WebClientServiceProvider;

public class BitbucketRepositoryClient
implements GitRepositoryClient {
    private final ComponentLog logger;
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String CONTENT_TYPE_HEADER = "Content-Type";
    private static final String BASIC = "Basic";
    private static final String BEARER = "Bearer";
    private final ObjectMapper objectMapper = JsonMapper.builder().build();
    private final String apiUrl;
    private final String apiVersion;
    private final String clientId;
    private final String workspace;
    private final String repoName;
    private final String repoPath;
    private WebClientServiceProvider webClient;
    private BitbucketToken<String> authToken;
    private final boolean canRead;
    private final boolean canWrite;

    private BitbucketRepositoryClient(Builder builder) throws FlowRegistryException {
        String permission;
        this.webClient = Objects.requireNonNull(builder.webClient, "Web Client is required");
        this.workspace = Objects.requireNonNull(builder.workspace, "Workspace is required");
        this.repoName = Objects.requireNonNull(builder.repoName, "Repository Name is required");
        this.logger = Objects.requireNonNull(builder.logger, "ComponentLog required");
        this.apiUrl = Objects.requireNonNull(builder.apiUrl, "API Instance is required");
        this.apiVersion = Objects.requireNonNull(builder.apiVersion, "API Version is required");
        this.clientId = Objects.requireNonNull(builder.clientId, "Client ID is required");
        this.repoPath = builder.repoPath;
        BitbucketAuthenticationType authenticationType = Objects.requireNonNull(builder.authenticationType, "Authentication type is required");
        switch (authenticationType) {
            case ACCESS_TOKEN: {
                Objects.requireNonNull(builder.accessToken, "Access Token is required");
                this.authToken = new AccessToken(this, builder.accessToken);
                break;
            }
            case BASIC_AUTH: {
                Objects.requireNonNull(builder.username, "Username is required");
                Objects.requireNonNull(builder.appPassword, "App Password is required");
                this.authToken = new BasicAuthToken(this, builder.username, builder.appPassword);
                break;
            }
            case OAUTH2: {
                Objects.requireNonNull(builder.oauthService, "OAuth 2.0 Token Provider is required");
                this.authToken = new OAuthToken(this, builder.oauthService);
            }
        }
        switch (permission = this.checkRepoPermissions(authenticationType)) {
            case "admin": 
            case "write": {
                this.canRead = true;
                this.canWrite = true;
                break;
            }
            case "read": {
                this.canRead = true;
                this.canWrite = false;
                break;
            }
            case "none": {
                this.canRead = false;
                this.canWrite = false;
                break;
            }
            default: {
                this.canRead = false;
                this.canWrite = false;
            }
        }
        this.logger.info("Created {} for clientId = [{}], repository [{}]", new Object[]{this.getClass().getSimpleName(), this.clientId, this.repoName});
    }

    public boolean hasReadPermission() {
        return this.canRead;
    }

    public boolean hasWritePermission() {
        return this.canWrite;
    }

    public String getWorkspace() {
        return this.workspace;
    }

    public String getRepoName() {
        return this.repoName;
    }

    public Set<String> getBranches() throws FlowRegistryException {
        JsonNode jsonResponse;
        this.logger.debug("Getting branches for repository [{}]", new Object[]{this.repoName});
        URI uri = this.getUriBuilder().addPathSegment("refs").addPathSegment("branches").build();
        HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
        if (response.statusCode() != 200) {
            throw new FlowRegistryException(String.format("Error while listing branches for repository [%s]: %s", this.repoName, this.getErrorMessage(response)));
        }
        try {
            jsonResponse = this.objectMapper.readTree(response.body());
        }
        catch (IOException e) {
            throw new FlowRegistryException("Could not parse response from Bitbucket API", (Throwable)e);
        }
        Iterator branches = jsonResponse.get("values").elements();
        HashSet<String> result = new HashSet<String>();
        while (branches.hasNext()) {
            JsonNode branch = (JsonNode)branches.next();
            result.add(branch.get("name").asText());
        }
        return result;
    }

    public Set<String> getTopLevelDirectoryNames(String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath("");
        this.logger.debug("Getting top-level directories for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Iterator<JsonNode> files = this.getFiles(branch, resolvedPath);
        HashSet<String> result = new HashSet<String>();
        while (files.hasNext()) {
            JsonNode file = files.next();
            if (!file.get("type").asText().equals("commit_directory")) continue;
            Path fullPath = Paths.get(file.get("path").asText(), new String[0]);
            result.add(fullPath.getFileName().toString());
        }
        return result;
    }

    public Set<String> getFileNames(String directory, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(directory);
        this.logger.debug("Getting filenames for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Iterator<JsonNode> files = this.getFiles(branch, resolvedPath);
        HashSet<String> result = new HashSet<String>();
        while (files.hasNext()) {
            JsonNode file = files.next();
            if (!file.get("type").asText().equals("commit_file")) continue;
            Path fullPath = Paths.get(file.get("path").asText(), new String[0]);
            result.add(fullPath.getFileName().toString());
        }
        return result;
    }

    public List<GitCommit> getCommits(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting commits for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Iterator<JsonNode> commits = this.getListCommits(branch, resolvedPath);
        ArrayList<GitCommit> result = new ArrayList<GitCommit>();
        while (commits.hasNext()) {
            JsonNode commit = commits.next();
            result.add(this.toGitCommit(commit));
        }
        return result;
    }

    public InputStream getContentFromBranch(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting content for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        Optional<String> lastCommit = this.getLatestCommit(branch, resolvedPath);
        if (lastCommit.isEmpty()) {
            throw new FlowRegistryException(String.format("Could not find committed files at %s on branch %s response from Bitbucket API", resolvedPath, branch));
        }
        return this.getContentFromCommit(path, lastCommit.get());
    }

    public InputStream getContentFromCommit(String path, String commitSha) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting content for path [{}] from commit [{}] in repository [{}]", new Object[]{resolvedPath, commitSha, this.repoName});
        URI uri = this.getUriBuilder().addPathSegment("src").addPathSegment(commitSha).addPathSegment(resolvedPath).build();
        HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
        if (response.statusCode() != 200) {
            throw new FlowRegistryException(String.format("Error while retrieving content for repository [%s] at path %s: %s", this.repoName, resolvedPath, this.getErrorMessage(response)));
        }
        return response.body();
    }

    public Optional<String> getContentSha(String path, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(path);
        this.logger.debug("Getting content SHA for path [{}] on branch [{}] in repository [{}]", new Object[]{resolvedPath, branch, this.repoName});
        return this.getLatestCommit(branch, resolvedPath);
    }

    public String createContent(GitCreateContentRequest request) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(request.getPath());
        String branch = request.getBranch();
        this.logger.debug("Creating content at path [{}] on branch [{}] in repository [{}] ", new Object[]{resolvedPath, branch, this.repoName});
        StandardMultipartFormDataStreamBuilder multipartBuilder = new StandardMultipartFormDataStreamBuilder();
        multipartBuilder.addPart(resolvedPath, (HttpContentType)StandardHttpContentType.APPLICATION_JSON, request.getContent().getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart("message", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, request.getMessage().getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart("branch", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, branch.getBytes(StandardCharsets.UTF_8));
        URI uri = this.getUriBuilder().addPathSegment("src").build();
        HttpResponseEntity response = this.webClient.getWebClientService().post().uri(uri).body(multipartBuilder.build(), OptionalLong.empty()).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).header(CONTENT_TYPE_HEADER, multipartBuilder.getHttpContentType().getContentType()).retrieve();
        if (response.statusCode() != 201) {
            throw new FlowRegistryException(String.format("Error while committing content for repository [%s] on branch %s at path %s: %s", this.repoName, branch, resolvedPath, this.getErrorMessage(response)));
        }
        Optional<String> lastCommit = this.getLatestCommit(branch, resolvedPath);
        if (lastCommit.isEmpty()) {
            throw new FlowRegistryException(String.format("Could not find commit for the file %s we just tried to commit on branch %s", resolvedPath, branch));
        }
        return lastCommit.get();
    }

    public InputStream deleteContent(String filePath, String commitMessage, String branch) throws FlowRegistryException {
        String resolvedPath = this.getResolvedPath(filePath);
        this.logger.debug("Deleting content at path [{}] on branch [{}] in repository [{}] ", new Object[]{resolvedPath, branch, this.repoName});
        InputStream fileToBeDeleted = this.getContentFromBranch(filePath, branch);
        StandardMultipartFormDataStreamBuilder multipartBuilder = new StandardMultipartFormDataStreamBuilder();
        multipartBuilder.addPart("files", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, resolvedPath.getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart("message", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, commitMessage.getBytes(StandardCharsets.UTF_8));
        multipartBuilder.addPart("branch", (HttpContentType)StandardHttpContentType.TEXT_PLAIN, branch.getBytes(StandardCharsets.UTF_8));
        URI uri = this.getUriBuilder().addPathSegment("src").build();
        HttpResponseEntity response = this.webClient.getWebClientService().post().uri(uri).body(multipartBuilder.build(), OptionalLong.empty()).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).header(CONTENT_TYPE_HEADER, multipartBuilder.getHttpContentType().getContentType()).retrieve();
        if (response.statusCode() != 201) {
            throw new FlowRegistryException(String.format("Error while deleting content for repository [%s] on branch %s at path %s: %s", this.repoName, branch, resolvedPath, this.getErrorMessage(response)));
        }
        return fileToBeDeleted;
    }

    private Iterator<JsonNode> getFiles(String branch, String resolvedPath) throws FlowRegistryException {
        Optional<String> lastCommit = this.getLatestCommit(branch, resolvedPath);
        if (lastCommit.isEmpty()) {
            throw new FlowRegistryException(String.format("Could not find committed files at %s on branch %s response from Bitbucket API", resolvedPath, branch));
        }
        URI uri = this.getUriBuilder().addPathSegment("src").addPathSegment(lastCommit.get()).addPathSegment(resolvedPath).build();
        String errorMessage = String.format("Error while listing content for repository [%s] on branch %s at path %s", this.repoName, branch, resolvedPath);
        return this.getPagedResponseValues(uri, errorMessage);
    }

    private Iterator<JsonNode> getListCommits(String branch, String path) throws FlowRegistryException {
        URI uri = this.getUriBuilder().addPathSegment("commits").addPathSegment(branch).addQueryParameter("path", path).build();
        String errorMessage = String.format("Error while listing commits for repository [%s] on branch %s", this.repoName, branch);
        return this.getPagedResponseValues(uri, errorMessage);
    }

    private Iterator<JsonNode> getPagedResponseValues(URI uri, String errorMessage) throws FlowRegistryException {
        ArrayList allValues = new ArrayList();
        URI nextUri = uri;
        while (nextUri != null) {
            JsonNode next;
            JsonNode root;
            HttpResponseEntity response = this.webClient.getWebClientService().get().uri(nextUri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
            if (response.statusCode() != 200) {
                String responseErrorMessage = this.getErrorMessage(response);
                String errorMessageFormat = errorMessage + ": %s";
                throw new FlowRegistryException(errorMessageFormat.formatted(responseErrorMessage));
            }
            try {
                root = this.objectMapper.readTree(response.body());
            }
            catch (IOException e) {
                throw new FlowRegistryException(String.format("Could not parse Bitbucket API response at %s", nextUri), (Throwable)e);
            }
            JsonNode values = root.get("values");
            if (values != null && values.isArray()) {
                values.forEach(allValues::add);
            }
            nextUri = (next = root.get("next")) != null && next.isTextual() ? URI.create(next.asText()) : null;
        }
        return allValues.iterator();
    }

    private Optional<String> getLatestCommit(String branch, String path) throws FlowRegistryException {
        Iterator<JsonNode> commits = this.getListCommits(branch, path);
        if (commits.hasNext()) {
            return Optional.of(commits.next().get("hash").asText());
        }
        return Optional.empty();
    }

    private String checkRepoPermissions(BitbucketAuthenticationType authenticationType) throws FlowRegistryException {
        switch (authenticationType) {
            case OAUTH2: {
                JsonNode jsonResponse;
                this.logger.debug("Retrieving information about current user");
                URI uri = this.webClient.getHttpUriBuilder().scheme("https").host(this.apiUrl).addPathSegment(this.apiVersion).addPathSegment("user").addPathSegment("permissions").addPathSegment("repositories").addQueryParameter("q", "repository.name=\"" + this.repoName + "\"").build();
                HttpResponseEntity response = this.webClient.getWebClientService().get().uri(uri).header(AUTHORIZATION_HEADER, this.authToken.getAuthzHeaderValue()).retrieve();
                if (response.statusCode() != 200) {
                    throw new FlowRegistryException(String.format("Error while retrieving permission metadata for specified repo - %s", this.getErrorMessage(response)));
                }
                try {
                    jsonResponse = this.objectMapper.readTree(response.body());
                }
                catch (IOException e) {
                    throw new FlowRegistryException("Could not parse response from Bitbucket API", (Throwable)e);
                }
                Iterator repoPermissions = jsonResponse.get("values").elements();
                if (repoPermissions.hasNext()) {
                    return ((JsonNode)repoPermissions.next()).get("permission").asText();
                }
                return "none";
            }
            case ACCESS_TOKEN: 
            case BASIC_AUTH: {
                try {
                    this.getBranches();
                    return "admin";
                }
                catch (FlowRegistryException e) {
                    return "none";
                }
            }
        }
        return "none";
    }

    private GitCommit toGitCommit(JsonNode commit) {
        return new GitCommit(commit.get("hash").asText(), commit.get("author").get("raw").asText(), commit.get("message").asText(), Instant.parse(commit.get("date").asText()));
    }

    private String getErrorMessage(HttpResponseEntity response) throws FlowRegistryException {
        JsonNode jsonResponse;
        try {
            jsonResponse = this.objectMapper.readTree(response.body());
        }
        catch (IOException e) {
            throw new FlowRegistryException("Could not parse response from Bitbucket API", (Throwable)e);
        }
        return String.format("[%s] - %s", jsonResponse.get("type").asText(), jsonResponse.get("error").get("message").asText());
    }

    private String getResolvedPath(String path) {
        return this.repoPath == null ? path : this.repoPath + "/" + path;
    }

    private HttpUriBuilder getUriBuilder() {
        return this.webClient.getHttpUriBuilder().scheme("https").host(this.apiUrl).addPathSegment(this.apiVersion).addPathSegment("repositories").addPathSegment(this.workspace).addPathSegment(this.repoName);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String clientId;
        private String apiUrl;
        private String apiVersion;
        private BitbucketAuthenticationType authenticationType;
        private String accessToken;
        private String username;
        private String appPassword;
        private OAuth2AccessTokenProvider oauthService;
        private WebClientServiceProvider webClient;
        private String workspace;
        private String repoName;
        private String repoPath;
        private ComponentLog logger;

        public Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public Builder apiUrl(String apiUrl) {
            this.apiUrl = apiUrl;
            return this;
        }

        public Builder apiVersion(String apiVersion) {
            this.apiVersion = apiVersion;
            return this;
        }

        public Builder authenticationType(BitbucketAuthenticationType authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        public Builder accessToken(String accessToken) {
            this.accessToken = accessToken;
            return this;
        }

        public Builder username(String username) {
            this.username = username;
            return this;
        }

        public Builder appPassword(String appPassword) {
            this.appPassword = appPassword;
            return this;
        }

        public Builder oauthService(OAuth2AccessTokenProvider oauthService) {
            this.oauthService = oauthService;
            return this;
        }

        public Builder webClient(WebClientServiceProvider webClient) {
            this.webClient = webClient;
            return this;
        }

        public Builder workspace(String workspace) {
            this.workspace = workspace;
            return this;
        }

        public Builder repoName(String repoName) {
            this.repoName = repoName;
            return this;
        }

        public Builder repoPath(String repoPath) {
            this.repoPath = repoPath;
            return this;
        }

        public Builder logger(ComponentLog logger) {
            this.logger = logger;
            return this;
        }

        public BitbucketRepositoryClient build() throws FlowRegistryException {
            return new BitbucketRepositoryClient(this);
        }
    }

    private class AccessToken
    implements BitbucketToken<String> {
        private String token;

        public AccessToken(BitbucketRepositoryClient bitbucketRepositoryClient, String token) {
            this.token = token;
        }

        @Override
        public String getAuthzHeaderValue() {
            return "Bearer " + this.token;
        }
    }

    private static interface BitbucketToken<T> {
        public T getAuthzHeaderValue();
    }

    private class BasicAuthToken
    implements BitbucketToken<String> {
        private String token;

        public BasicAuthToken(BitbucketRepositoryClient bitbucketRepositoryClient, String username, String appPassword) {
            String basicCreds = username + ":" + appPassword;
            byte[] basicCredsBytes = basicCreds.getBytes(StandardCharsets.UTF_8);
            Base64.Encoder encoder = Base64.getEncoder();
            this.token = encoder.encodeToString(basicCredsBytes);
        }

        @Override
        public String getAuthzHeaderValue() {
            return "Basic " + this.token;
        }
    }

    private class OAuthToken
    implements BitbucketToken<String> {
        private OAuth2AccessTokenProvider oauthService;

        public OAuthToken(BitbucketRepositoryClient bitbucketRepositoryClient, OAuth2AccessTokenProvider oauthService) {
            this.oauthService = oauthService;
        }

        @Override
        public String getAuthzHeaderValue() {
            return "Bearer " + this.oauthService.getAccessDetails().getAccessToken();
        }
    }
}

