/*
 * Decompiled with CFR 0.152.
 */
package org.apache.maven.scm.provider.git.jgit.command;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.scm.ScmFile;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmFileStatus;
import org.apache.maven.scm.provider.git.jgit.command.PushException;
import org.apache.maven.scm.provider.git.repository.GitScmProviderRepository;
import org.apache.maven.scm.util.FilenameUtils;
import org.eclipse.jgit.api.AddCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.api.RmCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.TransportConfigCallback;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.diff.DiffEntry;
import org.eclipse.jgit.diff.DiffFormatter;
import org.eclipse.jgit.diff.RawTextComparator;
import org.eclipse.jgit.errors.CorruptObjectException;
import org.eclipse.jgit.errors.IncorrectObjectTypeException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.errors.StopWalkException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryBuilder;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.lib.TextProgressMonitor;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevFlag;
import org.eclipse.jgit.revwalk.RevSort;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.revwalk.filter.CommitTimeRevFilter;
import org.eclipse.jgit.revwalk.filter.RevFilter;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.RemoteRefUpdate;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.eclipse.jgit.util.io.DisabledOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JGitUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(JGitUtils.class);

    private JGitUtils() {
    }

    public static Git openRepo(File basedir) throws IOException {
        return new Git(((RepositoryBuilder)((RepositoryBuilder)((RepositoryBuilder)new RepositoryBuilder().readEnvironment()).findGitDir(basedir)).setMustExist(true)).build());
    }

    public static void closeRepo(Git git) {
        if (git != null && git.getRepository() != null) {
            git.getRepository().close();
        }
    }

    public static ProgressMonitor getMonitor() {
        return new TextProgressMonitor();
    }

    public static CredentialsProvider prepareSession(Git git, GitScmProviderRepository repository) {
        StoredConfig config = git.getRepository().getConfig();
        config.setString("remote", "origin", "url", repository.getFetchUrl());
        config.setString("remote", "origin", "pushURL", repository.getPushUrl());
        LOGGER.info("fetch url: " + repository.getFetchUrlWithMaskedPassword());
        LOGGER.info("push url: " + repository.getPushUrlWithMaskedPassword());
        return JGitUtils.getCredentials(repository);
    }

    public static CredentialsProvider getCredentials(GitScmProviderRepository repository) {
        if (StringUtils.isNotBlank((CharSequence)repository.getUser()) && StringUtils.isNotBlank((CharSequence)repository.getPassword())) {
            return new UsernamePasswordCredentialsProvider(repository.getUser().trim(), repository.getPassword().trim());
        }
        return null;
    }

    public static Iterable<PushResult> push(Git git, GitScmProviderRepository repo, RefSpec refSpec, Set<RemoteRefUpdate.Status> successfulStatuses, Optional<TransportConfigCallback> transportConfigCallback) throws PushException {
        Iterable pushResultList;
        CredentialsProvider credentials = JGitUtils.prepareSession(git, repo);
        PushCommand command = (PushCommand)git.push().setRefSpecs(new RefSpec[]{refSpec}).setCredentialsProvider(credentials);
        transportConfigCallback.ifPresent(arg_0 -> ((PushCommand)command).setTransportConfigCallback(arg_0));
        try {
            pushResultList = command.call();
        }
        catch (GitAPIException e) {
            throw new PushException(repo.getPushUrlWithMaskedPassword(), e);
        }
        for (PushResult pushResult : pushResultList) {
            Collection ru = pushResult.getRemoteUpdates();
            for (RemoteRefUpdate remoteRefUpdate : ru) {
                if (!successfulStatuses.contains(remoteRefUpdate.getStatus())) {
                    throw new PushException(repo.getPushUrlWithMaskedPassword(), remoteRefUpdate);
                }
                LOGGER.debug("Push succeeded {}", (Object)remoteRefUpdate);
            }
        }
        return pushResultList;
    }

    public static boolean hasCommits(Repository repo) {
        if (repo != null && repo.getDirectory().exists()) {
            return new File(repo.getDirectory(), "objects").list().length > 2 || new File(repo.getDirectory(), "objects/pack").list().length > 0;
        }
        return false;
    }

    public static List<ScmFile> getFilesInCommit(Repository repository, RevCommit commit) throws MissingObjectException, IncorrectObjectTypeException, CorruptObjectException, IOException {
        return JGitUtils.getFilesInCommit(repository, commit, null);
    }

    public static List<ScmFile> getFilesInCommit(Repository repository, RevCommit commit, File baseDir) throws MissingObjectException, IncorrectObjectTypeException, CorruptObjectException, IOException {
        ArrayList<ScmFile> list = new ArrayList<ScmFile>();
        if (JGitUtils.hasCommits(repository)) {
            try (RevWalk rw = new RevWalk(repository);
                 DiffFormatter df = new DiffFormatter((OutputStream)DisabledOutputStream.INSTANCE);){
                RevCommit realParent = commit.getParentCount() > 0 ? commit.getParent(0) : commit;
                RevCommit parent = rw.parseCommit((AnyObjectId)realParent.getId());
                df.setRepository(repository);
                df.setDiffComparator(RawTextComparator.DEFAULT);
                df.setDetectRenames(true);
                List diffs = df.scan(parent.getTree(), commit.getTree());
                for (DiffEntry diff : diffs) {
                    String path = baseDir != null ? JGitUtils.relativize(baseDir, new File(repository.getWorkTree(), diff.getNewPath())).getPath() : diff.getNewPath();
                    list.add(new ScmFile(path, ScmFileStatus.CHECKED_IN));
                }
            }
        }
        return list;
    }

    public static ScmFileStatus getScmFileStatus(DiffEntry.ChangeType changeType) {
        switch (changeType) {
            case ADD: {
                return ScmFileStatus.ADDED;
            }
            case MODIFY: {
                return ScmFileStatus.MODIFIED;
            }
            case DELETE: {
                return ScmFileStatus.DELETED;
            }
            case RENAME: {
                return ScmFileStatus.RENAMED;
            }
            case COPY: {
                return ScmFileStatus.COPIED;
            }
        }
        return ScmFileStatus.UNKNOWN;
    }

    public static List<ScmFile> addAllFiles(Git git, ScmFileSet fileSet) throws GitAPIException {
        File workingCopyRootDirectory = git.getRepository().getWorkTree();
        AddCommand add = git.add();
        JGitUtils.getWorkingCopyRelativePaths(workingCopyRootDirectory, fileSet).stream().forEach(f -> add.addFilepattern(JGitUtils.toNormalizedFilePath(f)));
        add.call();
        Status status = git.status().call();
        HashSet<String> allInIndex = new HashSet<String>();
        allInIndex.addAll(status.getAdded());
        allInIndex.addAll(status.getChanged());
        return JGitUtils.getScmFilesForAllFileSetFilesContainedInRepoPath(workingCopyRootDirectory, fileSet, allInIndex, ScmFileStatus.ADDED);
    }

    public static List<ScmFile> removeAllFiles(Git git, ScmFileSet fileSet) throws GitAPIException {
        File workingCopyRootDirectory = git.getRepository().getWorkTree();
        RmCommand remove = git.rm();
        JGitUtils.getWorkingCopyRelativePaths(workingCopyRootDirectory, fileSet).stream().forEach(f -> remove.addFilepattern(JGitUtils.toNormalizedFilePath(f)));
        remove.call();
        Status status = git.status().call();
        HashSet<String> allInIndex = new HashSet<String>(status.getRemoved());
        return JGitUtils.getScmFilesForAllFileSetFilesContainedInRepoPath(workingCopyRootDirectory, fileSet, allInIndex, ScmFileStatus.DELETED);
    }

    public static List<File> getWorkingCopyRelativePaths(File workingCopyDirectory, ScmFileSet fileSet) {
        ArrayList<File> repositoryRelativePaths = new ArrayList<File>();
        for (File path : fileSet.getFileList()) {
            if (!path.isAbsolute()) {
                path = new File(fileSet.getBasedir().getPath(), path.getPath());
            }
            File repositoryRelativePath = JGitUtils.relativize(workingCopyDirectory, path);
            repositoryRelativePaths.add(repositoryRelativePath);
        }
        return repositoryRelativePaths;
    }

    public static String toNormalizedFilePath(File file) {
        return FilenameUtils.normalizeFilename((File)file);
    }

    private static List<ScmFile> getScmFilesForAllFileSetFilesContainedInRepoPath(File workingCopyDirectory, ScmFileSet fileSet, Set<String> repoFilePaths, ScmFileStatus fileStatus) {
        ArrayList<ScmFile> files = new ArrayList<ScmFile>(repoFilePaths.size());
        JGitUtils.getWorkingCopyRelativePaths(workingCopyDirectory, fileSet).stream().forEach(relativeFile -> {
            if (repoFilePaths.contains(JGitUtils.toNormalizedFilePath(relativeFile))) {
                ScmFile scmfile = new ScmFile(JGitUtils.relativize(fileSet.getBasedir(), new File(workingCopyDirectory, relativeFile.getPath())).getPath(), fileStatus);
                files.add(scmfile);
            }
        });
        return files;
    }

    private static File relativize(File baseDir, File file) {
        if (file.isAbsolute()) {
            return baseDir.toPath().relativize(file.toPath()).toFile();
        }
        return file;
    }

    public static List<RevCommit> getRevCommits(Repository repo, RevSort[] sortings, String fromRev, String toRev, final Date fromDate, final Date toDate, int maxLines) throws IOException, MissingObjectException, IncorrectObjectTypeException {
        ObjectId toRevId;
        ArrayList<RevCommit> revs = new ArrayList<RevCommit>();
        ObjectId fromRevId = fromRev != null ? repo.resolve(fromRev) : null;
        ObjectId objectId = toRevId = toRev != null ? repo.resolve(toRev) : null;
        if (sortings == null || sortings.length == 0) {
            sortings = new RevSort[]{RevSort.TOPO, RevSort.COMMIT_TIME_DESC};
        }
        try (RevWalk walk = new RevWalk(repo);){
            RevCommit real;
            RevCommit c;
            for (RevSort s : sortings) {
                walk.sort(s, true);
            }
            if (fromDate != null && toDate != null) {
                walk.setRevFilter(new RevFilter(){

                    public boolean include(RevWalk walker, RevCommit cmit) throws StopWalkException, MissingObjectException, IncorrectObjectTypeException, IOException {
                        int cmtTime = cmit.getCommitTime();
                        return (long)cmtTime >= fromDate.getTime() / 1000L && (long)cmtTime <= toDate.getTime() / 1000L;
                    }

                    public RevFilter clone() {
                        return this;
                    }
                });
            } else {
                if (fromDate != null) {
                    walk.setRevFilter(CommitTimeRevFilter.after((Date)fromDate));
                }
                if (toDate != null) {
                    walk.setRevFilter(CommitTimeRevFilter.before((Date)toDate));
                }
            }
            if (fromRevId != null) {
                c = walk.parseCommit((AnyObjectId)fromRevId);
                c.add(RevFlag.UNINTERESTING);
                RevCommit real2 = walk.parseCommit((AnyObjectId)c);
                walk.markUninteresting(real2);
            }
            if (toRevId != null) {
                c = walk.parseCommit((AnyObjectId)toRevId);
                c.remove(RevFlag.UNINTERESTING);
                real = walk.parseCommit((AnyObjectId)c);
                walk.markStart(real);
            } else {
                ObjectId head = repo.resolve("HEAD");
                if (head == null) {
                    throw new IOException("Cannot resolve HEAD");
                }
                real = walk.parseCommit((AnyObjectId)head);
                walk.markStart(real);
            }
            int n = 0;
            for (RevCommit c2 : walk) {
                if (maxLines != -1 && ++n > maxLines) break;
                revs.add(c2);
            }
            ArrayList<RevCommit> arrayList = revs;
            return arrayList;
        }
    }

    public static List<String> getTags(Repository repo, RevCommit commit) throws IOException {
        List refList = repo.getRefDatabase().getRefsByPrefix("refs/tags/");
        try (RevWalk revWalk = new RevWalk(repo);){
            ObjectId commitId = commit.getId();
            ArrayList<String> result = new ArrayList<String>();
            for (Ref ref : refList) {
                ObjectId tagId = ref.getObjectId();
                RevCommit tagCommit = revWalk.parseCommit((AnyObjectId)tagId);
                if (!commitId.equals((AnyObjectId)tagCommit.getId())) continue;
                result.add(ref.getName().substring("refs/tags/".length()));
            }
            ArrayList<String> arrayList = result;
            return arrayList;
        }
    }
}

