/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.PrintStream;
import java.net.InetSocketAddress;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.management.ObjectName;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.ha.HAServiceProtocol;
import org.apache.hadoop.ha.HAServiceStatus;
import org.apache.hadoop.ha.HealthCheckFailedException;
import org.apache.hadoop.ha.ServiceFailedException;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HAUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.protocol.ClientProtocol;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.BackupNode;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOp;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOpCodes;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormat;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.MetaRecoveryContext;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NameNodeHttpServer;
import org.apache.hadoop.hdfs.server.namenode.NameNodeRpcServer;
import org.apache.hadoop.hdfs.server.namenode.NameNodeStatusMXBean;
import org.apache.hadoop.hdfs.server.namenode.ha.ActiveState;
import org.apache.hadoop.hdfs.server.namenode.ha.BootstrapStandby;
import org.apache.hadoop.hdfs.server.namenode.ha.HAContext;
import org.apache.hadoop.hdfs.server.namenode.ha.HAState;
import org.apache.hadoop.hdfs.server.namenode.ha.StandbyState;
import org.apache.hadoop.hdfs.server.namenode.metrics.NameNodeMetrics;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StartupProgress;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StartupProgressMetrics;
import org.apache.hadoop.hdfs.server.protocol.DatanodeProtocol;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocol;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocols;
import org.apache.hadoop.hdfs.server.protocol.NamenodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.ipc.RefreshCallQueueProtocol;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.ipc.StandbyException;
import org.apache.hadoop.metrics2.lib.DefaultMetricsSystem;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.RefreshUserMappingsProtocol;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.authorize.RefreshAuthorizationPolicyProtocol;
import org.apache.hadoop.tools.GetUserMappingsProtocol;
import org.apache.hadoop.tracing.SpanReceiverHost;
import org.apache.hadoop.tracing.TraceAdminProtocol;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.JvmPauseMonitor;
import org.apache.hadoop.util.ServicePlugin;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class NameNode
implements NameNodeStatusMXBean {
    public static final String[] NAMENODE_SPECIFIC_KEYS;
    public static final String[] NAMESERVICE_SPECIFIC_KEYS;
    private static final String USAGE;
    public static final int DEFAULT_PORT = 8020;
    public static final Logger LOG;
    public static final Logger stateChangeLog;
    public static final Logger blockStateChangeLog;
    public static final HAState ACTIVE_STATE;
    public static final HAState STANDBY_STATE;
    protected FSNamesystem namesystem;
    protected final Configuration conf;
    protected final HdfsServerConstants.NamenodeRole role;
    private volatile HAState state;
    private final boolean haEnabled;
    private final HAContext haContext;
    protected final boolean allowStaleStandbyReads;
    private AtomicBoolean started = new AtomicBoolean(false);
    protected NameNodeHttpServer httpServer;
    private Thread emptier;
    protected boolean stopRequested = false;
    protected NamenodeRegistration nodeRegistration;
    private List<ServicePlugin> plugins;
    private NameNodeRpcServer rpcServer;
    private JvmPauseMonitor pauseMonitor;
    private ObjectName nameNodeStatusBeanName;
    SpanReceiverHost spanReceiverHost;
    private String clientNamenodeAddress;
    static NameNodeMetrics metrics;
    private static final StartupProgress startupProgress;

    public long getProtocolVersion(String protocol, long clientVersion) throws IOException {
        if (protocol.equals(ClientProtocol.class.getName())) {
            return 69L;
        }
        if (protocol.equals(DatanodeProtocol.class.getName())) {
            return 28L;
        }
        if (protocol.equals(NamenodeProtocol.class.getName())) {
            return 6L;
        }
        if (protocol.equals(RefreshAuthorizationPolicyProtocol.class.getName())) {
            return 1L;
        }
        if (protocol.equals(RefreshUserMappingsProtocol.class.getName())) {
            return 1L;
        }
        if (protocol.equals(RefreshCallQueueProtocol.class.getName())) {
            return 1L;
        }
        if (protocol.equals(GetUserMappingsProtocol.class.getName())) {
            return 1L;
        }
        if (protocol.equals(TraceAdminProtocol.class.getName())) {
            return 1L;
        }
        throw new IOException("Unknown protocol to name node: " + protocol);
    }

    public static void format(Configuration conf) throws IOException {
        NameNode.format(conf, true, true);
    }

    public FSNamesystem getNamesystem() {
        return this.namesystem;
    }

    public NamenodeProtocols getRpcServer() {
        return this.rpcServer;
    }

    static void initMetrics(Configuration conf, HdfsServerConstants.NamenodeRole role) {
        metrics = NameNodeMetrics.create(conf, role);
    }

    public static NameNodeMetrics getNameNodeMetrics() {
        return metrics;
    }

    public static StartupProgress getStartupProgress() {
        return startupProgress;
    }

    public String getTokenServiceName() {
        return this.getClientNamenodeAddress();
    }

    public void setClientNamenodeAddress(Configuration conf) {
        String nnAddr = conf.get("fs.defaultFS");
        if (nnAddr == null) {
            this.clientNamenodeAddress = null;
            return;
        }
        LOG.info("{} is {}", (Object)"fs.defaultFS", (Object)nnAddr);
        URI nnUri = URI.create(nnAddr);
        String nnHost = nnUri.getHost();
        if (nnHost == null) {
            this.clientNamenodeAddress = null;
            return;
        }
        if (DFSUtil.getNameServiceIds(conf).contains(nnHost)) {
            this.clientNamenodeAddress = nnHost;
        } else if (nnUri.getPort() > 0) {
            this.clientNamenodeAddress = nnUri.getAuthority();
        } else {
            this.clientNamenodeAddress = null;
            return;
        }
        LOG.info("Clients are to use {} to access this namenode/service.", (Object)this.clientNamenodeAddress);
    }

    public String getClientNamenodeAddress() {
        return this.clientNamenodeAddress;
    }

    public static InetSocketAddress getAddress(String address) {
        return NetUtils.createSocketAddr(address, 8020);
    }

    public static void setServiceAddress(Configuration conf, String address) {
        LOG.info("Setting ADDRESS {}", (Object)address);
        conf.set("dfs.namenode.servicerpc-address", address);
    }

    public static InetSocketAddress getServiceAddress(Configuration conf, boolean fallback) {
        String addr = conf.getTrimmed("dfs.namenode.servicerpc-address");
        if (addr == null || addr.isEmpty()) {
            return fallback ? NameNode.getAddress(conf) : null;
        }
        return NameNode.getAddress(addr);
    }

    public static InetSocketAddress getAddress(Configuration conf) {
        URI filesystemURI = FileSystem.getDefaultUri(conf);
        return NameNode.getAddress(filesystemURI);
    }

    public static InetSocketAddress getAddress(URI filesystemURI) {
        String authority = filesystemURI.getAuthority();
        if (authority == null) {
            throw new IllegalArgumentException(String.format("Invalid URI for NameNode address (check %s): %s has no authority.", "fs.defaultFS", filesystemURI.toString()));
        }
        if (!"hdfs".equalsIgnoreCase(filesystemURI.getScheme())) {
            throw new IllegalArgumentException(String.format("Invalid URI for NameNode address (check %s): %s is not of scheme '%s'.", "fs.defaultFS", filesystemURI.toString(), "hdfs"));
        }
        return NameNode.getAddress(authority);
    }

    public static URI getUri(InetSocketAddress namenode) {
        int port = namenode.getPort();
        String portString = port == 8020 ? "" : ":" + port;
        return URI.create("hdfs://" + namenode.getHostName() + portString);
    }

    public HdfsServerConstants.NamenodeRole getRole() {
        return this.role;
    }

    boolean isRole(HdfsServerConstants.NamenodeRole that) {
        return this.role.equals((Object)that);
    }

    protected InetSocketAddress getServiceRpcServerAddress(Configuration conf) {
        return NameNode.getServiceAddress(conf, false);
    }

    protected InetSocketAddress getRpcServerAddress(Configuration conf) {
        return NameNode.getAddress(conf);
    }

    protected String getServiceRpcServerBindHost(Configuration conf) {
        String addr = conf.getTrimmed("dfs.namenode.servicerpc-bind-host");
        if (addr == null || addr.isEmpty()) {
            return null;
        }
        return addr;
    }

    protected String getRpcServerBindHost(Configuration conf) {
        String addr = conf.getTrimmed("dfs.namenode.rpc-bind-host");
        if (addr == null || addr.isEmpty()) {
            return null;
        }
        return addr;
    }

    protected void setRpcServiceServerAddress(Configuration conf, InetSocketAddress serviceRPCAddress) {
        NameNode.setServiceAddress(conf, NetUtils.getHostPortString(serviceRPCAddress));
    }

    protected void setRpcServerAddress(Configuration conf, InetSocketAddress rpcAddress) {
        FileSystem.setDefaultUri(conf, NameNode.getUri(rpcAddress));
    }

    protected InetSocketAddress getHttpServerAddress(Configuration conf) {
        return NameNode.getHttpAddress(conf);
    }

    protected InetSocketAddress getHttpServerBindAddress(Configuration conf) {
        InetSocketAddress bindAddress = this.getHttpServerAddress(conf);
        String bindHost = conf.getTrimmed("dfs.namenode.http-bind-host");
        if (bindHost != null && !bindHost.isEmpty()) {
            bindAddress = new InetSocketAddress(bindHost, bindAddress.getPort());
        }
        return bindAddress;
    }

    public static InetSocketAddress getHttpAddress(Configuration conf) {
        return NetUtils.createSocketAddr(conf.getTrimmed("dfs.namenode.http-address", "0.0.0.0:50070"));
    }

    protected void loadNamesystem(Configuration conf) throws IOException {
        this.namesystem = FSNamesystem.loadFromDisk(conf);
    }

    NamenodeRegistration getRegistration() {
        return this.nodeRegistration;
    }

    NamenodeRegistration setRegistration() {
        this.nodeRegistration = new NamenodeRegistration(NetUtils.getHostPortString(this.rpcServer.getRpcAddress()), NetUtils.getHostPortString(this.getHttpAddress()), this.getFSImage().getStorage(), this.getRole());
        return this.nodeRegistration;
    }

    public static UserGroupInformation getRemoteUser() throws IOException {
        UserGroupInformation ugi = Server.getRemoteUser();
        return ugi != null ? ugi : UserGroupInformation.getCurrentUser();
    }

    void loginAsNameNodeUser(Configuration conf) throws IOException {
        InetSocketAddress socAddr = this.getRpcServerAddress(conf);
        SecurityUtil.login(conf, "dfs.namenode.keytab.file", "dfs.namenode.kerberos.principal", socAddr.getHostName());
    }

    protected void initialize(Configuration conf) throws IOException {
        String intervals;
        if (conf.get("hadoop.user.group.metrics.percentiles.intervals") == null && (intervals = conf.get("dfs.metrics.percentiles.intervals")) != null) {
            conf.set("hadoop.user.group.metrics.percentiles.intervals", intervals);
        }
        UserGroupInformation.setConfiguration(conf);
        this.loginAsNameNodeUser(conf);
        NameNode.initMetrics(conf, this.getRole());
        StartupProgressMetrics.register(startupProgress);
        if (HdfsServerConstants.NamenodeRole.NAMENODE == this.role) {
            this.startHttpServer(conf);
        }
        this.spanReceiverHost = SpanReceiverHost.get(conf, "dfs.htrace.");
        this.loadNamesystem(conf);
        this.rpcServer = this.createRpcServer(conf);
        if (this.clientNamenodeAddress == null) {
            this.clientNamenodeAddress = NetUtils.getHostPortString(this.rpcServer.getRpcAddress());
            LOG.info("Clients are to use " + this.clientNamenodeAddress + " to access" + " this namenode/service.");
        }
        if (HdfsServerConstants.NamenodeRole.NAMENODE == this.role) {
            this.httpServer.setNameNodeAddress(this.getNameNodeAddress());
            this.httpServer.setFSImage(this.getFSImage());
        }
        this.pauseMonitor = new JvmPauseMonitor(conf);
        this.pauseMonitor.start();
        metrics.getJvmMetrics().setPauseMonitor(this.pauseMonitor);
        this.startCommonServices(conf);
    }

    protected NameNodeRpcServer createRpcServer(Configuration conf) throws IOException {
        return new NameNodeRpcServer(conf, this);
    }

    private void startCommonServices(Configuration conf) throws IOException {
        this.namesystem.startCommonServices(conf, this.haContext);
        this.registerNNSMXBean();
        if (HdfsServerConstants.NamenodeRole.NAMENODE != this.role) {
            this.startHttpServer(conf);
            this.httpServer.setNameNodeAddress(this.getNameNodeAddress());
            this.httpServer.setFSImage(this.getFSImage());
        }
        this.rpcServer.start();
        this.plugins = conf.getInstances("dfs.namenode.plugins", ServicePlugin.class);
        for (ServicePlugin p : this.plugins) {
            try {
                p.start(this);
            }
            catch (Throwable t) {
                LOG.warn("ServicePlugin " + p + " could not be started", t);
            }
        }
        LOG.info((Object)((Object)this.getRole()) + " RPC up at: " + this.rpcServer.getRpcAddress());
        if (this.rpcServer.getServiceRpcAddress() != null) {
            LOG.info((Object)((Object)this.getRole()) + " service RPC up at: " + this.rpcServer.getServiceRpcAddress());
        }
    }

    private void stopCommonServices() {
        if (this.rpcServer != null) {
            this.rpcServer.stop();
        }
        if (this.namesystem != null) {
            this.namesystem.close();
        }
        if (this.pauseMonitor != null) {
            this.pauseMonitor.stop();
        }
        if (this.plugins != null) {
            for (ServicePlugin p : this.plugins) {
                try {
                    p.stop();
                }
                catch (Throwable t) {
                    LOG.warn("ServicePlugin " + p + " could not be stopped", t);
                }
            }
        }
        this.stopHttpServer();
    }

    private void startTrashEmptier(final Configuration conf) throws IOException {
        long trashInterval = conf.getLong("fs.trash.interval", 0L);
        if (trashInterval == 0L) {
            return;
        }
        if (trashInterval < 0L) {
            throw new IOException("Cannot start trash emptier with negative interval. Set fs.trash.interval to a positive value.");
        }
        FileSystem fs = SecurityUtil.doAsLoginUser(new PrivilegedExceptionAction<FileSystem>(){

            @Override
            public FileSystem run() throws IOException {
                return FileSystem.get(conf);
            }
        });
        this.emptier = new Thread(new Trash(fs, conf).getEmptier(), "Trash Emptier");
        this.emptier.setDaemon(true);
        this.emptier.start();
    }

    private void stopTrashEmptier() {
        if (this.emptier != null) {
            this.emptier.interrupt();
            this.emptier = null;
        }
    }

    private void startHttpServer(Configuration conf) throws IOException {
        this.httpServer = new NameNodeHttpServer(conf, this, this.getHttpServerBindAddress(conf));
        this.httpServer.start();
        this.httpServer.setStartupProgress(startupProgress);
    }

    private void stopHttpServer() {
        try {
            if (this.httpServer != null) {
                this.httpServer.stop();
            }
        }
        catch (Exception e) {
            LOG.error("Exception while stopping httpserver", e);
        }
    }

    public NameNode(Configuration conf) throws IOException {
        this(conf, HdfsServerConstants.NamenodeRole.NAMENODE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected NameNode(Configuration conf, HdfsServerConstants.NamenodeRole role) throws IOException {
        this.conf = conf;
        this.role = role;
        this.setClientNamenodeAddress(conf);
        String nsId = this.getNameServiceId(conf);
        String namenodeId = HAUtil.getNameNodeId(conf, nsId);
        this.haEnabled = HAUtil.isHAEnabled(conf, nsId);
        this.state = this.createHAState(NameNode.getStartupOption(conf));
        this.allowStaleStandbyReads = HAUtil.shouldAllowStandbyReads(conf);
        this.haContext = this.createHAContext();
        try {
            NameNode.initializeGenericKeys(conf, nsId, namenodeId);
            this.initialize(conf);
            try {
                this.haContext.writeLock();
                this.state.prepareToEnterState(this.haContext);
                this.state.enterState(this.haContext);
            }
            finally {
                this.haContext.writeUnlock();
            }
        }
        catch (IOException e) {
            this.stop();
            throw e;
        }
        catch (HadoopIllegalArgumentException e) {
            this.stop();
            throw e;
        }
        this.started.set(true);
    }

    protected HAState createHAState(HdfsServerConstants.StartupOption startOpt) {
        if (!this.haEnabled || startOpt == HdfsServerConstants.StartupOption.UPGRADE || startOpt == HdfsServerConstants.StartupOption.UPGRADEONLY) {
            return ACTIVE_STATE;
        }
        return STANDBY_STATE;
    }

    protected HAContext createHAContext() {
        return new NameNodeHAContext();
    }

    public void join() {
        try {
            this.rpcServer.join();
        }
        catch (InterruptedException ie) {
            LOG.info("Caught interrupted exception ", ie);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        NameNode nameNode = this;
        synchronized (nameNode) {
            if (this.stopRequested) {
                return;
            }
            this.stopRequested = true;
        }
        try {
            if (this.state != null) {
                this.state.exitState(this.haContext);
            }
        }
        catch (ServiceFailedException e) {
            LOG.warn("Encountered exception while exiting state ", e);
        }
        finally {
            this.stopCommonServices();
            if (metrics != null) {
                metrics.shutdown();
            }
            if (this.namesystem != null) {
                this.namesystem.shutdown();
            }
            if (this.nameNodeStatusBeanName != null) {
                MBeans.unregister(this.nameNodeStatusBeanName);
                this.nameNodeStatusBeanName = null;
            }
            if (this.spanReceiverHost != null) {
                this.spanReceiverHost.closeReceivers();
            }
        }
    }

    synchronized boolean isStopRequested() {
        return this.stopRequested;
    }

    public boolean isInSafeMode() {
        return this.namesystem.isInSafeMode();
    }

    @VisibleForTesting
    public FSImage getFSImage() {
        return this.namesystem.getFSImage();
    }

    public InetSocketAddress getNameNodeAddress() {
        return this.rpcServer.getRpcAddress();
    }

    public String getNameNodeAddressHostPortString() {
        return NetUtils.getHostPortString(this.rpcServer.getRpcAddress());
    }

    public InetSocketAddress getServiceRpcAddress() {
        InetSocketAddress serviceAddr = this.rpcServer.getServiceRpcAddress();
        return serviceAddr == null ? this.rpcServer.getRpcAddress() : serviceAddr;
    }

    public InetSocketAddress getHttpAddress() {
        return this.httpServer.getHttpAddress();
    }

    public InetSocketAddress getHttpsAddress() {
        return this.httpServer.getHttpsAddress();
    }

    private static boolean format(Configuration conf, boolean force, boolean isInteractive) throws IOException {
        String nsId = DFSUtil.getNamenodeNameServiceId(conf);
        String namenodeId = HAUtil.getNameNodeId(conf, nsId);
        NameNode.initializeGenericKeys(conf, nsId, namenodeId);
        NameNode.checkAllowFormat(conf);
        if (UserGroupInformation.isSecurityEnabled()) {
            InetSocketAddress socAddr = NameNode.getAddress(conf);
            SecurityUtil.login(conf, "dfs.namenode.keytab.file", "dfs.namenode.kerberos.principal", socAddr.getHostName());
        }
        Collection<URI> nameDirsToFormat = FSNamesystem.getNamespaceDirs(conf);
        List<URI> sharedDirs = FSNamesystem.getSharedEditsDirs(conf);
        ArrayList<URI> dirsToPrompt = new ArrayList<URI>();
        dirsToPrompt.addAll(nameDirsToFormat);
        dirsToPrompt.addAll(sharedDirs);
        List<URI> editDirsToFormat = FSNamesystem.getNamespaceEditsDirs(conf);
        String clusterId = HdfsServerConstants.StartupOption.FORMAT.getClusterId();
        if (clusterId == null || clusterId.equals("")) {
            clusterId = NNStorage.newClusterID();
        }
        System.out.println("Formatting using clusterid: " + clusterId);
        FSImage fsImage = new FSImage(conf, nameDirsToFormat, editDirsToFormat);
        try {
            FSNamesystem fsn = new FSNamesystem(conf, fsImage);
            fsImage.getEditLog().initJournalsForWrite();
            if (!fsImage.confirmFormat(force, isInteractive)) {
                return true;
            }
            fsImage.format(fsn, clusterId);
        }
        catch (IOException ioe) {
            LOG.warn("Encountered exception during format: ", ioe);
            fsImage.close();
            throw ioe;
        }
        return false;
    }

    public static void checkAllowFormat(Configuration conf) throws IOException {
        if (!conf.getBoolean("dfs.namenode.support.allow.format", true)) {
            throw new IOException("The option dfs.namenode.support.allow.format is set to false for this filesystem, so it cannot be formatted. You will need to set dfs.namenode.support.allow.format parameter to true in order to format this filesystem");
        }
    }

    @VisibleForTesting
    public static boolean initializeSharedEdits(Configuration conf) throws IOException {
        return NameNode.initializeSharedEdits(conf, true);
    }

    @VisibleForTesting
    public static boolean initializeSharedEdits(Configuration conf, boolean force) throws IOException {
        return NameNode.initializeSharedEdits(conf, force, false);
    }

    private static Configuration getConfigurationWithoutSharedEdits(Configuration conf) throws IOException {
        List<URI> editsDirs = FSNamesystem.getNamespaceEditsDirs(conf, false);
        String editsDirsString = Joiner.on(",").join(editsDirs);
        Configuration confWithoutShared = new Configuration(conf);
        confWithoutShared.unset("dfs.namenode.shared.edits.dir");
        confWithoutShared.setStrings("dfs.namenode.edits.dir", editsDirsString);
        return confWithoutShared;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean initializeSharedEdits(Configuration conf, boolean force, boolean interactive) throws IOException {
        String nsId = DFSUtil.getNamenodeNameServiceId(conf);
        String namenodeId = HAUtil.getNameNodeId(conf, nsId);
        NameNode.initializeGenericKeys(conf, nsId, namenodeId);
        if (conf.get("dfs.namenode.shared.edits.dir") == null) {
            LOG.error("No shared edits directory configured for namespace " + nsId + " namenode " + namenodeId);
            return false;
        }
        if (UserGroupInformation.isSecurityEnabled()) {
            InetSocketAddress socAddr = NameNode.getAddress(conf);
            SecurityUtil.login(conf, "dfs.namenode.keytab.file", "dfs.namenode.kerberos.principal", socAddr.getHostName());
        }
        NNStorage existingStorage = null;
        FSImage sharedEditsImage = null;
        try {
            FSNamesystem fsns = FSNamesystem.loadFromDisk(NameNode.getConfigurationWithoutSharedEdits(conf));
            existingStorage = fsns.getFSImage().getStorage();
            NamespaceInfo nsInfo = existingStorage.getNamespaceInfo();
            List<URI> sharedEditsDirs = FSNamesystem.getSharedEditsDirs(conf);
            sharedEditsImage = new FSImage(conf, Lists.newArrayList(), sharedEditsDirs);
            sharedEditsImage.getEditLog().initJournalsForWrite();
            if (!sharedEditsImage.confirmFormat(force, interactive)) {
                boolean bl = true;
                return bl;
            }
            NNStorage newSharedStorage = sharedEditsImage.getStorage();
            newSharedStorage.format(nsInfo);
            sharedEditsImage.getEditLog().formatNonFileJournals(nsInfo);
            fsns.getFSImage().getEditLog().close();
            fsns.getFSImage().getEditLog().initJournalsForWrite();
            fsns.getFSImage().getEditLog().recoverUnclosedStreams();
            NameNode.copyEditLogSegmentsToSharedDir(fsns, sharedEditsDirs, newSharedStorage, conf);
        }
        catch (IOException ioe) {
            LOG.error("Could not initialize shared edits dir", ioe);
            boolean bl = true;
            return bl;
        }
        finally {
            if (sharedEditsImage != null) {
                try {
                    sharedEditsImage.close();
                }
                catch (IOException ioe) {
                    LOG.warn("Could not close sharedEditsImage", ioe);
                }
            }
            if (existingStorage != null) {
                try {
                    existingStorage.unlockAll();
                }
                catch (IOException ioe) {
                    LOG.warn("Could not unlock storage directories", ioe);
                    return true;
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void copyEditLogSegmentsToSharedDir(FSNamesystem fsns, Collection<URI> sharedEditsDirs, NNStorage newSharedStorage, Configuration conf) throws IOException {
        Preconditions.checkArgument(!sharedEditsDirs.isEmpty(), "No shared edits specified");
        ArrayList<URI> sharedEditsUris = new ArrayList<URI>(sharedEditsDirs);
        FSEditLog newSharedEditLog = new FSEditLog(conf, newSharedStorage, sharedEditsUris);
        newSharedEditLog.initJournalsForWrite();
        newSharedEditLog.recoverUnclosedStreams();
        FSEditLog sourceEditLog = fsns.getFSImage().editLog;
        long fromTxId = fsns.getFSImage().getMostRecentCheckpointTxId();
        Collection<EditLogInputStream> streams = null;
        try {
            streams = sourceEditLog.selectInputStreams(fromTxId + 1L, 0L);
            newSharedEditLog.setNextTxId(fromTxId + 1L);
            for (EditLogInputStream stream : streams) {
                FSEditLogOp op;
                LOG.debug("Beginning to copy stream " + stream + " to shared edits");
                boolean segmentOpen = false;
                while ((op = stream.readOp()) != null) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("copying op: " + op);
                    }
                    if (!segmentOpen) {
                        newSharedEditLog.startLogSegment(op.txid, false);
                        segmentOpen = true;
                    }
                    newSharedEditLog.logEdit(op);
                    if (op.opCode != FSEditLogOpCodes.OP_END_LOG_SEGMENT) continue;
                    newSharedEditLog.logSync();
                    newSharedEditLog.endCurrentLogSegment(false);
                    LOG.debug("ending log segment because of END_LOG_SEGMENT op in " + stream);
                    segmentOpen = false;
                }
                if (!segmentOpen) continue;
                LOG.debug("ending log segment because of end of stream in " + stream);
                newSharedEditLog.logSync();
                newSharedEditLog.endCurrentLogSegment(false);
                segmentOpen = false;
            }
        }
        finally {
            if (streams != null) {
                FSEditLog.closeAllStreams(streams);
            }
        }
    }

    @VisibleForTesting
    public static boolean doRollback(Configuration conf, boolean isConfirmationNeeded) throws IOException {
        String nsId = DFSUtil.getNamenodeNameServiceId(conf);
        String namenodeId = HAUtil.getNameNodeId(conf, nsId);
        NameNode.initializeGenericKeys(conf, nsId, namenodeId);
        FSNamesystem nsys = new FSNamesystem(conf, new FSImage(conf));
        System.err.print("\"rollBack\" will remove the current state of the file system,\nreturning you to the state prior to initiating your recent.\nupgrade. This action is permanent and cannot be undone. If you\nare performing a rollback in an HA environment, you should be\ncertain that no NameNode process is running on any host.");
        if (isConfirmationNeeded && !ToolRunner.confirmPrompt("Roll back file system state?")) {
            System.err.println("Rollback aborted.");
            return true;
        }
        nsys.getFSImage().doRollback(nsys);
        return false;
    }

    private static void printUsage(PrintStream out) {
        out.println(USAGE + "\n");
    }

    @VisibleForTesting
    static HdfsServerConstants.StartupOption parseArguments(String[] args) {
        int argsLen = args == null ? 0 : args.length;
        HdfsServerConstants.StartupOption startOpt = HdfsServerConstants.StartupOption.REGULAR;
        for (int i = 0; i < argsLen; ++i) {
            String cmd = args[i];
            if (HdfsServerConstants.StartupOption.FORMAT.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.FORMAT;
                ++i;
                while (i < argsLen) {
                    if (args[i].equalsIgnoreCase(HdfsServerConstants.StartupOption.CLUSTERID.getName())) {
                        if (++i >= argsLen) {
                            LOG.error("Must specify a valid cluster ID after the " + HdfsServerConstants.StartupOption.CLUSTERID.getName() + " flag");
                            return null;
                        }
                        String clusterId = args[i];
                        if (clusterId.isEmpty() || clusterId.equalsIgnoreCase(HdfsServerConstants.StartupOption.FORCE.getName()) || clusterId.equalsIgnoreCase(HdfsServerConstants.StartupOption.NONINTERACTIVE.getName())) {
                            LOG.error("Must specify a valid cluster ID after the " + HdfsServerConstants.StartupOption.CLUSTERID.getName() + " flag");
                            return null;
                        }
                        startOpt.setClusterId(clusterId);
                    }
                    if (args[i].equalsIgnoreCase(HdfsServerConstants.StartupOption.FORCE.getName())) {
                        startOpt.setForceFormat(true);
                    }
                    if (args[i].equalsIgnoreCase(HdfsServerConstants.StartupOption.NONINTERACTIVE.getName())) {
                        startOpt.setInteractiveFormat(false);
                    }
                    ++i;
                }
                continue;
            }
            if (HdfsServerConstants.StartupOption.GENCLUSTERID.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.GENCLUSTERID;
                continue;
            }
            if (HdfsServerConstants.StartupOption.REGULAR.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.REGULAR;
                continue;
            }
            if (HdfsServerConstants.StartupOption.BACKUP.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.BACKUP;
                continue;
            }
            if (HdfsServerConstants.StartupOption.CHECKPOINT.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.CHECKPOINT;
                continue;
            }
            if (HdfsServerConstants.StartupOption.UPGRADE.getName().equalsIgnoreCase(cmd) || HdfsServerConstants.StartupOption.UPGRADEONLY.getName().equalsIgnoreCase(cmd)) {
                HdfsServerConstants.StartupOption startupOption = startOpt = HdfsServerConstants.StartupOption.UPGRADE.getName().equalsIgnoreCase(cmd) ? HdfsServerConstants.StartupOption.UPGRADE : HdfsServerConstants.StartupOption.UPGRADEONLY;
                while (i + 1 < argsLen) {
                    String flag = args[i + 1];
                    if (flag.equalsIgnoreCase(HdfsServerConstants.StartupOption.CLUSTERID.getName())) {
                        if (i + 2 < argsLen) {
                            startOpt.setClusterId(args[i += 2]);
                            continue;
                        }
                        LOG.error("Must specify a valid cluster ID after the " + HdfsServerConstants.StartupOption.CLUSTERID.getName() + " flag");
                        return null;
                    }
                    if (flag.equalsIgnoreCase(HdfsServerConstants.StartupOption.RENAMERESERVED.getName())) {
                        if (i + 2 < argsLen) {
                            FSImageFormat.setRenameReservedPairs(args[i + 2]);
                            i += 2;
                            continue;
                        }
                        FSImageFormat.useDefaultRenameReservedPairs();
                        ++i;
                        continue;
                    }
                    LOG.error("Unknown upgrade flag " + flag);
                    return null;
                }
                continue;
            }
            if (HdfsServerConstants.StartupOption.ROLLINGUPGRADE.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.ROLLINGUPGRADE;
                if (++i >= argsLen) {
                    LOG.error("Must specify a rolling upgrade startup option " + HdfsServerConstants.RollingUpgradeStartupOption.getAllOptionString());
                    return null;
                }
                startOpt.setRollingUpgradeStartupOption(args[i]);
                continue;
            }
            if (HdfsServerConstants.StartupOption.ROLLBACK.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.ROLLBACK;
                continue;
            }
            if (HdfsServerConstants.StartupOption.FINALIZE.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.FINALIZE;
                continue;
            }
            if (HdfsServerConstants.StartupOption.IMPORT.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.IMPORT;
                continue;
            }
            if (HdfsServerConstants.StartupOption.BOOTSTRAPSTANDBY.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.BOOTSTRAPSTANDBY;
                return startOpt;
            }
            if (HdfsServerConstants.StartupOption.INITIALIZESHAREDEDITS.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.INITIALIZESHAREDEDITS;
                ++i;
                while (i < argsLen) {
                    if (HdfsServerConstants.StartupOption.NONINTERACTIVE.getName().equals(args[i])) {
                        startOpt.setInteractiveFormat(false);
                    } else if (HdfsServerConstants.StartupOption.FORCE.getName().equals(args[i])) {
                        startOpt.setForceFormat(true);
                    } else {
                        LOG.error("Invalid argument: " + args[i]);
                        return null;
                    }
                    ++i;
                }
                return startOpt;
            }
            if (HdfsServerConstants.StartupOption.RECOVER.getName().equalsIgnoreCase(cmd)) {
                if (startOpt != HdfsServerConstants.StartupOption.REGULAR) {
                    throw new RuntimeException("Can't combine -recover with other startup options.");
                }
                startOpt = HdfsServerConstants.StartupOption.RECOVER;
                while (++i < argsLen) {
                    if (args[i].equalsIgnoreCase(HdfsServerConstants.StartupOption.FORCE.getName())) {
                        startOpt.setForce(1);
                        continue;
                    }
                    throw new RuntimeException("Error parsing recovery options: can't understand option \"" + args[i] + "\"");
                }
                continue;
            }
            if (HdfsServerConstants.StartupOption.METADATAVERSION.getName().equalsIgnoreCase(cmd)) {
                startOpt = HdfsServerConstants.StartupOption.METADATAVERSION;
                continue;
            }
            return null;
        }
        return startOpt;
    }

    private static void setStartupOption(Configuration conf, HdfsServerConstants.StartupOption opt) {
        conf.set("dfs.namenode.startup", opt.name());
    }

    static HdfsServerConstants.StartupOption getStartupOption(Configuration conf) {
        return HdfsServerConstants.StartupOption.valueOf(conf.get("dfs.namenode.startup", HdfsServerConstants.StartupOption.REGULAR.toString()));
    }

    private static void doRecovery(HdfsServerConstants.StartupOption startOpt, Configuration conf) throws IOException {
        String nsId = DFSUtil.getNamenodeNameServiceId(conf);
        String namenodeId = HAUtil.getNameNodeId(conf, nsId);
        NameNode.initializeGenericKeys(conf, nsId, namenodeId);
        if (startOpt.getForce() < 2 && !ToolRunner.confirmPrompt("You have selected Metadata Recovery mode.  This mode is intended to recover lost metadata on a corrupt filesystem.  Metadata recovery mode often permanently deletes data from your HDFS filesystem.  Please back up your edit log and fsimage before trying this!\n\nAre you ready to proceed? (Y/N)\n")) {
            System.err.println("Recovery aborted at user request.\n");
            return;
        }
        MetaRecoveryContext.LOG.info("starting recovery...");
        UserGroupInformation.setConfiguration(conf);
        NameNode.initMetrics(conf, startOpt.toNodeRole());
        try (FSNamesystem fsn = null;){
            fsn = FSNamesystem.loadFromDisk(conf);
            fsn.getFSImage().saveNamespace(fsn);
            MetaRecoveryContext.LOG.info("RECOVERY COMPLETE");
        }
    }

    private static boolean printMetadataVersion(Configuration conf) throws IOException {
        String nsId = DFSUtil.getNamenodeNameServiceId(conf);
        String namenodeId = HAUtil.getNameNodeId(conf, nsId);
        NameNode.initializeGenericKeys(conf, nsId, namenodeId);
        FSImage fsImage = new FSImage(conf);
        FSNamesystem fs = new FSNamesystem(conf, fsImage, false);
        return fsImage.recoverTransitionRead(HdfsServerConstants.StartupOption.METADATAVERSION, fs, null);
    }

    public static NameNode createNameNode(String[] argv, Configuration conf) throws IOException {
        HdfsServerConstants.StartupOption startOpt;
        LOG.info("createNameNode " + Arrays.asList(argv));
        if (conf == null) {
            conf = new HdfsConfiguration();
        }
        if ((startOpt = NameNode.parseArguments(argv)) == null) {
            NameNode.printUsage(System.err);
            return null;
        }
        NameNode.setStartupOption(conf, startOpt);
        switch (startOpt) {
            case FORMAT: {
                boolean aborted = NameNode.format(conf, startOpt.getForceFormat(), startOpt.getInteractiveFormat());
                ExitUtil.terminate(aborted ? 1 : 0);
                return null;
            }
            case GENCLUSTERID: {
                System.err.println("Generating new cluster id:");
                System.out.println(NNStorage.newClusterID());
                ExitUtil.terminate(0);
                return null;
            }
            case FINALIZE: {
                System.err.println("Use of the argument '" + (Object)((Object)HdfsServerConstants.StartupOption.FINALIZE) + "' is no longer supported. To finalize an upgrade, start the NN " + " and then run `hdfs dfsadmin -finalizeUpgrade'");
                ExitUtil.terminate(1);
                return null;
            }
            case ROLLBACK: {
                boolean aborted = NameNode.doRollback(conf, true);
                ExitUtil.terminate(aborted ? 1 : 0);
                return null;
            }
            case BOOTSTRAPSTANDBY: {
                String[] toolArgs = Arrays.copyOfRange(argv, 1, argv.length);
                int rc = BootstrapStandby.run(toolArgs, conf);
                ExitUtil.terminate(rc);
                return null;
            }
            case INITIALIZESHAREDEDITS: {
                boolean aborted = NameNode.initializeSharedEdits(conf, startOpt.getForceFormat(), startOpt.getInteractiveFormat());
                ExitUtil.terminate(aborted ? 1 : 0);
                return null;
            }
            case BACKUP: 
            case CHECKPOINT: {
                HdfsServerConstants.NamenodeRole role = startOpt.toNodeRole();
                DefaultMetricsSystem.initialize(role.toString().replace(" ", ""));
                return new BackupNode(conf, role);
            }
            case RECOVER: {
                NameNode.doRecovery(startOpt, conf);
                return null;
            }
            case METADATAVERSION: {
                NameNode.printMetadataVersion(conf);
                ExitUtil.terminate(0);
                return null;
            }
            case UPGRADEONLY: {
                DefaultMetricsSystem.initialize("NameNode");
                new NameNode(conf);
                ExitUtil.terminate(0);
                return null;
            }
        }
        DefaultMetricsSystem.initialize("NameNode");
        return new NameNode(conf);
    }

    public static void initializeGenericKeys(Configuration conf, String nameserviceId, String namenodeId) {
        if (nameserviceId != null && !nameserviceId.isEmpty() || namenodeId != null && !namenodeId.isEmpty()) {
            if (nameserviceId != null) {
                conf.set("dfs.nameservice.id", nameserviceId);
            }
            if (namenodeId != null) {
                conf.set("dfs.ha.namenode.id", namenodeId);
            }
            DFSUtil.setGenericConf(conf, nameserviceId, namenodeId, NAMENODE_SPECIFIC_KEYS);
            DFSUtil.setGenericConf(conf, nameserviceId, null, NAMESERVICE_SPECIFIC_KEYS);
        }
        if (conf.get("dfs.namenode.rpc-address") != null) {
            URI defaultUri = URI.create("hdfs://" + conf.get("dfs.namenode.rpc-address"));
            conf.set("fs.defaultFS", defaultUri.toString());
            LOG.debug("Setting fs.defaultFS to " + defaultUri.toString());
        }
    }

    protected String getNameServiceId(Configuration conf) {
        return DFSUtil.getNamenodeNameServiceId(conf);
    }

    public static void main(String[] argv) throws Exception {
        if (DFSUtil.parseHelpArgument(argv, USAGE, System.out, true)) {
            System.exit(0);
        }
        try {
            StringUtils.startupShutdownMessage(NameNode.class, argv, LOG);
            NameNode namenode = NameNode.createNameNode(argv, null);
            if (namenode != null) {
                namenode.join();
            }
        }
        catch (Throwable e) {
            LOG.error("Failed to start namenode.", e);
            ExitUtil.terminate(1, e);
        }
    }

    synchronized void monitorHealth() throws HealthCheckFailedException, AccessControlException {
        this.namesystem.checkSuperuserPrivilege();
        if (!this.haEnabled) {
            return;
        }
        this.getNamesystem().checkAvailableResources();
        if (!this.getNamesystem().nameNodeHasResourcesAvailable()) {
            throw new HealthCheckFailedException("The NameNode has no resources available");
        }
    }

    synchronized void transitionToActive() throws ServiceFailedException, AccessControlException {
        this.namesystem.checkSuperuserPrivilege();
        if (!this.haEnabled) {
            throw new ServiceFailedException("HA for namenode is not enabled");
        }
        this.state.setState(this.haContext, ACTIVE_STATE);
    }

    synchronized void transitionToStandby() throws ServiceFailedException, AccessControlException {
        this.namesystem.checkSuperuserPrivilege();
        if (!this.haEnabled) {
            throw new ServiceFailedException("HA for namenode is not enabled");
        }
        this.state.setState(this.haContext, STANDBY_STATE);
    }

    synchronized HAServiceStatus getServiceStatus() throws ServiceFailedException, AccessControlException {
        this.namesystem.checkSuperuserPrivilege();
        if (!this.haEnabled) {
            throw new ServiceFailedException("HA for namenode is not enabled");
        }
        if (this.state == null) {
            return new HAServiceStatus(HAServiceProtocol.HAServiceState.INITIALIZING);
        }
        HAServiceProtocol.HAServiceState retState = this.state.getServiceState();
        HAServiceStatus ret = new HAServiceStatus(retState);
        if (retState == HAServiceProtocol.HAServiceState.STANDBY) {
            String safemodeTip = this.namesystem.getSafeModeTip();
            if (!safemodeTip.isEmpty()) {
                ret.setNotReadyToBecomeActive("The NameNode is in safemode. " + safemodeTip);
            } else {
                ret.setReadyToBecomeActive();
            }
        } else if (retState == HAServiceProtocol.HAServiceState.ACTIVE) {
            ret.setReadyToBecomeActive();
        } else {
            ret.setNotReadyToBecomeActive("State is " + this.state);
        }
        return ret;
    }

    synchronized HAServiceProtocol.HAServiceState getServiceState() {
        if (this.state == null) {
            return HAServiceProtocol.HAServiceState.INITIALIZING;
        }
        return this.state.getServiceState();
    }

    private void registerNNSMXBean() {
        this.nameNodeStatusBeanName = MBeans.register("NameNode", "NameNodeStatus", this);
    }

    @Override
    public String getNNRole() {
        String roleStr = "";
        HdfsServerConstants.NamenodeRole role = this.getRole();
        if (null != role) {
            roleStr = role.toString();
        }
        return roleStr;
    }

    @Override
    public String getState() {
        String servStateStr = "";
        HAServiceProtocol.HAServiceState servState = this.getServiceState();
        if (null != servState) {
            servStateStr = servState.toString();
        }
        return servStateStr;
    }

    @Override
    public String getHostAndPort() {
        return this.getNameNodeAddressHostPortString();
    }

    @Override
    public boolean isSecurityEnabled() {
        return UserGroupInformation.isSecurityEnabled();
    }

    @Override
    public long getLastHATransitionTime() {
        return this.state.getLastHATransitionTime();
    }

    protected synchronized void doImmediateShutdown(Throwable t) throws ExitUtil.ExitException {
        String message = "Error encountered requiring NN shutdown. Shutting down immediately.";
        try {
            LOG.error(message, t);
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        ExitUtil.terminate(1, t);
    }

    public boolean isStandbyState() {
        return this.state.equals(STANDBY_STATE);
    }

    public boolean isActiveState() {
        return this.state.equals(ACTIVE_STATE);
    }

    boolean isStarted() {
        return this.started.get();
    }

    void checkHaStateChange(HAServiceProtocol.StateChangeRequestInfo req) throws AccessControlException {
        boolean autoHaEnabled = this.conf.getBoolean("dfs.ha.automatic-failover.enabled", false);
        switch (req.getSource()) {
            case REQUEST_BY_USER: {
                if (!autoHaEnabled) break;
                throw new AccessControlException("Manual HA control for this NameNode is disallowed, because automatic HA is enabled.");
            }
            case REQUEST_BY_USER_FORCED: {
                if (!autoHaEnabled) break;
                LOG.warn("Allowing manual HA control from " + Server.getRemoteAddress() + " even though automatic HA is enabled, because the user " + "specified the force flag");
                break;
            }
            case REQUEST_BY_ZKFC: {
                if (autoHaEnabled) break;
                throw new AccessControlException("Request from ZK failover controller at " + Server.getRemoteAddress() + " denied since automatic HA " + "is not enabled");
            }
        }
    }

    static {
        HdfsConfiguration.init();
        NAMENODE_SPECIFIC_KEYS = new String[]{"dfs.namenode.rpc-address", "dfs.namenode.rpc-bind-host", "dfs.namenode.name.dir", "dfs.namenode.edits.dir", "dfs.namenode.shared.edits.dir", "dfs.namenode.checkpoint.dir", "dfs.namenode.checkpoint.edits.dir", "dfs.namenode.servicerpc-address", "dfs.namenode.servicerpc-bind-host", "dfs.namenode.http-address", "dfs.namenode.https-address", "dfs.namenode.http-bind-host", "dfs.namenode.https-bind-host", "dfs.namenode.keytab.file", "dfs.namenode.secondary.http-address", "dfs.namenode.secondary.https-address", "dfs.secondary.namenode.keytab.file", "dfs.namenode.backup.address", "dfs.namenode.backup.http-address", "dfs.namenode.backup.dnrpc-address", "dfs.namenode.kerberos.principal", "dfs.namenode.kerberos.internal.spnego.principal", "dfs.ha.fencing.methods", "dfs.ha.zkfc.port", "dfs.ha.fencing.methods"};
        NAMESERVICE_SPECIFIC_KEYS = new String[]{"dfs.ha.automatic-failover.enabled"};
        USAGE = "Usage: java NameNode [" + HdfsServerConstants.StartupOption.BACKUP.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.CHECKPOINT.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.FORMAT.getName() + " [" + HdfsServerConstants.StartupOption.CLUSTERID.getName() + " cid ] [" + HdfsServerConstants.StartupOption.FORCE.getName() + "] [" + HdfsServerConstants.StartupOption.NONINTERACTIVE.getName() + "] ] | \n\t[" + HdfsServerConstants.StartupOption.UPGRADE.getName() + " [" + HdfsServerConstants.StartupOption.CLUSTERID.getName() + " cid]" + " [" + HdfsServerConstants.StartupOption.RENAMERESERVED.getName() + "<k-v pairs>] ] | \n\t[" + HdfsServerConstants.StartupOption.UPGRADEONLY.getName() + " [" + HdfsServerConstants.StartupOption.CLUSTERID.getName() + " cid]" + " [" + HdfsServerConstants.StartupOption.RENAMERESERVED.getName() + "<k-v pairs>] ] | \n\t[" + HdfsServerConstants.StartupOption.ROLLBACK.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.ROLLINGUPGRADE.getName() + " " + HdfsServerConstants.RollingUpgradeStartupOption.getAllOptionString() + " ] | \n\t[" + HdfsServerConstants.StartupOption.FINALIZE.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.IMPORT.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.INITIALIZESHAREDEDITS.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.BOOTSTRAPSTANDBY.getName() + "] | \n\t[" + HdfsServerConstants.StartupOption.RECOVER.getName() + " [ " + HdfsServerConstants.StartupOption.FORCE.getName() + "] ] | \n\t[" + HdfsServerConstants.StartupOption.METADATAVERSION.getName() + " ] " + " ]";
        LOG = LoggerFactory.getLogger(NameNode.class.getName());
        stateChangeLog = LoggerFactory.getLogger("org.apache.hadoop.hdfs.StateChange");
        blockStateChangeLog = LoggerFactory.getLogger("BlockStateChange");
        ACTIVE_STATE = new ActiveState();
        STANDBY_STATE = new StandbyState();
        startupProgress = new StartupProgress();
    }

    protected class NameNodeHAContext
    implements HAContext {
        protected NameNodeHAContext() {
        }

        @Override
        public void setState(HAState s2) {
            NameNode.this.state = s2;
        }

        @Override
        public HAState getState() {
            return NameNode.this.state;
        }

        @Override
        public void startActiveServices() throws IOException {
            try {
                NameNode.this.namesystem.startActiveServices();
                NameNode.this.startTrashEmptier(NameNode.this.conf);
            }
            catch (Throwable t) {
                NameNode.this.doImmediateShutdown(t);
            }
        }

        @Override
        public void stopActiveServices() throws IOException {
            try {
                if (NameNode.this.namesystem != null) {
                    NameNode.this.namesystem.stopActiveServices();
                }
                NameNode.this.stopTrashEmptier();
            }
            catch (Throwable t) {
                NameNode.this.doImmediateShutdown(t);
            }
        }

        @Override
        public void startStandbyServices() throws IOException {
            try {
                NameNode.this.namesystem.startStandbyServices(NameNode.this.conf);
            }
            catch (Throwable t) {
                NameNode.this.doImmediateShutdown(t);
            }
        }

        @Override
        public void prepareToStopStandbyServices() throws ServiceFailedException {
            try {
                NameNode.this.namesystem.prepareToStopStandbyServices();
            }
            catch (Throwable t) {
                NameNode.this.doImmediateShutdown(t);
            }
        }

        @Override
        public void stopStandbyServices() throws IOException {
            try {
                if (NameNode.this.namesystem != null) {
                    NameNode.this.namesystem.stopStandbyServices();
                }
            }
            catch (Throwable t) {
                NameNode.this.doImmediateShutdown(t);
            }
        }

        @Override
        public void writeLock() {
            NameNode.this.namesystem.writeLock();
            NameNode.this.namesystem.lockRetryCache();
        }

        @Override
        public void writeUnlock() {
            NameNode.this.namesystem.unlockRetryCache();
            NameNode.this.namesystem.writeUnlock();
        }

        @Override
        public void checkOperation(OperationCategory op) throws StandbyException {
            NameNode.this.state.checkOperation(NameNode.this.haContext, op);
        }

        @Override
        public boolean allowStaleReads() {
            return NameNode.this.allowStaleStandbyReads;
        }
    }

    public static enum OperationCategory {
        UNCHECKED,
        READ,
        WRITE,
        CHECKPOINT,
        JOURNAL;

    }
}

