/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kudu.hive.metastore;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.MetaStoreEventListener;
import org.apache.hadoop.hive.metastore.TableType;
import org.apache.hadoop.hive.metastore.api.EnvironmentContext;
import org.apache.hadoop.hive.metastore.api.FieldSchema;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.metastore.events.AlterTableEvent;
import org.apache.hadoop.hive.metastore.events.CreateTableEvent;
import org.apache.hadoop.hive.metastore.events.DropTableEvent;
import org.apache.hadoop.hive.metastore.events.ListenerEvent;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.kudu.client.HiveMetastoreConfig;
import org.apache.kudu.client.KuduClient;
import org.apache.kudu.client.KuduException;
import org.apache.kudu.shaded.com.google.common.annotations.VisibleForTesting;

public class KuduMetastorePlugin
extends MetaStoreEventListener {
    @VisibleForTesting
    static final String KUDU_STORAGE_HANDLER = "org.apache.hadoop.hive.kudu.KuduStorageHandler";
    @VisibleForTesting
    static final String LEGACY_KUDU_STORAGE_HANDLER = "com.cloudera.kudu.hive.KuduStorageHandler";
    @VisibleForTesting
    static final String KUDU_CLUSTER_ID_KEY = "kudu.cluster_id";
    @VisibleForTesting
    static final String KUDU_TABLE_ID_KEY = "kudu.table_id";
    @VisibleForTesting
    static final String KUDU_TABLE_NAME_KEY = "kudu.table_name";
    @VisibleForTesting
    static final String KUDU_MASTER_ADDRS_KEY = "kudu.master_addresses";
    @VisibleForTesting
    static final String KUDU_MASTER_EVENT_KEY = "kudu.master_event";
    @VisibleForTesting
    static final String KUDU_CHECK_ID_KEY = "kudu.check_id";
    @VisibleForTesting
    static final String EXTERNAL_TABLE_KEY = "EXTERNAL";
    static final String EXTERNAL_PURGE_KEY = "external.table.purge";
    static final String COMMENT_KEY = "comment";
    static final String SKIP_VALIDATION_ENV = "KUDU_SKIP_HMS_PLUGIN_VALIDATION";
    static final String SYNC_ENABLED_ENV = "KUDU_HMS_SYNC_ENABLED";
    static final String SASL_PROTOCOL_NAME_ENV = "KUDU_SASL_PROTOCOL_NAME";
    private static final Map<String, KuduClient> KUDU_CLIENTS = new ConcurrentHashMap<String, KuduClient>();

    public KuduMetastorePlugin(Configuration config) {
        super(config);
    }

    public void onCreateTable(CreateTableEvent tableEvent) throws MetaException {
        super.onCreateTable(tableEvent);
        if (KuduMetastorePlugin.skipsValidation()) {
            return;
        }
        Table table = tableEvent.getTable();
        if (!KuduMetastorePlugin.isSynchronizedTable(table)) {
            return;
        }
        if (!KuduMetastorePlugin.isKuduTable(table)) {
            KuduMetastorePlugin.checkNoKuduProperties(table);
            return;
        }
        KuduMetastorePlugin.checkMasterAddrsProperty(table);
        if (!KuduMetastorePlugin.kuduSyncEnabled((ListenerEvent)tableEvent, table)) {
            return;
        }
        KuduMetastorePlugin.checkKuduProperties(table);
        if (!KuduMetastorePlugin.isKuduMasterAction((ListenerEvent)tableEvent)) {
            throw new MetaException("Kudu tables may not be created through Hive");
        }
    }

    public void onDropTable(DropTableEvent tableEvent) throws MetaException {
        String targetTableId;
        super.onDropTable(tableEvent);
        if (KuduMetastorePlugin.skipsValidation()) {
            return;
        }
        Table table = tableEvent.getTable();
        if (!KuduMetastorePlugin.isSynchronizedTable(table)) {
            return;
        }
        EnvironmentContext environmentContext = tableEvent.getEnvironmentContext();
        String string = targetTableId = environmentContext == null ? null : (String)environmentContext.getProperties().get(KUDU_TABLE_ID_KEY);
        if (!KuduMetastorePlugin.isKuduTable(table)) {
            if (targetTableId != null) {
                throw new MetaException("Kudu table ID does not match the non-Kudu HMS entry");
            }
            return;
        }
        if (!KuduMetastorePlugin.kuduSyncEnabled((ListenerEvent)tableEvent, table)) {
            return;
        }
        if (targetTableId == null) {
            return;
        }
        if (!targetTableId.equals(table.getParameters().get(KUDU_TABLE_ID_KEY))) {
            throw new MetaException("Kudu table ID does not match the HMS entry");
        }
    }

    public void onAlterTable(AlterTableEvent tableEvent) throws MetaException {
        super.onAlterTable(tableEvent);
        if (KuduMetastorePlugin.skipsValidation()) {
            return;
        }
        Table oldTable = tableEvent.getOldTable();
        Table newTable = tableEvent.getNewTable();
        if (!KuduMetastorePlugin.isKuduTable(oldTable) && !KuduMetastorePlugin.isLegacyKuduTable(oldTable)) {
            KuduMetastorePlugin.checkNoKuduProperties(newTable);
            return;
        }
        if (KuduMetastorePlugin.kuduMetadataUnchanged(oldTable, newTable)) {
            return;
        }
        if (!KuduMetastorePlugin.kuduSyncEnabled((ListenerEvent)tableEvent, oldTable)) {
            return;
        }
        if (KuduMetastorePlugin.isKuduTable(oldTable) && !KuduMetastorePlugin.isKuduMasterAction((ListenerEvent)tableEvent) && KuduMetastorePlugin.isSynchronizedTable(oldTable) != KuduMetastorePlugin.isSynchronizedTable(newTable)) {
            throw new MetaException("Kudu table type may not be altered");
        }
        if (!KuduMetastorePlugin.isSynchronizedTable(oldTable)) {
            return;
        }
        if (KuduMetastorePlugin.isLegacyKuduTable(oldTable)) {
            if (KuduMetastorePlugin.isKuduTable(newTable)) {
                KuduMetastorePlugin.checkKuduProperties(newTable);
                KuduMetastorePlugin.checkOnlyKuduMasterCanAlterSchema(tableEvent, oldTable, newTable);
                return;
            }
            KuduMetastorePlugin.checkNoKuduProperties(newTable);
        } else if (KuduMetastorePlugin.isKuduTable(oldTable)) {
            if (KuduMetastorePlugin.isLegacyKuduTable(newTable)) {
                KuduMetastorePlugin.checkNoKuduProperties(newTable);
                KuduMetastorePlugin.checkOnlyKuduMasterCanAlterSchema(tableEvent, oldTable, newTable);
                return;
            }
            KuduMetastorePlugin.checkKuduProperties(newTable);
            KuduMetastorePlugin.checkOnlyKuduMasterCanAlterSchema(tableEvent, oldTable, newTable);
            if (KuduMetastorePlugin.checkTableID((ListenerEvent)tableEvent)) {
                String oldTableId = (String)oldTable.getParameters().get(KUDU_TABLE_ID_KEY);
                String newTableId = (String)newTable.getParameters().get(KUDU_TABLE_ID_KEY);
                if (!newTableId.equals(oldTableId)) {
                    throw new MetaException("Kudu table ID does not match the existing HMS entry");
                }
            }
        }
    }

    private static boolean isKuduTable(Table table) {
        String storageHandler = (String)table.getParameters().get("storage_handler");
        return KUDU_STORAGE_HANDLER.equals(storageHandler);
    }

    private static boolean isLegacyKuduTable(Table table) {
        return LEGACY_KUDU_STORAGE_HANDLER.equals(table.getParameters().get("storage_handler"));
    }

    private static boolean isExternalTable(Table table) {
        String isExternal = (String)table.getParameters().get(EXTERNAL_TABLE_KEY);
        if (isExternal == null) {
            return false;
        }
        return Boolean.parseBoolean(isExternal);
    }

    private static boolean isPurgeTable(Table table) {
        boolean externalPurge = Boolean.parseBoolean(table.getParameters().getOrDefault(EXTERNAL_PURGE_KEY, "false"));
        return TableType.MANAGED_TABLE.name().equals(table.getTableType()) || externalPurge;
    }

    private static boolean isSynchronizedTable(Table table) {
        return TableType.MANAGED_TABLE.name().equals(table.getTableType()) || KuduMetastorePlugin.isExternalTable(table) && KuduMetastorePlugin.isPurgeTable(table);
    }

    private static void checkKuduProperties(Table table) throws MetaException {
        if (!KuduMetastorePlugin.isKuduTable(table)) {
            throw new MetaException(String.format("Kudu table entry must contain a Kudu storage handler property (%s=%s)", "storage_handler", KUDU_STORAGE_HANDLER));
        }
        String tableId = (String)table.getParameters().get(KUDU_TABLE_ID_KEY);
        if (tableId == null || tableId.isEmpty()) {
            throw new MetaException(String.format("Kudu table entry must contain a table ID property (%s)", KUDU_TABLE_ID_KEY));
        }
        KuduMetastorePlugin.checkMasterAddrsProperty(table);
    }

    private static void checkMasterAddrsProperty(Table table) throws MetaException {
        String masterAddresses = (String)table.getParameters().get(KUDU_MASTER_ADDRS_KEY);
        if (masterAddresses == null || masterAddresses.isEmpty()) {
            throw new MetaException(String.format("Kudu table entry must contain a Master addresses property (%s)", KUDU_MASTER_ADDRS_KEY));
        }
    }

    private static void checkNoKuduProperties(Table table) throws MetaException {
        if (KuduMetastorePlugin.isKuduTable(table)) {
            throw new MetaException(String.format("non-Kudu table entry must not contain the Kudu storage handler (%s=%s)", "storage_handler", KUDU_STORAGE_HANDLER));
        }
        if (table.getParameters().containsKey(KUDU_TABLE_ID_KEY)) {
            throw new MetaException(String.format("non-Kudu table entry must not contain a table ID property (%s)", KUDU_TABLE_ID_KEY));
        }
        if (table.getParameters().containsKey(KUDU_CLUSTER_ID_KEY)) {
            throw new MetaException(String.format("non-Kudu table entry must not contain a cluster ID property (%s)", KUDU_CLUSTER_ID_KEY));
        }
    }

    @VisibleForTesting
    static boolean kuduMetadataUnchanged(Table before, Table after) {
        Map beforeParams = before.getParameters();
        Map afterParams = after.getParameters();
        if (!(Objects.equals(beforeParams.get("storage_handler"), afterParams.get("storage_handler")) && Objects.equals(beforeParams.get(KUDU_MASTER_ADDRS_KEY), afterParams.get(KUDU_MASTER_ADDRS_KEY)) && Objects.equals(beforeParams.get(KUDU_TABLE_ID_KEY), afterParams.get(KUDU_TABLE_ID_KEY)) && Objects.equals(beforeParams.get(KUDU_TABLE_NAME_KEY), afterParams.get(KUDU_TABLE_NAME_KEY)) && Objects.equals(beforeParams.get(KUDU_CLUSTER_ID_KEY), afterParams.get(KUDU_CLUSTER_ID_KEY)))) {
            return false;
        }
        if (KuduMetastorePlugin.isSynchronizedTable(before) != KuduMetastorePlugin.isSynchronizedTable(after)) {
            return false;
        }
        if (!(Objects.equals(before.getDbName(), after.getDbName()) && Objects.equals(before.getTableName(), after.getTableName()) && Objects.equals(before.getOwner(), after.getOwner()) && Objects.equals(beforeParams.get(COMMENT_KEY), afterParams.get(COMMENT_KEY)))) {
            return false;
        }
        List beforeCols = before.getSd().getCols();
        List afterCols = after.getSd().getCols();
        if (beforeCols.size() != afterCols.size()) {
            return false;
        }
        for (int i = 0; i < beforeCols.size(); ++i) {
            FieldSchema beforeCol = (FieldSchema)beforeCols.get(i);
            FieldSchema afterCol = (FieldSchema)afterCols.get(i);
            if (Objects.equals(beforeCol.getName(), afterCol.getName()) && Objects.equals(beforeCol.getType(), afterCol.getType()) && Objects.equals(beforeCol.getComment(), afterCol.getComment())) continue;
            return false;
        }
        return true;
    }

    private static void checkOnlyKuduMasterCanAlterSchema(AlterTableEvent tableEvent, Table oldTable, Table newTable) throws MetaException {
        if (!KuduMetastorePlugin.isKuduMasterAction((ListenerEvent)tableEvent) && !oldTable.getSd().getCols().equals(newTable.getSd().getCols())) {
            throw new MetaException("Kudu table columns may not be altered through Hive");
        }
    }

    private static boolean isKuduMasterAction(ListenerEvent event) {
        EnvironmentContext environmentContext = event.getEnvironmentContext();
        if (environmentContext == null) {
            return false;
        }
        Map properties = environmentContext.getProperties();
        if (properties == null) {
            return false;
        }
        if (!properties.containsKey(KUDU_MASTER_EVENT_KEY)) {
            return false;
        }
        return Boolean.parseBoolean((String)properties.get(KUDU_MASTER_EVENT_KEY));
    }

    private static boolean checkTableID(ListenerEvent event) {
        EnvironmentContext environmentContext = event.getEnvironmentContext();
        if (environmentContext == null) {
            return true;
        }
        Map properties = environmentContext.getProperties();
        if (properties == null) {
            return true;
        }
        if (!properties.containsKey(KUDU_CHECK_ID_KEY)) {
            return true;
        }
        return Boolean.parseBoolean((String)properties.get(KUDU_CHECK_ID_KEY));
    }

    private static boolean skipsValidation() {
        String skipValidation = System.getenv(SKIP_VALIDATION_ENV);
        return skipValidation != null && !skipValidation.isEmpty() && Integer.parseInt(skipValidation) != 0;
    }

    private static boolean kuduSyncEnabled(ListenerEvent event, Table table) throws MetaException {
        HiveMetastoreConfig hmsConfig;
        String envEnabled = System.getenv(SYNC_ENABLED_ENV);
        if (envEnabled != null && !envEnabled.isEmpty()) {
            return Integer.parseInt(envEnabled) == 1;
        }
        if (KuduMetastorePlugin.isKuduMasterAction(event)) {
            return true;
        }
        String masterAddresses = (String)table.getParameters().get(KUDU_MASTER_ADDRS_KEY);
        if (masterAddresses == null || masterAddresses.isEmpty()) {
            return false;
        }
        KuduClient kuduClient = KuduMetastorePlugin.getKuduClient(masterAddresses);
        try {
            hmsConfig = kuduClient.getHiveMetastoreConfig();
        }
        catch (KuduException e) {
            throw new MetaException(String.format("Error determining if Kudu's integration with the Hive Metastore is enabled: %s", e.getMessage()));
        }
        return hmsConfig != null;
    }

    private static KuduClient getKuduClient(String kuduMasters) {
        KuduClient client = KUDU_CLIENTS.get(kuduMasters);
        if (client == null) {
            try {
                client = (KuduClient)UserGroupInformation.getLoginUser().doAs(() -> new KuduClient.KuduClientBuilder(kuduMasters).saslProtocolName(KuduMetastorePlugin.getSaslProtocolName()).build());
            }
            catch (IOException | InterruptedException e) {
                throw new RuntimeException("Failed to create the Kudu client");
            }
            KUDU_CLIENTS.put(kuduMasters, client);
        }
        return client;
    }

    private static String getSaslProtocolName() {
        String saslProtocolName = System.getenv(SASL_PROTOCOL_NAME_ENV);
        if (saslProtocolName == null || saslProtocolName.isEmpty()) {
            saslProtocolName = "kudu";
        }
        return saslProtocolName;
    }
}

