/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.state.internals;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.utils.Bytes;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.state.internals.LRUCacheEntry;
import org.apache.kafka.streams.state.internals.ThreadCache;
import org.apache.kafka.streams.state.internals.ThreadCacheMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class NamedCache {
    private static final Logger log = LoggerFactory.getLogger(NamedCache.class);
    private final String name;
    private final TreeMap<Bytes, LRUNode> cache = new TreeMap();
    private final Set<Bytes> dirtyKeys = new LinkedHashSet<Bytes>();
    private ThreadCache.DirtyEntryFlushListener listener;
    private LRUNode tail;
    private LRUNode head;
    private long currentSizeBytes;
    private ThreadCacheMetrics metrics;
    private Sensor hitRatio = null;
    private long numReadHits = 0L;
    private long numReadMisses = 0L;
    private long numOverwrites = 0L;
    private long numFlushes = 0L;

    NamedCache(String name) {
        this(name, null);
    }

    NamedCache(String name, ThreadCacheMetrics metrics) {
        this.name = name;
        this.metrics = metrics != null ? metrics : new ThreadCache.NullThreadCacheMetrics();
        this.hitRatio = this.metrics.addCacheSensor(name, "hitRatio", new String[0]);
    }

    synchronized long hits() {
        return this.numReadHits;
    }

    synchronized long misses() {
        return this.numReadMisses;
    }

    synchronized long overwrites() {
        return this.numOverwrites;
    }

    synchronized long flushes() {
        return this.numFlushes;
    }

    synchronized LRUCacheEntry get(Bytes key) {
        LRUNode node = this.getInternal(key);
        if (node == null) {
            return null;
        }
        this.updateLRU(node);
        return node.entry;
    }

    synchronized void setListener(ThreadCache.DirtyEntryFlushListener listener) {
        this.listener = listener;
    }

    synchronized void flush() {
        this.flush(null);
    }

    private void flush(LRUNode evicted) {
        ++this.numFlushes;
        log.debug("Named cache {} stats on flush: #hits={}, #misses={}, #overwrites={}, #flushes={}", new Object[]{this.name, this.hits(), this.misses(), this.overwrites(), this.flushes()});
        if (this.listener == null) {
            throw new IllegalArgumentException("No listener for namespace " + this.name + " registered with cache");
        }
        if (this.dirtyKeys.isEmpty()) {
            return;
        }
        ArrayList<ThreadCache.DirtyEntry> entries = new ArrayList<ThreadCache.DirtyEntry>();
        if (evicted != null) {
            entries.add(new ThreadCache.DirtyEntry(evicted.key, ((LRUNode)evicted).entry.value, evicted.entry));
            this.dirtyKeys.remove(evicted.key);
        }
        for (Bytes key : this.dirtyKeys) {
            LRUNode node = this.getInternal(key);
            if (node == null) {
                throw new IllegalStateException("Key = " + key + " found in dirty key set, but entry is null");
            }
            entries.add(new ThreadCache.DirtyEntry(key, ((LRUNode)node).entry.value, node.entry));
            node.entry.markClean();
        }
        this.dirtyKeys.clear();
        this.listener.apply(entries);
    }

    synchronized void put(Bytes key, LRUCacheEntry value) {
        if (!value.isDirty && this.dirtyKeys.contains(key)) {
            throw new IllegalStateException(String.format("Attempting to put a clean entry for key [%s] into NamedCache [%s] when it already contains a dirty entry for the same key", key, this.name));
        }
        LRUNode node = this.cache.get(key);
        if (node != null) {
            ++this.numOverwrites;
            this.currentSizeBytes -= node.size();
            node.update(value);
            this.updateLRU(node);
        } else {
            node = new LRUNode(key, value);
            this.putHead(node);
            this.cache.put(key, node);
        }
        if (value.isDirty()) {
            this.dirtyKeys.remove(key);
            this.dirtyKeys.add(key);
        }
        this.currentSizeBytes += node.size();
    }

    synchronized long sizeInBytes() {
        return this.currentSizeBytes;
    }

    private LRUNode getInternal(Bytes key) {
        LRUNode node = this.cache.get(key);
        if (node == null) {
            ++this.numReadMisses;
            return null;
        }
        ++this.numReadHits;
        this.metrics.recordCacheSensor(this.hitRatio, (double)this.numReadHits / (double)(this.numReadHits + this.numReadMisses));
        return node;
    }

    private void updateLRU(LRUNode node) {
        this.remove(node);
        this.putHead(node);
    }

    private void remove(LRUNode node) {
        if (node.previous != null) {
            node.previous.next = node.next;
        } else {
            this.head = node.next;
        }
        if (node.next != null) {
            node.next.previous = node.previous;
        } else {
            this.tail = node.previous;
        }
    }

    private void putHead(LRUNode node) {
        node.next = this.head;
        node.previous = null;
        if (this.head != null) {
            this.head.previous = node;
        }
        this.head = node;
        if (this.tail == null) {
            this.tail = this.head;
        }
    }

    synchronized void evict() {
        if (this.tail == null) {
            return;
        }
        LRUNode eldest = this.tail;
        this.currentSizeBytes -= eldest.size();
        this.remove(eldest);
        this.cache.remove(eldest.key);
        if (eldest.entry.isDirty()) {
            this.flush(eldest);
        }
    }

    synchronized LRUCacheEntry putIfAbsent(Bytes key, LRUCacheEntry value) {
        LRUCacheEntry originalValue = this.get(key);
        if (originalValue == null) {
            this.put(key, value);
        }
        return originalValue;
    }

    synchronized void putAll(List<KeyValue<byte[], LRUCacheEntry>> entries) {
        for (KeyValue<byte[], LRUCacheEntry> entry : entries) {
            this.put(Bytes.wrap((byte[])((byte[])entry.key)), (LRUCacheEntry)entry.value);
        }
    }

    synchronized LRUCacheEntry delete(Bytes key) {
        LRUNode node = this.cache.remove(key);
        if (node == null) {
            return null;
        }
        this.remove(node);
        this.cache.remove(key);
        this.dirtyKeys.remove(key);
        this.currentSizeBytes -= node.size();
        return node.entry();
    }

    public long size() {
        return this.cache.size();
    }

    synchronized Iterator<Bytes> keyRange(Bytes from, Bytes to) {
        return this.keySetIterator(this.cache.navigableKeySet().subSet(from, true, to, true));
    }

    private Iterator<Bytes> keySetIterator(Set<Bytes> keySet) {
        TreeSet<Bytes> copy = new TreeSet<Bytes>();
        copy.addAll(keySet);
        return copy.iterator();
    }

    synchronized Iterator<Bytes> allKeys() {
        return this.keySetIterator(this.cache.navigableKeySet());
    }

    synchronized LRUCacheEntry first() {
        if (this.head == null) {
            return null;
        }
        return this.head.entry;
    }

    synchronized LRUCacheEntry last() {
        if (this.tail == null) {
            return null;
        }
        return this.tail.entry;
    }

    synchronized LRUNode head() {
        return this.head;
    }

    synchronized LRUNode tail() {
        return this.tail;
    }

    synchronized long dirtySize() {
        return this.dirtyKeys.size();
    }

    class LRUNode {
        private final Bytes key;
        private LRUCacheEntry entry;
        private LRUNode previous;
        private LRUNode next;

        LRUNode(Bytes key, LRUCacheEntry entry) {
            this.key = key;
            this.entry = entry;
        }

        LRUCacheEntry entry() {
            return this.entry;
        }

        Bytes key() {
            return this.key;
        }

        long size() {
            return (long)(this.key.get().length + 8 + 8 + 8) + this.entry.size();
        }

        LRUNode next() {
            return this.next;
        }

        LRUNode previous() {
            return this.previous;
        }

        private void update(LRUCacheEntry entry) {
            this.entry = entry;
        }
    }
}

