/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.nessie;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.exceptions.CommitStateUnknownException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.nessie.NessieUtil;
import org.apache.iceberg.nessie.UpdateableReference;
import org.apache.iceberg.relocated.com.google.common.base.Suppliers;
import org.apache.iceberg.util.Tasks;
import org.projectnessie.client.api.CommitMultipleOperationsBuilder;
import org.projectnessie.client.api.CreateNamespaceBuilder;
import org.projectnessie.client.api.DeleteNamespaceBuilder;
import org.projectnessie.client.api.GetMultipleNamespacesBuilder;
import org.projectnessie.client.api.GetNamespaceBuilder;
import org.projectnessie.client.api.NessieApiV1;
import org.projectnessie.client.api.OnReferenceBuilder;
import org.projectnessie.client.api.UpdateNamespaceBuilder;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.error.BaseNessieClientServerException;
import org.projectnessie.error.NessieConflictException;
import org.projectnessie.error.NessieNamespaceAlreadyExistsException;
import org.projectnessie.error.NessieNamespaceNotEmptyException;
import org.projectnessie.error.NessieNamespaceNotFoundException;
import org.projectnessie.error.NessieNotFoundException;
import org.projectnessie.error.NessieReferenceNotFoundException;
import org.projectnessie.model.Branch;
import org.projectnessie.model.CommitMeta;
import org.projectnessie.model.Content;
import org.projectnessie.model.ContentKey;
import org.projectnessie.model.EntriesResponse;
import org.projectnessie.model.GetNamespacesResponse;
import org.projectnessie.model.IcebergTable;
import org.projectnessie.model.ImmutableCommitMeta;
import org.projectnessie.model.ImmutableIcebergTable;
import org.projectnessie.model.Operation;
import org.projectnessie.model.Reference;
import org.projectnessie.model.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NessieIcebergClient
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(NessieIcebergClient.class);
    private final NessieApiV1 api;
    private final Supplier<UpdateableReference> reference;
    private final Map<String, String> catalogOptions;

    public NessieIcebergClient(NessieApiV1 api, String requestedRef, String requestedHash, Map<String, String> catalogOptions) {
        this.api = api;
        this.catalogOptions = catalogOptions;
        this.reference = Suppliers.memoize(() -> this.loadReference(requestedRef, requestedHash));
    }

    public NessieApiV1 getApi() {
        return this.api;
    }

    UpdateableReference getRef() {
        return this.reference.get();
    }

    public Reference getReference() {
        return this.reference.get().getReference();
    }

    public void refresh() throws NessieNotFoundException {
        this.getRef().refresh(this.api);
    }

    public NessieIcebergClient withReference(String requestedRef, String hash) {
        if (null == requestedRef || this.getRef().getReference().getName().equals(requestedRef) && this.getRef().getHash().equals(hash)) {
            return this;
        }
        return new NessieIcebergClient(this.getApi(), requestedRef, hash, this.catalogOptions);
    }

    private UpdateableReference loadReference(String requestedRef, String hash) {
        try {
            Object ref;
            Object object = ref = requestedRef == null ? this.api.getDefaultBranch() : this.api.getReference().refName(requestedRef).get();
            if (hash != null) {
                ref = ref instanceof Branch ? Branch.of((String)ref.getName(), (String)hash) : Tag.of((String)ref.getName(), (String)hash);
            }
            return new UpdateableReference((Reference)ref, hash != null);
        }
        catch (NessieNotFoundException ex) {
            if (requestedRef != null) {
                throw new IllegalArgumentException(String.format("Nessie ref '%s' does not exist", requestedRef), ex);
            }
            throw new IllegalArgumentException(String.format("Nessie does not have an existing default branch. Either configure an alternative ref via '%s' or create the default branch on the server.", "nessie.ref"), ex);
        }
    }

    public List<TableIdentifier> listTables(Namespace namespace) {
        try {
            return this.withReference(this.api.getEntries()).get().getEntries().stream().filter(this.namespacePredicate(namespace)).filter(e -> Content.Type.ICEBERG_TABLE == e.getType()).map(this::toIdentifier).collect(Collectors.toList());
        }
        catch (NessieNotFoundException ex) {
            throw new NoSuchNamespaceException((Throwable)ex, "Unable to list tables due to missing ref '%s'", new Object[]{this.getRef().getName()});
        }
    }

    private Predicate<EntriesResponse.Entry> namespacePredicate(Namespace ns) {
        if (ns == null) {
            return e -> true;
        }
        List<String> namespace = Arrays.asList(ns.levels());
        return e -> {
            List names = e.getName().getElements();
            if (names.size() <= namespace.size()) {
                return false;
            }
            return namespace.equals(names.subList(0, namespace.size()));
        };
    }

    private TableIdentifier toIdentifier(EntriesResponse.Entry entry) {
        List elements = entry.getName().getElements();
        return TableIdentifier.of((String[])elements.toArray(new String[elements.size()]));
    }

    public IcebergTable table(TableIdentifier tableIdentifier) {
        try {
            ContentKey key = NessieUtil.toKey(tableIdentifier);
            Content table = (Content)this.withReference(this.api.getContent().key(key)).get().get(key);
            return table != null ? (IcebergTable)table.unwrap(IcebergTable.class).orElse(null) : null;
        }
        catch (NessieNotFoundException e) {
            return null;
        }
    }

    public void createNamespace(Namespace namespace, Map<String, String> metadata) {
        try {
            this.getRef().checkMutable();
            this.withReference(((CreateNamespaceBuilder)this.getApi().createNamespace().namespace(org.projectnessie.model.Namespace.of((String[])namespace.levels()))).properties(metadata)).create();
            this.refresh();
        }
        catch (NessieNamespaceAlreadyExistsException e) {
            throw new AlreadyExistsException((Throwable)e, "Namespace already exists: %s", new Object[]{namespace});
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot create Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public List<Namespace> listNamespaces(Namespace namespace) throws NoSuchNamespaceException {
        try {
            GetNamespacesResponse response = this.withReference((GetMultipleNamespacesBuilder)this.getApi().getMultipleNamespaces().namespace(org.projectnessie.model.Namespace.of((String[])namespace.levels()))).get();
            return response.getNamespaces().stream().map(ns -> Namespace.of((String[])ns.getElements().toArray(new String[0]))).filter(ns -> ns.length() == namespace.length() + 1).collect(Collectors.toList());
        }
        catch (NessieReferenceNotFoundException e) {
            throw new RuntimeException(String.format("Cannot list Namespaces starting from '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public boolean dropNamespace(Namespace namespace) throws NamespaceNotEmptyException {
        try {
            this.getRef().checkMutable();
            this.withReference((DeleteNamespaceBuilder)this.getApi().deleteNamespace().namespace(org.projectnessie.model.Namespace.of((String[])namespace.levels()))).delete();
            this.refresh();
            return true;
        }
        catch (NessieNamespaceNotFoundException e) {
            return false;
        }
        catch (NessieNotFoundException e) {
            LOG.error("Cannot drop Namespace '{}': ref '{}' is no longer valid.", new Object[]{namespace, this.getRef().getName(), e});
            return false;
        }
        catch (NessieNamespaceNotEmptyException e) {
            throw new NamespaceNotEmptyException((Throwable)e, "Namespace '%s' is not empty. One or more tables exist.", new Object[]{namespace});
        }
    }

    public Map<String, String> loadNamespaceMetadata(Namespace namespace) throws NoSuchNamespaceException {
        try {
            return this.withReference((GetNamespaceBuilder)this.getApi().getNamespace().namespace(org.projectnessie.model.Namespace.of((String[])namespace.levels()))).get().getProperties();
        }
        catch (NessieNamespaceNotFoundException e) {
            throw new NoSuchNamespaceException((Throwable)e, "Namespace does not exist: %s", new Object[]{namespace});
        }
        catch (NessieReferenceNotFoundException e) {
            throw new RuntimeException(String.format("Cannot load Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public boolean setProperties(Namespace namespace, Map<String, String> properties) {
        try {
            this.withReference(((UpdateNamespaceBuilder)this.getApi().updateProperties().namespace(org.projectnessie.model.Namespace.of((String[])namespace.levels()))).updateProperties(properties)).update();
            this.refresh();
            return true;
        }
        catch (NessieNamespaceNotFoundException e) {
            throw new NoSuchNamespaceException((Throwable)e, "Namespace does not exist: %s", new Object[]{namespace});
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot update properties on Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public boolean removeProperties(Namespace namespace, Set<String> properties) {
        try {
            this.withReference(((UpdateNamespaceBuilder)this.getApi().updateProperties().namespace(org.projectnessie.model.Namespace.of((String[])namespace.levels()))).removeProperties(properties)).update();
            this.refresh();
            return true;
        }
        catch (NessieNamespaceNotFoundException e) {
            throw new NoSuchNamespaceException((Throwable)e, "Namespace does not exist: %s", new Object[]{namespace});
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot remove properties from Namespace '%s': ref '%s' is no longer valid.", namespace, this.getRef().getName()), e);
        }
    }

    public void renameTable(TableIdentifier from, TableIdentifier to) {
        this.getRef().checkMutable();
        IcebergTable existingFromTable = this.table(from);
        if (existingFromTable == null) {
            throw new NoSuchTableException("Table does not exist: %s", new Object[]{from.name()});
        }
        IcebergTable existingToTable = this.table(to);
        if (existingToTable != null) {
            throw new AlreadyExistsException("Table already exists: %s", new Object[]{to.name()});
        }
        CommitMultipleOperationsBuilder operations = this.getApi().commitMultipleOperations().commitMeta(NessieUtil.buildCommitMetadata(String.format("Iceberg rename table from '%s' to '%s'", from, to), this.catalogOptions)).operation((Operation)Operation.Delete.of((ContentKey)NessieUtil.toKey(from))).operation((Operation)Operation.Put.of((ContentKey)NessieUtil.toKey(to), (Content)existingFromTable));
        try {
            Tasks.foreach((Object[])new CommitMultipleOperationsBuilder[]{operations}).retry(5).stopRetryOn(new Class[]{NessieNotFoundException.class}).throwFailureWhenFinished().onFailure((o, exception) -> this.refresh()).run(ops -> {
                Branch branch = ((CommitMultipleOperationsBuilder)ops.branch((Branch)this.getRef().getReference())).commit();
                this.getRef().updateReference((Reference)branch);
            }, BaseNessieClientServerException.class);
        }
        catch (NessieNotFoundException e) {
            throw new RuntimeException(String.format("Cannot rename table '%s' to '%s': ref '%s' no longer exists.", from.name(), to.name(), this.getRef().getName()), e);
        }
        catch (BaseNessieClientServerException e) {
            throw new CommitFailedException((Throwable)e, "Cannot rename table '%s' to '%s': the current reference is not up to date.", new Object[]{from.name(), to.name()});
        }
        catch (HttpClientException ex) {
            throw new CommitStateUnknownException((Throwable)ex);
        }
    }

    public boolean dropTable(TableIdentifier identifier, boolean purge) {
        this.getRef().checkMutable();
        IcebergTable existingTable = this.table(identifier);
        if (existingTable == null) {
            return false;
        }
        if (purge) {
            LOG.info("Purging data for table {} was set to true but is ignored", (Object)identifier.toString());
        }
        CommitMultipleOperationsBuilder commitBuilderBase = this.getApi().commitMultipleOperations().commitMeta(NessieUtil.buildCommitMetadata(String.format("Iceberg delete table %s", identifier), this.catalogOptions)).operation((Operation)Operation.Delete.of((ContentKey)NessieUtil.toKey(identifier)));
        boolean threw = true;
        try {
            Tasks.foreach((Object[])new CommitMultipleOperationsBuilder[]{commitBuilderBase}).retry(5).stopRetryOn(new Class[]{NessieNotFoundException.class}).throwFailureWhenFinished().onFailure((o, exception) -> this.refresh()).run(commitBuilder -> {
                Branch branch = ((CommitMultipleOperationsBuilder)commitBuilder.branch((Branch)this.getRef().getReference())).commit();
                this.getRef().updateReference((Reference)branch);
            }, BaseNessieClientServerException.class);
            threw = false;
        }
        catch (NessieConflictException e) {
            LOG.error("Cannot drop table: failed after retry (update ref '{}' and retry)", (Object)this.getRef().getName(), (Object)e);
        }
        catch (NessieNotFoundException e) {
            LOG.error("Cannot drop table: ref '{}' is no longer valid.", (Object)this.getRef().getName(), (Object)e);
        }
        catch (BaseNessieClientServerException e) {
            LOG.error("Cannot drop table: unknown error", (Throwable)e);
        }
        return !threw;
    }

    public void commitTable(TableMetadata base, TableMetadata metadata, String newMetadataLocation, IcebergTable expectedContent, ContentKey key) throws NessieConflictException, NessieNotFoundException {
        Snapshot snapshot;
        String metadataCommitId;
        Branch current;
        UpdateableReference updateableReference = this.getRef();
        updateableReference.checkMutable();
        Branch expectedHead = current = (Branch)updateableReference.getReference();
        if (base != null && (metadataCommitId = base.property("nessie.commit.id", expectedHead.getHash())) != null) {
            expectedHead = Branch.of((String)expectedHead.getName(), (String)metadataCommitId);
        }
        ImmutableIcebergTable.Builder newTableBuilder = ImmutableIcebergTable.builder();
        if (expectedContent != null) {
            newTableBuilder.id(expectedContent.getId());
        }
        long snapshotId = (snapshot = metadata.currentSnapshot()) != null ? snapshot.snapshotId() : -1L;
        ImmutableIcebergTable newTable = newTableBuilder.snapshotId(snapshotId).schemaId(metadata.currentSchemaId()).specId(metadata.defaultSpecId()).sortOrderId(metadata.defaultSortOrderId()).metadataLocation(newMetadataLocation).build();
        LOG.debug("Committing '{}' against '{}', current is '{}': {}", new Object[]{key, expectedHead, current.getHash(), newTable});
        ImmutableCommitMeta.Builder builder = ImmutableCommitMeta.builder();
        builder.message(this.buildCommitMsg(base, metadata, key.toString()));
        if (this.isSnapshotOperation(base, metadata)) {
            builder.putProperties("iceberg.operation", snapshot.operation());
        }
        Branch branch = ((CommitMultipleOperationsBuilder)this.getApi().commitMultipleOperations().operation((Operation)Operation.Put.of((ContentKey)key, (Content)newTable, (Content)expectedContent)).commitMeta((CommitMeta)NessieUtil.catalogOptions(builder, this.catalogOptions).build()).branch(expectedHead)).commit();
        LOG.info("Committed '{}' against '{}', expected commit-id was '{}'", new Object[]{key, branch, expectedHead.getHash()});
        updateableReference.updateReference((Reference)branch);
    }

    private boolean isSnapshotOperation(TableMetadata base, TableMetadata metadata) {
        Snapshot snapshot = metadata.currentSnapshot();
        return snapshot != null && (base == null || base.currentSnapshot() == null || snapshot.snapshotId() != base.currentSnapshot().snapshotId());
    }

    private <T extends OnReferenceBuilder<?>> T withReference(T builder) {
        UpdateableReference ref = this.getRef();
        if (!ref.isMutable()) {
            builder.reference(ref.getReference());
        } else {
            builder.refName(ref.getName());
        }
        return builder;
    }

    private String buildCommitMsg(TableMetadata base, TableMetadata metadata, String tableName) {
        if (this.isSnapshotOperation(base, metadata)) {
            return String.format("Iceberg %s against %s", metadata.currentSnapshot().operation(), tableName);
        }
        if (base != null && metadata.currentSchemaId() != base.currentSchemaId()) {
            return String.format("Iceberg schema change against %s", tableName);
        }
        if (base == null) {
            return String.format("Iceberg table created/registered with name %s", tableName);
        }
        return String.format("Iceberg commit against %s", tableName);
    }

    public String refName() {
        return this.getRef().getName();
    }

    @Override
    public void close() {
        if (null != this.api) {
            this.api.close();
        }
    }
}

