/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.parquet;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Deque;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.apache.iceberg.expressions.PathUtil;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.shaded.org.apache.parquet.io.api.Binary;
import org.apache.iceberg.shaded.org.apache.parquet.schema.GroupType;
import org.apache.iceberg.shaded.org.apache.parquet.schema.LogicalTypeAnnotation;
import org.apache.iceberg.shaded.org.apache.parquet.schema.PrimitiveType;
import org.apache.iceberg.shaded.org.apache.parquet.schema.Type;
import org.apache.iceberg.shaded.org.apache.parquet.schema.Types;
import org.apache.iceberg.types.Comparators;
import org.apache.iceberg.util.BinaryUtil;
import org.apache.iceberg.util.UUIDUtil;
import org.apache.iceberg.util.UnicodeUtil;
import org.apache.iceberg.variants.PhysicalType;
import org.apache.iceberg.variants.VariantArray;
import org.apache.iceberg.variants.VariantMetadata;
import org.apache.iceberg.variants.VariantObject;
import org.apache.iceberg.variants.VariantPrimitive;
import org.apache.iceberg.variants.VariantValue;
import org.apache.iceberg.variants.VariantVisitor;
import org.apache.iceberg.variants.Variants;

class ParquetVariantUtil {
    private ParquetVariantUtil() {
    }

    static PhysicalType convert(PrimitiveType primitive) {
        LogicalTypeAnnotation annotation = primitive.getLogicalTypeAnnotation();
        if (annotation != null) {
            return annotation.accept(PhysicalTypeConverter.INSTANCE).orElse(null);
        }
        switch (primitive.getPrimitiveTypeName()) {
            case BOOLEAN: {
                return PhysicalType.BOOLEAN_TRUE;
            }
            case INT32: {
                return PhysicalType.INT32;
            }
            case INT64: {
                return PhysicalType.INT64;
            }
            case FLOAT: {
                return PhysicalType.FLOAT;
            }
            case DOUBLE: {
                return PhysicalType.DOUBLE;
            }
            case BINARY: {
                return PhysicalType.BINARY;
            }
        }
        return null;
    }

    static PhysicalType convert(LogicalTypeAnnotation.TimestampLogicalTypeAnnotation timestamp) {
        switch (timestamp.getUnit()) {
            case MICROS: {
                return timestamp.isAdjustedToUTC() ? PhysicalType.TIMESTAMPTZ : PhysicalType.TIMESTAMPNTZ;
            }
            case NANOS: {
                return timestamp.isAdjustedToUTC() ? PhysicalType.TIMESTAMPTZ_NANOS : PhysicalType.TIMESTAMPNTZ_NANOS;
            }
        }
        throw new UnsupportedOperationException("Invalid unit for shredded timestamp: " + String.valueOf((Object)timestamp.getUnit()));
    }

    static ByteBuffer toByteBuffer(VariantMetadata metadata, VariantValue value) {
        ByteBuffer buffer = ByteBuffer.allocate(metadata.sizeInBytes() + value.sizeInBytes()).order(ByteOrder.LITTLE_ENDIAN);
        metadata.writeTo(buffer, 0);
        value.writeTo(buffer, metadata.sizeInBytes());
        return buffer;
    }

    static <T> T convertValue(PhysicalType primitive, int scale, Object value) {
        switch (primitive) {
            case BOOLEAN_FALSE: 
            case BOOLEAN_TRUE: 
            case INT32: 
            case INT64: 
            case DATE: 
            case TIMESTAMPTZ: 
            case TIMESTAMPNTZ: 
            case TIMESTAMPTZ_NANOS: 
            case TIMESTAMPNTZ_NANOS: 
            case TIME: 
            case FLOAT: 
            case DOUBLE: {
                return (T)value;
            }
            case INT8: {
                return (T)Byte.valueOf(((Number)value).byteValue());
            }
            case INT16: {
                return (T)Short.valueOf(((Number)value).shortValue());
            }
            case DECIMAL4: 
            case DECIMAL8: {
                return (T)BigDecimal.valueOf(((Number)value).longValue(), scale);
            }
            case DECIMAL16: {
                return (T)new BigDecimal(new BigInteger(((Binary)value).getBytes()), scale);
            }
            case BINARY: {
                return (T)((Binary)value).toByteBuffer();
            }
            case STRING: {
                return (T)((Binary)value).toStringUsingUTF8();
            }
            case UUID: {
                return (T)UUIDUtil.convert(((Binary)value).getBytes());
            }
        }
        throw new IllegalStateException("Invalid bound type: " + String.valueOf((Object)primitive));
    }

    static <T> Comparator<T> comparator(PhysicalType primitive) {
        if (primitive == PhysicalType.BINARY) {
            return Comparators.unsignedBytes();
        }
        return Comparator.naturalOrder();
    }

    static int scale(PrimitiveType primitive) {
        LogicalTypeAnnotation annotation = primitive.getLogicalTypeAnnotation();
        if (annotation instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
            return ((LogicalTypeAnnotation.DecimalLogicalTypeAnnotation)annotation).getScale();
        }
        return 0;
    }

    static Type toParquetSchema(VariantValue value) {
        return VariantVisitor.visit(value, new ParquetSchemaProducer());
    }

    private static class PhysicalTypeConverter
    implements LogicalTypeAnnotation.LogicalTypeAnnotationVisitor<PhysicalType> {
        private static final PhysicalTypeConverter INSTANCE = new PhysicalTypeConverter();

        private PhysicalTypeConverter() {
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.StringLogicalTypeAnnotation ignored) {
            return Optional.of(PhysicalType.STRING);
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.DecimalLogicalTypeAnnotation decimal) {
            if (decimal.getPrecision() <= 9) {
                return Optional.of(PhysicalType.DECIMAL4);
            }
            if (decimal.getPrecision() <= 18) {
                return Optional.of(PhysicalType.DECIMAL8);
            }
            return Optional.of(PhysicalType.DECIMAL16);
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.DateLogicalTypeAnnotation ignored) {
            return Optional.of(PhysicalType.DATE);
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.TimeLogicalTypeAnnotation ignored) {
            return Optional.of(PhysicalType.TIME);
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.TimestampLogicalTypeAnnotation timestamps) {
            switch (timestamps.getUnit()) {
                case MICROS: {
                    if (timestamps.isAdjustedToUTC()) {
                        return Optional.of(PhysicalType.TIMESTAMPTZ);
                    }
                    return Optional.of(PhysicalType.TIMESTAMPNTZ);
                }
                case NANOS: {
                    if (timestamps.isAdjustedToUTC()) {
                        return Optional.of(PhysicalType.TIMESTAMPTZ_NANOS);
                    }
                    return Optional.of(PhysicalType.TIMESTAMPNTZ_NANOS);
                }
            }
            return Optional.empty();
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.IntLogicalTypeAnnotation ints) {
            if (!ints.isSigned()) {
                return Optional.empty();
            }
            switch (ints.getBitWidth()) {
                case 8: {
                    return Optional.of(PhysicalType.INT8);
                }
                case 16: {
                    return Optional.of(PhysicalType.INT16);
                }
                case 32: {
                    return Optional.of(PhysicalType.INT32);
                }
                case 64: {
                    return Optional.of(PhysicalType.INT64);
                }
            }
            return Optional.empty();
        }

        @Override
        public Optional<PhysicalType> visit(LogicalTypeAnnotation.UUIDLogicalTypeAnnotation uuidLogicalType) {
            return Optional.of(PhysicalType.UUID);
        }
    }

    private static class ParquetSchemaProducer
    extends VariantVisitor<Type> {
        private ParquetSchemaProducer() {
        }

        @Override
        public Type object(VariantObject object, List<String> names, List<Type> typedValues) {
            if (object.numFields() < 1) {
                return null;
            }
            ArrayList<GroupType> fields = Lists.newArrayList();
            int index = 0;
            for (String name : names) {
                Type typedValue = typedValues.get(index);
                fields.add(ParquetSchemaProducer.field(name, typedValue));
                ++index;
            }
            return ParquetSchemaProducer.objectFields(fields);
        }

        @Override
        public Type array(VariantArray array, List<Type> elementResults) {
            if (elementResults.isEmpty()) {
                return null;
            }
            Type shredType = elementResults.get(0);
            if (shredType != null && elementResults.stream().allMatch(type -> Objects.equals(type, shredType))) {
                return ParquetSchemaProducer.list(shredType);
            }
            return null;
        }

        private static GroupType list(Type shreddedType) {
            GroupType elementType = ParquetSchemaProducer.field("element", shreddedType);
            ParquetSchemaProducer.checkField(elementType);
            return (GroupType)Types.optionalList().element(elementType).named("typed_value");
        }

        @Override
        public Type primitive(VariantPrimitive<?> primitive) {
            switch (primitive.type()) {
                case NULL: {
                    return null;
                }
                case BOOLEAN_FALSE: 
                case BOOLEAN_TRUE: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.BOOLEAN);
                }
                case INT8: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT32, LogicalTypeAnnotation.intType(8));
                }
                case INT16: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT32, LogicalTypeAnnotation.intType(16));
                }
                case INT32: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT32);
                }
                case INT64: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64);
                }
                case FLOAT: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.FLOAT);
                }
                case DOUBLE: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.DOUBLE);
                }
                case DECIMAL4: {
                    BigDecimal decimal4 = (BigDecimal)primitive.get();
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT32, LogicalTypeAnnotation.decimalType(decimal4.scale(), 9));
                }
                case DECIMAL8: {
                    BigDecimal decimal8 = (BigDecimal)primitive.get();
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64, LogicalTypeAnnotation.decimalType(decimal8.scale(), 18));
                }
                case DECIMAL16: {
                    BigDecimal decimal16 = (BigDecimal)primitive.get();
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.BINARY, LogicalTypeAnnotation.decimalType(decimal16.scale(), 38));
                }
                case DATE: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT32, LogicalTypeAnnotation.dateType());
                }
                case TIMESTAMPTZ: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64, LogicalTypeAnnotation.timestampType(true, LogicalTypeAnnotation.TimeUnit.MICROS));
                }
                case TIMESTAMPNTZ: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64, LogicalTypeAnnotation.timestampType(false, LogicalTypeAnnotation.TimeUnit.MICROS));
                }
                case BINARY: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.BINARY);
                }
                case STRING: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.BINARY, LogicalTypeAnnotation.stringType());
                }
                case TIME: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64, LogicalTypeAnnotation.timeType(false, LogicalTypeAnnotation.TimeUnit.MICROS));
                }
                case TIMESTAMPTZ_NANOS: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64, LogicalTypeAnnotation.timestampType(true, LogicalTypeAnnotation.TimeUnit.NANOS));
                }
                case TIMESTAMPNTZ_NANOS: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.INT64, LogicalTypeAnnotation.timestampType(false, LogicalTypeAnnotation.TimeUnit.NANOS));
                }
                case UUID: {
                    return ParquetSchemaProducer.shreddedPrimitive(PrimitiveType.PrimitiveTypeName.FIXED_LEN_BYTE_ARRAY, LogicalTypeAnnotation.uuidType(), 16);
                }
            }
            throw new UnsupportedOperationException("Unsupported shredding type: " + String.valueOf((Object)primitive.type()));
        }

        private static GroupType objectFields(List<GroupType> fields) {
            Types.GroupBuilder<GroupType> builder = Types.buildGroup(Type.Repetition.OPTIONAL);
            for (GroupType field : fields) {
                ParquetSchemaProducer.checkField(field);
                builder.addField(field);
            }
            return (GroupType)builder.named("typed_value");
        }

        private static void checkField(GroupType fieldType) {
            Preconditions.checkArgument(fieldType.isRepetition(Type.Repetition.REQUIRED), "Invalid field type repetition: %s should be REQUIRED", (Object)fieldType.getRepetition());
        }

        private static GroupType field(String name, Type shreddedType) {
            Types.GroupBuilder builder = (Types.GroupBuilder)Types.buildGroup(Type.Repetition.REQUIRED).optional(PrimitiveType.PrimitiveTypeName.BINARY).named("value");
            if (shreddedType != null) {
                ParquetSchemaProducer.checkShreddedType(shreddedType);
                builder.addField(shreddedType);
            }
            return (GroupType)builder.named(name);
        }

        private static void checkShreddedType(Type shreddedType) {
            Preconditions.checkArgument(shreddedType.getName().equals("typed_value"), "Invalid shredded type name: %s should be typed_value", (Object)shreddedType.getName());
            Preconditions.checkArgument(shreddedType.isRepetition(Type.Repetition.OPTIONAL), "Invalid shredded type repetition: %s should be OPTIONAL", (Object)shreddedType.getRepetition());
        }

        private static Type shreddedPrimitive(PrimitiveType.PrimitiveTypeName primitive) {
            return (Type)Types.optional(primitive).named("typed_value");
        }

        private static Type shreddedPrimitive(PrimitiveType.PrimitiveTypeName primitive, LogicalTypeAnnotation annotation) {
            return (Type)((Types.PrimitiveBuilder)Types.optional(primitive).as(annotation)).named("typed_value");
        }

        private static Type shreddedPrimitive(PrimitiveType.PrimitiveTypeName primitive, LogicalTypeAnnotation annotation, int length) {
            return (Type)((Types.PrimitiveBuilder)((Types.PrimitiveBuilder)Types.optional(primitive).as(annotation)).length(length)).named("typed_value");
        }
    }

    static class VariantMetrics {
        private final long valueCount;
        private final long nullCount;
        private final VariantValue lowerBound;
        private final VariantValue upperBound;
        private final Deque<String> fieldNames = Lists.newLinkedList();
        private String lazyFieldName = null;

        VariantMetrics(long valueCount, long nullCount) {
            this.valueCount = valueCount;
            this.nullCount = nullCount;
            this.lowerBound = null;
            this.upperBound = null;
        }

        VariantMetrics(long valueCount, long nullCount, VariantValue lowerBound, VariantValue upperBound) {
            this.valueCount = valueCount;
            this.nullCount = nullCount;
            this.lowerBound = VariantMetrics.truncateLowerBound(lowerBound);
            this.upperBound = VariantMetrics.truncateUpperBound(upperBound);
        }

        VariantMetrics prependFieldName(String name) {
            this.lazyFieldName = null;
            this.fieldNames.addFirst(name);
            return this;
        }

        public String fieldName() {
            if (null == this.lazyFieldName) {
                this.lazyFieldName = PathUtil.toNormalizedPath(this.fieldNames);
            }
            return this.lazyFieldName;
        }

        public long valueCount() {
            return this.valueCount;
        }

        public long nullCount() {
            return this.nullCount;
        }

        public VariantValue lowerBound() {
            return this.lowerBound;
        }

        public VariantValue upperBound() {
            return this.upperBound;
        }

        private static VariantValue truncateLowerBound(VariantValue value) {
            switch (value.type()) {
                case STRING: {
                    return Variants.of(PhysicalType.STRING, UnicodeUtil.truncateStringMin((String)value.asPrimitive().get(), 16));
                }
                case BINARY: {
                    return Variants.of(PhysicalType.BINARY, BinaryUtil.truncateBinaryMin((ByteBuffer)value.asPrimitive().get(), 16));
                }
            }
            return value;
        }

        private static VariantValue truncateUpperBound(VariantValue value) {
            switch (value.type()) {
                case STRING: {
                    String truncatedString = UnicodeUtil.truncateStringMax((String)value.asPrimitive().get(), 16);
                    return truncatedString != null ? Variants.of(PhysicalType.STRING, truncatedString) : null;
                }
                case BINARY: {
                    ByteBuffer truncatedBuffer = BinaryUtil.truncateBinaryMin((ByteBuffer)value.asPrimitive().get(), 16);
                    return truncatedBuffer != null ? Variants.of(PhysicalType.BINARY, truncatedBuffer) : null;
                }
            }
            return value;
        }
    }
}

